/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FlashHeuristicScore.h"
#include "URI.h"

using namespace std;

int
FlashHeuristicScore::getNumericScore() const
{
	return m_score;
}

int
FlashHeuristicScore::getNumericSizeScore() const
{
	return m_sizeScore;
}

HeuristicScore::Status
FlashHeuristicScore::getStatus() const
{
	int score = getNumericScore();
	if (score < 0 /* changed from 20 */) {
		return NOT_AD;
	} else if (score < 60) {
		return PROBABLY_NOT_AD;
	} else if (score < 100) {
		return PROBABLY_AD;
	} else {
		return AD;
	}
	
	/*
	The difference between this and the base class version
	is the NOT_AD threshold lowered from 20 to 0.
	This was done to allow self-hosted, innocent-looking
	objects to be analyzed.
	*/ 
}

void
FlashHeuristicScore::processSize(int width, int height)
{
	if (m_sizeScore) {
		m_score -= m_sizeScore;
		m_sizeScore = 0;
	}
	m_sizeScore = getSizeScore(width, height);
	if (m_sizeScore == 0 && getAspect(width, height) >= 2.5) {
		m_sizeScore = 30;
	}
	m_score += m_sizeScore;
}

void
FlashHeuristicScore::unprocessSize()
{
	m_score -= m_sizeScore;
	m_sizeScore = 0;
}

void
FlashHeuristicScore::processUrl(URI const& abs, URI const& rel, URI const& base)
{
	m_score += getHintModifier(abs);
	UrlRelationship rship = getUrlRelationship(rel, base);
	if (rship == URLS_UNRELATED) {
		m_score += 70;
	} else if (rship == URLS_SAME_HOST) {
		m_score -= 20;
	}
	UrlStatus const status = getUrlStatus(rel);
	if (status == URL_INNOCENT) {
		m_score -= 10;
	} else if (status == URL_SUSPICIOUS) {
		m_score += 30;
	}
}

void
FlashHeuristicScore::processOptions(bool loop, bool menu)
{
	if (loop) {
		m_score += 40;
	}
	if (!menu) {
		m_score += 30;
	}
}

void
FlashHeuristicScore::processScriptLevel(int level, bool external)
{
	if (level > 1) {
		m_score += 70;
		if (external) {
			m_score += 30;
		}
	} else if (level > 0) {
		m_score += 30;
		if (external) {
			m_score += 20;
		}
	}
}

void
FlashHeuristicScore::processScriptRedirect(bool val)
{
	if (val) {
		m_score += 40;
	}
}

void
FlashHeuristicScore::processScriptPersistentCookies(bool val)
{
	if (val) {
		m_score += 40;
	}
}

double
FlashHeuristicScore::getAspect(int width, int height)
{
	if (!(width > 0 && height > 0)) {
		return 0.0;
	}
	if (width > height) {
		return double(width) / double(height);
	} else {
		return double(height) / double(width);
	}
}
