/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpFetcher.h"
#include "AbstractRequestHandler.h"
#include "AbstractResponseHandler.h"
#include "FilterTryList.h"
#include "ErrorDescriptor.h"
#include "DownloadSizeLimiter.h"
#include "Server.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpStatusLine.h"
#include "IntrusivePtr.h"
#include "URI.h"
#include "Debug.h"
#include <memory>
#include <stddef.h>

class DownloadProgress;

using namespace std;

class HttpFetcher::ResponseHandler : public AbstractResponseHandler, private NonCopyable
{
public:
	ResponseHandler(HttpFetcher& owner) : m_rOwner(owner) {}
private:
	virtual void processProvisionalResponse(std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processResponseMetadata(std::auto_ptr<HttpResponseMetadata> metadata);
	
	// postcondition: data is empty
	virtual void processBodyData(SplittableBuffer& data, bool eof, DownloadProgress const&);
	
	virtual void processError(std::auto_ptr<ErrorDescriptor> edesc);
	
	HttpFetcher& m_rOwner;
};


/*======================= HttpFetcher::ResponseHandler ====================*/

void
HttpFetcher::ResponseHandler::processProvisionalResponse(auto_ptr<HttpResponseMetadata> metadata)
{
}

void
HttpFetcher::ResponseHandler::processResponseMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	if (metadata->getBodyStatus() == HttpResponseMetadata::BODY_SIZE_KNOWN
	    && metadata->getBodySize() > m_rOwner.m_maxBodySize) {
		m_rOwner.handleError(
			auto_ptr<ErrorDescriptor>(new ErrorDescriptor(
				ErrorCodes::RESPONSE_BODY_TOO_BIG,
				"response body is too big"
			))
		);
	}
	m_rOwner.m_ptrResponseMetadata = metadata;
}

void
HttpFetcher::ResponseHandler::processBodyData(
	SplittableBuffer& data, bool eof, DownloadProgress const&)
{
	if (m_rOwner.getStatus() == IN_PROGRESS) {
		m_rOwner.handleData(data, eof);
	} else {
		data.clear();
	}
}

void
HttpFetcher::ResponseHandler::processError(auto_ptr<ErrorDescriptor> edesc)
{
	if (m_rOwner.getStatus() == IN_PROGRESS) {
		m_rOwner.handleError(edesc);
	}
}


/*============================= HttpFetcher ===============================*/

HttpFetcher::HttpFetcher(
	Server& server, ConstRequestPtr const& request,
	size_t max_body_size, size_t max_fetch_size)
:	m_rServer(server),
	m_ptrRequest(request),
	m_ptrResponseHandler(new ResponseHandler(*this)),
	m_status(IN_PROGRESS),
	m_maxBodySize(max_body_size),
	m_maxFetchSize(max_fetch_size),
	m_receivedBodySize(0)
{
	auto_ptr<FilterTryList> filters(new FilterTryList);
	if (max_fetch_size != std::numeric_limits<size_t>::max()) {
		filters->append(
			FilterTryList::FilterConstructorPtr(
				new FilterTryList::FilterConstructor(
					sigc::bind(
						sigc::ptr_fun(&HttpFetcher::createSizeLimiterFilter),
						m_maxFetchSize
					)
				)
			)
		);
	}
	m_ptrRequestHandler = m_rServer.submitRequest(
		request, m_ptrResponseHandler, filters
	);
	SplittableBuffer body;
	m_ptrRequestHandler->appendBodyData(body, true);
}

HttpFetcher::~HttpFetcher()
{
	m_rServer.reset();
}

void
HttpFetcher::handleData(SplittableBuffer& data, bool eof)
{
	m_receivedBodySize += data.size();
	m_responseBody.appendDestructive(data);
	if (eof) {
		m_status = COMPLETE;
	} else if (m_receivedBodySize > m_maxBodySize) {
		handleError(
			auto_ptr<ErrorDescriptor>(new ErrorDescriptor(
				ErrorCodes::RESPONSE_BODY_TOO_BIG,
				"response body is too big"
			))
		); 
	}
}

void
HttpFetcher::handleError(auto_ptr<ErrorDescriptor> edesc)
{
	DEBUGLOG(
		"Client::ResponseHandler::processError(): "
		<< edesc->getErrorMessage() << "\r\nURL: "
		<< m_ptrRequest->requestLine().getURI()
	);
	m_status = FAILED;
	m_ptrErrDesc = edesc;
}

void
HttpFetcher::createSizeLimiterFilter(
	ResponseFilterChain& chain,
	HttpResponseMetadata const& metadata, size_t limit)
{
	chain.appendFilter(IntrusivePtr<AbstractResponseFilter>(
		new DownloadSizeLimiter(chain, limit)
	));
}
