/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ResponseFilterChain.h"
#include "AbstractResponseFilter.h"
#include "ResponseFilterBase.h"
#include "FilterTryList.h"
#include "ErrorDescriptor.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestMetadata.h"
#include "JsFilterContext.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include <cassert>

using namespace std;

class ResponseFilterChain::Terminator : public ResponseFilterBase
{
public:
	Terminator(ResponseFilterChain& chain);
	
	virtual ~Terminator();

	virtual void processMetadata(std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processBodyData(SplittableBuffer& data, bool eof);
};


class ResponseFilterChain::RcJsFilterContext :
	public JsFilterContext,
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
};


ResponseFilterChain::ResponseFilterChain(
	ServiceContext& context,
	ConstRequestPtr const& request,
	auto_ptr<FilterTryList> filter_try_list,
	IntrusivePtr<AbstractResponseHandler> const& final_recipient,
	ResponseErrorInitiator& error_initiator)
:	m_rContext(context),
	m_ptrRequest(request),
	m_ptrFilterTryList(filter_try_list),
	m_ptrFinalRecipient(final_recipient),
	m_rErrorInitiator(error_initiator),
	m_ptrTerminator(new Terminator(*this))
{
	m_ptrFilterTryList->append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::ensureTransferEncodingUnderstood)
		)
	));
	m_ptrFilterTryList->append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::ensureContentEncodingUnderstood)
		)
	));
}

ResponseFilterChain::~ResponseFilterChain()
{
}

void
ResponseFilterChain::processProvisionalResponse(auto_ptr<HttpResponseMetadata> metadata)
{
	m_ptrFinalRecipient->processProvisionalResponse(metadata);
}

void
ResponseFilterChain::processResponseMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	if (!m_ptrFirstFilter) {
		m_ptrFirstFilter = nextFilter(*metadata);
	}
	m_ptrFirstFilter->processMetadata(metadata);
}

void
ResponseFilterChain::processBodyData(SplittableBuffer& data, bool eof, DownloadProgress const& progress)
{
	assert(m_ptrFirstFilter.get());
	m_downloadProgress = progress;
	m_ptrFirstFilter->processBodyData(data, eof);
}

void
ResponseFilterChain::processError(auto_ptr<ErrorDescriptor> edesc)
{
	m_ptrFinalRecipient->processError(edesc);
}

void
ResponseFilterChain::appendFilter(IntrusivePtr<AbstractResponseFilter> const& filter)
{
	m_filterQueue.push_back(filter);
}

IntrusivePtr<AbstractResponseFilter>
ResponseFilterChain::nextFilter(HttpResponseMetadata const& metadata)
{
	while (m_filterQueue.empty() && !m_ptrFilterTryList->empty()) {
		FilterTryList::FilterConstructorPtr filter_constructor(
			m_ptrFilterTryList->front()
		);
		m_ptrFilterTryList->shift();
		// The filter constructor must be removed from the FilterTryList
		// before being called, because it may alter this list.
		(*filter_constructor)(*this, metadata);
	}
	if (!m_filterQueue.empty()) {
		IntrusivePtr<AbstractResponseFilter> filter = m_filterQueue.front();
		m_filterQueue.pop_front();
		return filter;
	}
	return m_ptrTerminator;
}

bool
ResponseFilterChain::isFlagSet(
	FilterGroupTag const& group, std::string const& flag) const
{
	return (m_flags.find(FilterFlag(group, flag)) != m_flags.end());
}

void
ResponseFilterChain::setFlag(
	FilterGroupTag const& group, std::string const& flag)
{
	m_flags.insert(FilterFlag(group, flag));
}

void
ResponseFilterChain::clearFlag(
	FilterGroupTag const& group, std::string const& flag)
{
	m_flags.erase(FilterFlag(group, flag));
}


JsFilterContext&
ResponseFilterChain::getJsContextFor(FilterGroupTag const& group)
{
	IntrusivePtr<RcJsFilterContext>& context = m_jsContexts[group];
	if (!context) {
		context.reset(new RcJsFilterContext);
	}
	return *context;
}


/*======================= ResponseFilterChain::Terminator ======================*/

ResponseFilterChain::Terminator::Terminator(ResponseFilterChain& chain)
:	ResponseFilterBase(chain)
{
}

ResponseFilterChain::Terminator::~Terminator()
{
}

void
ResponseFilterChain::Terminator::processMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	getFinalRecipient().processResponseMetadata(metadata);
}

void
ResponseFilterChain::Terminator::processBodyData(SplittableBuffer& data, bool eof)
{
	getFinalRecipient().processBodyData(data, eof, getFilterChain().getDownloadProgress());
}
