/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef SERVER_H_
#define SERVER_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpResponseParser.h"
#include "AsyncConnectionListener.h"
#include "SplittableBuffer.h"
#include "CraftedResponseGenerator.h"
#include "DownloadProgress.h"
#include "ServerTimeouts.h"
#include "HttpVersion.h"
#include "Reactor.h"
#include "EventHandler.h"
#include "NullRefCounter.h"
#include "IntrusivePtr.h"
#include "RequestPtr.h"
#include "WeakPtr.h"
#include "StrongPtr.h"
#include "NonCopyable.h"
#include "Timer.h"
#include <sigc++/sigc++.h>
#include <stddef.h>
#include <string>
#include <deque>
#include <memory>

class ServiceContext;
class ServerConnection;
class HttpRequestMetadata;
class HttpResponseMetadata;
class AbstractRequestHandler;
class AbstractResponseHandler;
class ErrorDescriptor;
class FilterTryList;

class Server :
	private HttpResponseParser,
	private AsyncConnectionListener,
	private EventHandler<NullRefCounter>,
	public sigc::trackable
{
	DECLARE_NON_COPYABLE(Server)
public:
	Server(ServiceContext& context, ServerTimeouts const& timeouts);
	
	virtual ~Server();

	virtual IntrusivePtr<AbstractRequestHandler> submitRequest(
		ConstRequestPtr const& request_metadata,
		IntrusivePtr<AbstractResponseHandler> const& handler);
	
	IntrusivePtr<AbstractRequestHandler> submitRequest(
		ConstRequestPtr const& request_metadata,
		IntrusivePtr<AbstractResponseHandler> const& handler,
		std::auto_ptr<FilterTryList> filter_try_list);
	
	IntrusivePtr<AbstractRequestHandler> submitRequest(
		ConstRequestPtr const& request_metadata,
		IntrusivePtr<AbstractResponseHandler> const& handler,
		std::auto_ptr<ErrorDescriptor> error_response);
	
	IntrusivePtr<AbstractRequestHandler> submitRequest(
		ConstRequestPtr const& request_metadata,
		IntrusivePtr<AbstractResponseHandler> const& handler,
		CraftedResponseGenerator const& response_generator);
	
	void reset();
	
	int getNumPendingRequests() const { return m_requestQueue.size(); }
	
	int getMaxPendingRequests() const { return MAX_PIPELINE_SIZE; }
private:
	enum { MAX_PIPELINE_SIZE = 4 };
	enum { READ_BUF_SIZE = 16*1024 };
	enum { MAX_BUFFERED_REQUEST_BODY = 1024*1024 };
	enum { MAX_WRITE_BUF_SIZE = 24000 };
	
	enum Flags {
		READING_SUSPENDED = 1,
		WRITE_BUFFER_FULL = 2
	};
	
	typedef WeakPtr<ACE_NULL_SYNCH, Server, Server*> ServerWeakPtr;
	
	class Request;
	friend class Server::Request;
	class PhantomRequest;
	friend class Server::PhantomRequest;
	class ScopedSuspender;
	friend class Server::ScopedSuspender;
	
	void updatePipeline();
	
	static void inspectRequest(Request& req);
	
	bool isNewConnectionRequiredFor(Request const& req) const;
	
	void requestConnectionFor(Request const& req);
	
	bool isConnectionReady() const;
	
	bool isOutputReady() const;
	
	size_t getPendingOutputSize() const;
	
	void onNewOutput();
	
	void processOutputData();
	
	size_t writeRequestData(Request& req);
	
	void processInputData(bool eof);
	
	void shiftRequestQueue();
	
	bool connectionMayBeStored() const;
	
	void storeConnection();
	
	void dropConnection();
	
	std::auto_ptr<ServerConnection> retrieveConnection();
	
	void restartPendingRequests();
	
	void answerWithError(std::auto_ptr<ErrorDescriptor> error_response);
	
	bool isRequestAtPipelineStart(Request const* req) const;
	
	void suspend() { m_flags |= READING_SUSPENDED; }
	
	void resume() { m_flags &= ~READING_SUSPENDED; }
	
	void resetFlags() { m_flags &= READING_SUSPENDED; }
	
	void onReadTimeout();
	
	void onWriteTimeout();
	
	void onTimeout();
	
	void updateState();
	
	bool processLeadingPhantomResponse();
	
	bool processLeadingErrorResponse();
	
	void processArtificialResponses();
	
	bool isArtificialResponseRequired() const;
	
	virtual void eventGotProvisionalResponse(
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void eventGotMetadata(
		std::auto_ptr<HttpResponseMetadata> metadata, bool is_persistent);
	
	virtual void connectionEstablished();
	
	virtual void connectionFailed(FailureCode fcode, int errnum);
	
	virtual void handleRead(ACE_HANDLE);
	
	virtual void handleWrite(ACE_HANDLE);
	
	void onDelayedAction();
	
	ServiceContext& m_rContext;
	ServerTimeouts m_timeouts;
	std::deque<IntrusivePtr<Request> > m_requestQueue;
	std::deque<IntrusivePtr<PhantomRequest> > m_phantomRequestQueue;
	size_t m_pipelineSize;
	std::auto_ptr<ServerConnection> m_ptrConnection;
	SplittableBuffer m_inputData;
	DownloadProgress m_downloadProgress;
	HttpVersion m_lastResponseVersion;
	ReactorHandlerId m_handlerId;
	Timer m_readTimer;
	Timer m_writeTimer;
	Timer m_delayedActionTimer;
	Reactor::IOEvents m_eventMask;
	int m_flags;
	int m_isWaitingForWrite;
	StrongPtr<ACE_NULL_SYNCH, Server, Server*> m_ptrSelf;
};

#endif
