/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ServerConnectionPool.h"
#include "ServerConnection.h"
#include "IntrusivePtr.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include "InetAddr.h"
#include <ace/Synch.h>
#include <ace/Singleton.h>
#include <ace/OS_NS_sys_time.h>
#include <ace/Time_Value.h>
#include <boost/multi_index_container.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/composite_key.hpp>
#include <iterator>
#include <utility>
#include <cassert>

using namespace std;
using namespace boost::multi_index;

class ServerConnectionPool::ConnRC :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >,
	public std::auto_ptr<ServerConnection>
{
public:
	ConnRC(std::auto_ptr<ServerConnection> conn)
	: std::auto_ptr<ServerConnection>(conn) {}
};


struct ServerConnectionPool::Entry
{	
	Entry(InetAddr const& addr, ACE_Time_Value const& timeout,
		IntrusivePtr<ConnRC> const& connection)
	:	addr(addr),
		timeout(timeout),
		connection(connection)
	{
	}
	
	InetAddr addr;
	ACE_Time_Value timeout; // absolute time value
	IntrusivePtr<ConnRC> connection;
};


class ServerConnectionPool::Impl : public ServerConnectionPool
{
public:
	Impl();
	
	virtual ~Impl();
	
	virtual void storeConnection(std::auto_ptr<ServerConnection> conn);
	
	virtual std::auto_ptr<ServerConnection> retrieveConnection(InetAddr const& addr);
private:
	class AddrTag {};
	class TimeoutTag {};
	
	typedef ACE_Thread_Mutex Mutex;
	typedef multi_index_container<
		Entry,
		indexed_by<
			ordered_non_unique<
				tag<AddrTag>,
				composite_key<
					Entry,
					member<Entry, InetAddr, &Entry::addr>,
					member<Entry, ACE_Time_Value, &Entry::timeout>
				>
			>,
			ordered_non_unique<
				tag<TimeoutTag>,
				member<Entry, ACE_Time_Value, &Entry::timeout>
			>
		>
	> Container;
	typedef Container::index<AddrTag>::type AddrIdx;
	typedef Container::index<TimeoutTag>::type TimeoutIdx;
	
	void removeTimedOut();
	
	void removeExcess();
	
	bool isConnAlive(ServerConnection& conn);
	
	Mutex m_mutex;
	Container m_container;
};



ServerConnectionPool*
ServerConnectionPool::instance()
{
	return ACE_Singleton<Impl, ACE_Recursive_Thread_Mutex>::instance();
}


/*======================== ServerConnectionPool::Impl =====================*/

ServerConnectionPool::Impl::Impl()
{
}

ServerConnectionPool::Impl::~Impl()
{
}

void
ServerConnectionPool::Impl::storeConnection(auto_ptr<ServerConnection> conn)
{
	assert(conn.get());
	
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, );
	
	removeTimedOut();
	
	ACE_Time_Value timeout(TIMEOUT);
	timeout += ACE_OS::gettimeofday();
	
	IntrusivePtr<ConnRC> conn_ptr(new ConnRC(conn)); // conn is now null
	Entry entry((*conn_ptr)->getAddress(), timeout, conn_ptr);
	m_container.insert(entry);
	
	removeExcess();
}

std::auto_ptr<ServerConnection>
ServerConnectionPool::Impl::retrieveConnection(InetAddr const& addr)
{
	auto_ptr<ServerConnection> res;
	
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, res);
	
	removeTimedOut();
	
	AddrIdx& idx = m_container.get<AddrTag>();
	while (true) {
		AddrIdx::iterator it = idx.lower_bound(boost::make_tuple(addr));
		if (it != idx.end() && it->addr == addr) {
			IntrusivePtr<ConnRC> conn_ptr(it->connection);
			idx.erase(it);
			if (isConnAlive(**conn_ptr)) {
				res = *conn_ptr;
			} else {
				continue;
			}
		}
		break;
	}
	
	return res;
}

void
ServerConnectionPool::Impl::removeTimedOut()
{
	// we are protected by the caller's guard
	
	ACE_Time_Value now(ACE_OS::gettimeofday());
	TimeoutIdx& idx = m_container.get<TimeoutTag>();
	
	TimeoutIdx::iterator it = idx.upper_bound(now);
	idx.erase(idx.begin(), it);
}

void
ServerConnectionPool::Impl::removeExcess()
{
	// we are protected by the caller's guard
	
	int excess = int(m_container.size()) - CAPACITY;
	if (excess > 0) {
		TimeoutIdx& idx = m_container.get<TimeoutTag>();
		TimeoutIdx::iterator end(idx.begin());
		std::advance(end, excess);
		idx.erase(idx.begin(), end);
	}
}

bool
ServerConnectionPool::Impl::isConnAlive(ServerConnection& conn)
{
	int res = ACE::handle_read_ready(
		conn.peer().get_handle(), &ACE_Time_Value::zero
	);
	return (res == -1 && errno == ETIME);
}
