/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef APPLICATION_H_
#define APPLICATION_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef DEBUG
#include "DebugAgent.h"
#endif
#include "ConfigFile.h"
#include "UrlPatternsFile.h"
#include "ContentFilterGroup.h"
#include "IntrusivePtr.h"
#include "AcceptorThread.h"
#include "InterthreadCommandQueue.h"
#include "NetworkActivityHandler.h"
#include "FilterJsLogHandler.h"
#include <sigc++/sigc++.h>
#include <gtkmm/main.h>
#include <glibmm/dispatcher.h>
#include <string>
#include <iosfwd>
#include <memory>
#include <deque>
#include <set>
#include <time.h>

class Config;
class UrlPatterns;
class ContentFilters;
class TrayIcon;
class TrayMenu;
class DebugWindow;
class ACE_FILE_IO;

class Application :
	private InterthreadCommandQueue::Notifier,
	public sigc::trackable
{
private:
	Application(int argc, char* argv[]);
	
	~Application();
	
	void run();
public:
	typedef std::deque<IntrusivePtr<ContentFilterGroup> > ContentFilterList;
	
	static void entryPoint(int argc, char* argv[]);
	
	static Application* instance() { return m_spInstance; }
	
	void showTrayMenu(guint button, guint32 activate_time);
	
	void showAboutDialog();
	
	void showLogDialog();
	
	void showBasicConfigDialog();
	
	void showAdvancedConfigWindow();
	
	void showFilterConfigWindow();
#ifdef DEBUG
	void showDebugWindow();
	
	DebugWindow* debugWindow() { return m_ptrDebugWindow.get(); }
#endif
	void setFilteringEnabled(bool val);
	
	void setScriptFilteringEnabled(bool val);
	
	void requestAppExit();
	
	static Glib::ustring localeToUtf8(std::string const& text);
	
	static std::string localeFromUtf8(Glib::ustring const& text);
	
	static Glib::ustring filenameToUtf8(std::string const& fname);
	
	static bool fileToStream(ACE_FILE_IO& file, std::ostream& strm);
	
	static bool readFile(
		std::string const& abs_fname, std::string& target,
		time_t* mtime = 0, bool log_errors = true);
	
	static bool writeFile(
		std::string const& abs_fname, std::string const& data,
		time_t* mtime = 0, bool log_errors = true);
	
	static bool renameFile(
		std::string const& from, std::string const& to,
		bool log_errors = true);
	
	static bool deleteFile(std::string const& fname, bool log_errors = true);
	
	static bool processConfig(
		std::string const& text, Config& target,
		ConfigFile& file_structure);
	
	bool applyConfig(Config const& config);
	
	bool loadConfig();
	
	static bool processUrlPatterns(
		std::string const& text, std::string const& fname,
		UrlPatterns& target, UrlPatternsFile& file_structure);
	
	static void applyStandardUrlPatterns(UrlPatterns const& patterns);
	
	static void applyLocalUrlPatterns(UrlPatterns const& patterns);
	
	bool loadStandardUrlPatterns();
	
	bool loadLocalUrlPatterns();
	
	bool loadContentFilters();
	
	bool loadFilterFnames(std::string const& dirname, std::deque<std::string>& fnames);
	
	void loadEnabledFilterList(
		ContentFilterGroup& group, std::string const& dirname);
	
	void appendContentFilterGroup(
		IntrusivePtr<ContentFilterGroup> const& group);
	
	bool removeContentFilterGroup(
		IntrusivePtr<ContentFilterGroup> const& group);
	
	void sortContentFiltersByFileName();
	
	void updateGlobalContentFilters();
	
	std::string getConfigFileName() const;
	
	std::string getConfigDefaultFileName() const;
	
	std::string getStandardUrlPatternsFileName() const;
	
	std::string getLocalUrlPatternsFileName() const;
	
	std::string getGlobalFiltersDir() const;
	
	std::string getUserFiltersDir() const;
	
	ConfigFile& configFile() { return m_configFile; }
	
	ConfigFile const& configFile() const { return m_configFile; }
	
	UrlPatternsFile& standardUrlPatternsFile() {
		return m_standardUrlPatternsFile;
	}
	
	UrlPatternsFile const& standardUrlPatternsFile() const {
		return m_standardUrlPatternsFile;
	}
	
	UrlPatternsFile& localUrlPatternsFile() {
		return m_localUrlPatternsFile;
	}
	
	UrlPatternsFile const& localUrlPatternsFile() const {
		return m_localUrlPatternsFile;
	}
	
	ContentFilterList& contentFilters() {
		return m_contentFilters;
	}
	
	ContentFilterList const& contentFilters() const {
		return m_contentFilters;
	}
	
	bool isAcceptingConnections() const {
		return m_isAcceptingConnections;
	}
	
	sigc::signal<void, bool>& acceptingConnectionsSignal() {
		return m_acceptingConnectionsSignal;
	}
	
	sigc::signal<void>& destroySignal() { return m_destroySignal; }
	
	void handleNetworkActivity();
private:
	class TransparentWindow;
	class ConfigErrorHandler;
	struct ContentFilterFnameComparator;
	
	enum { COMMAND_QUEUE_CAPACITY = 32 };
	
	virtual void notifyCommandsPending();
	
	virtual void notifyQueueEmpty();
	
	void processCommands();
	
	bool createMissingUserConfigs();
	
	static bool copyFile(std::string const& source, std::string const& target);
	
	static bool createDir(std::string const& dirname);
	
	void setAcceptingConnections(bool val);
	
	static Application* m_spInstance;
	bool m_isExiting;
	int m_commandNestLevel;
	sigc::signal<void> m_destroySignal;
	Gtk::Main m_gtkMain; // many other things depend on it
	Glib::Dispatcher m_dispatcher;
	InterthreadCommandQueue m_commandQueue;
	NetworkActivityHandler m_networkActivityHandler;
	FilterJsLogHandler m_filterJsLogHandler;
	std::string m_userConfDir;
	std::string m_globalConfDir;
	ConfigFile m_configFile;
	UrlPatternsFile m_standardUrlPatternsFile;
	UrlPatternsFile m_localUrlPatternsFile;
	ContentFilterList m_contentFilters;
	sigc::signal<void, bool> m_acceptingConnectionsSignal;
	std::auto_ptr<TrayIcon> m_ptrTrayIcon;
	std::auto_ptr<TrayMenu> m_ptrTrayMenu;
#ifdef DEBUG
	DebugAgent m_debugAgent;
	std::auto_ptr<DebugWindow> m_ptrDebugWindow;
#endif
	bool m_isAcceptingConnections;
	AcceptorThread m_acceptorThread;
	// Must be the last thing to construct and first to destroy
	// because we don't want worker threads to act on destroyed objects.
};

#endif
