/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FilterConfigWindow.h"
#include "Application.h"
#include "OperationLog.h"
#include "AutoIndentingTextView.h"
#include "AutoScrollingWindow.h"
#include "CompiledImages.h"
#include "ContentFilterGroup.h"
#include "ContentFiltersFile.h"
#include "RegexFilterDescriptor.h"
#include "FilterTag.h"
#include "FilterGroupTag.h"
#include "TextPattern.h"
#include "IntrusivePtr.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include "ScopedIncDec.h"
#include "StringUtils.h"
#include "CompiledImages.h"
#include <ace/OS_NS_unistd.h>
#include <boost/lexical_cast.hpp>
#include <boost/regex.hpp>
#include <gtkmm/textview.h>
#include <gtkmm/treeview.h>
#include <gtkmm/treestore.h>
#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/table.h>
#include <gtkmm/label.h>
#include <gtkmm/entry.h>
#include <gtkmm/button.h>
#include <gtkmm/radiobutton.h>
#include <gtkmm/radiobuttongroup.h>
#include <gtkmm/alignment.h>
#include <gtkmm/frame.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/menu.h>
#include <gtkmm/image.h>
#include <gtkmm/stock.h>
#include <gtkmm/messagedialog.h>
#include <gdkmm/pixbuf.h>
#include <glibmm/ustring.h>
#include <glibmm/convert.h>
#include <pangomm/fontdescription.h>
#include <gdk/gdk.h> // for GdkEvent, key codes
#include <cassert>
#include <algorithm>
#include <iterator>
#include <sstream>
#include <memory>
#include <vector>

using namespace std;

// EditState represents a saved edit form.
struct FilterConfigWindow::EditState :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
	Operation operation; // CREATING or EDITING
	bool isModified; // always true for CREATING
	Glib::ustring order;
	Glib::ustring matchCountLimit;
	Glib::ustring url;
	Glib::ustring contentType;
	Glib::ustring search;
	Glib::ustring replacement;
	RegexFilterDescriptor::ReplacementType replacementType;
	Glib::ustring ifFlag;
	Glib::ustring setFlag;
	Glib::ustring clearFlag;
	
	// used when creating a filter
	EditState();
	
	// used when editing a filter
	EditState(RegexFilterDescriptor const& filter);
	
	~EditState();
};


class FilterConfigWindow::MyCellRendererToggle :
	public Gtk::CellRendererToggle
{
protected:
	virtual bool activate_vfunc(
		GdkEvent* event, Gtk::Widget& widget, Glib::ustring const& path,
		Gdk::Rectangle const& background_area,
		Gdk::Rectangle const& cell_area, Gtk::CellRendererState flags);
};


class FilterConfigWindow::NoItemContextMenu : public Gtk::Menu
{
public:
	NoItemContextMenu(FilterTree& tree);
	
	virtual ~NoItemContextMenu();
};


class FilterConfigWindow::FileContextMenu : public Gtk::Menu
{
public:
	FileContextMenu(FilterTree& tree);
	
	virtual ~FileContextMenu();
};


class FilterConfigWindow::FilterContextMenu : public Gtk::Menu
{
public:
	FilterContextMenu(FilterTree& tree);
	
	virtual ~FilterContextMenu();
	
	void enableMoveUpItem(bool enabled);
	
	void enableMoveDownItem(bool enabled);
private:
	Gtk::MenuItem* m_pMoveUpItem;
	Gtk::MenuItem* m_pMoveDownItem;
};


struct FilterConfigWindow::FileNode :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
	IntrusivePtr<ContentFilterGroup> filterGroup; // always set
	
	FileNode(IntrusivePtr<ContentFilterGroup> const& group);
	
	virtual ~FileNode();
};


struct FilterConfigWindow::FilterNode :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
	IntrusivePtr<EditState> editState; // always set
	IntrusivePtr<RegexFilterDescriptor> filter;
	// the filter we are editing, or null if creating a filter
	
	// used when creating a new filter
	FilterNode();
	
	// used when editing an existing filter
	FilterNode(IntrusivePtr<RegexFilterDescriptor> const& filter);
	
	virtual ~FilterNode();
};


struct FilterConfigWindow::FilterTreeColumns :
	public Gtk::TreeModelColumnRecord
{
	Gtk::TreeModelColumn<CheckState> ENABLED;
	Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > ICON;
	Gtk::TreeModelColumn<Glib::ustring> NAME;
	Gtk::TreeModelColumn<bool> MODIFIED;
	Gtk::TreeModelColumn<IntrusivePtr<FileNode> > FILE_NODE;
	Gtk::TreeModelColumn<IntrusivePtr<FilterNode> > FILTER_NODE;
	/*
	If both FILE_NODE and FILTER_NODE are null, it means
	the file or filter is in the process of being created
	(we are editing the item's label).
	*/
	
	FilterTreeColumns() {
		add(ENABLED); add(ICON); add(NAME);
		add(MODIFIED); add(FILE_NODE); add(FILTER_NODE);
	}
};


class FilterConfigWindow::FilterTreeModel : public Gtk::TreeStore
{
public:
	enum EnablePolicy { NORMAL, ALL_ENABLED, ALL_DISABLED };
	
	static Glib::RefPtr<FilterTreeModel> create();
	
	FilterTreeColumns const& getCols() const { return m_columns; }
	
	sigc::slot<void, Glib::ustring const&> getCheckBoxHandler();
	
	bool isFileItem(Row const& row) const;
	
	bool isFilterItem(Row const& row) const;
	
	iterator erase(iterator const& iter);
	
	sigc::signal<void, Gtk::TreeIter const&>& signalItemBeingDeleted() {
		return m_itemBeingDeletedSignal;
	}
	
	void updateFileCheckState(Row const& file_row);
	
	void updateFileModifiedState(Row const& file_row);
	
	void markFilterAsModified(Row const& row);

	void markFilterAsUnmodified(Row const& row);
	
	bool haveModifiedItems() const;
	
	IntrusivePtr<RegexFilterDescriptor> findExistingFollower(iterator const& filter_item);
	
	void sortFilesByName();
	
	static bool saveEnabledFilterList(
		ContentFilterGroup const& group, EnablePolicy policy = NORMAL);
private:
	FilterTreeModel();
	
	virtual ~FilterTreeModel();
	
	void loadFilterGroups();
	
	CheckState loadIndividualFilters(
		ContentFilters const& filters, Row& parent_row);
	
	void onCheckboxToggled(Glib::ustring const& path);
	
	void onFileCheckStateChanging(
		Row const& row, CheckState old_state, CheckState new_state);
	
	void onFilterCheckStateChanging(
		Row const& row, CheckState old_state, CheckState new_state);
	
	FilterTreeColumns m_columns;
	sigc::signal<void, Gtk::TreeIter const&> m_itemBeingDeletedSignal;
};


class FilterConfigWindow::FilterTree : public Gtk::TreeView
{
public:
	FilterTree(FilterConfigWindow& owner,
		Glib::RefPtr<FilterTreeModel> const& model);
	
	virtual ~FilterTree();
	
	bool isFilterGroupBeingEdited(FilterGroupTag const& group_tag);
	
	void onNewFile();
	
	void onFileRename();
	
	void onFileDelete();
	
	void onNewFilter();
	
	void onFilterRename();
	
	void onFilterMoveUp();
	
	void onFilterMoveDown();
	
	void onFilterDelete();
private:
	void translateEnabledProperty(
		Gtk::CellRenderer* renderer, Gtk::TreeIter const& it);
	
	void translateModifiedProperty(
		Gtk::CellRenderer* renderer, Gtk::TreeIter const& it);
	
	virtual bool on_button_press_event(GdkEventButton* evt);
	
	void swapFilterPositions(
		Gtk::TreeIter const& item1, Gtk::TreeIter const& item2);
	
	bool isEditingInProgress() const;
	
	void startEditing(Gtk::TreeIter const& it);
	
	void onEditingCanceled();
	
	void onEditingFinished(
		Glib::ustring const& old_text, Glib::ustring const& new_text);
	
	void onItemBeingDeleted(Gtk::TreeIter const& item);
	
	void onSelChanged();
	
	FilterConfigWindow& m_rOwner;
	Glib::RefPtr<FilterTreeModel> m_ptrModel;
	Gtk::TreeViewColumn m_column;
	MyCellRendererToggle m_cbCellRenderer;
	Gtk::CellRendererText m_nameCellRenderer;
	NoItemContextMenu m_noItemMenu;
	FileContextMenu m_fileMenu;
	FilterContextMenu m_filterMenu;
	Gtk::TreeIter m_selectedItem;
};


class FilterConfigWindow::FilterPanel : public Gtk::VBox
{
public:
	FilterPanel(FilterConfigWindow& owner);
	
	virtual ~FilterPanel();
	
	Operation getOperation() const { return m_operation; }
	
	void setOperation(Operation op);
	
	bool isModified() const { return m_isModified; }
	
	void setModified(bool modified);
	
	void load(EditState const& state);
	
	void save(EditState& state);
	
	bool validateAndApply(RegexFilterDescriptor& filter);
private:
	void onModified();
	
	FilterConfigWindow& m_rOwner;
	Operation m_operation;
	bool m_isModified;
	int m_modifyEventsBlocked;
	Gtk::Entry m_orderCtrl;
	Gtk::Entry m_matchCountLimitCtrl;
	Gtk::Entry m_contentTypeCtrl;
	Gtk::Entry m_urlCtrl;
	Gtk::TextView m_searchCtrl;
	AutoIndentingTextView m_replaceCtrl;
	Gtk::RadioButtonGroup m_replacementTypeGroup;
	Gtk::RadioButton m_plainTextRB;
	Gtk::RadioButton m_expressionRB;
	Gtk::RadioButton m_javaScriptRB;
	Gtk::Entry m_ifFlagCtrl;
	Gtk::Entry m_setFlagCtrl;
	Gtk::Entry m_clearFlagCtrl;
	Gtk::Button m_resetButton;
	Gtk::Button m_saveButton;
};


FilterConfigWindow* FilterConfigWindow::m_spInstance = 0;


FilterConfigWindow::FilterConfigWindow()
:	AbstractLogView(*OperationLog::instance()),
	m_ptrFilterTreeModel(FilterTreeModel::create()),
	m_pFilterTree(0),
	m_pFilterPanel(0),
	m_pLogView(0)
{
	signal_focus_in_event().connect(
		sigc::mem_fun(*this, &FilterConfigWindow::onFocusChange)
	);
	signal_focus_out_event().connect(
		sigc::mem_fun(*this, &FilterConfigWindow::onFocusChange)
	);
	
	set_title("Filter Configuration");
	set_icon(CompiledImages::window_icon_png.getPixbuf());
	
	Gtk::VBox* top_vbox = manage(new Gtk::VBox);
	add(*top_vbox);
	
	Gtk::HBox* lr_box = manage(new Gtk::HBox);
	top_vbox->pack_start(*lr_box);
	
	Gtk::VBox* tree_box = manage(new Gtk::VBox);
	lr_box->pack_start(*tree_box, Gtk::PACK_SHRINK);
	
	Gtk::ScrolledWindow* tree_scroller = manage(new Gtk::ScrolledWindow);
	tree_box->pack_start(*tree_scroller);
	tree_scroller->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	tree_scroller->set_shadow_type(Gtk::SHADOW_IN);
	tree_scroller->set_size_request(205, 276);
	tree_scroller->set_border_width(1);
	
	m_pFilterTree = manage(new FilterTree(*this, m_ptrFilterTreeModel));
	tree_scroller->add(*m_pFilterTree);
	
	Gtk::Alignment* clear_log_align = manage(new Gtk::Alignment(0.0, 0.0, 0.0, 0.0));
	tree_box->pack_start(*clear_log_align, Gtk::PACK_SHRINK);
	clear_log_align->set_padding(4, 0, 4, 4);

	Gtk::Button* clear_log_btn = manage(new Gtk::Button("Clear Log"));
	clear_log_align->add(*clear_log_btn);
	clear_log_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &FilterConfigWindow::onClearLog)
	);	
	
	Gtk::Alignment* fpanel_align = manage(new Gtk::Alignment);
	lr_box->pack_start(*fpanel_align);
	fpanel_align->set_padding(4, 0, 4, 4);
	
	m_pFilterPanel = manage(new FilterPanel(*this));
	fpanel_align->add(*m_pFilterPanel);
	m_pFilterPanel->hide();
	
	Gtk::Frame* log_frame = manage(new Gtk::Frame(" Log "));
	top_vbox->pack_start(*log_frame, Gtk::PACK_SHRINK);
	log_frame->set_border_width(1);
	
	AutoScrollingWindow* log_scroll_window = manage(new AutoScrollingWindow);
	log_frame->add(*log_scroll_window);
	log_scroll_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	log_scroll_window->set_shadow_type(Gtk::SHADOW_IN);
	log_scroll_window->set_size_request(625, 100);
	log_scroll_window->set_border_width(2);
	
	m_pLogView = manage(
		new Gtk::TextView(OperationLog::instance()->getTextBuffer()
	));
	log_scroll_window->add(*m_pLogView);
	m_pLogView->set_editable(false);
	m_pLogView->set_cursor_visible(false);
	m_pLogView->set_wrap_mode(Gtk::WRAP_WORD);
	
	show_all_children();
	
	Application::instance()->destroySignal().connect(
		sigc::mem_fun(*this, &FilterConfigWindow::hide)
	);
}

FilterConfigWindow::~FilterConfigWindow()
{
	assert(m_spInstance);
	m_spInstance = 0;
}

bool
FilterConfigWindow::isFilterGroupBeingEdited(FilterGroupTag const& group_tag)
{
	if (!m_spInstance) {
		return false;
	}
	return m_spInstance->m_pFilterTree->isFilterGroupBeingEdited(group_tag);
}

void
FilterConfigWindow::showWindow()
{
	if (m_spInstance) {
		m_spInstance->present();
	} else {
		(m_spInstance = new FilterConfigWindow)->show();
	}
}

bool
FilterConfigWindow::on_delete_event(GdkEventAny*)
{
	if (prepareForWindowDestruction()) {
		delete this;
	}
	return true; // stop event propagation
}

bool
FilterConfigWindow::prepareForWindowDestruction()
{
	if (!m_ptrFilterTreeModel->haveModifiedItems()) {
		return true;
	}
	
	Gtk::MessageDialog dialog(
		*this,
		"Are you sure you want to close this window?\n"
		"Any <b>unsaved changes</b> will be lost.",
		true, /* use_markup */
		Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO,
		true /* modal */
	);
	int const response = dialog.run();
	if (response != Gtk::RESPONSE_YES) {
		return false;
	}
	
	return true;
}

FilterConfigWindow::FilterTreeColumns const&
FilterConfigWindow::getCols() const
{
	return m_ptrFilterTreeModel->getCols();
}

bool
FilterConfigWindow::onFocusChange(GdkEventFocus*)
{
	AbstractLogView::reportVisibility(property_is_active());
	return false; // propagate event further
}

void
FilterConfigWindow::onClearLog()
{
	OperationLog::instance()->clear();
}

void
FilterConfigWindow::onReset()
{
	assert(m_pFilterPanel->getOperation() == EDITING);
	Gtk::TreeIter item = m_pFilterTree->get_selection()->get_selected();
	assert(item);
	Gtk::TreeRow row = *item;
	assert(m_ptrFilterTreeModel->isFilterItem(row));
	IntrusivePtr<FilterNode> node = row[getCols().FILTER_NODE];
	assert(node);
	assert(node->editState);
	assert(node->editState->operation == EDITING);
	
	node->editState->isModified = false;
	m_pFilterPanel->load(*node->editState);
	m_ptrFilterTreeModel->markFilterAsUnmodified(row);
}

void
FilterConfigWindow::onSave()
{
	Gtk::TreeIter item = m_pFilterTree->get_selection()->get_selected();
	assert(item);
	Gtk::TreeRow row = *item;
	assert(m_ptrFilterTreeModel->isFilterItem(row));
	IntrusivePtr<FilterNode> node = row[getCols().FILTER_NODE];
	assert(node);
	assert(node->editState);
	Gtk::TreeRow file_row = *row.parent();
	IntrusivePtr<FileNode> file_node = file_row[getCols().FILE_NODE];
	assert(file_node);
	assert(file_node->filterGroup);
	ContentFilterGroup& group = *file_node->filterGroup;
	
	IntrusivePtr<RegexFilterDescriptor> new_filter;
	if (node->filter) {
		new_filter.reset(new RegexFilterDescriptor(*node->filter));
	} else {
		new_filter.reset(new RegexFilterDescriptor(
			FilterTag::create(),
			file_node->filterGroup->getTag(),
			Application::localeFromUtf8(row[getCols().NAME])
		));
	}
	
	new_filter->setEnabled(row[getCols().ENABLED] == CHECKED);
	if (!m_pFilterPanel->validateAndApply(*new_filter)) {
		return;
	}
	
	ContentFilterGroup new_group(group);
	
	if (node->editState->operation == CREATING) {
		assert(!node->filter);
		IntrusivePtr<RegexFilterDescriptor> follower
			= m_ptrFilterTreeModel->findExistingFollower(item);
		if (!follower) {
			new_group.filters().filters().push_back(new_filter);
		} else {
			typedef ContentFilters::FilterList::iterator Iter;
			Iter it = new_group.filters().find(follower);
			assert(it != new_group.filters().filters().end());
			new_group.filters().filters().insert(it, new_filter);
		}
	} else {
		assert(node->filter);
		typedef ContentFilters::FilterList::iterator Iter;
		Iter it = new_group.filters().find(node->filter);
		assert(it != new_group.filters().filters().end());
		*it = new_filter;
	}
	new_group.fileStructure().updateWith(new_group.filters());
	
	if (!saveFilterGroup(new_group, &group)) {
		return;
	}
	
	node->filter = new_filter;
	group.swap(new_group);
	Application::instance()->updateGlobalContentFilters();
	m_pFilterPanel->setOperation(EDITING);
	m_pFilterPanel->setModified(false);
	m_pFilterPanel->save(*node->editState);
	m_ptrFilterTreeModel->markFilterAsUnmodified(row);
}

void
FilterConfigWindow::markCurrentFilterAsModified()
{
	Gtk::TreeIter item = m_pFilterTree->get_selection()->get_selected();
	assert(item);
	Gtk::TreeRow row = *item;
	assert(m_ptrFilterTreeModel->isFilterItem(row));
	m_ptrFilterTreeModel->markFilterAsModified(row);
}

void
FilterConfigWindow::handleItemRename(
	Gtk::TreeIter const& item, Glib::ustring const& new_name)
{
	Gtk::TreeRow row = *item;
	bool done = false;
	if (m_ptrFilterTreeModel->isFileItem(row)) {
		IntrusivePtr<FileNode> node = row[m_ptrFilterTreeModel->getCols().FILE_NODE];
		if (node) {
			done = renameFile(row, new_name);
		} else {
			done = createFile(row, new_name);
			if (!done) {
				m_ptrFilterTreeModel->erase(item);
			}
		}
		if (done) {
			m_ptrFilterTreeModel->sortFilesByName();
		}
	} else {
		// filter item
		IntrusivePtr<FilterNode> node = row[m_ptrFilterTreeModel->getCols().FILTER_NODE];
		Gtk::TreeRow parent_row = *row.parent();
		if (node) {
			done = renameFilter(row, parent_row, new_name);
		} else {
			done = createFilter(row, parent_row, new_name);
			if (!done) {
				m_ptrFilterTreeModel->erase(item);
				m_ptrFilterTreeModel->updateFileCheckState(parent_row);
			}
		}
	}
}

bool
FilterConfigWindow::createFile(
	Gtk::TreeRow const& row, Glib::ustring const& name)
{
	assert(m_ptrFilterTreeModel->isFileItem(row));
	string fname;
	
	if (!verifyFileNameValid(name)) {
		return false;
	}
	if (!verifyFileNameLocaleConvertable(name, &fname)) {
		return false;
	}
	if (!verifyFileDoesntExist(fname)) {
		return false;
	}
	
	IntrusivePtr<ContentFilterGroup> group(
		new ContentFilterGroup(
			FilterGroupTag::create(), fname,
			ContentFilterGroup::USER_DIR
		)
	);
	
	if (!saveFilterGroup(*group)) {
		return false;
	}
	
	IntrusivePtr<FileNode> node(new FileNode(group));
	row[getCols().NAME] = name;
	row[getCols().FILE_NODE] = node;
	Application::instance()->appendContentFilterGroup(group);
	return true;
}

bool
FilterConfigWindow::renameFile(
	Gtk::TreeRow const& row, Glib::ustring const& new_name)
{
	assert(m_ptrFilterTreeModel->isFileItem(row));
	IntrusivePtr<FileNode> node = row[getCols().FILE_NODE];
	assert(node);
	assert(node->filterGroup);
	ContentFilterGroup& group = *node->filterGroup;
	string fname;
	
	if (!verifyFileNameValid(new_name)) {
		return false;
	}
	if (!verifyFileNameLocaleConvertable(new_name, &fname)) {
		return false;
	}
	if (group.fileName() == fname) {
		return false;
	}
	if (!verifyFileDoesntExist(fname)) {
		return false;
	}
	
	if (!renameFilterGroupFile(group, fname)) {
		return false;
	}
	
	group.fileName() = fname;
	row[getCols().NAME] = new_name;
	return true;
}

bool
FilterConfigWindow::createFilter(Gtk::TreeRow const& row,
	Gtk::TreeRow const& parent_row, Glib::ustring const& name)
{
	assert(m_ptrFilterTreeModel->isFilterItem(row));
	
	if (!verifyFilterNameValid(name)) {
		return false;	
	}
	if (!verifyFilterNameLocaleConvertable(name)) {
		return false;
	}
	if (!verifyFilterDoesntExist(parent_row, name)) {
		return false;
	}
	
	IntrusivePtr<FilterNode> node(new FilterNode);
	row[getCols().NAME] = name;
	row[getCols().FILTER_NODE] = node;
	row[getCols().MODIFIED] = true;
	parent_row[getCols().MODIFIED] = true;
	m_pFilterPanel->load(*node->editState);
	m_pFilterPanel->show();
	return true;
}

bool
FilterConfigWindow::renameFilter(Gtk::TreeRow const& row,
	Gtk::TreeRow const& parent_row, Glib::ustring const& new_name)
{
	assert(m_ptrFilterTreeModel->isFilterItem(row));
	IntrusivePtr<FilterNode> node = row[getCols().FILTER_NODE];
	assert(node);
	assert(node->editState);
	string new_name_str;
	
	if (row[getCols().NAME] == new_name) {
		return false;
	}
	if (!verifyFilterNameValid(new_name)) {
		return false;
	}
	if (!verifyFilterNameLocaleConvertable(new_name, &new_name_str)) {
		return false;
	}
	if (!verifyFilterDoesntExist(parent_row, new_name)) {
		return false;
	}
	
	if (!node->filter) {
		assert(node->editState->operation == CREATING);
		row[getCols().NAME] = new_name;
		return true;
	}
	
	IntrusivePtr<FileNode> parent_node = parent_row[getCols().FILE_NODE];
	ContentFilterGroup& group = *parent_node->filterGroup;
	IntrusivePtr<RegexFilterDescriptor> new_filter(
		new RegexFilterDescriptor(*node->filter)
	);
	new_filter->name() = new_name_str;
	
	size_t filter_pos = std::distance(
		group.filters().filters().begin(),
		group.filters().find(node->filter)
	);
	ContentFilterGroup new_group(group);
	new_group.filters().filters()[filter_pos] = new_filter;
	new_group.fileStructure().renameFilter(filter_pos, new_name_str);
	
	if (!saveFilterGroup(new_group, &group)) {
		return false;
	}
	
	row[getCols().NAME] = new_name;
	group.swap(new_group);
	node->filter = new_filter;
	Application::instance()->updateGlobalContentFilters();
	return true;
}

bool
FilterConfigWindow::verifyFileNameValid(Glib::ustring const& name)
{
	if (name.empty()) {
		showError("File name can't be empty");
		return false;
	}
	if (name.find_first_of(".:/\\") != name.npos) {
		showError(
			"Filter file names must not contain\n"
			"the following characters: . : / \\"
		);
		return false;
	}
	return true;
}

bool
FilterConfigWindow::verifyFileNameLocaleConvertable(
	Glib::ustring const& name, std::string* result)
{
	try {
		if (result) {
			*result = Glib::filename_from_utf8(name);
		} else {
			Glib::filename_from_utf8(name);
		}
	} catch (Glib::ConvertError& e) {
		showError("File name can't be converted to your locale");
		return false;
	}
	return true;
}

bool
FilterConfigWindow::verifyFileDoesntExist(std::string const& fname)
{
	string abs_fname;
	if (Glib::path_is_absolute(fname)) {
		abs_fname = fname;
	} else {
		Application& app = *Application::instance();
		abs_fname = Glib::build_filename(app.getUserFiltersDir(), fname);
	}
	if (Glib::file_test(abs_fname, Glib::FILE_TEST_EXISTS)) {
		showError("File already exists");
		return false;
	}
	return true;
}

bool
FilterConfigWindow::verifyFilterNameValid(Glib::ustring const& name)
{
	if (name.empty()) {
		showError("Filter name can't be empty");
		return false;
	}
	return true;
}

bool
FilterConfigWindow::verifyFilterNameLocaleConvertable(
	Glib::ustring const& name, std::string* result)
{
	try {
		if (result) {
			*result = Glib::locale_from_utf8(name);
		} else {
			Glib::locale_from_utf8(name);
		}
	} catch (Glib::ConvertError& e) {
		showError("Filter name can't be converted to your locale");
		return false;
	}
	return true;
}

bool
FilterConfigWindow::verifyFilterDoesntExist(
	Gtk::TreeRow const& parent_row, Glib::ustring const& name)
{
	Gtk::TreeNodeChildren children = parent_row.children();
	Gtk::TreeIter it = children.begin();
	Gtk::TreeIter const end = children.end();
	for (; it != end; ++it) {
		Gtk::TreeRow row = *it;
		if (name == row[m_ptrFilterTreeModel->getCols().NAME]) {
			showError("Duplicate filter name");
			return false;	
		}
	}
	return true;
}

void
FilterConfigWindow::onItemUnselected(Gtk::TreeIter const& item)
{
	if (item) {
		Gtk::TreeRow row = *item;
		if (m_ptrFilterTreeModel->isFilterItem(row)) {
			IntrusivePtr<FilterNode> node = row[getCols().FILTER_NODE];
			if (node) {
				assert(node->editState);
				m_pFilterPanel->save(*node->editState);
			}
		}
	}
}

void
FilterConfigWindow::onItemSelected(Gtk::TreeIter const& item)
{
	bool show = false;
	if (item) {
		Gtk::TreeRow row = *item;
		if (m_ptrFilterTreeModel->isFilterItem(row)) {
			IntrusivePtr<FilterNode> node = row[getCols().FILTER_NODE];
			if (node) {
				assert(node->editState);
				m_pFilterPanel->load(*node->editState);
				show = true;
			}
		}
	}
	if (show) {
		m_pFilterPanel->show();
	} else {
		m_pFilterPanel->hide();
	}
}

void
FilterConfigWindow::showError(Glib::ustring const& msg, bool markup)
{
	Gtk::MessageDialog dialog(
		*this, msg, markup,
		Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK,
		true /* modal */
	);
	dialog.run();
}

Glib::RefPtr<Gdk::Pixbuf>
FilterConfigWindow::getFileIcon()
{
	static Glib::RefPtr<Gdk::Pixbuf> icon(CompiledImages::file_png.getPixbuf());
	return icon;
}

Glib::RefPtr<Gdk::Pixbuf>
FilterConfigWindow::getFilterIcon()
{
	static Glib::RefPtr<Gdk::Pixbuf> icon(CompiledImages::filter_png.getPixbuf());
	return icon;
}

bool
FilterConfigWindow::renameFilterGroupFile(
	ContentFilterGroup const& group, std::string const& new_name)
{
	Application& app = *Application::instance();
	string const dir = app.getUserFiltersDir();
	string const old_filter_fname = Glib::build_filename(dir, group.fileName());
	string const new_filter_fname = Glib::build_filename(dir, new_name);
	string const old_enabled_fname = old_filter_fname + ".enabled";
	string const new_enabled_fname = new_filter_fname + ".enabled";
	
	bool enabled_file_exists = Glib::file_test(old_filter_fname, Glib::FILE_TEST_EXISTS);
	if (enabled_file_exists) {
		if (!app.renameFile(old_enabled_fname, new_enabled_fname)) {
			return false;
		}
	}
	if (!app.renameFile(old_filter_fname, new_filter_fname)) {
		if (enabled_file_exists) {
			app.renameFile(new_enabled_fname, old_enabled_fname);
		}
		return false;
	}
	return true;
}

bool
FilterConfigWindow::saveFilterGroup(
	ContentFilterGroup const& new_group,
	ContentFilterGroup const* old_group)
{
	Application& app = *Application::instance();
	
	if (!FilterTreeModel::saveEnabledFilterList(new_group)) {
		return false;
	}
	
	bool const skip_content = (old_group &&
		new_group.fileStructure() == old_group->fileStructure()
	);
	if (!skip_content) {
		string fname = Glib::build_filename(app.getUserFiltersDir(), new_group.fileName());
		ostringstream strm;
		strm << new_group.fileStructure();
		if (!app.writeFile(fname, strm.str())) {
			if (old_group) {
				FilterTreeModel::saveEnabledFilterList(*old_group);
			} else {
				app.deleteFile(fname + ".enabled");
			}
			return false;
		}
	}
	
	return true;
}


/*===================== FilterConfigWindow::EditState =====================*/

FilterConfigWindow::EditState::EditState()
:	operation(CREATING),
	isModified(true), // always true for CREATING
	replacementType(RegexFilterDescriptor::TEXT)
{
}

FilterConfigWindow::EditState::EditState(RegexFilterDescriptor const& filter)
:	operation(EDITING),
	isModified(false)
{
	if (filter.order() != 0) {
		order = StringUtils::fromNumber(filter.order());
	}
	if (filter.matchCountLimit() != -1) {
		matchCountLimit = StringUtils::fromNumber(filter.matchCountLimit());
	}
	if (filter.urlPattern().get()) {
		url = Application::localeToUtf8(filter.urlPattern()->source());
	}
	if (filter.contentTypePattern().get()) {
		contentType = Application::localeToUtf8(filter.contentTypePattern()->source());
	}
	if (filter.searchPattern().get()) {
		search = Application::localeToUtf8(filter.searchPattern()->source());
	}
	if (filter.replacement().get()) {
		replacement = Application::localeToUtf8(*filter.replacement());
	}
	replacementType = filter.replacementType();
	ifFlag = Application::localeToUtf8(filter.ifFlag());
	setFlag = Application::localeToUtf8(filter.setFlag());
	clearFlag = Application::localeToUtf8(filter.clearFlag());
}

FilterConfigWindow::EditState::~EditState()
{
}


/*=============== FilterConfigWindow::MyCellRendererToggle ===============*/

bool
FilterConfigWindow::MyCellRendererToggle::activate_vfunc(
	GdkEvent* event, Gtk::Widget& widget, Glib::ustring const& path,
	Gdk::Rectangle const& background_area,
	Gdk::Rectangle const& cell_area, Gtk::CellRendererState flags)
{
	/*
	GtkTreeView has a nasty feature: it doesn't care which cell renderer
	was clicked, it only cares which column was. Since we pack all of our
	cell renderers into a single column, the effect is:
	A user clicks anywhere in a row and the checkbox af the left of the row
	is toggled.
	And no, we can't just put the checkbox into its own column. That would
	create gaps between that column and the next one denepding on the depth
	level of the row.
	In this workaround we assume that we are the first cell renderer
	in a column. We also assume the checkbox width is 16 pixers, which is
	actually 12 plus a padding of 2 pixels from each side.
	*/
	if (event && event->type == GDK_BUTTON_PRESS) {
		if (event->button.x > cell_area.get_x() + 16) {
			return false;
		}
	}
	return Gtk::CellRendererToggle::activate_vfunc(
		event, widget, path, background_area, cell_area, flags
	);
}


/*================= FilterConfigWindow::FilterTreeModel ==================*/

FilterConfigWindow::FilterTreeModel::FilterTreeModel()
{
	set_column_types(m_columns);
	loadFilterGroups();
}

FilterConfigWindow::FilterTreeModel::~FilterTreeModel()
{
}

Glib::RefPtr<FilterConfigWindow::FilterTreeModel>
FilterConfigWindow::FilterTreeModel::create()
{
	return Glib::RefPtr<FilterTreeModel>(new FilterTreeModel);
}

sigc::slot<void, Glib::ustring const&>
FilterConfigWindow::FilterTreeModel::getCheckBoxHandler()
{
	return sigc::mem_fun(*this, &FilterTreeModel::onCheckboxToggled);
}

void
FilterConfigWindow::FilterTreeModel::loadFilterGroups()
{
	typedef Application::ContentFilterList GroupList;
	GroupList const& groups = Application::instance()->contentFilters();
	GroupList::const_iterator it = groups.begin();
	GroupList::const_iterator const end = groups.end();
	for (; it != end; ++it) {
		ContentFilterGroup const& group = **it;
		Glib::ustring fname = Application::filenameToUtf8(group.fileName());
		Row row = *append();
		row[m_columns.ICON] = getFileIcon();
		row[m_columns.NAME] = fname;
		row[m_columns.FILE_NODE] = IntrusivePtr<FileNode>(new FileNode(*it));
		CheckState state = loadIndividualFilters(group.filters(), row);
		row[m_columns.ENABLED] = state;
	}
}

FilterConfigWindow::CheckState
FilterConfigWindow::FilterTreeModel::loadIndividualFilters(
	ContentFilters const& filters, Row& parent_row)
{
	typedef ContentFilters::FilterList FilterList;
	
	bool have_enabled_nodes = false;
	bool have_disabled_nodes = false;
	FilterList const& fts = filters.filters(); 
	FilterList::const_iterator it = fts.begin();
	FilterList::const_iterator const end = fts.end();
	for (; it != end; ++it) {
		RegexFilterDescriptor const& filter = **it;
		Glib::ustring name = Application::localeToUtf8(filter.name());
		Row row = *append(parent_row.children());
		row[m_columns.ICON] = getFilterIcon();
		row[m_columns.NAME] = name;
		row[m_columns.FILTER_NODE] = IntrusivePtr<FilterNode>(new FilterNode(*it));
		row[m_columns.ENABLED] = filter.isEnabled() ? CHECKED : UNCHECKED;
		if (filter.isEnabled()) {
			have_enabled_nodes = true;
		} else {
			have_disabled_nodes = true;
		}
	}
	if (have_enabled_nodes && have_disabled_nodes) {
		return MIXED;
	} else if (have_enabled_nodes) {
		return CHECKED;
	} else {
		return UNCHECKED;
	}
}

bool
FilterConfigWindow::FilterTreeModel::isFileItem(Row const& row) const
{
	return !row.parent();
}

bool
FilterConfigWindow::FilterTreeModel::isFilterItem(Row const& row) const
{
	return row.parent();
}

Gtk::TreeStore::iterator
FilterConfigWindow::FilterTreeModel::erase(iterator const& iter)
{
	m_itemBeingDeletedSignal.emit(iter);
	return Gtk::TreeStore::erase(iter);
}

void
FilterConfigWindow::FilterTreeModel::updateFileCheckState(Row const& file_row)
{
	assert(isFileItem(file_row));
	bool have_enabled_children = false;
	bool have_disabled_children = false;	
	
	Children children = file_row.children(); 
	Children::iterator it = children.begin();
	Children::iterator const end = children.end();
	for (; it != end; ++it) {
		Row child = *it;
		if (child[m_columns.ENABLED] == CHECKED) {
			have_enabled_children = true;
		} else {
			have_disabled_children = true;
		}
	}
	
	CheckState state = UNCHECKED;
	if (have_enabled_children && have_disabled_children) {
		state = MIXED;
	} else if (have_enabled_children) {
		state = CHECKED;
	}
	file_row[m_columns.ENABLED] = state;
}

void
FilterConfigWindow::FilterTreeModel::updateFileModifiedState(Row const& file_row)
{
	assert(isFileItem(file_row));
	bool mod_found = false;
	
	Children children = file_row.children(); 
	Children::iterator it = children.begin();
	Children::iterator const end = children.end();
	for (; it != end; ++it) {
		Row child = *it;
		if (child[m_columns.MODIFIED]) {
			mod_found = true;
			break;
		}
	}
	
	file_row[m_columns.MODIFIED] = mod_found;
}

void
FilterConfigWindow::FilterTreeModel::markFilterAsModified(Row const& row)
{
	assert(isFilterItem(row));
	Gtk::TreeRow file_row = *row.parent();
	row[m_columns.MODIFIED] = true;
	file_row[m_columns.MODIFIED] = true;
}

void
FilterConfigWindow::FilterTreeModel::markFilterAsUnmodified(Row const& row)
{
	assert(isFilterItem(row));
	Gtk::TreeRow file_row = *row.parent();
	row[m_columns.MODIFIED] = false;
	updateFileModifiedState(file_row);
}

bool
FilterConfigWindow::FilterTreeModel::haveModifiedItems() const
{
	// It's enough to check for toplevel (file) nodes.
	Children nodes = children();
	Children::iterator it = nodes.begin();
	Children::iterator const end = nodes.end();
	for (; it != end; ++it) {
		Row child = *it;
		if (child[m_columns.MODIFIED]) {
			return true;
		}
	}
	return false;
}

IntrusivePtr<RegexFilterDescriptor>
FilterConfigWindow::FilterTreeModel::findExistingFollower(
	iterator const& filter_item)
{
	assert(isFilterItem(*filter_item));
	
	IntrusivePtr<RegexFilterDescriptor> res;
	iterator it = filter_item;
	for (; it; ++it) {
		Row row = *it;
		IntrusivePtr<FilterNode> node = row[m_columns.FILTER_NODE];
		if (node && node->filter) {
			res = node->filter;
			break;
		}
	}
	
	return res;
}

namespace {
	class FilePosition
	{
	public:
		FilePosition(std::string const& fname, int pos)
		: m_fileName(fname), m_pos(pos) {}
		
		bool operator<(FilePosition const& other) const {
			return m_fileName < other.m_fileName;
		}
		
		operator int() const { return m_pos; }
	private:
		std::string m_fileName;
		int m_pos;
	};
}

void
FilterConfigWindow::FilterTreeModel::sortFilesByName()
{
	vector<FilePosition> order;
	Children nodes = children();
	order.reserve(nodes.size());
	
	Children::iterator it = nodes.begin();
	Children::iterator const end = nodes.end();
	for (int pos = 0; it != end; ++it, ++pos) {
		Row row = *it;
		IntrusivePtr<FileNode> node = row[m_columns.FILE_NODE];
		assert(node);
		assert(node->filterGroup);
		string fname = node->filterGroup->fileName();
		order.push_back(FilePosition(fname, pos));
	}
	
	std::stable_sort(order.begin(), order.end());
	vector<int> new_order(order.begin(), order.end());
	
	reorder(nodes, new_order);
	Application& app = *Application::instance();
	app.sortContentFiltersByFileName();
	app.updateGlobalContentFilters();
}

bool
FilterConfigWindow::FilterTreeModel::saveEnabledFilterList(
	ContentFilterGroup const& group, EnablePolicy policy)
{
	Application& app = *Application::instance();
	std::string dir = app.getUserFiltersDir();
	std::string fname = Glib::build_filename(dir, group.fileName() + ".enabled");
	
	string file_data;
	
	if (policy == ALL_ENABLED) {
		file_data = "*";
	} else if (policy == NORMAL) {
		ostringstream strm;
		ContentFilters const& filters = group.filters();
		bool have_enabled = false;
		bool have_disabled = false;
		ContentFilters::FilterList::const_iterator it = filters.filters().begin();
		ContentFilters::FilterList::const_iterator end = filters.filters().end();
		for (; it != end; ++it) {
			bool const enabled = (*it)->isEnabled();
			if (enabled) {
				strm << (*it)->name() << "\r\n";
				have_enabled = true;
			} else {
				have_disabled = true;
			}
		}
		if (have_enabled && !have_disabled) {
			file_data = "*";
		} else {
			file_data = strm.str();
		}
	}
	
	return app.writeFile(fname, file_data);
}

void
FilterConfigWindow::FilterTreeModel::onCheckboxToggled(Glib::ustring const& path)
{
	Row row = *get_iter(path);
	CheckState old_state = row[m_columns.ENABLED];
	CheckState new_state = (old_state == UNCHECKED) ? CHECKED : UNCHECKED;
	if (isFileItem(row)) {
		onFileCheckStateChanging(row, old_state, new_state);
	} else {
		onFilterCheckStateChanging(row, old_state, new_state);
	}
	
}

void
FilterConfigWindow::FilterTreeModel::onFileCheckStateChanging(
	Row const& row, CheckState old_state, CheckState new_state)
{
	IntrusivePtr<FileNode> file_node = row[m_columns.FILE_NODE];
	if (!file_node || row.children().empty()) {
		return;
	}
	
	ContentFilterGroup& group = *file_node->filterGroup;
	bool const checked = (new_state == CHECKED);
	
	if (!saveEnabledFilterList(group, checked ? ALL_ENABLED : ALL_DISABLED)) {
		return;
	}
	
	row[m_columns.ENABLED] = new_state;
	
	Children children = row.children(); 
	Children::iterator it = children.begin();
	Children::iterator const end = children.end();
	for (; it != end; ++it) {
		Row child = *it;
		child[m_columns.ENABLED] = new_state;
		IntrusivePtr<FilterNode> filter_node = child[m_columns.FILTER_NODE];
		if (filter_node && filter_node->filter) {
			filter_node->filter->setEnabled(checked);
		}
	}
	
	Application::instance()->updateGlobalContentFilters();
}

void
FilterConfigWindow::FilterTreeModel::onFilterCheckStateChanging(
	Row const& row, CheckState old_state, CheckState new_state)
{
	IntrusivePtr<FilterNode> filter_node = row[m_columns.FILTER_NODE];
	iterator file_item = row.parent();
	IntrusivePtr<FileNode> file_node = (*file_item)[m_columns.FILE_NODE];
	assert(file_node);
	assert(file_node->filterGroup);
	
	if (!(filter_node && filter_node->filter)) {
		return;
	}
	
	ContentFilterGroup& group = *file_node->filterGroup;
	
	filter_node->filter->setEnabled(new_state == CHECKED);
	if (!saveEnabledFilterList(group)) {
		filter_node->filter->setEnabled(old_state == CHECKED);
		return;
	}
	
	row[m_columns.ENABLED] = new_state;
	updateFileCheckState(*file_item);
	
	Application::instance()->updateGlobalContentFilters();
}


/*===================== FilterConfigWindow::FilterTree ===================*/

FilterConfigWindow::FilterTree::FilterTree(
	FilterConfigWindow& owner, Glib::RefPtr<FilterTreeModel> const& model)
:	Gtk::TreeView(model),
	m_rOwner(owner),
	m_ptrModel(model),
	m_noItemMenu(*this),
	m_fileMenu(*this),
	m_filterMenu(*this),
	m_selectedItem(model->children().end())
{
	set_headers_visible(false);
	
	m_column.pack_start(m_cbCellRenderer, false);
	m_column.pack_start(model->getCols().ICON, false);
	m_column.pack_start(m_nameCellRenderer, true);
	
	m_nameCellRenderer.signal_editing_canceled().connect(
		sigc::mem_fun(*this, &FilterTree::onEditingCanceled)
	);
	m_nameCellRenderer.signal_edited().connect(
		sigc::mem_fun(*this, &FilterTree::onEditingFinished)
	);
	m_column.add_attribute(m_nameCellRenderer.property_text(), model->getCols().NAME);
	m_column.set_cell_data_func(
		m_nameCellRenderer,
		sigc::mem_fun(*this, &FilterTree::translateModifiedProperty)
	);
	m_column.set_cell_data_func(
		m_cbCellRenderer,
		sigc::mem_fun(*this, &FilterTree::translateEnabledProperty)
	);
	m_cbCellRenderer.signal_toggled().connect(m_ptrModel->getCheckBoxHandler());
	
	append_column(m_column);
	
	m_ptrModel->signalItemBeingDeleted().connect(
		sigc::mem_fun(*this, &FilterTree::onItemBeingDeleted)
	);
	get_selection()->signal_changed().connect(
		sigc::mem_fun(*this, &FilterTree::onSelChanged)
	);
}

FilterConfigWindow::FilterTree::~FilterTree()
{
}

bool
FilterConfigWindow::FilterTree::isFilterGroupBeingEdited(FilterGroupTag const& group_tag)
{
	Gtk::TreeIter item = get_selection()->get_selected();
	if (!item) {
		return false;
	}
	
	Gtk::TreeRow row = *item;
	
	IntrusivePtr<FilterNode> node = row[m_ptrModel->getCols().FILTER_NODE];
	if (node) {
		return (node->filter && node->filter->getGroupTag() == group_tag);
	}
	IntrusivePtr<FileNode> file_node = row[m_ptrModel->getCols().FILE_NODE];
	return (file_node && file_node->filterGroup->getTag() == group_tag);
}

void
FilterConfigWindow::FilterTree::onNewFile()
{
	Gtk::TreeIter sel = get_selection()->get_selected();
	Gtk::TreeIter new_el;
	if (sel) {
		assert(m_ptrModel->isFileItem(*sel));
		new_el = m_ptrModel->insert_after(sel);
	} else {
		new_el = m_ptrModel->append();
	}
	Gtk::TreeRow row = *new_el;
	row[m_ptrModel->getCols().ENABLED] = UNCHECKED;
	row[m_ptrModel->getCols().ICON] = getFileIcon();
	Gtk::TreePath path(row);
	expand_to_path(path);
	scroll_to_row(path);
	get_selection()->select(row);
	startEditing(row);
}

void
FilterConfigWindow::FilterTree::onFileRename()
{
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	assert(m_ptrModel->isFileItem(*sel));
	startEditing(sel);
}

void
FilterConfigWindow::FilterTree::onFileDelete()
{
	Application& app = *Application::instance();
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	Gtk::TreeRow row = *sel;
	assert(m_ptrModel->isFileItem(row));
	IntrusivePtr<FileNode> node = row[m_ptrModel->getCols().FILE_NODE];
	assert(node);
	assert(node->filterGroup);
	ContentFilterGroup& group = *node->filterGroup;
	string filter_fname;
	
	if (group.fileLocation() == group.USER_DIR) {
		string dir = app.getUserFiltersDir();
		filter_fname = Glib::build_filename(dir, group.fileName());
	} else {
		string dir = app.getGlobalFiltersDir();
		filter_fname = Glib::build_filename(dir, group.fileName());
		if (ACE_OS::access(dir.c_str(), W_OK) == -1) {
			m_rOwner.showError(
				"This filter is global and you don't have permission to delete it."
			);
			return;
		}
	}
	
	Glib::ustring question("Really delete this file?");
	if (group.fileLocation() == group.GLOBAL_DIR) {
		question += "\nNote that this filter is <b>global</b> and will probably reapper after upgrade.";
	}
	Gtk::MessageDialog dialog(
		m_rOwner, question,
		true, /* use_markup */
		Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO,
		true /* modal */
	);
	int const response = dialog.run();
	if (response != Gtk::RESPONSE_YES) {
		return;
	}
	
	string enabled_fname(filter_fname + ".enabled");
	if (!app.deleteFile(enabled_fname)) {
		return;
	}
	if (!app.deleteFile(filter_fname)) {
		m_ptrModel->saveEnabledFilterList(group);
		return;
	}
	
	bool removed = app.removeContentFilterGroup(node->filterGroup);
	assert(removed);
	m_ptrModel->erase(sel);
	app.updateGlobalContentFilters();
}

void
FilterConfigWindow::FilterTree::onNewFilter()
{
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	Gtk::TreeIter new_el;
	if (m_ptrModel->isFileItem(*sel)) {
		Gtk::TreeRow file_row = *sel;
		new_el = m_ptrModel->append(file_row.children());
	} else {
		new_el = m_ptrModel->insert_after(sel);
	}
	Gtk::TreeRow row = *new_el;
	row[m_ptrModel->getCols().ENABLED] = UNCHECKED;
	row[m_ptrModel->getCols().ICON] = getFilterIcon();
	Gtk::TreePath path(row);
	expand_to_path(path);
	scroll_to_row(path);
	get_selection()->select(row);
	startEditing(row);
}

void
FilterConfigWindow::FilterTree::onFilterRename()
{
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	assert(m_ptrModel->isFilterItem(*sel));
	startEditing(sel);
}

void
FilterConfigWindow::FilterTree::onFilterMoveUp()
{
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	Gtk::TreeIter prev(sel);
	--prev;
	if (!prev) {
		return;
	}
	swapFilterPositions(sel, prev);
}

void
FilterConfigWindow::FilterTree::onFilterMoveDown()
{
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	Gtk::TreeIter next(sel);
	++next;
	if (!next) {
		return;
	}
	swapFilterPositions(sel, next);
}

void
FilterConfigWindow::FilterTree::onFilterDelete()
{
	Application& app = *Application::instance();
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	Gtk::TreeRow row = *sel;
	assert(m_ptrModel->isFilterItem(row));
	Gtk::TreeRow parent_row = *row.parent();
	IntrusivePtr<FilterNode> node = row[m_ptrModel->getCols().FILTER_NODE];
	IntrusivePtr<FileNode> file_node = parent_row[m_ptrModel->getCols().FILE_NODE];
	assert(file_node);
	assert(file_node->filterGroup);
	ContentFilterGroup& group = *file_node->filterGroup;
	
	Gtk::MessageDialog dialog(
		m_rOwner, "Really delete this filter?",
		true, /* use_markup */
		Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO,
		true /* modal */
	);
	int const response = dialog.run();
	if (response != Gtk::RESPONSE_YES) {
		return;
	}
	
	if (!node || node->editState->operation == CREATING) {
		m_ptrModel->erase(sel);
		m_ptrModel->updateFileCheckState(parent_row);
		m_ptrModel->updateFileModifiedState(parent_row);
		return;
	} else {
		assert(node->filter);
	}
	
	ContentFilterGroup new_group(group);
	typedef ContentFilters::FilterList::iterator Iter;
	Iter it = new_group.filters().find(node->filter);
	assert(it != new_group.filters().filters().end());
	new_group.filters().filters().erase(it);
	new_group.fileStructure().updateWith(new_group.filters());
	
	if (!saveFilterGroup(new_group, &group)) {
		return;
	}
	
	m_ptrModel->erase(sel);
	m_ptrModel->updateFileCheckState(parent_row);
	m_ptrModel->updateFileModifiedState(parent_row);
	
	group.swap(new_group);
	app.updateGlobalContentFilters();
}

void
FilterConfigWindow::FilterTree::translateEnabledProperty(
	Gtk::CellRenderer* renderer, Gtk::TreeIter const& it)
{
	MyCellRendererToggle* rend = dynamic_cast<MyCellRendererToggle*>(renderer);
	Gtk::TreeRow row = *it;
	CheckState state = row[m_ptrModel->getCols().ENABLED];
	rend->set_active(state != UNCHECKED);
	rend->set_property("inconsistent", (state == MIXED));
}

void
FilterConfigWindow::FilterTree::translateModifiedProperty(
	Gtk::CellRenderer* renderer, Gtk::TreeIter const& it)
{
	Gtk::CellRendererText* rend = dynamic_cast<Gtk::CellRendererText*>(renderer);
	Gtk::TreeRow row = *it;
	int weight = PANGO_WEIGHT_NORMAL;
	if (row[m_ptrModel->getCols().MODIFIED]) {
		weight = PANGO_WEIGHT_BOLD;
	}
	rend->property_weight() = weight;
}

bool
FilterConfigWindow::FilterTree::on_button_press_event(GdkEventButton* evt)
{
	bool rval = Gtk::TreeView::on_button_press_event(evt);
	
	if (evt->type == GDK_BUTTON_PRESS && evt->button == 3) {
		if (isEditingInProgress()) {
			return rval;
		}
		
		Gtk::TreeIter sel = get_selection()->get_selected();
		if (!sel) {
			// no item is selected
			m_noItemMenu.popup(evt->button, evt->time);
			return rval;
		}
		
		if (m_ptrModel->isFileItem(*sel)) {
			m_fileMenu.popup(evt->button, evt->time);
		} else {
			Gtk::TreeRow parent_row = *sel->parent();
			Gtk::TreeNodeChildren siblings = parent_row.children();
			m_filterMenu.enableMoveUpItem(sel != siblings.begin());
			Gtk::TreeIter next = sel;
			++next;
			m_filterMenu.enableMoveDownItem(next != siblings.end());
			m_filterMenu.popup(evt->button, evt->time);
		}
	}
	
	return rval;
}

void
FilterConfigWindow::FilterTree::swapFilterPositions(
	Gtk::TreeIter const& item1, Gtk::TreeIter const& item2)
{
	assert(m_ptrModel->isFilterItem(*item1));
	assert(m_ptrModel->isFilterItem(*item2));
	Gtk::TreeRow row1 = *item1;
	Gtk::TreeRow row2 = *item2;
	assert(row1.parent() == row2.parent());
	IntrusivePtr<FilterNode> node1 = row1[m_ptrModel->getCols().FILTER_NODE];
	IntrusivePtr<FilterNode> node2 = row2[m_ptrModel->getCols().FILTER_NODE];
	Gtk::TreeRow file_row(*row1.parent());
	IntrusivePtr<FileNode> file_node = file_row[m_ptrModel->getCols().FILE_NODE];
	ContentFilterGroup& group = *file_node->filterGroup;
	
	if (node1 && node2 && node1->filter && node2->filter) {
		assert(node1->editState->operation == EDITING);
		assert(node2->editState->operation == EDITING);
		ContentFilterGroup new_group(group);
		typedef ContentFilters::FilterList::iterator Iter;
		Iter it1 = new_group.filters().find(node1->filter);
		Iter it2 = new_group.filters().find(node2->filter);
		assert(it1 != new_group.filters().filters().end());
		assert(it2 != new_group.filters().filters().end());
		swap(*it1, *it2);
		new_group.fileStructure().updateWith(new_group.filters());
		
		if (!saveFilterGroup(new_group, &group)) {
			return;
		}
		
		group.swap(new_group);
		Application::instance()->updateGlobalContentFilters();
	} // otherwise at least one of the filters exists only 'on paper'
	
	m_ptrModel->iter_swap(item1, item2);
}

bool
FilterConfigWindow::FilterTree::isEditingInProgress() const
{
	// We only set property_editable when we actually edit something.
	return m_nameCellRenderer.property_editable();
}

void
FilterConfigWindow::FilterTree::startEditing(Gtk::TreeIter const& it)
{
	m_nameCellRenderer.property_editable() = true;
	set_cursor(
		Gtk::TreePath(it), m_column, m_nameCellRenderer,
		true // start_editing
	);
}

void
FilterConfigWindow::FilterTree::onEditingCanceled()
{
	m_nameCellRenderer.property_editable() = false;
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	Gtk::TreeRow row = *sel;
	IntrusivePtr<FileNode> file_node = row[m_ptrModel->getCols().FILE_NODE];
	IntrusivePtr<FilterNode> filter_node = row[m_ptrModel->getCols().FILTER_NODE];
	if (!file_node && !filter_node) {
		m_ptrModel->erase(sel);
	}
}

void
FilterConfigWindow::FilterTree::onEditingFinished(
	Glib::ustring const& old_text, Glib::ustring const& new_text)
{
	m_nameCellRenderer.property_editable() = false;
	Gtk::TreeIter sel = get_selection()->get_selected();
	assert(sel);
	m_rOwner.handleItemRename(sel, new_text);
}

void
FilterConfigWindow::FilterTree::onItemBeingDeleted(Gtk::TreeIter const& item)
{
	if (item == m_selectedItem) {
		//m_rOwner.onItemUnselected(item); // not really necessary
		m_selectedItem = m_ptrModel->children().end();
	}
}

void
FilterConfigWindow::FilterTree::onSelChanged()
{
	m_rOwner.onItemUnselected(m_selectedItem);
	m_selectedItem = get_selection()->get_selected();
	m_rOwner.onItemSelected(m_selectedItem);
}


/*================ FilterConfigWindow::NoItemContextMenu =================*/

FilterConfigWindow::NoItemContextMenu::NoItemContextMenu(FilterTree& tree)
{
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"New File",
			*manage(new Gtk::Image(getFileIcon())),
			sigc::mem_fun(tree, &FilterTree::onNewFile)
		)
	);
}

FilterConfigWindow::NoItemContextMenu::~NoItemContextMenu()
{
}


/*================= FilterConfigWindow::FileContextMenu =================*/

FilterConfigWindow::FileContextMenu::FileContextMenu(FilterTree& tree)
{
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"New File",
			*manage(new Gtk::Image(getFileIcon())),
			sigc::mem_fun(tree, &FilterTree::onNewFile)
		)
	);
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"New Filter",
			*manage(new Gtk::Image(getFilterIcon())),
			sigc::mem_fun(tree, &FilterTree::onNewFilter)
		)
	);
	
	items().push_back(Gtk::Menu_Helpers::SeparatorElem());
	
	items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Rename",
			sigc::mem_fun(tree, &FilterTree::onFileRename)
		)
	);
	
	items().push_back(Gtk::Menu_Helpers::SeparatorElem());
	
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"Delete",
			*manage(new Gtk::Image(Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(tree, &FilterTree::onFileDelete)
		)
	);
}

FilterConfigWindow::FileContextMenu::~FileContextMenu()
{
}


/*================ FilterConfigWindow::FilterContextMenu ================*/

FilterConfigWindow::FilterContextMenu::FilterContextMenu(FilterTree& tree)
{
	/*
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"New Filter",
			*manage(new Gtk::Image(getFilterIcon())),
			sigc::mem_fun(tree, &FilterTree::onNewFilter)
		)
	);
	*/
	items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Rename",
			sigc::mem_fun(tree, &FilterTree::onFilterRename)
		)
	);
	
	items().push_back(Gtk::Menu_Helpers::SeparatorElem());
	
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"Move Up",
			*manage(new Gtk::Image(Gtk::Stock::GO_UP, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(tree, &FilterTree::onFilterMoveUp)
		)
	);
	m_pMoveUpItem = &items().back();
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"Move Down",
			*manage(new Gtk::Image(Gtk::Stock::GO_DOWN, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(tree, &FilterTree::onFilterMoveDown)
		)
	);
	m_pMoveDownItem = &items().back();
	
	items().push_back(Gtk::Menu_Helpers::SeparatorElem());
	
	items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"Delete",
			*manage(new Gtk::Image(Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(tree, &FilterTree::onFilterDelete)
		)
	);
}

FilterConfigWindow::FilterContextMenu::~FilterContextMenu()
{
}

void
FilterConfigWindow::FilterContextMenu::enableMoveUpItem(bool enabled)
{
	m_pMoveUpItem->set_sensitive(enabled);
}

void
FilterConfigWindow::FilterContextMenu::enableMoveDownItem(bool enabled)
{
	m_pMoveDownItem->set_sensitive(enabled);
}


/*===================== FilterConfigWindow::FileNode =====================*/

FilterConfigWindow::FileNode::FileNode(
	IntrusivePtr<ContentFilterGroup> const& group)
:	filterGroup(group)
{
}

FilterConfigWindow::FileNode::~FileNode()
{
}


/*==================== FilterConfigWindow::FilterNode ====================*/

FilterConfigWindow::FilterNode::FilterNode()
:	editState(new EditState)
{
}

FilterConfigWindow::FilterNode::FilterNode(
	IntrusivePtr<RegexFilterDescriptor> const& ftr)
:	editState(new EditState(*ftr)),
	filter(ftr)
{
}

FilterConfigWindow::FilterNode::~FilterNode()
{
}

/*=================== FilterConfigWindow::FilterPanel ====================*/

FilterConfigWindow::FilterPanel::FilterPanel(FilterConfigWindow& owner)
:	m_rOwner(owner),
	m_operation(CREATING),
	m_isModified(true), // always true for CREATING
	m_modifyEventsBlocked(0),
	m_orderCtrl(),
	m_matchCountLimitCtrl(),
	m_contentTypeCtrl(),
	m_urlCtrl(),
	m_searchCtrl(),
	m_replaceCtrl(),
	m_replacementTypeGroup(),
	m_plainTextRB(m_replacementTypeGroup, "Plain Text"),
	m_expressionRB(m_replacementTypeGroup, "Expression"),
	m_javaScriptRB(m_replacementTypeGroup, "JavaScript"),
	m_ifFlagCtrl(),
	m_setFlagCtrl(),
	m_clearFlagCtrl(),
	m_resetButton("Reset"),
	m_saveButton("Save")
{
	m_saveButton.add_accelerator(
		"clicked", owner.get_accel_group(),
		GDK_s, Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE
	);
	
	Gtk::Table* table = manage(new Gtk::Table(7, 2));
	pack_start(*table);
	table->set_col_spacings(3);
	table->set_row_spacings(2);
	
	table->attach(
		*manage(new Gtk::Label("Order", Gtk::ALIGN_RIGHT)),
		0, 1, 0, 1, Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	
	Gtk::Alignment* order_align = manage(new Gtk::Alignment(0.0, 0.5, 0.0, 0.0));
	table->attach(
		*order_align, 1, 2, 0, 1,
		Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	order_align->add(m_orderCtrl);
	m_orderCtrl.set_size_request(30, -1);
	
	table->attach(
		*manage(new Gtk::Label("Max matches", Gtk::ALIGN_RIGHT)),
		0, 1, 1, 2, Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	
	Gtk::Alignment* limit_align = manage(new Gtk::Alignment(0.0, 0.5, 0.0, 0.0));
	table->attach(
		*limit_align, 1, 2, 1, 2,
		Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	limit_align->add(m_matchCountLimitCtrl);
	m_matchCountLimitCtrl.set_size_request(30, -1);
	
	table->attach(
		*manage(new Gtk::Label("Content type", Gtk::ALIGN_RIGHT)),
		0, 1, 2, 3, Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	
	Gtk::Alignment* ctype_align = manage(new Gtk::Alignment(0.0, 0.5, 0.0, 0.0));
	table->attach(
		*ctype_align, 1, 2, 2, 3,
		Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	ctype_align->add(m_contentTypeCtrl);
	m_contentTypeCtrl.set_size_request(260, -1);
	
	table->attach(
		*manage(new Gtk::Label("URL", Gtk::ALIGN_RIGHT)),
		0, 1, 3, 4, Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	
	Gtk::Alignment* url_align = manage(new Gtk::Alignment(0.0, 0.5, 0.0, 0.0));
	table->attach(
		*url_align, 1, 2, 3, 4,
		Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	url_align->add(m_urlCtrl);
	m_urlCtrl.set_size_request(260, -1);
	
	Gtk::Table* flags_table = manage(new Gtk::Table(2, 3));
	table->attach(
		*flags_table, 0, 2, 4, 5,
		Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	flags_table->set_col_spacings(1);
	flags_table->set_row_spacings(1);
	
	flags_table->attach(
		*manage(new Gtk::Label("If")),
		0, 1, 0, 1
	);
	flags_table->attach(
		*manage(new Gtk::Label("Set")),
		1, 2, 0, 1
	);
	flags_table->attach(
		*manage(new Gtk::Label("Clear")),
		2, 3, 0, 1
	);
	m_ifFlagCtrl.set_size_request(50, -1);
	m_setFlagCtrl.set_size_request(50, -1);
	m_clearFlagCtrl.set_size_request(50, -1);
	flags_table->attach(m_ifFlagCtrl, 0, 1, 1, 2);
	flags_table->attach(m_setFlagCtrl, 1, 2, 1, 2);
	flags_table->attach(m_clearFlagCtrl, 2, 3, 1, 2);
	
	Gtk::VBox* search_vbox = manage(new Gtk::VBox);
	table->attach(
		*search_vbox, 0, 2, 5, 6,
		Gtk::EXPAND|Gtk::FILL, Gtk::SHRINK|Gtk::FILL
	);
	search_vbox->pack_start(
		*manage(new Gtk::Label("Search", Gtk::ALIGN_LEFT)),
		Gtk::PACK_SHRINK
	);
	Gtk::ScrolledWindow* search_scroller = manage(new Gtk::ScrolledWindow);
	search_vbox->pack_start(*search_scroller);
	search_scroller->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	search_scroller->set_shadow_type(Gtk::SHADOW_IN);
	search_scroller->set_size_request(-1, 43);
	search_scroller->add(m_searchCtrl);
	m_searchCtrl.set_accepts_tab(false);
	
	Gtk::VBox* replace_vbox = manage(new Gtk::VBox);
	table->attach(
		*replace_vbox, 0, 2, 6, 7,
		Gtk::EXPAND|Gtk::FILL, Gtk::EXPAND|Gtk::FILL
	);
	Gtk::HBox* replace_hbox = manage(new Gtk::HBox(false, 8));
	replace_vbox->pack_start(*replace_hbox, Gtk::PACK_SHRINK);
	replace_hbox->pack_start(*manage(new Gtk::Label("Replace", Gtk::ALIGN_LEFT)));
	replace_hbox->pack_start(m_plainTextRB, Gtk::PACK_SHRINK);
	replace_hbox->pack_start(m_expressionRB, Gtk::PACK_SHRINK);
	replace_hbox->pack_start(m_javaScriptRB, Gtk::PACK_SHRINK);
	Gtk::ScrolledWindow* replace_scroller = manage(new Gtk::ScrolledWindow);
	replace_vbox->pack_start(*replace_scroller);
	replace_scroller->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	replace_scroller->set_shadow_type(Gtk::SHADOW_IN);
	replace_scroller->set_size_request(-1, 60);
	replace_scroller->add(m_replaceCtrl);
	
	Gtk::Alignment* buttons_align = manage(new Gtk::Alignment(0.5, 0.5, 0.0, 0.0));
	pack_start(*buttons_align, Gtk::PACK_SHRINK);
	buttons_align->set_padding(4, 0, 0, 0);
	Gtk::HButtonBox* bbox = manage(new Gtk::HButtonBox);
	buttons_align->add(*bbox);
	bbox->set_spacing(10);
	bbox->pack_start(m_resetButton);
	bbox->pack_start(m_saveButton);
	m_resetButton.set_sensitive(false);
	// The Reset button is disabled while the Save button is enabled
	// because that's how it's supposed to be with operation == CREATING
	// and modified == true.

	Pango::FontDescription monospaced("monospace");
	m_contentTypeCtrl.modify_font(monospaced);
	m_urlCtrl.modify_font(monospaced);
	m_searchCtrl.modify_font(monospaced);
	m_replaceCtrl.modify_font(monospaced);
	
	m_resetButton.signal_clicked().connect(
		sigc::mem_fun(m_rOwner, &FilterConfigWindow::onReset)
	);
	m_saveButton.signal_clicked().connect(
		sigc::mem_fun(m_rOwner, &FilterConfigWindow::onSave)
	);
	
	sigc::slot<void> mod_slot(sigc::mem_fun(*this, &FilterPanel::onModified));
	m_orderCtrl.signal_changed().connect(mod_slot);
	m_matchCountLimitCtrl.signal_changed().connect(mod_slot);
	m_contentTypeCtrl.signal_changed().connect(mod_slot);
	m_urlCtrl.signal_changed().connect(mod_slot);
	m_searchCtrl.get_buffer()->signal_changed().connect(mod_slot);
	m_replaceCtrl.get_buffer()->signal_changed().connect(mod_slot);
	m_plainTextRB.signal_toggled().connect(mod_slot);
	m_expressionRB.signal_toggled().connect(mod_slot);
	m_javaScriptRB.signal_toggled().connect(mod_slot);
	m_ifFlagCtrl.signal_changed().connect(mod_slot);
	m_setFlagCtrl.signal_changed().connect(mod_slot);
	m_clearFlagCtrl.signal_changed().connect(mod_slot);
	
	show_all_children();
}

FilterConfigWindow::FilterPanel::~FilterPanel()
{
}

void
FilterConfigWindow::FilterPanel::setOperation(Operation op)
{
	if (m_operation == op) {
		return;	
	}
	m_operation = op;
	m_resetButton.set_sensitive(op == EDITING && m_isModified);
}

void
FilterConfigWindow::FilterPanel::setModified(bool modified)
{
	if (m_isModified == modified) {
		return;	
	}
	m_isModified = modified;
	m_resetButton.set_sensitive(m_isModified && m_operation == EDITING);
	m_saveButton.set_sensitive(m_isModified);
}

void
FilterConfigWindow::FilterPanel::load(EditState const& state)
{
	typedef RegexFilterDescriptor RFD;
	ScopedIncDec<int> blocker(m_modifyEventsBlocked);
	m_orderCtrl.set_text(state.order);
	m_matchCountLimitCtrl.set_text(state.matchCountLimit);
	m_contentTypeCtrl.set_text(state.contentType);
	m_urlCtrl.set_text(state.url);
	m_searchCtrl.get_buffer()->set_text(state.search);
	m_replaceCtrl.get_buffer()->set_text(state.replacement);
	m_plainTextRB.set_active(state.replacementType == RFD::TEXT);
	m_expressionRB.set_active(state.replacementType == RFD::EXPRESSION);
	m_javaScriptRB.set_active(state.replacementType == RFD::JS);
	m_ifFlagCtrl.set_text(state.ifFlag);
	m_setFlagCtrl.set_text(state.setFlag);
	m_clearFlagCtrl.set_text(state.clearFlag);
	setOperation(state.operation);
	setModified(state.isModified);
}

void
FilterConfigWindow::FilterPanel::save(EditState& state)
{
	typedef RegexFilterDescriptor RFD;
	state.operation = m_operation;
	state.isModified = m_isModified;
	state.order = m_orderCtrl.get_text();
	state.matchCountLimit = m_matchCountLimitCtrl.get_text();
	state.contentType = m_contentTypeCtrl.get_text();
	state.url = m_urlCtrl.get_text();
	state.search = m_searchCtrl.get_buffer()->get_text();
	state.replacement = m_replaceCtrl.get_buffer()->get_text();
	if (m_javaScriptRB.get_active()) {
		state.replacementType = RFD::JS;
	} else if (m_expressionRB.get_active()) {
		state.replacementType = RFD::EXPRESSION;
	} else {
		state.replacementType = RFD::TEXT;
	}
	state.ifFlag = m_ifFlagCtrl.get_text();
	state.setFlag = m_setFlagCtrl.get_text();
	state.clearFlag = m_clearFlagCtrl.get_text();
}

bool
FilterConfigWindow::FilterPanel::validateAndApply(
	RegexFilterDescriptor& filter)
{
	namespace rc = boost::regex_constants;
	boost::regex::flag_type const regex_flags =
		rc::normal|rc::icase|rc::optimize;
	char const* field = 0;
	Gtk::Widget* widget = 0;
	
	EditState state;
	save(state);
	
	try {
		field = "Order";
		widget = &m_orderCtrl;
		if (state.order.empty()) {
			filter.order() = 0;
		} else {
			filter.order() = boost::lexical_cast<int>(state.order.c_str());
		}
		
		field = "Max matches";
		widget = &m_matchCountLimitCtrl;
		if (state.matchCountLimit.empty()) {
			filter.matchCountLimit() = -1;
		} else {
			filter.matchCountLimit() = boost::lexical_cast<int>(state.matchCountLimit.c_str());
		}
		
		field = "Content type";
		widget = &m_contentTypeCtrl;
		if (state.contentType.empty()) {
			filter.contentTypePattern().reset(0);
		} else {
			IntrusivePtr<TextPattern> pattern(new TextPattern(
				Glib::locale_from_utf8(state.contentType),
				regex_flags
			));
			filter.contentTypePattern() = pattern;
		}
		
		field = "URL";
		widget = &m_urlCtrl;
		if (state.url.empty()) {
			filter.urlPattern().reset(0);
		} else {
			IntrusivePtr<TextPattern> pattern(new TextPattern(
				Glib::locale_from_utf8(state.url), regex_flags
			));
			filter.urlPattern() = pattern;
		}
		
		field = "Search";
		widget = &m_searchCtrl;
		if (state.search.empty()) {
			m_rOwner.showError("\"Search\" can't be empty.");
			m_searchCtrl.grab_focus();
			return false;
		} else {
			IntrusivePtr<TextPattern> pattern(new TextPattern(
				Glib::locale_from_utf8(state.search),
				regex_flags
			));
			filter.searchPattern() = pattern;
		}
		
		field = "Replace";
		widget = &m_replaceCtrl;
		filter.replacement().reset(new string(
			Glib::locale_from_utf8(state.replacement)
		));
		
		filter.replacementType() = state.replacementType;
		
		field = "If";
		widget = &m_ifFlagCtrl;
		filter.ifFlag() = Glib::locale_from_utf8(state.ifFlag);
		
		field = "Set";
		widget = &m_setFlagCtrl;
		filter.setFlag() = Glib::locale_from_utf8(state.setFlag);
		
		field = "Clear";
		widget = &m_clearFlagCtrl;
		filter.clearFlag() = Glib::locale_from_utf8(state.clearFlag);
	} catch (Glib::ConvertError& e) {
		m_rOwner.showError(
			"The value for \""+Glib::ustring(field)+"\" "
			"could not be converted to your locale."
		);
		widget->grab_focus();
		return false;
	} catch (boost::bad_lexical_cast& e) {
		m_rOwner.showError(
			"Illegal value for \""+Glib::ustring(field)+"\"."
		);
		widget->grab_focus();
		return false;
	} catch (boost::bad_expression& e) {
		m_rOwner.showError(
			"Illegal value for \""+Glib::ustring(field)+"\".\n"
			"Bad regex: "+Application::localeToUtf8(e.what())
		);
		widget->grab_focus();
		return false;
	}
	return true;
}

void
FilterConfigWindow::FilterPanel::onModified()
{
	if (m_isModified || m_modifyEventsBlocked) {
		return;
	}
	
	setModified(true);
	m_rOwner.markCurrentFilterAsModified();
}
