/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AdvancedConfigWindow.h"
#include "Application.h"
#include "OperationLog.h"
#include "LogWidget.h"
#include "Conf.h"
#include "ConfigFile.h"
#include "UrlPatterns.h"
#include "UrlPatternsFile.h"
#include "SharedPtr.h"
#include "GlobalState.h"
#include "EffectiveFileTimestamps.h"
#include <ace/OS_NS_sys_stat.h>
#include <wx/string.h>
#include <wx/button.h>
#include <wx/notebook.h>
#include <wx/panel.h>
#include <wx/textctrl.h>
#include <wx/statusbr.h>
#include <wx/event.h>
#include <wx/icon.h>
#include <wx/font.h>
#include <wx/accel.h>
#include <memory>
#include <string>
#include <stddef.h>
#include <time.h>

using namespace std;

class AdvancedConfigWindow::Button : public wxButton
{
public:
	Button(wxWindow* parent, wxWindowID id, const wxString& label = wxEmptyString)
	: wxButton(parent, id, label), m_isBold(false) {}
	
	void setBold(bool bold = true);
private:
	bool m_isBold;
};


class AdvancedConfigWindow::TextControl : public wxTextCtrl
{
public:
	TextControl(
		wxWindow* parent, wxWindowID id,
		wxString const& value = wxEmptyString,
		wxPoint const& pos = wxDefaultPosition,
		wxSize const& size = wxDefaultSize,
		long style = 0);
	
	~TextControl();
	
	void notifyTextModified();
private:
	void onTextChange(wxCommandEvent& evt);
	
	void onChar(wxKeyEvent& evt);
	
	void onClick(wxMouseEvent& evt);
	
	void notifyCaretMove();
	
	DECLARE_EVENT_TABLE()
};


class AdvancedConfigWindow::Page : public wxPanel
{
public:
	Page(AdvancedConfigWindow* window, bool readonly,
		wxNotebook* notebook, size_t notebook_page);
	
	~Page();
	
	bool isReadOnly() const { return m_isReadOnly; }
	
	bool isPageSelected() const;
	
	void getCaretPosition(long* line, long* col);
	
	std::string getText() const;
	
	void setText(wxString const& text);
	
	void focusText() { m_pTextControl->SetFocus(); }
	
	bool isTextModified() const { return m_pTextControl->IsModified(); }
	
	void setTextUnmodified() { m_pTextControl->DiscardEdits(); }
	
	bool isModified() const { return m_isModified; }
	
	time_t getFileLoadTimestamp() const { return m_fileLoadTimestamp; }
	
	void setFileLoadTimestamp(time_t val) { m_fileLoadTimestamp = val; }
	
	time_t getFileCurrentTimestamp() const { return m_fileCurrentTimestamp; }

	void setFileCurrentTimestamp(time_t val) { m_fileCurrentTimestamp = val; }
	
	void unselectTextOnce();
	
	void updateState();
	
	bool updateFileCurrentTimestamp(); // returns true if changed
	
	virtual wxString const& getTabName() const = 0;
	
	virtual time_t getFileEffectiveTimestamp() const { return 0; }
	
	virtual time_t readFileCurrentTimestamp() const { return 0; }
	
	virtual void loadFile() = 0;
	
	virtual void onReload() {}
	
	virtual void onSave() {}
private:
	void onTextModified(wxCommandEvent& evt);
	
	AdvancedConfigWindow* const m_pWindow;
	bool const m_isReadOnly;
	wxNotebook* const m_pNotebook;
	size_t const m_notebookPage;
	TextControl* m_pTextControl;
	State m_state;
	bool m_textHasBeenUnselected;
	time_t m_fileLoadTimestamp;
	time_t m_fileCurrentTimestamp;
	bool m_isModified;
	
	DECLARE_EVENT_TABLE()
};


class AdvancedConfigWindow::ConfigPage : public AdvancedConfigWindow::Page
{
public:
	ConfigPage(
		AdvancedConfigWindow* window, bool readonly,
		wxNotebook* notebook, size_t notebook_page);
	
	~ConfigPage();
	
	virtual wxString const& getTabName() const;
	
	virtual time_t getFileEffectiveTimestamp() const;
	
	virtual time_t readFileCurrentTimestamp() const;
	
	virtual void loadFile();
	
	virtual void onReload();
	
	virtual void onSave();
private:
	bool restoreOldConfig(Config const& config);
	
	wxString m_unmodifiedTabName;
	wxString m_modifiedTabName;
};


class AdvancedConfigWindow::ConfigDefaultPage : public AdvancedConfigWindow::Page
{
public:
	ConfigDefaultPage(
		AdvancedConfigWindow* window, bool readonly,
		wxNotebook* notebook, size_t notebook_page);
	
	~ConfigDefaultPage();
	
	virtual wxString const& getTabName() const;
	
	virtual void loadFile();
private:
	wxString m_tabName;
};


class AdvancedConfigWindow::UrlsPage : public AdvancedConfigWindow::Page
{
public:
	UrlsPage(
		AdvancedConfigWindow* window, bool readonly,
		wxNotebook* notebook, size_t notebook_page);
	
	~UrlsPage();
	
	virtual wxString const& getTabName() const;
	
	virtual void loadFile();
private:
	wxString m_tabName;
};


class AdvancedConfigWindow::UrlsLocalPage : public AdvancedConfigWindow::Page
{
public:
	UrlsLocalPage(
		AdvancedConfigWindow* window, bool readonly,
		wxNotebook* notebook, size_t notebook_page);
	
	~UrlsLocalPage();
	
	virtual wxString const& getTabName() const;
	
	virtual time_t getFileEffectiveTimestamp() const;
	
	virtual time_t readFileCurrentTimestamp() const;
	
	virtual void loadFile();
	
	virtual void onReload();
	
	virtual void onSave();
private:
	wxString m_unmodifiedTabName;
	wxString m_modifiedTabName;
};


enum {
	ID_LOG_CLEAR_BTN = wxID_HIGHEST + 1
};

BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_LOCAL_EVENT_TYPE(myEVT_CARET_MOVE, 0)
	DECLARE_LOCAL_EVENT_TYPE(myEVT_TEXT_MODIFIED, 1)
END_DECLARE_EVENT_TYPES()
DEFINE_EVENT_TYPE(myEVT_CARET_MOVE)
DEFINE_EVENT_TYPE(myEVT_TEXT_MODIFIED)

BEGIN_EVENT_TABLE(AdvancedConfigWindow, wxFrame)
	EVT_ACTIVATE(onWindowActivate)
	EVT_CLOSE(onWindowClose)
	EVT_NOTEBOOK_PAGE_CHANGED(wxID_ANY, onTabChanged)
	EVT_BUTTON(wxID_BACKWARD, onReload)
	EVT_BUTTON(wxID_FORWARD, onSave)
	EVT_MENU(wxID_FORWARD, onSave)
	EVT_BUTTON(ID_LOG_CLEAR_BTN, onClearLog)
	EVT_COMMAND(wxID_ANY, myEVT_CARET_MOVE, onCaretMove)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(AdvancedConfigWindow::Page, wxPanel)
	EVT_COMMAND(wxID_ANY, myEVT_TEXT_MODIFIED, onTextModified)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(AdvancedConfigWindow::TextControl, wxTextCtrl)
	EVT_TEXT(wxID_ANY, onTextChange)
	EVT_CHAR(onChar)
	EVT_LEFT_DOWN(onClick)
	EVT_LEFT_UP(onClick)
	EVT_MIDDLE_DOWN(onClick)
	EVT_MIDDLE_UP(onClick)
	EVT_RIGHT_DOWN(onClick)
	EVT_RIGHT_UP(onClick)
END_EVENT_TABLE()


AdvancedConfigWindow* AdvancedConfigWindow::m_spInstance = 0;


AdvancedConfigWindow::AdvancedConfigWindow()
:	wxFrame(wxGetApp().GetTopWindow(), -1, _T("Advanced Configuration")),
	AbstractLogView(*OperationLog::instance()),
	m_isConstructionFinished(false),
	m_pReloadButton(0),
	m_pSaveButton(0),
	m_pLogWidget(0),
	m_caretLine(-1),
	m_caretCol(-1)
{
	SetExtraStyle(wxWS_EX_BLOCK_EVENTS);
	SetIcon(wxIcon(_T("AppIcon"), wxBITMAP_TYPE_ICO_RESOURCE, 16, 16));
	
	wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	wxPanel *panel = new wxPanel(this, -1);
	topsizer->Add(panel, 1, wxEXPAND, 0);
	wxBoxSizer *panelsizer = new wxBoxSizer(wxVERTICAL);
	panel->SetSizer(panelsizer);
	
	m_pNotebook = new wxNotebook(panel, -1);
	panelsizer->Add(m_pNotebook, 1, wxEXPAND, 0);
	Page* config_page = new ConfigPage(this, false, m_pNotebook, 0);
	m_pNotebook->AddPage(config_page, config_page->getTabName());
	Page* config_default_page = new ConfigDefaultPage(this, true, m_pNotebook, 1);
	m_pNotebook->AddPage(config_default_page, config_default_page->getTabName());
	Page* urls_page = new UrlsPage(this, true, m_pNotebook, 2);
	m_pNotebook->AddPage(urls_page, urls_page->getTabName());
	Page* urls_local_page = new UrlsLocalPage(this, false, m_pNotebook, 3);
	m_pNotebook->AddPage(urls_local_page, urls_local_page->getTabName());
	
	wxBoxSizer* buttonsizer = new wxBoxSizer(wxHORIZONTAL);
	panelsizer->Add(buttonsizer, 0, wxEXPAND);
	
	wxBoxSizer* clearlog_sizer = new wxBoxSizer(wxHORIZONTAL);
	buttonsizer->Add(clearlog_sizer, 1, wxTOP|wxLEFT|wxRIGHT, 5);
	clearlog_sizer->Add(
		new wxButton(panel, ID_LOG_CLEAR_BTN,
		_T("Clear Log")), 0, wxALIGN_LEFT
	);
	
	m_pReloadButton = new Button(panel, wxID_BACKWARD, _T("Reload"));
	buttonsizer->Add(m_pReloadButton, 0, wxTOP|wxLEFT|wxRIGHT, 5);
	m_pReloadButton->Disable();
	m_pSaveButton = new Button(panel, wxID_FORWARD, _T("Save"));
	buttonsizer->Add(m_pSaveButton, 0, wxTOP|wxLEFT|wxRIGHT, 5);
	m_pSaveButton->Disable();
	
	buttonsizer->Add(1, 1, 1);
	
	wxStaticBoxSizer* logsizer = new wxStaticBoxSizer(
		new wxStaticBox(panel, -1, _T("Log")), wxVERTICAL
	);
	panelsizer->Add(logsizer, 0, wxEXPAND);
	m_pLogWidget = new LogWidget(
		OperationLog::instance(),
		panel, -1, wxEmptyString,
		wxDefaultPosition, wxSize(500,100)
	);
	logsizer->Add(m_pLogWidget, 1, wxEXPAND);
	
	wxStatusBar *sbar = new wxStatusBar(this, -1);
	sbar->SetFieldsCount(3);
	static const int widths[] = {-1, 53, 63};
	sbar->SetStatusWidths(3, widths);
	SetStatusBar(sbar);
	
	config_page->loadFile();
	config_default_page->loadFile();
	urls_page->loadFile();
	urls_local_page->loadFile();
	
	updateLineColInfo(config_page);
	
	panelsizer->SetSizeHints(panel);
	topsizer->SetSizeHints(this);
	
	wxAcceleratorEntry save_accel(wxACCEL_CTRL, 'S', wxID_FORWARD);
	wxAcceleratorTable accel_tbl(1, &save_accel);
	SetAcceleratorTable(accel_tbl);
	
	m_isConstructionFinished = true;
}

AdvancedConfigWindow::~AdvancedConfigWindow()
{
}

void
AdvancedConfigWindow::show()
{
	if (!m_spInstance) {
		m_spInstance = new AdvancedConfigWindow;
		m_spInstance->Show();
	} else {
		m_spInstance->Show();
		m_spInstance->Raise();
	}
}

void
AdvancedConfigWindow::onWindowClose(wxCloseEvent& evt)
{
	if (evt.CanVeto()) {
		if (!prepareForWindowDestruction()) {
			evt.Veto();
			return;
		}
	}
	
	if (m_spInstance == this) {
		m_spInstance = 0;
	}
	Destroy();
}

bool
AdvancedConfigWindow::prepareForWindowDestruction()
{
	if (!haveModifiedPages()) {
		return true;
	}
	
	int res = wxMessageBox(
		_T("Are you sure you want to close this window?\n")
		_T("Any unsaved changes will be lost."),
		_T("Question"),
		wxYES_NO, this
	);
	return (res == wxYES);
}

void
AdvancedConfigWindow::onWindowActivate(wxActivateEvent& evt)
{
	bool const active = evt.GetActive();
	AbstractLogView::reportVisibility(active);
	
	if (active) {
		for (int i = m_pNotebook->GetPageCount() - 1; i >= 0; --i) {
			Page* page = getPage(i);
			if (page->updateFileCurrentTimestamp()) {
				page->updateState();
			}
		}
	}
}

AdvancedConfigWindow::Page*
AdvancedConfigWindow::getPage(int idx)
{
	void* page = m_pNotebook->GetPage(idx)->GetClientData();
	return reinterpret_cast<Page*>(page);
}

AdvancedConfigWindow::Page const*
AdvancedConfigWindow::getPage(int idx) const
{
	void const* page = m_pNotebook->GetPage(idx)->GetClientData();
	return reinterpret_cast<Page const*>(page);
}

AdvancedConfigWindow::Page*
AdvancedConfigWindow::getCurrentPage()
{
	int selection = m_pNotebook->GetSelection();
	return getPage(m_pNotebook->GetSelection());
}

bool
AdvancedConfigWindow::haveModifiedPages() const
{
	for (int i = m_pNotebook->GetPageCount() - 1; i >= 0; --i) {
		if (getPage(i)->isTextModified()) {
			return true;
		}
	}
	return false;
}

void
AdvancedConfigWindow::updateLineColInfo(Page* page)
{
	long line = 0, col = 0;
	page->getCaretPosition(&line, &col);
	if (line != m_caretLine) {
		m_caretLine = line;
		wxString text(_T("Line: "));
		text << line;
		GetStatusBar()->SetStatusText(text, 1);
	}
	if (col != m_caretCol) {
		m_caretCol = col;
		wxString text(_T("Col: "));
		text << col;
		GetStatusBar()->SetStatusText(text, 2);
	}
}

void
AdvancedConfigWindow::onTabChanged(wxNotebookEvent& evt)
{
	int selection = evt.GetSelection();
	Page* page = getPage(selection);
	page->unselectTextOnce();
	onTabChanged(page);
}

void
AdvancedConfigWindow::onTabChanged(Page* page) {
	if (!m_isConstructionFinished) {
		return;
	}
	updateLineColInfo(page);
	page->updateState();
}

void
AdvancedConfigWindow::onCaretMove(wxCommandEvent& evt)
{
	updateLineColInfo(getCurrentPage());
}

void
AdvancedConfigWindow::onReload(wxCommandEvent& evt)
{
	Page* page = getCurrentPage();
	page->onReload();
	page->focusText();
}

void
AdvancedConfigWindow::onSave(wxCommandEvent& evt)
{
	// we can also be called from a Ctrl+S accelerator
	if (m_pSaveButton->IsEnabled()) {
		Page* page = getCurrentPage();
		page->onSave();
		page->focusText();
	}
}

void
AdvancedConfigWindow::onClearLog(wxCommandEvent& evt)
{
	OperationLog::instance()->clear();
}


/*================ AdvancedConfigWindow::TextControl =====================*/

AdvancedConfigWindow::TextControl::TextControl(
	wxWindow* parent, wxWindowID id,
	wxString const& value, wxPoint const& pos,
	wxSize const& size, long style)
:	wxTextCtrl(parent, id, value, pos, size, style)
{
}

AdvancedConfigWindow::TextControl::~TextControl()
{
}

void
AdvancedConfigWindow::TextControl::onTextChange(wxCommandEvent& evt)
{
	notifyTextModified();
	notifyCaretMove();
}

void
AdvancedConfigWindow::TextControl::onChar(wxKeyEvent& evt)
{
	evt.Skip();
	notifyCaretMove();
}

void
AdvancedConfigWindow::TextControl::onClick(wxMouseEvent& evt)
{
	evt.Skip();
	notifyCaretMove();
}

void
AdvancedConfigWindow::TextControl::notifyCaretMove()
{
	AddPendingEvent(wxCommandEvent(myEVT_CARET_MOVE));
}

void
AdvancedConfigWindow::TextControl::notifyTextModified()
{
	AddPendingEvent(wxCommandEvent(myEVT_TEXT_MODIFIED));
}


/*================== AdvancedConfigWindow::Button =======================*/

void
AdvancedConfigWindow::Button::setBold(bool bold)
{
	if (m_isBold != bold) {
		m_isBold = bold;
		wxFont font = GetFont();
		font.SetWeight(bold ? wxFONTWEIGHT_BOLD : wxFONTWEIGHT_NORMAL);
		SetFont(font);
	}
}


/*=================== AdvancedConfigWindow::Page ========================*/

AdvancedConfigWindow::Page::Page(
	AdvancedConfigWindow* window, bool readonly,
	wxNotebook* notebook, size_t notebook_page)
:	wxPanel(notebook),
	m_pWindow(window),
	m_isReadOnly(readonly),
	m_pNotebook(notebook),
	m_notebookPage(notebook_page),
	m_pTextControl(0),
	m_textHasBeenUnselected(false),
	m_fileLoadTimestamp(0),
	m_fileCurrentTimestamp(0),
	m_isModified(false)
{
	SetClientData(this);
	wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	m_pTextControl = new TextControl(
		this, -1, wxEmptyString, wxDefaultPosition, wxSize(500, 200),
		wxTE_PROCESS_TAB|wxTE_MULTILINE|wxTE_DONTWRAP|
		(readonly ? wxTE_READONLY : 0)
	);
	topsizer->Add(m_pTextControl, 1, wxEXPAND);
	
	topsizer->SetSizeHints(this);
}

AdvancedConfigWindow::Page::~Page()
{
}

void
AdvancedConfigWindow::Page::getCaretPosition(long* line, long* col)
{
	long ins_pos = m_pTextControl->GetInsertionPoint();
	long x = 0, y = 0;
	m_pTextControl->PositionToXY(ins_pos, &x, &y);
	*line = y + 1;
	*col = x + 1;
}

bool
AdvancedConfigWindow::Page::isPageSelected() const
{
	return m_pNotebook->GetSelection() == m_notebookPage;
}

std::string
AdvancedConfigWindow::Page::getText() const
{
	std::string str(m_pTextControl->GetValue().mb_str());
	return str;
}

void
AdvancedConfigWindow::Page::setText(wxString const& text)
{
	m_pTextControl->SetValue(text);
	m_pTextControl->notifyTextModified();
}

void
AdvancedConfigWindow::Page::unselectTextOnce()
{
	// workaround a bug in wxWidgets
	if (!m_textHasBeenUnselected) {
		m_pTextControl->SetSelection(0, 0);
		m_textHasBeenUnselected = true;
	}
}

void
AdvancedConfigWindow::Page::updateState()
{
	bool const text_modified = isTextModified();
	bool const file_is_newer =
		(m_fileCurrentTimestamp != m_fileLoadTimestamp);
	bool const file_not_effective =
		(getFileEffectiveTimestamp() != m_fileLoadTimestamp);
	
	bool const was_modified = m_isModified;
	m_isModified = text_modified || file_not_effective;
	if (was_modified != m_isModified) {
		m_pNotebook->SetPageText(m_notebookPage, getTabName());
	}
	
	if (isPageSelected()) {
		Button* reload_btn = m_pWindow->getReloadButton();
		Button* save_btn = m_pWindow->getSaveButton();
		if (!m_isReadOnly) {
			reload_btn->Enable(text_modified || file_is_newer);
			reload_btn->setBold(file_is_newer);
			save_btn->Enable(m_isModified);
		}
		reload_btn->Show(!m_isReadOnly);
		save_btn->Show(!m_isReadOnly);
	}
}

bool
AdvancedConfigWindow::Page::updateFileCurrentTimestamp()
{
	time_t const new_tstamp = readFileCurrentTimestamp();
	time_t const old_tstamp = getFileCurrentTimestamp();
	setFileCurrentTimestamp(new_tstamp);
	return (old_tstamp != new_tstamp);
}

void
AdvancedConfigWindow::Page::onTextModified(wxCommandEvent& evt)
{
	updateState();
}


/*================ AdvancedConfigWindow::ConfigPage ====================*/

AdvancedConfigWindow::ConfigPage::ConfigPage(
	AdvancedConfigWindow* window, bool readonly,
	wxNotebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_unmodifiedTabName(_T("config")),
	m_modifiedTabName(_T("config *"))
{
}

AdvancedConfigWindow::ConfigPage::~ConfigPage()
{
}

wxString const&
AdvancedConfigWindow::ConfigPage::getTabName() const
{
	return isModified() ? m_modifiedTabName : m_unmodifiedTabName;
}

time_t
AdvancedConfigWindow::ConfigPage::getFileEffectiveTimestamp() const
{
	return EffectiveFileTimestamps::config;
}

time_t
AdvancedConfigWindow::ConfigPage::readFileCurrentTimestamp() const
{
	Application& app = wxGetApp();
	ACE_stat st;
	wxString const path = app.getConfigFileName().GetFullPath();
	if (ACE_OS::stat(path.c_str(), &st) != -1) {
		return st.st_mtime;
	}
	return 0;
}

void
AdvancedConfigWindow::ConfigPage::loadFile()
{
	Application& app = wxGetApp();
	string content;
	time_t mtime = 0;
	app.readFile(app.getConfigFileName(), content, &mtime);
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	setText(content.c_str());
}

void
AdvancedConfigWindow::ConfigPage::onReload()
{
	loadFile();
}

void
AdvancedConfigWindow::ConfigPage::onSave()
{
	Application& app = wxGetApp();
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Applying new config ... "));
	size_t num_records = log->getNumRecords();
	
	string const text = getText();
	Config new_config;
	ConfigFile new_config_file;
	
	if (!app.processConfig(text, new_config, new_config_file)) {
		return;
	}
	
	Config old_config(GlobalState::ReadAccessor()->config());
	
	if (!app.applyConfig(new_config)) {
		restoreOldConfig(old_config);
		return;
	}
	
	time_t mtime = 0;
	if (!app.writeFile(app.getConfigFileName(), text, &mtime)) {
		restoreOldConfig(old_config);
		return;
	}
	
	app.configFile().swap(new_config_file);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	EffectiveFileTimestamps::config = mtime;
	setTextUnmodified();
	updateState();
}

bool
AdvancedConfigWindow::ConfigPage::restoreOldConfig(Config const& config)
{
	Application& app = wxGetApp();
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Restoring the old config ... "));
	size_t num_records = log->getNumRecords();
	
	if (!app.applyConfig(config)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	return true;
}


/*============= AdvancedConfigWindow::ConfigDefaultPage =================*/

AdvancedConfigWindow::ConfigDefaultPage::ConfigDefaultPage(
	AdvancedConfigWindow* window, bool readonly,
	wxNotebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_tabName(_T("config.default"))
{
}

AdvancedConfigWindow::ConfigDefaultPage::~ConfigDefaultPage()
{
}

wxString const&
AdvancedConfigWindow::ConfigDefaultPage::getTabName() const
{
	return m_tabName;
}

void
AdvancedConfigWindow::ConfigDefaultPage::loadFile()
{
	Application& app = wxGetApp();
	string content;
	app.readFile(app.getConfigDefaultFileName(), content);
	setText(content.c_str());
}


/*================= AdvancedConfigWindow::UrlsPage ===================*/

AdvancedConfigWindow::UrlsPage::UrlsPage(
	AdvancedConfigWindow* window, bool readonly,
	wxNotebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_tabName(_T("urls"))
{
}

AdvancedConfigWindow::UrlsPage::~UrlsPage()
{
}

wxString const&
AdvancedConfigWindow::UrlsPage::getTabName() const
{
	return m_tabName;
}

void
AdvancedConfigWindow::UrlsPage::loadFile()
{
	Application& app = wxGetApp();
	string content;
	app.readFile(app.getStandardUrlPatternsFileName(), content);
	setText(content.c_str());
}


/*=============== AdvancedConfigWindow::UrlsLocalPage ===================*/

AdvancedConfigWindow::UrlsLocalPage::UrlsLocalPage(
	AdvancedConfigWindow* window, bool readonly,
	wxNotebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_unmodifiedTabName(_T("urls.local")),
	m_modifiedTabName(_T("urls.local *"))
{
}

AdvancedConfigWindow::UrlsLocalPage::~UrlsLocalPage()
{
}

wxString const&
AdvancedConfigWindow::UrlsLocalPage::getTabName() const
{
	return isModified() ? m_modifiedTabName : m_unmodifiedTabName;
}

time_t
AdvancedConfigWindow::UrlsLocalPage::getFileEffectiveTimestamp() const
{
	return EffectiveFileTimestamps::urls_local;
}

time_t
AdvancedConfigWindow::UrlsLocalPage::readFileCurrentTimestamp() const
{
	Application& app = wxGetApp();
	ACE_stat st;
	wxString const path = app.getLocalUrlPatternsFileName().GetFullPath();
	if (ACE_OS::stat(path.c_str(), &st) != -1) {
		return st.st_mtime;
	}
	return 0;
}

void
AdvancedConfigWindow::UrlsLocalPage::loadFile()
{
	Application& app = wxGetApp();
	string content;
	time_t mtime = 0;
	app.readFile(app.getLocalUrlPatternsFileName(), content, &mtime);
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	setText(content.c_str());
}

void
AdvancedConfigWindow::UrlsLocalPage::onReload()
{
	loadFile();
}

void
AdvancedConfigWindow::UrlsLocalPage::onSave()
{
	Application& app = wxGetApp();
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Applying new local url patterns ... "));
	size_t num_records = log->getNumRecords();
	
	string const text = getText();
	UrlPatterns new_patterns;
	UrlPatternsFile new_patterns_file;
	
	if (!app.processUrlPatterns(text, _T("urls.local"),
		                    new_patterns, new_patterns_file)) {
		return;
	}
	
	time_t mtime = 0;
	if (!app.writeFile(app.getLocalUrlPatternsFileName(), text, &mtime)) {
		return;
	}
	
	app.applyLocalUrlPatterns(new_patterns);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	EffectiveFileTimestamps::urls_local = mtime;
	setTextUnmodified();
	updateState();
}
