/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FilterConfigWindow.h"
#include "Application.h"
#include "OperationLog.h"
#include "LogWidget.h"
#include "AutoIndentingTextCtrl.h"
#include "CheckTreeView.h"
#include "ContentFilterGroup.h"
#include "ContentFiltersFile.h"
#include "RegexFilterDescriptor.h"
#include "FilterTag.h"
#include "FilterGroupTag.h"
#include "TextPattern.h"
#include "IntrusivePtr.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include "ScopedIncDec.h"
#include <ace/Synch_Traits.h>
#include <boost/lexical_cast.hpp>
#include <boost/regex.hpp>
#include <wx/sizer.h>
#include <wx/gbsizer.h>
#include <wx/panel.h>
#include <wx/treectrl.h>
#include <wx/textctrl.h>
#include <wx/font.h>
#include <wx/imaglist.h>
#include <wx/button.h>
#include <wx/radiobut.h>
#include <wx/icon.h>
#include <wx/event.h>
#include <wx/file.h>
#include <wx/filename.h>
#include <wx/menu.h>
#include <wx/menuitem.h>
#include <wx/msgdlg.h>
#include <wx/gdicmn.h>
#include <wx/utils.h>
#include <wx/accel.h>
#include <cassert>
#include <algorithm>
#include <iterator>
#include <memory>
#include <string>

using namespace std;

// EditState represents a saved edit form.
struct FilterConfigWindow::EditState :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
	Operation operation; // CREATING or EDITING
	bool isModified; // always true for CREATING
	wxString order;
	wxString matchCountLimit;
	wxString url;
	wxString contentType;
	wxString search;
	wxString replacement;
	RegexFilterDescriptor::ReplacementType replacementType;
	wxString ifFlag;
	wxString setFlag;
	wxString clearFlag;
	
	// used when creating a filter
	EditState();
	
	// used when editing a filter
	EditState(RegexFilterDescriptor const& filter);
	
	~EditState();
};


class FilterConfigWindow::FilterTree : public CheckTreeView
{
public:
	enum EnablePolicy { NORMAL, ALL_ENABLED, ALL_DISABLED };
	
	FilterTree(FilterConfigWindow& owner, wxWindow* parent);
	
	virtual ~FilterTree();
	
	bool isFilterGroupBeingEdited(FilterGroupTag const& group_tag);
	
	bool isFileItem(wxTreeItemId const& item) const;
	
	bool isFilterItem(wxTreeItemId const& item) const;
	
	FileNode* getFileNode(wxTreeItemId const& item);
	
	FilterNode* getFilterNode(wxTreeItemId const& item);
	
	void markFilterAsModified(wxTreeItemId const& item);
	
	void markFilterAsUnmodified(wxTreeItemId const& item);
	
	bool haveModifiedItems() const;
	
	void updateFileModifiedState(wxTreeItemId const& file_item);
	
	void updateFileCheckState(wxTreeItemId const& file_item);
	
	static bool saveEnabledFilterList(
		ContentFilterGroup const& group, EnablePolicy policy = NORMAL);
	
	IntrusivePtr<RegexFilterDescriptor> findExistingFollower(
		wxTreeItemId const& filter_item);
private:
	enum { FILE_ICON = 0, FILTER_ICON = 1 };
	
	void loadFilterGroups();
	
	CheckState loadIndividualFilters(
		ContentFilters const& filters, wxTreeItemId parent_item);
	
	void onItemCheckStateChanging(CheckTreeEvent& evt);
	
	void onFileCheckStateChanging(CheckTreeEvent& evt);
	
	void onFilterCheckStateChanging(CheckTreeEvent& evt);
	
	void onItemCheckStateChanged(CheckTreeEvent& evt);
	
	void onSelChanging(wxTreeEvent& evt);
	
	void onItemMenu(wxTreeEvent& evt);
	
	void onNewFile(wxCommandEvent& evt);
	
	void onNewFilter(wxCommandEvent& evt);
	
	void onFilterUp(wxCommandEvent& evt);
	
	void onFilterDown(wxCommandEvent& evt);
	
	void onRename(wxCommandEvent& evt);
	
	void onLabelEditEnd(wxTreeEvent& evt);
	
	void onFileDelete(wxCommandEvent& evt);
	
	void onFilterDelete(wxCommandEvent& evt);
	
	void onFileNameChanged(wxCommandEvent& evt);
	
	bool createFile(wxTreeItemId const& item, wxString const& name);
	
	bool createFilter(wxTreeItemId const& item,
		wxTreeItemId const& parent, wxString const& name);
	
	bool renameFile(wxTreeItemId const& item, wxString const& new_name);
	
	bool renameFilter(wxTreeItemId const& item,
		wxTreeItemId const& parent, wxString const& new_name);
	
	void popupNoItemContextMenu(int x, int y);
	
	void popupItemContextMenu(wxTreeItemId const& item, int x, int y);
	
	void popupFileContextMenu(wxTreeItemId const& item, int x, int y);
	
	void popupFilterContextMenu(wxTreeItemId const& item, int x, int y);
	
	bool verifyFileNameValid(wxString const& fname);
	
	bool verifyFilterNameValid(wxString const& name);
	
	bool verifyFilterFileDoesntExist(wxString const& fname);
	
	bool verifyFilterDoesntExist(
		wxTreeItemId const& parent, wxString const& name);
	
	FilterConfigWindow& m_rOwner;
	wxImageList m_imageList;
	
	DECLARE_EVENT_TABLE()
};

/*
If FilterTree::GetItemData() returns null, it means the file or filter
is in the process of being created (we are editing the item's label). 
*/

struct FilterConfigWindow::FileNode : public wxTreeItemData
{
	IntrusivePtr<ContentFilterGroup> filterGroup; // always set
	
	FileNode(IntrusivePtr<ContentFilterGroup> const& group);
	
	virtual ~FileNode();
};


struct FilterConfigWindow::FilterNode : public wxTreeItemData
{
	IntrusivePtr<EditState> editState; // always set
	IntrusivePtr<RegexFilterDescriptor> filter;
	// the filter we are editing, or null if creating a filter
	
	// used when creating a new filter
	FilterNode();
	
	// used when editing an existing filter
	FilterNode(IntrusivePtr<RegexFilterDescriptor> const& filter);
	
	virtual ~FilterNode();
	
	void swap(FilterNode& other);
};


class FilterConfigWindow::FilterPanel : public wxPanel
{
public:
	FilterPanel(FilterConfigWindow& owner, wxWindow* parent);
	
	virtual ~FilterPanel();
	
	Operation getOperation() const { return m_operation; }
	
	void setOperation(Operation op);
	
	bool isModified() const { return m_isModified; }
	
	void setModified(bool modified);
	
	void load(EditState const& state);
	
	void save(EditState& state) const;
	
	bool validateAndApply(RegexFilterDescriptor& filter);
private:
	void onModify(wxCommandEvent& evt);
	
	void onSave(wxCommandEvent& evt);
	
	FilterConfigWindow& m_rOwner;
	Operation m_operation;
	bool m_isModified;
	int m_textEventsBlocked;
	wxTextCtrl* m_pOrderCtrl;
	wxTextCtrl* m_pMatchCountLimitCtrl;
	wxTextCtrl* m_pContentTypeCtrl;
	wxTextCtrl* m_pUrlCtrl;
	wxTextCtrl* m_pSearchCtrl;
	wxTextCtrl* m_pReplaceCtrl;
	wxRadioButton* m_pPlainTextRB;
	wxRadioButton* m_pExpressionRB;
	wxRadioButton* m_pJavaScriptRB;
	wxTextCtrl* m_pIfFlagCtrl;
	wxTextCtrl* m_pSetFlagCtrl;
	wxTextCtrl* m_pClearFlagCtrl;
	wxButton* m_pResetButton;
	wxButton* m_pSaveButton;
	
	DECLARE_EVENT_TABLE()
};


enum {
	ID_LOG_CLEAR_BTN,
	ID_FILTER_TREE,
	ID_PLAIN_TEXT_RB,
	ID_EXPRESSION_RB,
	ID_JAVASCRIPT_RB,
	ID_MENU_NEW_FILE,
	ID_MENU_NEW_FILTER,
	ID_MENU_RENAME,
	ID_MENU_DELETE_FILE,
	ID_MENU_DELETE_FILTER,
	ID_MENU_MOVE_UP,
	ID_MENU_MOVE_DOWN
};

BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_LOCAL_EVENT_TYPE(myEVT_FILE_NAME_CHANGED, 0)
END_DECLARE_EVENT_TYPES()
DEFINE_EVENT_TYPE(myEVT_FILE_NAME_CHANGED)

BEGIN_EVENT_TABLE(FilterConfigWindow, wxFrame)
	EVT_ACTIVATE(onWindowActivate)
	EVT_CLOSE(onWindowClose)
	EVT_BUTTON(ID_LOG_CLEAR_BTN, onClearLog)
	EVT_TREE_SEL_CHANGED(ID_FILTER_TREE, onSelChanged)
	EVT_BUTTON(wxID_BACKWARD, onReset)
	EVT_BUTTON(wxID_FORWARD, onSave)
	EVT_MENU(ID_MENU_DELETE_FILE, onFileDelete)
	EVT_MENU(ID_MENU_DELETE_FILTER, onFilterDelete)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(FilterConfigWindow::FilterTree, CheckTreeView)
	EVT_TREE_CHECK_STATE_CHANGING(ID_FILTER_TREE, onItemCheckStateChanging)
	EVT_TREE_CHECK_STATE_CHANGED(ID_FILTER_TREE, onItemCheckStateChanged)
	EVT_TREE_SEL_CHANGING(ID_FILTER_TREE, onSelChanging)
	EVT_TREE_ITEM_MENU(ID_FILTER_TREE, onItemMenu)
	EVT_TREE_END_LABEL_EDIT(ID_FILTER_TREE, onLabelEditEnd)
	EVT_COMMAND(wxID_ANY, myEVT_FILE_NAME_CHANGED, onFileNameChanged)
	EVT_MENU(ID_MENU_NEW_FILE, onNewFile)
	EVT_MENU(ID_MENU_NEW_FILTER, onNewFilter)
	EVT_MENU(ID_MENU_MOVE_UP, onFilterUp)
	EVT_MENU(ID_MENU_MOVE_DOWN, onFilterDown)
	EVT_MENU(ID_MENU_RENAME, onRename)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(FilterConfigWindow::FilterPanel, wxPanel)
	EVT_TEXT(wxID_ANY, onModify)
	EVT_RADIOBUTTON(wxID_ANY, onModify)
	EVT_MENU(wxID_FORWARD, onSave)
END_EVENT_TABLE()


FilterConfigWindow* FilterConfigWindow::m_spInstance = 0;


FilterConfigWindow::FilterConfigWindow()
:	wxFrame(wxGetApp().GetTopWindow(), -1, _T("Filter Configuration")),
	AbstractLogView(*OperationLog::instance()),
	m_pFilterTree(0),
	m_pFilterPanel(0),
	m_pLogWidget(0)
{
	SetExtraStyle(wxWS_EX_BLOCK_EVENTS);
	SetIcon(wxIcon(_T("AppIcon"), wxBITMAP_TYPE_ICO_RESOURCE, 16, 16));
	
	wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	wxPanel* panel = new wxPanel(this, -1);
	topsizer->Add(panel, 1, wxGROW);
	wxBoxSizer* panelsizer = new wxBoxSizer(wxVERTICAL);
	panel->SetSizer(panelsizer);
	
	wxBoxSizer* lr_sizer = new wxBoxSizer(wxHORIZONTAL);
	panelsizer->Add(lr_sizer, 1, wxGROW);
	
	wxBoxSizer* tree_sizer = new wxBoxSizer(wxVERTICAL);
	lr_sizer->Add(tree_sizer, 0, wxGROW);
	
	m_pFilterTree = new FilterTree(*this, panel);
	tree_sizer->Add(m_pFilterTree, 1);
	
	tree_sizer->Add(
		new wxButton(panel, ID_LOG_CLEAR_BTN, _T("Clear Log")),
		0, wxTOP|wxLEFT|wxRIGHT, 5
	);	
	
	m_pFilterPanel = new FilterPanel(*this, panel);
	lr_sizer->Add(m_pFilterPanel, 1, wxGROW|wxTOP|wxLEFT|wxRIGHT, 4);
	
	wxStaticBoxSizer* logsizer = new wxStaticBoxSizer(
		new wxStaticBox(panel, -1, _T("Log")), wxVERTICAL
	);
	panelsizer->Add(logsizer, 0, wxEXPAND);
	
	m_pLogWidget = new LogWidget(
		OperationLog::instance(),
		panel, -1, wxEmptyString,
		wxDefaultPosition, wxSize(618,100)
	);
	logsizer->Add(m_pLogWidget, 1, wxEXPAND);
	
	topsizer->SetSizeHints(this);
	m_pFilterPanel->Hide();
}

FilterConfigWindow::~FilterConfigWindow()
{
}

void
FilterConfigWindow::show()
{
	if (!m_spInstance) {
		m_spInstance = new FilterConfigWindow;
		m_spInstance->Show();
	} else {
		m_spInstance->Show();
		m_spInstance->Raise();
	}
}

bool
FilterConfigWindow::isFilterGroupBeingEdited(FilterGroupTag const& group_tag)
{
	if (!m_spInstance) {
		return false;
	}
	return m_spInstance->m_pFilterTree->isFilterGroupBeingEdited(group_tag);
}

void
FilterConfigWindow::onWindowActivate(wxActivateEvent& evt)
{
	AbstractLogView::reportVisibility(evt.GetActive());
}

void
FilterConfigWindow::onWindowClose(wxCloseEvent& evt)
{
	if (evt.CanVeto()) {
		if (!prepareForWindowDestruction()) {
			evt.Veto();
			return;
		}
	}
	
	if (m_spInstance == this) {
		m_spInstance = 0;
	}
	Destroy();
}

bool
FilterConfigWindow::prepareForWindowDestruction()
{
	if (!m_pFilterTree->haveModifiedItems()) {
		return true;
	}
	
	int res = wxMessageBox(
		_T("Are you sure you want to close this window?\n")
		_T("Any unsaved changes will be lost."),
		_T("Question"),
		wxYES_NO, this
	);
	if (res != wxYES) {
		return false;
	}
	
	return true;
}

void
FilterConfigWindow::onClearLog(wxCommandEvent& evt)
{
	OperationLog::instance()->clear();
}

void
FilterConfigWindow::onSelChanged(wxTreeEvent& evt)
{
	onItemUnselected(evt.GetOldItem());
	onItemSelected(evt.GetItem());
}

void
FilterConfigWindow::onItemUnselected(wxTreeItemId const& item)
{
	if (item) {
		if (m_pFilterTree->isFilterItem(item)) {
			FilterNode* node = m_pFilterTree->getFilterNode(item);
			if (node) {
				assert(node->editState);
				m_pFilterPanel->save(*node->editState);
			}
		}
	}
}

void
FilterConfigWindow::onItemSelected(wxTreeItemId const& item)
{
	bool show = false;
	if (item) {
		if (m_pFilterTree->isFilterItem(item)) {
			FilterNode* node = m_pFilterTree->getFilterNode(item);
			if (node) {
				assert(node->editState);
				m_pFilterPanel->load(*node->editState);
				show = true;
			}
		}
	}
	m_pFilterPanel->Show(show);
}

void
FilterConfigWindow::onReset(wxCommandEvent& evt)
{
	assert(m_pFilterPanel->getOperation() == EDITING);
	wxTreeItemId item = m_pFilterTree->GetSelection();
	assert(item);
	assert(m_pFilterTree->isFilterItem(item));
	FilterNode* node = m_pFilterTree->getFilterNode(item);
	assert(node);
	assert(node->editState);
	assert(node->editState->operation == EDITING);
	
	node->editState->isModified = false;
	m_pFilterPanel->load(*node->editState);
	m_pFilterTree->markFilterAsUnmodified(item);
}

void
FilterConfigWindow::onSave(wxCommandEvent& evt)
{
	wxTreeItemId item = m_pFilterTree->GetSelection();
	assert(item);
	assert(m_pFilterTree->isFilterItem(item));
	FilterNode* node = m_pFilterTree->getFilterNode(item);
	assert(node);
	assert(node->editState);
	wxTreeItemId parent = m_pFilterTree->GetItemParent(item);
	FileNode* file_node = m_pFilterTree->getFileNode(parent);
	assert(file_node);
	assert(file_node->filterGroup);
	ContentFilterGroup& group = *file_node->filterGroup;
	
	IntrusivePtr<RegexFilterDescriptor> new_filter;
	if (node->filter) {
		new_filter.reset(new RegexFilterDescriptor(*node->filter));
	} else {
		new_filter.reset(new RegexFilterDescriptor(
			FilterTag::create(),
			file_node->filterGroup->getTag(),
			m_pFilterTree->GetItemText(item).mb_str()
		));
	}
	
	bool const enabled =
		(m_pFilterTree->GetCheckState(item) == CheckTreeView::CB_CHECKED);
	new_filter->setEnabled(enabled);
	if (!m_pFilterPanel->validateAndApply(*new_filter)) {
		return;
	}
	
	ContentFilterGroup new_group(group);
	
	if (node->editState->operation == CREATING) {
		assert(!node->filter);
		IntrusivePtr<RegexFilterDescriptor> follower
			= m_pFilterTree->findExistingFollower(item);
		if (!follower) {
			new_group.filters().filters().push_back(new_filter);
		} else {
			typedef ContentFilters::FilterList::iterator Iter;
			Iter it = new_group.filters().find(follower);
			assert(it != new_group.filters().filters().end());
			new_group.filters().filters().insert(it, new_filter);
		}
	} else {
		assert(node->filter);
		typedef ContentFilters::FilterList::iterator Iter;
		Iter it = new_group.filters().find(node->filter);
		assert(it != new_group.filters().filters().end());
		*it = new_filter;
	}
	
	new_group.fileStructure().updateWith(new_group.filters());
	
	if (!saveFilterGroup(new_group, &group)) {
		return;
	}
	
	node->filter = new_filter;
	group.swap(new_group);
	wxGetApp().updateGlobalContentFilters();
	m_pFilterPanel->setOperation(EDITING);
	m_pFilterPanel->setModified(false);
	m_pFilterPanel->save(*node->editState);
	m_pFilterTree->markFilterAsUnmodified(item);
}

void
FilterConfigWindow::onFileDelete(wxCommandEvent& evt)
{
	wxTreeItemId item = m_pFilterTree->GetSelection();
	assert(item);
	assert(m_pFilterTree->isFileItem(item));
	FileNode* node = m_pFilterTree->getFileNode(item);
	assert(node);
	assert(node->filterGroup);
	ContentFilterGroup& group = *node->filterGroup;
	
	int res = wxMessageBox(
		_T("Really delete this file?"),
		_T("File Delete Confirmation"),
		wxYES_NO, this
	);
	if (res != wxYES) {
		return;
	}
	
	Application& app = wxGetApp();
	wxFileName filter_fname(app.getFiltersDir());
	filter_fname.SetName(group.fileName());
	wxFileName enabled_fname(filter_fname);
	enabled_fname.SetName(filter_fname.GetName() + _T(".enabled"));
	
	if (!app.deleteFile(enabled_fname)) {
		return;
	}
	if (!app.deleteFile(filter_fname)) {
		m_pFilterTree->saveEnabledFilterList(group);
		return;
	}
	
	bool removed = app.removeContentFilterGroup(node->filterGroup);
	assert(removed);
	m_pFilterTree->Delete(item);
	app.updateGlobalContentFilters();
}

void
FilterConfigWindow::onFilterDelete(wxCommandEvent& evt)
{
	wxTreeItemId item = m_pFilterTree->GetSelection();
	assert(item);
	assert(m_pFilterTree->isFilterItem(item));
	FilterNode* node = m_pFilterTree->getFilterNode(item);
	wxTreeItemId parent = m_pFilterTree->GetItemParent(item);
	FileNode* file_node = m_pFilterTree->getFileNode(parent);
	assert(file_node);
	assert(file_node->filterGroup);
	ContentFilterGroup& group = *file_node->filterGroup;
	
	int res = wxMessageBox(
		_T("Really delete this filter?"),
		_T("Filter Delete Confirmation"),
		wxYES_NO, this
	);
	if (res != wxYES) {
		return;
	}
	
	if (!node || node->editState->operation == CREATING) {
		m_pFilterTree->Delete(item);
		m_pFilterTree->updateFileCheckState(parent);
		m_pFilterTree->updateFileModifiedState(parent);
		return;
	} else {
		assert(node->filter);
	}
	
	ContentFilterGroup new_group(group);
	typedef ContentFilters::FilterList::iterator Iter;
	Iter it = new_group.filters().find(node->filter);
	assert(it != new_group.filters().filters().end());
	new_group.filters().filters().erase(it);
	new_group.fileStructure().updateWith(new_group.filters());
	
	if (!saveFilterGroup(new_group, &group)) {
		return;
	}
	
	m_pFilterTree->Delete(item);
	m_pFilterTree->updateFileCheckState(parent);
	m_pFilterTree->updateFileModifiedState(parent);
	
	group.swap(new_group);
	wxGetApp().updateGlobalContentFilters();
}

void
FilterConfigWindow::onFilterCreated(
	wxTreeItemId const& item, FilterNode& node)
{
	m_pFilterTree->markFilterAsModified(item);
	m_pFilterPanel->load(*node.editState);
	m_pFilterPanel->Show();
}

void
FilterConfigWindow::markCurrentFilterAsModified()
{
	wxTreeItemId item = m_pFilterTree->GetSelection();
	assert(item);
	assert(m_pFilterTree->isFilterItem(item));
	m_pFilterTree->markFilterAsModified(item);
}

bool
FilterConfigWindow::renameFilterGroupFile(
	ContentFilterGroup const& group, wxString const& new_name)
{
	Application& app = wxGetApp();
	wxFileName old_filter_fname = app.getFiltersDir();
	old_filter_fname.SetName(group.fileName());
	wxFileName old_enabled_fname(old_filter_fname);
	old_enabled_fname.SetName(old_filter_fname.GetName()+_T(".enabled"));
	wxFileName new_filter_fname(old_filter_fname);
	new_filter_fname.SetName(new_name);
	wxFileName new_enabled_fname(new_filter_fname);
	new_enabled_fname.SetName(new_filter_fname.GetName()+_T(".enabled"));
	
	bool const enabled_file_exists = old_enabled_fname.FileExists();
	if (enabled_file_exists) {
		if (!app.renameFile(old_enabled_fname, new_enabled_fname)) {
			return false;
		}
	}
	if (!app.renameFile(old_filter_fname, new_filter_fname)) {
		if (enabled_file_exists) {
			app.renameFile(new_enabled_fname, old_enabled_fname);
		}
		return false;
	}
	return true;
}

bool
FilterConfigWindow::saveFilterGroup(
	ContentFilterGroup const& new_group,
	ContentFilterGroup const* old_group)
{
	Application& app = wxGetApp();
	
	if (!FilterTree::saveEnabledFilterList(new_group)) {
		return false;
	}
	
	bool const skip_content = (old_group &&
		new_group.fileStructure() == old_group->fileStructure()
	);
	if (!skip_content) {
		wxFileName fname = app.getFiltersDir();
		fname.SetName(new_group.fileName());
		ostringstream strm;
		strm << new_group.fileStructure();
		if (!app.writeFile(fname, strm.str())) {
			if (old_group) {
				FilterTree::saveEnabledFilterList(*old_group);
			} else {
				wxFileName enabled_fname(fname);
				enabled_fname.SetName(fname.GetName()+_T(".enabled"));
				app.deleteFile(enabled_fname);
			}
			return false;
		}
	}
	
	return true;
}


/*===================== FilterConfigWindow::EditState =====================*/

FilterConfigWindow::EditState::EditState()
:	operation(CREATING),
	isModified(true), // always true for CREATING
	replacementType(RegexFilterDescriptor::TEXT)
{
}

FilterConfigWindow::EditState::EditState(RegexFilterDescriptor const& filter)
:	operation(EDITING),
	isModified(false)
{
	if (filter.order() != 0) {
		order << filter.order();
	}
	if (filter.matchCountLimit() != -1) {
		matchCountLimit << filter.matchCountLimit();
	}
	if (filter.urlPattern().get()) {
		url = filter.urlPattern()->source().c_str();	
	}
	if (filter.contentTypePattern().get()) {
		contentType = filter.contentTypePattern()->source().c_str();
	}
	if (filter.searchPattern().get()) {
		search = filter.searchPattern()->source().c_str();
	}
	if (filter.replacement().get()) {
		replacement = filter.replacement()->c_str();
	}
	replacementType = filter.replacementType();
	ifFlag = filter.ifFlag().c_str();
	setFlag = filter.setFlag().c_str();
	clearFlag = filter.clearFlag().c_str();
}

FilterConfigWindow::EditState::~EditState()
{	
}


/*===================== FilterConfigWindow::FilterTree ===================*/

FilterConfigWindow::FilterTree::FilterTree(
	FilterConfigWindow& owner, wxWindow* parent)
:	CheckTreeView(
		parent, ID_FILTER_TREE, wxDefaultPosition, wxSize(205, 276),
		wxTR_EDIT_LABELS|wxTR_HAS_BUTTONS|wxTR_HIDE_ROOT|
		wxTR_LINES_AT_ROOT|wxTR_SINGLE
	),
	m_rOwner(owner),
	m_imageList(16, 16, true, 2)
{
	m_imageList.Add(wxIcon("FileIcon", wxBITMAP_TYPE_ICO_RESOURCE));
	m_imageList.Add(wxIcon("FilterIcon", wxBITMAP_TYPE_ICO_RESOURCE));
	SetImageList(&m_imageList);
	
	AddRoot(wxEmptyString);
	loadFilterGroups();
}

FilterConfigWindow::FilterTree::~FilterTree()
{
	SetImageList(0);
}

bool
FilterConfigWindow::FilterTree::isFilterGroupBeingEdited(
	FilterGroupTag const& group_tag)
{
	wxTreeItemId item = GetSelection();
	if (!item) {
		return false;
	}
	
	if (isFilterItem(item)) {
		FilterNode* node = getFilterNode(item);
		return (node && node->filter && node->filter->getGroupTag() == group_tag);
	} else {
		FileNode* node = getFileNode(item);
		return (node && node->filterGroup->getTag() == group_tag);
	}
}

bool
FilterConfigWindow::FilterTree::isFileItem(wxTreeItemId const& item) const
{
	wxTreeItemId root = GetRootItem();
	return item != root && GetItemParent(item) == root;
}

bool
FilterConfigWindow::FilterTree::isFilterItem(wxTreeItemId const& item) const
{
	wxTreeItemId root = GetRootItem();
	return item != root && GetItemParent(item) != root;
}

FilterConfigWindow::FileNode*
FilterConfigWindow::FilterTree::getFileNode(wxTreeItemId const& item)
{
	return dynamic_cast<FileNode*>(GetItemData(item));
}

FilterConfigWindow::FilterNode*
FilterConfigWindow::FilterTree::getFilterNode(wxTreeItemId const& item)
{
	return dynamic_cast<FilterNode*>(GetItemData(item));
}

void
FilterConfigWindow::FilterTree::markFilterAsModified(
	wxTreeItemId const& item)
{
	assert(isFilterItem(item));
	SetItemBold(item);
	SetItemBold(GetItemParent(item));
}

void
FilterConfigWindow::FilterTree::markFilterAsUnmodified(
	wxTreeItemId const& item)
{
	assert(isFilterItem(item));
	SetItemBold(item, false);
	wxTreeItemId parent = GetItemParent(item);
	updateFileModifiedState(parent);
}

bool
FilterConfigWindow::FilterTree::haveModifiedItems() const
{
	// It's enough to check for toplevel (file) nodes.
	wxTreeItemIdValue cookie;
	wxTreeItemId child = GetFirstChild(GetRootItem(), cookie);
	bool modified_found = false;
	for (; (child); child = GetNextSibling(child)) {
		if (IsBold(child)) {
			return true;
		}
	}
	
	return false;
}

void
FilterConfigWindow::FilterTree::updateFileModifiedState(
	wxTreeItemId const& file_item)
{
	assert(isFileItem(file_item));
	wxTreeItemIdValue cookie;
	wxTreeItemId child = GetFirstChild(file_item, cookie);
	bool modified_found = false;
	for (; (child); child = GetNextSibling(child)) {
		if (IsBold(child)) {
			modified_found = true;
			break;
		}
	}
	SetItemBold(file_item, modified_found);
}

void
FilterConfigWindow::FilterTree::updateFileCheckState(
	wxTreeItemId const& file_item)
{
	assert(isFileItem(file_item));
	bool have_enabled_children = false;
	bool have_disabled_children = false;	
	wxTreeItemIdValue cookie;
	wxTreeItemId child = GetFirstChild(file_item, cookie);
	for (; (child); child = GetNextSibling(child)) {
		CheckTreeView::CheckState state = GetCheckState(child);
		if (state == CB_CHECKED) {
			have_enabled_children = true;
		} else {
			have_disabled_children = true;
		}
	}
	
	CheckState state = CB_UNCHECKED;
	if (have_enabled_children && have_disabled_children) {
		state = CheckTreeView::CB_MIXED;
	} else if (have_enabled_children) {
		state = CheckTreeView::CB_CHECKED;
	}
	SetCheckState(file_item, state);
}

bool
FilterConfigWindow::FilterTree::saveEnabledFilterList(
	ContentFilterGroup const& group, EnablePolicy policy)
{
	Application& app = wxGetApp();
	wxFileName fname = app.getFiltersDir();
	fname.SetName(group.fileName() + _T(".enabled"));
	
	string file_data;
	
	if (policy == ALL_ENABLED) {
		file_data = "*";
	} else if (policy == NORMAL) {
		ostringstream strm;
		ContentFilters const& filters = group.filters();
		bool have_enabled = false;
		bool have_disabled = false;
		ContentFilters::FilterList::const_iterator it = filters.filters().begin();
		ContentFilters::FilterList::const_iterator end = filters.filters().end();
		for (; it != end; ++it) {
			bool const enabled = (*it)->isEnabled();
			if (enabled) {
				strm << (*it)->name() << "\r\n";
				have_enabled = true;
			} else {
				have_disabled = true;
			}
		}
		if (have_enabled && !have_disabled) {
			file_data = "*";
		} else {
			file_data = strm.str();
		}
	}
	
	return app.writeFile(fname, file_data);
}

IntrusivePtr<RegexFilterDescriptor>
FilterConfigWindow::FilterTree::findExistingFollower(
	wxTreeItemId const& filter_item)
{
	assert(isFilterItem(filter_item));
	
	IntrusivePtr<RegexFilterDescriptor> res;
	wxTreeItemId it = filter_item;
	while ((it = GetNextSibling(it))) {
		if (FilterNode* node = getFilterNode(it)) {
			if (node->filter) {
				res = node->filter;
				break;
			}
		}
	}
	
	return res;
}

void
FilterConfigWindow::FilterTree::loadFilterGroups()
{
	typedef Application::ContentFilterList GroupList;
	
	wxTreeItemId root = GetRootItem();
	GroupList const& groups = wxGetApp().contentFilters();
	GroupList::const_iterator it = groups.begin();
	GroupList::const_iterator const end = groups.end();
	for (; it != end; ++it) {
		ContentFilterGroup const& group = **it;
		wxString const& fname = group.fileName();
		FileNode* node = new FileNode(*it);
		wxTreeItemId item = AppendItem(root, fname, FILE_ICON, -1, node);
		CheckState state = loadIndividualFilters(group.filters(), item);
		SetCheckState(item, state);
	}
}

CheckTreeView::CheckState
FilterConfigWindow::FilterTree::loadIndividualFilters(
	ContentFilters const& filters, wxTreeItemId parent_item)
{
	typedef ContentFilters::FilterList FilterList;
	
	bool have_enabled_nodes = false;
	bool have_disabled_nodes = false;
	FilterList const& fts = filters.filters(); 
	FilterList::const_iterator it = fts.begin();
	FilterList::const_iterator const end = fts.end();
	for (; it != end; ++it) {
		RegexFilterDescriptor const& filter = **it;
		char const* name = filter.name().c_str();
		FilterNode* node = new FilterNode(*it);
		wxTreeItemId item = AppendItem(
			parent_item, name, FILTER_ICON, -1, node
		);
		if (filter.isEnabled()) {
			SetCheckState(item, CB_CHECKED);
			have_enabled_nodes = true;
		} else {
			SetCheckState(item, CB_UNCHECKED);
			have_disabled_nodes = true;
		}
	}
	if (have_enabled_nodes && have_disabled_nodes) {
		return CB_MIXED;
	} else if (have_enabled_nodes) {
		return CB_CHECKED;
	} else {
		return CB_UNCHECKED;
	}
}

void
FilterConfigWindow::FilterTree::onItemCheckStateChanging(CheckTreeEvent& evt)
{
	if (!evt.isActionByUser()) {
		return;
	}
	
	if (isFileItem(evt.getItem())) {
		onFileCheckStateChanging(evt);
	} else {
		onFilterCheckStateChanging(evt);
	}
}

void
FilterConfigWindow::FilterTree::onFileCheckStateChanging(CheckTreeEvent& evt)
{
	wxTreeItemId file_item = evt.getItem();
	FileNode* file_node = getFileNode(file_item);
	if (!file_node || GetChildrenCount(file_item) == 0) {
		evt.Veto();
		return;
	}
	
	ContentFilterGroup& group = *file_node->filterGroup;
	bool const checked = (evt.getNewState() == CB_CHECKED);
	
	if (!saveEnabledFilterList(group, checked ? ALL_ENABLED : ALL_DISABLED)) {
		evt.Veto();
		return;
	}
	
	wxTreeItemIdValue cookie;
	wxTreeItemId child = GetFirstChild(file_item, cookie);
	for (; (child); child = GetNextSibling(child)) {
		SetCheckState(child, evt.getNewState());
		if (FilterNode* node = getFilterNode(child)) {
			if (node->filter) {
				node->filter->setEnabled(checked);
			}
		}
	}
	
	wxGetApp().updateGlobalContentFilters();
}

void
FilterConfigWindow::FilterTree::onFilterCheckStateChanging(CheckTreeEvent& evt)
{
	wxTreeItemId filter_item = evt.getItem();
	FilterNode* filter_node = getFilterNode(filter_item);
	wxTreeItemId file_item = GetItemParent(filter_item);
	FileNode* file_node = getFileNode(file_item);
	assert(file_node);
	assert(file_node->filterGroup);
	
	if (!(filter_node && filter_node->filter)) {
		return;
	}
	
	ContentFilterGroup& group = *file_node->filterGroup;
	
	filter_node->filter->setEnabled(evt.getNewState() == CB_CHECKED);
	if (!saveEnabledFilterList(group)) {
		filter_node->filter->setEnabled(evt.getOldState() == CB_CHECKED);
		evt.Veto();
		return;
	}
	
	wxGetApp().updateGlobalContentFilters();
}

void
FilterConfigWindow::FilterTree::onItemCheckStateChanged(CheckTreeEvent& evt)
{
	if (!evt.isActionByUser()) {
		return;
	}
	
	if (!isFileItem(evt.getItem())) {
		updateFileCheckState(GetItemParent(evt.getItem()));
	}
}

void
FilterConfigWindow::FilterTree::onSelChanging(wxTreeEvent& evt)
{
	if (GetEditControl()) {
		evt.Veto();
	}
}

void
FilterConfigWindow::FilterTree::onItemMenu(wxTreeEvent& evt)
{
	if (GetEditControl()) {
		return;
	}
	
	wxPoint mouse = ScreenToClient(wxGetMousePosition());
	wxTreeItemId item = evt.GetItem();
	if (!item) {
		// no item is selected
		popupNoItemContextMenu(mouse.x, mouse.y);
		return;
	}
	
	wxRect rect;
	GetBoundingRect(item, rect);
	if (rect.Inside(mouse)) {
		popupItemContextMenu(item, mouse.x, mouse.y);
	} else {
		// probably caused by the Menu button on the keyboard
		GetBoundingRect(item, rect, /* text_only  = */ true);
		popupItemContextMenu(item, rect.GetLeft(), rect.GetBottom());
	}
}

void
FilterConfigWindow::FilterTree::onNewFile(wxCommandEvent& evt)
{
	wxTreeItemId item = GetSelection();
	wxTreeItemId parent = item ? GetItemParent(item) : GetRootItem();
	
	wxTreeItemId new_item = InsertItem(
		parent, item, wxEmptyString, FILE_ICON
	);
	SetCheckState(new_item, CB_UNCHECKED);
	SelectItem(new_item);
	EditLabel(new_item);
}

void
FilterConfigWindow::FilterTree::onNewFilter(wxCommandEvent& evt)
{
	wxTreeItemId item = GetSelection();
	if (!item) {
		return;
	}
	
	wxTreeItemId new_item;
	wxTreeItemId parent;
	if (isFileItem(item)) {
		parent = item;
		new_item = AppendItem(parent, wxEmptyString, FILTER_ICON);	
	} else {
		parent = GetItemParent(item);
		new_item = InsertItem(parent, item, wxEmptyString, FILTER_ICON);
	}
	SetCheckState(new_item, CB_UNCHECKED);
	updateFileCheckState(parent);
	EnsureVisible(new_item);
	SelectItem(new_item);
	EditLabel(new_item);
}

void
FilterConfigWindow::FilterTree::onFilterUp(wxCommandEvent& evt)
{
	wxTreeItemId item = GetSelection();
	assert(item);
	assert(isFilterItem(item));
	wxTreeItemId prev = GetPrevSibling(item);
	assert(prev);
	assert(isFilterItem(prev));
	wxTreeItemId parent = GetItemParent(item);
	FileNode* file_node = getFileNode(parent);
	assert(file_node);
	ContentFilterGroup& group = *file_node->filterGroup;
	
	FilterNode* item_node = getFilterNode(item);
	FilterNode* prev_node = getFilterNode(prev);
	if (!item_node || !prev_node) {
		// one of them is still being created
		return;
	}
	
	if (item_node->filter && prev_node->filter) {
		assert(item_node->editState->operation == EDITING);
		assert(prev_node->editState->operation == EDITING);
		ContentFilterGroup new_group(group);
		typedef ContentFilters::FilterList::iterator Iter;
		Iter item_it = new_group.filters().find(item_node->filter);
		assert(item_it != new_group.filters().filters().end());
		Iter prev_it = item_it;
		assert(prev_it != new_group.filters().filters().begin());
		--prev_it;
		swap(*item_it, *prev_it);
		
		new_group.fileStructure().updateWith(new_group.filters());
		
		if (!saveFilterGroup(new_group, &group)) {
			return;
		}
		
		group.swap(new_group);
		wxGetApp().updateGlobalContentFilters();
	}
	
	wxString label = GetItemText(prev);
	CheckState check_state = GetCheckState(prev);
	auto_ptr<FilterNode> data(new FilterNode);
	prev_node->swap(*data);
	
	Delete(prev);
	wxTreeItemId moved = InsertItem(
		parent, item, label, FILTER_ICON, -1, data.release()
	);
	SetCheckState(moved, check_state);
}

void
FilterConfigWindow::FilterTree::onFilterDown(wxCommandEvent& evt)
{
	wxTreeItemId item = GetSelection();
	assert(item);
	assert(isFilterItem(item));
	wxTreeItemId next = GetNextSibling(item);
	assert(next);
	assert(isFilterItem(next));
	wxTreeItemId parent = GetItemParent(item);
	FileNode* file_node = getFileNode(parent);
	assert(file_node);
	ContentFilterGroup& group = *file_node->filterGroup;
	
	FilterNode* item_node = getFilterNode(item);
	FilterNode* next_node = getFilterNode(next);
	if (!item_node || !next_node) {
		// one of them is still being created
		return;
	}
	
	if (item_node->filter && next_node->filter) {
		assert(item_node->editState->operation == EDITING);
		assert(next_node->editState->operation == EDITING);
		ContentFilterGroup new_group(group);
		typedef ContentFilters::FilterList::iterator Iter;
		Iter item_it = new_group.filters().find(item_node->filter);
		assert(item_it != new_group.filters().filters().end());
		Iter next_it = item_it;
		++next_it;
		assert(next_it != new_group.filters().filters().end());
		swap(*item_it, *next_it);
		
		new_group.fileStructure().updateWith(new_group.filters());
		
		if (!saveFilterGroup(new_group, &group)) {
			return;
		}
		
		group.swap(new_group);
		wxGetApp().updateGlobalContentFilters();
	}
	
	wxString label = GetItemText(next);
	CheckState check_state = GetCheckState(next);
	auto_ptr<FilterNode> data(new FilterNode);
	next_node->swap(*data);
	
	Delete(next);
	wxTreeItemId moved = InsertItem(
		parent, GetPrevSibling(item), label,
		FILTER_ICON, -1, data.release()
	);
	SetCheckState(moved, check_state);
}

void
FilterConfigWindow::FilterTree::onRename(wxCommandEvent& evt)
{
	wxTreeItemId item = GetSelection();
	assert(item);
	EditLabel(item);
}

void
FilterConfigWindow::FilterTree::onLabelEditEnd(wxTreeEvent& evt)
{
	wxTreeItemId item = evt.GetItem();
	if (evt.IsEditCancelled()) {
		if (!GetItemData(item)) {
			// creating a file or a filter has been cancelled
			Delete(item);
		}
		return;
	}
	
	bool done = false;
	if (isFileItem(item)) {
		FileNode* node = getFileNode(item);
		if (node) {
			done = renameFile(item, evt.GetLabel());
		} else {
			done = createFile(item, evt.GetLabel());
			if (!done) {
				Delete(item);	
			}
		}
		if (done) {
			AddPendingEvent(wxCommandEvent(myEVT_FILE_NAME_CHANGED));
		}
	} else {
		// filter item
		FilterNode* node = getFilterNode(item);
		wxTreeItemId parent = GetItemParent(item);
		if (node) {
			done = renameFilter(item, parent, evt.GetLabel());
		} else {
			done = createFilter(item, parent, evt.GetLabel());
			if (!done) {
				Delete(item);
				updateFileCheckState(parent);
			}
		}
	}
	
	if (!done) {
		evt.Veto();
	}
}

void
FilterConfigWindow::FilterTree::onFileNameChanged(wxCommandEvent& evt)
{
	SortChildren(GetRootItem());
	wxTreeItemId item = GetSelection();
	if (item) {
		EnsureVisible(item);
	}
	
	Application& app = wxGetApp();
	app.sortContentFiltersByFileName();
	app.updateGlobalContentFilters();
}

bool
FilterConfigWindow::FilterTree::createFile(
	wxTreeItemId const& item, wxString const& name)
{
	assert(isFileItem(item));
	
	if (!verifyFileNameValid(name)) {
		return false;	
	}
	if (!verifyFilterFileDoesntExist(name)) {
		return false;	
	}
	
	IntrusivePtr<ContentFilterGroup> group(
		new ContentFilterGroup(FilterGroupTag::create(), name)
	);
	
	if (!saveFilterGroup(*group)) {
		return false;
	}
	
	FileNode* node = new FileNode(group);
	SetItemData(item, node);
	wxGetApp().appendContentFilterGroup(group);
	return true;
}

bool
FilterConfigWindow::FilterTree::createFilter(
	wxTreeItemId const& item, wxTreeItemId const& parent,
	wxString const& name)
{
	assert(isFilterItem(item));
	
	if (!verifyFilterNameValid(name)) {
		return false;	
	}
	if (!verifyFilterDoesntExist(parent, name)) {
		return false;	
	}
	
	FilterNode* node = new FilterNode;
	SetItemData(item, node);
	m_rOwner.onFilterCreated(item, *node);
	return true;
}

bool
FilterConfigWindow::FilterTree::renameFile(
	wxTreeItemId const& item, wxString const& new_name)
{
	assert(isFileItem(item));
	FileNode* node = getFileNode(item);
	assert(node);
	assert(node->filterGroup);
	ContentFilterGroup& group = *node->filterGroup;
	
	if (!verifyFileNameValid(new_name)) {
		return false;
	}
	if (group.fileName() == new_name) {
		return false;
	}
	if (!verifyFilterFileDoesntExist(new_name)) {
		return false;	
	}
	
	if (!renameFilterGroupFile(group, new_name)) {
		return false;
	}
	
	group.fileName() = new_name;
	return true;
}

bool
FilterConfigWindow::FilterTree::renameFilter(wxTreeItemId const& item,
	wxTreeItemId const& parent, wxString const& new_name)
{
	assert(isFilterItem(item));
	FilterNode* node = getFilterNode(item);
	assert(node);
	assert(node->editState);	

	if (GetItemText(item) == new_name) {
		return false;
	}
	if (!verifyFilterNameValid(new_name)) {
		return false;
	}
	if (!verifyFilterDoesntExist(parent, new_name)) {
		return false;
	}

	if (!node->filter) {
		assert(node->editState->operation == CREATING);
		return true;
	}
	
	FileNode* parent_node = getFileNode(parent);
	ContentFilterGroup& group = *parent_node->filterGroup;
	string const new_name_str = new_name.mb_str();
	IntrusivePtr<RegexFilterDescriptor> new_filter(
		new RegexFilterDescriptor(*node->filter)
	);
	new_filter->name() = new_name_str;
	
	size_t filter_pos = std::distance(
		group.filters().filters().begin(),
		group.filters().find(node->filter)
	);
	ContentFilterGroup new_group(group);
	new_group.filters().filters()[filter_pos] = new_filter;
	new_group.fileStructure().renameFilter(filter_pos, new_name_str);
	
	if (!saveFilterGroup(new_group, &group)) {
		return false;
	}
	
	group.swap(new_group);
	node->filter = new_filter;
	wxGetApp().updateGlobalContentFilters();
	return true;
}

void
FilterConfigWindow::FilterTree::popupNoItemContextMenu(int x, int y)
{
	wxMenu menu;
	menu.Append(ID_MENU_NEW_FILE, _T("New File"));
	PopupMenu(&menu, x, y);
}

void
FilterConfigWindow::FilterTree::popupItemContextMenu(
	wxTreeItemId const& item, int x, int y)
{
	if (isFileItem(item)) {
		popupFileContextMenu(item, x, y);
	} else {
		popupFilterContextMenu(item, x, y);
	}
}

void
FilterConfigWindow::FilterTree::popupFileContextMenu(
	wxTreeItemId const& item, int x, int y)
{
	wxMenu menu;
	menu.Append(ID_MENU_NEW_FILE, _T("New File"));
	menu.Append(ID_MENU_NEW_FILTER, _T("New Filter"));
	menu.AppendSeparator();
	menu.Append(ID_MENU_RENAME, _T("Rename"));
	menu.AppendSeparator();
	menu.Append(ID_MENU_DELETE_FILE, _T("Delete"));
	PopupMenu(&menu, x, y);
}

void
FilterConfigWindow::FilterTree::popupFilterContextMenu(
	wxTreeItemId const& item, int x, int y)
{
	wxMenu menu;
	//menu.Append(ID_MENU_NEW_FILTER, _T("New Filter"));
	menu.Append(ID_MENU_RENAME, _T("Rename"));
	menu.AppendSeparator();
	menu.Append(ID_MENU_MOVE_UP, _T("Move Up"));
	menu.Append(ID_MENU_MOVE_DOWN, _T("Move Down"));
	menu.AppendSeparator();
	menu.Append(ID_MENU_DELETE_FILTER, _T("Delete"));
	
	if (!GetPrevSibling(item)) {
		menu.Enable(ID_MENU_MOVE_UP, false);
	}
	if (!GetNextSibling(item)) {
		menu.Enable(ID_MENU_MOVE_DOWN, false);
	}
	
	PopupMenu(&menu, x, y);
}

bool
FilterConfigWindow::FilterTree::verifyFileNameValid(wxString const& fname)
{
	if (fname.IsEmpty()) {
		wxMessageBox(
			_T("File name can't be empty"),
			_T("Error"), wxOK|wxICON_ERROR, this
		);
		return false;
	}
	if (fname.Find(_T('.')) != -1 ||
	    fname.Find(_T(':')) != -1 ||
	    fname.Find(_T('/')) != -1 ||
		fname.Find(_T('\\')) != -1) {
		wxMessageBox(
			_T("Filter file names must not contain\n")
			_T("the following characters: . : / \\"),
			_T("Error"), wxOK|wxICON_ERROR, this
		);
		return false;
	}
	return true;
}

bool
FilterConfigWindow::FilterTree::verifyFilterNameValid(wxString const& name)
{
	if (name.IsEmpty()) {
		wxMessageBox(
			_T("Filter name can't be empty"),
			_T("Error"), wxOK|wxICON_ERROR, this
		);
		return false;
	}
	return true;
}

bool
FilterConfigWindow::FilterTree::verifyFilterFileDoesntExist(
	wxString const& fname)
{
	wxFileName abs_fname(wxGetApp().getFiltersDir());
	abs_fname.SetName(fname);
	if (abs_fname.FileExists()) {
		wxMessageBox(
			_T("File already exists"),
			_T("Error"), wxOK|wxICON_ERROR, this
		);
		return false;
	}
	return true;
}

bool
FilterConfigWindow::FilterTree::verifyFilterDoesntExist(
	wxTreeItemId const& parent, wxString const& name)
{
	wxTreeItemIdValue cookie;
	wxTreeItemId item = GetFirstChild(parent, cookie);
	for (; item; item = GetNextSibling(item)) {
		if (GetItemText(item) == name) {
			wxMessageBox(
				_T("Duplicate filter name"),
				_T("Error"), wxOK|wxICON_ERROR, this
			);
			return false;	
		}
	}
	return true;
}


/*===================== FilterConfigWindow::FileNode =====================*/

FilterConfigWindow::FileNode::FileNode(
	IntrusivePtr<ContentFilterGroup> const& group)
:	filterGroup(group)
{
}

FilterConfigWindow::FileNode::~FileNode()
{
}


/*==================== FilterConfigWindow::FilterNode ====================*/

FilterConfigWindow::FilterNode::FilterNode()
:	editState(new EditState)
{
}

FilterConfigWindow::FilterNode::FilterNode(
	IntrusivePtr<RegexFilterDescriptor> const& ftr)
:	editState(new EditState(*ftr)),
	filter(ftr)
{
}

FilterConfigWindow::FilterNode::~FilterNode()
{
}

void
FilterConfigWindow::FilterNode::swap(FilterNode& other)
{
	editState.swap(other.editState);
	filter.swap(other.filter);
}

/*=================== FilterConfigWindow::FilterPanel ====================*/

FilterConfigWindow::FilterPanel::FilterPanel(
	FilterConfigWindow& owner, wxWindow* parent)
:	wxPanel(parent, -1),
	m_rOwner(owner),
	m_operation(CREATING),
	m_isModified(true), // always true for CREATING
	m_textEventsBlocked(0),
	m_pOrderCtrl(0),
	m_pMatchCountLimitCtrl(0),
	m_pContentTypeCtrl(0),
	m_pUrlCtrl(0),
	m_pSearchCtrl(0),
	m_pReplaceCtrl(0),
	m_pPlainTextRB(0),
	m_pExpressionRB(0),
	m_pJavaScriptRB(0),
	m_pIfFlagCtrl(0),
	m_pSetFlagCtrl(0),
	m_pClearFlagCtrl(0),
	m_pResetButton(0),
	m_pSaveButton(0)
{
	wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	wxGridBagSizer* gb_sizer = new wxGridBagSizer(3, 3);
	topsizer->Add(gb_sizer, 1, wxGROW);
	gb_sizer->AddGrowableCol(1, 1);
	gb_sizer->AddGrowableRow(6, 1);
	
	gb_sizer->Add(
		new wxStaticText(this, -1, _T("Order")),
		wxGBPosition(0, 0), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL|wxALIGN_RIGHT
	);
	m_pOrderCtrl = new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(30, -1)
	);
	gb_sizer->Add(
		m_pOrderCtrl,
		wxGBPosition(0, 1), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL
	);
	
	gb_sizer->Add(
		new wxStaticText(this, -1, _T("Max matches")),
		wxGBPosition(1, 0), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL|wxALIGN_RIGHT
	);
	m_pMatchCountLimitCtrl = new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(30, -1)
	);
	gb_sizer->Add(
		m_pMatchCountLimitCtrl,
		wxGBPosition(1, 1), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL
	);
	
	gb_sizer->Add(
		new wxStaticText(this, -1, _T("Content type")),
		wxGBPosition(2, 0), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL|wxALIGN_RIGHT
	);
	m_pContentTypeCtrl = new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(260, -1)
	);
	gb_sizer->Add(
		m_pContentTypeCtrl,
		wxGBPosition(2, 1), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL
	);
	
	gb_sizer->Add(
		new wxStaticText(this, -1, _T("URL")),
		wxGBPosition(3, 0), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL|wxALIGN_RIGHT
	);
	m_pUrlCtrl = new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(260, -1)
	);
	gb_sizer->Add(
		m_pUrlCtrl,
		wxGBPosition(3, 1), wxDefaultSpan,
		wxALIGN_CENTER_VERTICAL
	);
	
	wxFlexGridSizer* flags_sizer = new wxFlexGridSizer(2, 3, 1, 1);
	gb_sizer->Add(flags_sizer, wxGBPosition(4, 0), wxGBSpan(1, 2), wxGROW);
	for (int i = 0; i < 3; ++i) {
		flags_sizer->AddGrowableCol(i, 1);
	}
	
	flags_sizer->Add(
		new wxStaticText(this, -1, _T("If")),
		0, wxALIGN_CENTER
	);
	flags_sizer->Add(
		new wxStaticText(this, -1, _T("Set")),
		0, wxALIGN_CENTER
	);
	flags_sizer->Add(
		new wxStaticText(this, -1, _T("Clear")),
		0, wxALIGN_CENTER
	);
	
	m_pIfFlagCtrl = new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(50, -1)
	);
	flags_sizer->Add(m_pIfFlagCtrl, 0, wxGROW);
	
	m_pSetFlagCtrl =new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(50, -1)
	);
	flags_sizer->Add(m_pSetFlagCtrl, 0, wxGROW);
	
	m_pClearFlagCtrl = new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(50, -1)
	);
	flags_sizer->Add(m_pClearFlagCtrl, 0, wxGROW);
	
	wxBoxSizer* sizer1 = new wxBoxSizer(wxVERTICAL);
	gb_sizer->Add(sizer1, wxGBPosition(5, 0), wxGBSpan(1, 2), wxGROW);
	sizer1->Add(new wxStaticText(this, -1, _T("Search")));
	m_pSearchCtrl = new wxTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(-1, 47),
		wxTE_MULTILINE
	);
	sizer1->Add(m_pSearchCtrl, 0, wxGROW|wxTOP, 1);
	
	wxBoxSizer* sizer2 = new wxBoxSizer(wxVERTICAL);
	gb_sizer->Add(sizer2, wxGBPosition(6, 0), wxGBSpan(1, 2), wxGROW);
	wxBoxSizer* sizer3 = new wxBoxSizer(wxHORIZONTAL);
	sizer2->Add(sizer3, 0, wxGROW);
	sizer3->Add(
		new wxStaticText(this, -1, _T("Replace")),
		1, wxALIGN_LEFT|wxALIGN_BOTTOM
	);
	m_pPlainTextRB = new wxRadioButton(
		this, ID_PLAIN_TEXT_RB, _T("Plain Text"),
		wxDefaultPosition, wxDefaultSize,
		wxRB_GROUP
	);
	sizer3->Add(m_pPlainTextRB, 0, wxALIGN_BOTTOM);
	sizer3->AddSpacer(8);
	m_pExpressionRB = new wxRadioButton(
		this, ID_EXPRESSION_RB, _T("Expression")
	);
	sizer3->Add(m_pExpressionRB, 0, wxALIGN_BOTTOM);
	sizer3->AddSpacer(8);
	m_pJavaScriptRB = new wxRadioButton(
		this, ID_JAVASCRIPT_RB, _T("JavaScript")
	);
	sizer3->Add(m_pJavaScriptRB, 0, wxALIGN_BOTTOM);
	m_pReplaceCtrl = new AutoIndentingTextCtrl(
		this, -1, wxEmptyString,
		wxDefaultPosition, wxSize(-1, 47),
		wxTE_MULTILINE|wxTE_PROCESS_TAB
	);
	sizer2->Add(m_pReplaceCtrl, 1, wxGROW|wxTOP, 1);
	
	wxBoxSizer* buttonsizer = new wxBoxSizer(wxHORIZONTAL);
	topsizer->Add(buttonsizer, 0, wxALIGN_CENTER);
	m_pResetButton = new wxButton(this, wxID_BACKWARD, _T("Reset"));
	buttonsizer->Add(m_pResetButton, 0, wxTOP|wxLEFT|wxRIGHT, 5);
	m_pResetButton->Disable();
	m_pSaveButton = new wxButton(this, wxID_FORWARD, _T("Save"));
	buttonsizer->Add(m_pSaveButton, 0, wxTOP|wxLEFT|wxRIGHT, 5);
	// The Reset button is disabled while the Save button is enabled
	// because that's how it's supposed to be with operation == CREATING
	// and modified == true.
	
	wxFont font = m_pOrderCtrl->GetFont();
	font.SetFamily(wxFONTFAMILY_MODERN);
	font.SetFaceName(_T("Courier"));
	m_pContentTypeCtrl->SetFont(font);
	m_pUrlCtrl->SetFont(font);
	m_pSearchCtrl->SetFont(font);
	m_pReplaceCtrl->SetFont(font);
	
	topsizer->SetSizeHints(this);
	
	wxAcceleratorEntry save_accel(wxACCEL_CTRL, 'S', wxID_FORWARD);
	wxAcceleratorTable accel_tbl(1, &save_accel);
	SetAcceleratorTable(accel_tbl);
}

FilterConfigWindow::FilterPanel::~FilterPanel()
{
}

void
FilterConfigWindow::FilterPanel::setOperation(Operation op)
{
	if (m_operation == op) {
		return;
	}
	m_operation = op;
	m_pResetButton->Enable(op == EDITING && m_isModified);
}

void
FilterConfigWindow::FilterPanel::setModified(bool modified)
{
	if (m_isModified == modified) {
		return;
	}
	m_isModified = modified;
	m_pResetButton->Enable(m_isModified && m_operation == EDITING);
	m_pSaveButton->Enable(m_isModified);
}

void
FilterConfigWindow::FilterPanel::load(EditState const& state)
{
	typedef RegexFilterDescriptor RFD;
	ScopedIncDec<int> blocker(m_textEventsBlocked);
	m_pOrderCtrl->SetValue(state.order);
	m_pMatchCountLimitCtrl->SetValue(state.matchCountLimit);
	m_pContentTypeCtrl->SetValue(state.contentType);
	m_pUrlCtrl->SetValue(state.url);
	m_pSearchCtrl->SetValue(state.search);
	m_pReplaceCtrl->SetValue(state.replacement);
	m_pPlainTextRB->SetValue(state.replacementType == RFD::TEXT);
	m_pExpressionRB->SetValue(state.replacementType == RFD::EXPRESSION);
	m_pJavaScriptRB->SetValue(state.replacementType == RFD::JS);
	m_pIfFlagCtrl->SetValue(state.ifFlag);
	m_pSetFlagCtrl->SetValue(state.setFlag);
	m_pClearFlagCtrl->SetValue(state.clearFlag);
	setOperation(state.operation);
	setModified(state.isModified);
}

void
FilterConfigWindow::FilterPanel::save(EditState& state) const
{
	typedef RegexFilterDescriptor RFD;
	state.operation = m_operation;
	state.isModified = m_isModified;
	state.order = m_pOrderCtrl->GetValue();
	state.matchCountLimit = m_pMatchCountLimitCtrl->GetValue();
	state.contentType = m_pContentTypeCtrl->GetValue();
	state.url = m_pUrlCtrl->GetValue();
	state.search = m_pSearchCtrl->GetValue();
	state.replacement = m_pReplaceCtrl->GetValue();
	if (m_pJavaScriptRB->GetValue()) {
		state.replacementType = RFD::JS;
	} else if (m_pExpressionRB->GetValue()) {
		state.replacementType = RFD::EXPRESSION;
	} else {
		state.replacementType = RFD::TEXT;
	}
	state.ifFlag = m_pIfFlagCtrl->GetValue();
	state.setFlag = m_pSetFlagCtrl->GetValue();
	state.clearFlag = m_pClearFlagCtrl->GetValue();
}

bool
FilterConfigWindow::FilterPanel::validateAndApply(
	RegexFilterDescriptor& filter)
{
	namespace rc = boost::regex_constants;
	boost::regex::flag_type const regex_flags =
		rc::normal|rc::icase|rc::optimize;
	int const err_style = wxOK|wxICON_ERROR;
	wxString const err_caption(_T("Error"));
	wxChar const* field = 0;
	wxWindow* widget = 0;
	
	EditState state;
	save(state);
	
	try {
		field = _T("Order");
		widget = m_pOrderCtrl;
		if (state.order.IsEmpty()) {
			filter.order() = 0;
		} else {
			filter.order() = boost::lexical_cast<int>(state.order);
		}
		
		field = _T("Max Matches");
		widget = m_pMatchCountLimitCtrl;
		if (state.matchCountLimit.IsEmpty()) {
			filter.matchCountLimit() = -1;
		} else {
			filter.matchCountLimit() = boost::lexical_cast<int>(state.matchCountLimit);
		}
		
		field = _T("Content type");
		widget = m_pContentTypeCtrl;
		if (state.contentType.IsEmpty()) {
			filter.contentTypePattern().reset(0);
		} else {
			IntrusivePtr<TextPattern> pattern(new TextPattern(
				state.contentType.mb_str(), regex_flags
			));
			filter.contentTypePattern() = pattern;
		}
		
		field = _T("URL");
		widget = m_pUrlCtrl;
		if (state.url.IsEmpty()) {
			filter.urlPattern().reset(0);
		} else {
			IntrusivePtr<TextPattern> pattern(new TextPattern(
				state.url.mb_str(), regex_flags
			));
			filter.urlPattern() = pattern;
		}
		
		field = _T("Search");
		widget = m_pSearchCtrl;
		if (state.search.IsEmpty()) {
			wxMessageBox(
				_T("\"Search\" can't be empty."),
				err_caption, err_style, this
			);
			m_pSearchCtrl->SetFocus();
			return false;
		} else {
			IntrusivePtr<TextPattern> pattern(new TextPattern(
				state.search.mb_str(), regex_flags
			));
			filter.searchPattern() = pattern;
		}
		
		filter.replacement().reset(new string(state.replacement.mb_str()));	
		filter.replacementType() = state.replacementType;
		filter.ifFlag() = state.ifFlag.mb_str();
		filter.setFlag() = state.setFlag.mb_str();
		filter.clearFlag() = state.clearFlag.mb_str();
	} catch (boost::bad_lexical_cast& e) {
		wxMessageBox(
			_T("Illegal value for \"")+wxString(field)+_T("\"."),
			err_caption, err_style, this
		);
		widget->SetFocus();
		return false;
	} catch (boost::bad_expression& e) {
		wxMessageBox(
			_T("Illegal value for \"")+wxString(field)+_T("\".\n")
			_T("Bad regex: ")+wxString::FromAscii(e.what()),
			err_caption, err_style, this
		);
		widget->SetFocus();
		return false;
	}
	return true;
}

void
FilterConfigWindow::FilterPanel::onModify(wxCommandEvent& evt)
{
	if (m_isModified || m_textEventsBlocked) {
		return;
	}
	
	setModified(true);
	m_rOwner.markCurrentFilterAsModified();
}

void
FilterConfigWindow::FilterPanel::onSave(wxCommandEvent& evt)
{
	// We can be called as a result a button press,
	// or a Ctrl+S shortcut.
	if (IsShown() && m_pSaveButton->IsEnabled()) {
		evt.Skip();
	}
}
