/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef LOG_H_
#define LOG_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "IntrusivePtr.h"
#include "TextStyle.h"
#include "StyledText.h"
#include <sigc++/sigc++.h>
#include <ace/config.h>
#include <wx/string.h>
#include <stddef.h>
#include <deque>
#include <set>

class LogObserver;

class Log : public sigc::trackable, private NonCopyable
{
public:
	typedef IntrusivePtr<TextStyle> StylePtr;
	typedef IntrusivePtr<TextStyle const> ConstStylePtr;

	Log();

	~Log();

	void attachObserver(LogObserver* observer);
	
	void detachObserver(LogObserver* observer);

	void appendRecord(StyledText const& text);

	void appendRecord(wxString const& text);
	
	void appendRecord(wxString const& text, ConstStylePtr const& style);

	template<typename Iter>
	void appendRecord(Iter begin, Iter end);

	void appendToLastRecord(StyledText const& text);
	
	void appendToLastRecord(wxString const& text);

	void appendToLastRecord(wxString const& text, ConstStylePtr const& style);
	
	template<typename Iter>
	void appendToLastRecord(Iter begin, Iter end);

	void clear();
	
	bool hasUnseenWarnings() const { return m_haveUnseenWarnings; }
	
	sigc::slot<void, bool>& getViewVisibilityReporter() {
		return m_visibilityReporter;
	}
	
	sigc::signal<void, bool>& getUnseenWarningsSignal() {
		return m_unseenWarningsSignal;
	}
	
	ConstStylePtr getDefaultStyle() const { return m_ptrDefaultStyle; }
	
	ConstStylePtr getSuccessStyle() const { return m_ptrSuccessStyle; }

	ConstStylePtr getWarningStyle() const { return m_ptrWarningStyle; }

	ConstStylePtr getErrorStyle() const { return m_ptrErrorStyle; }
	
	size_t getNumRecords() const { return m_numRecords; }

	std::deque<StyledText> const& getText() const { return m_text; }
protected:
	void setDefaultStyle(StylePtr const& style) { m_ptrDefaultStyle = style; }
	
	void setSuccessStyle(StylePtr const& style) { m_ptrSuccessStyle = style; }

	void setWarningStyle(StylePtr const& style) { m_ptrWarningStyle = style; }

	void setErrorStyle(StylePtr const& style) { m_ptrErrorStyle = style; }
private:
	void appendText(StyledText const& text, bool prepend_newline);
	
	void onViewVisibilityChange(bool visible);
	
	std::deque<StyledText> m_text;
	size_t m_numRecords;
	std::set<LogObserver *> m_observers;
	int m_numVisibleViews;
	bool m_haveUnseenWarnings; // or errors
	sigc::slot<void, bool> m_visibilityReporter;
	sigc::signal<void, bool> m_unseenWarningsSignal;
	StylePtr m_ptrDefaultStyle;
	StylePtr m_ptrSuccessStyle;
	StylePtr m_ptrWarningStyle;
	StylePtr m_ptrErrorStyle;
};


inline void
Log::attachObserver(LogObserver* observer)
{
	m_observers.insert(observer);
}

inline void
Log::detachObserver(LogObserver* observer)
{
	m_observers.erase(observer);
}

inline void
Log::appendRecord(wxString const& text)
{
	appendRecord(StyledText(text, m_ptrDefaultStyle));
}

inline void
Log::appendRecord(wxString const& text, ConstStylePtr const& style)
{
	appendRecord(StyledText(text, style));
}

inline void
Log::appendToLastRecord(wxString const& text)
{
	appendToLastRecord(StyledText(text, m_ptrDefaultStyle));
}

inline void
Log::appendToLastRecord(wxString const& text, ConstStylePtr const& style)
{
	appendToLastRecord(StyledText(text, style));
}

template<typename Iter>
void
Log::appendRecord(Iter begin, Iter end)
{
	if (begin == end) {
		appendRecord(wxEmptyString);
	} else {
		appendRecord(*begin);
		++begin;
		for (; begin != end; ++begin) {
			appendToLastRecord(*begin);
		}
	}
}

template<typename Iter>
void
Log::appendToLastRecord(Iter begin, Iter end)
{
	for (; begin != end; ++begin) {
		appendToLastRecord(*begin);
	}
}

#endif
