/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iostream>
#include <fstream>
#include <string>
#include <algorithm>

void process(std::istream& in, std::ostream& out);
void output_token(const std::string& token, std::ostream& out);

int main(int argc, char** argv)
{
	if (argc < 2) {
		std::cerr << "Usage: mkskel <infile> [outfile]" << std::endl;
		return 1;
	}
	
	std::ifstream in(argv[1]);
	if (!in.good()) {
		std::cerr << "Could not open \"" << argv[1] << "\" for reading" << std::endl;
		return 1;
	}
	
	if (argc > 2) {
		std::ofstream out(argv[2]);
		if (!out.good()) {
			std::cerr << "Could not open \"" << argv[2] << "\" for writing" << std::endl;
			return 1;
		}
		process(in, out);
	} else {
		process(in, std::cout);
	}
	
	return 0;
}

void process(std::istream& in, std::ostream& out)
{
	char buf[1024];
	std::string token;
	out << '{' << std::endl;
	while (in.read(buf, sizeof(buf)).gcount()) {
		const char* begin = buf;
		const char* const end = buf + in.gcount();
		while (true) {
			const char* pos = std::find(begin, end, '\n');
			pos = std::find(begin, pos, '@');
			if (pos == end) {
				token.append(begin, end);
				break;
			}
			if (*pos == '\n') {
				token.append(begin, pos+1);
				output_token(token, out);
				token.resize(0);
			} else { // *pos == '@'
				if (token.c_str()[0] == '@') {
					token.append(begin, pos+1);
					output_token(token, out);
					token.resize(0);
				} else {
					token.append(begin, pos);
					output_token(token, out);
					token.resize(0);
					token += '@';
				}
			}
			begin = pos + 1;
		}
	}
	output_token(token, out);
	out << "0" << std::endl << "};" << std::endl;
}

void output_token(const std::string& token, std::ostream& out)
{
	if (token.length() == 0) {
		return;
	}
	
	out << '"';
	
	const char* begin = token.c_str();
	const char* const end = begin + token.length();
	bool has_nl = false;
	while (true) {
		const char* pos = std::find(begin, end, '\n');
		pos = std::find(begin, pos, '\r');
		pos = std::find(begin, pos, '"');
		pos = std::find(begin, pos, '\\');
		out.write(begin, pos - begin);
		if (pos == end) {
			break;
		}
		if (*pos == '\n') {
			has_nl = true;
			out << "\\n";
		} else if (*pos == '\r') {
			out << "\\r";
		} else if (*pos == '"') {
			out << "\\\"";
		} else { // *pos == '\\'
			out << "\\\\";
		}
		begin = pos + 1;
	}
	
	out << "\", ";
	if (has_nl) {
		out << std::endl;
	}
}

