/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "WFMOReactor.h"

#ifdef HAVE_WFMO_REACTOR

#include "EventHandler.h"
#include "ReactorHelpers.h"
#include "SynchFactory.h"
#include "MtGuard.h"
#include "AtomicOps.h"
#include <ace/Lock.h>
#include <ace/IPC_SAP.h>
#include <cassert>
#include <cstdlib>
#include <cerrno>
#include <limits>

#define EXCLUSIVE_WAITSET_ACCESS()    \
MtGuard<ACE_Lock> guard(*m_ptrMutex); \
if (m_isInsideDemux) wakeup();        \
MtGuard<ACE_Lock> demux_guard(*m_ptrDemuxMutex);

using namespace std;
using namespace ReactorHelpers;

/*
Handler::opaque holds the WSAEVENT.
Handler::flags is laid out like this:
Upper bit: EOF_REACHED_FLAG
Next bit: EVENTS_CHANGED_FLAG
Next bit: REVIVE_READ_EVENT_FLAG
Rest of bits: the index to m_events. Index 0 means the event is not yet
represented in m_events (m_events[0] is holds the wakeup event).
*/
static unsigned const EOF_REACHED_FLAG = unsigned(1) << (sizeof(unsigned)*8 - 1);
static unsigned const EVENTS_CHANGED_FLAG = EOF_REACHED_FLAG >> 1;
static unsigned const REVIVE_READ_EVENT_FLAG = EVENTS_CHANGED_FLAG >> 1;
static unsigned const EVENT_IDX_MASK = ~(EOF_REACHED_FLAG|EVENTS_CHANGED_FLAG|REVIVE_READ_EVENT_FLAG);

static long const WFMO_READ = FD_READ|FD_CLOSE|FD_ACCEPT;
static long const WFMO_WRITE = FD_WRITE|FD_CONNECT;
static long const WFMO_EXCEPT = FD_OOB;

// returns true if EOF have been reached
static bool reviveReadEvent(ACE_HANDLE handle);

static long translateEventsToWFMO(Reactor::IOEvents events);


WFMOReactor::WFMOReactor(
	AbstractSynchFactory const& synch_factory)
:	m_ptrHandlers(new HandlerRepository),
	m_ptrTimers(new TimerQueue),
	m_ptrMutex(synch_factory.createMutex()),
	m_ptrDemuxMutex(synch_factory.createMutex()),
	m_dispatchWave(0),
	m_ioIter(),
	m_isInsideDemux(false),
	m_isStopped(false),
	m_isStoppedAtomic(0)
{
	m_events[0] = ::WSACreateEvent();
	if (m_events[0] == WSA_INVALID_EVENT) {
		throw Exception("WSACreateEvent failed");
	}
	m_handlers[0] = 0;
}

WFMOReactor::~WFMOReactor()
{
	unregisterAllHandlers();
	::WSACloseEvent(m_events[0]);
}

ReactorHandlerId
WFMOReactor::findHandler(ACE_HANDLE handle) const
{
	return m_ptrHandlers->findByHandle(handle);
}

ReactorHandlerId
WFMOReactor::registerHandler(
	ACE_HANDLE handle, EventHandlerPtr const& handler, IOEvents events)
{
	assert(handler);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	if (m_ptrHandlers->size() == MAXIMUM_WAIT_OBJECTS - 1) {
		throw CapacityException();
	}
	
	if (m_isInsideDemux) {
		wakeup();
	}
	
	WSAEVENT evt = ::WSACreateEvent();
	if (evt == WSA_INVALID_EVENT) {
		throw Exception("WSACreateEvent faled");
	}
	
	long const wfmo_events = translateEventsToWFMO(events);
	if (::WSAEventSelect((SOCKET)handle, evt, wfmo_events) == SOCKET_ERROR) {
		::WSACloseEvent(evt);
		throw Exception("WSAEventSelect failed");
	}
	
	unsigned flags = 0;
	if (events & READ) {
		flags |= REVIVE_READ_EVENT_FLAG;
	}
	
	try {
		Handler new_handler(handle, handler, events, flags, evt);
		return m_ptrHandlers->add(new_handler);
	} catch (...) {
		::WSAEventSelect((SOCKET)handle, evt, 0);
		::WSACloseEvent(evt);
		throw;
	}
}

void
WFMOReactor::unregisterHandler(ReactorHandlerId const& id)
{
	assert(id);
	EXCLUSIVE_WAITSET_ACCESS();
	
	destroyEvent(*HandlerRepository::iterFromId(id));
	m_ptrHandlers->remove(id);
}

void
WFMOReactor::unregisterAllHandlers()
{
	EXCLUSIVE_WAITSET_ACCESS();
	
	HandlerRepository::iterator it = m_ptrHandlers->begin();
	HandlerRepository::iterator const end = m_ptrHandlers->end();
	for (; it != end; ++it) {
		destroyEvent(*it);
	}
	
	m_ptrHandlers->clear();
	m_ptrHandlers->first_inactive = m_ptrHandlers->end();
	m_ioIter.reset(0, 0);
}

void
WFMOReactor::enableEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	IOEvents old_events = it->events;
	IOEvents new_events = old_events | events;
	m_ptrHandlers->setEvents(it, new_events);
	if (old_events != new_events) {
		it->flags |= EVENTS_CHANGED_FLAG;
		if ((new_events & READ) && !(old_events & READ)) {
			it->flags |= REVIVE_READ_EVENT_FLAG;
		}
	}
}

void
WFMOReactor::disableEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	IOEvents old_events = it->events;
	IOEvents new_events = old_events & ~events;
	m_ptrHandlers->setEvents(it, new_events);
	if (old_events != new_events) {
		it->flags |= EVENTS_CHANGED_FLAG;
	}
}

void
WFMOReactor::setEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	IOEvents old_events = it->events;
	m_ptrHandlers->setEvents(it, events);
	if (old_events != events) {
		it->flags |= EVENTS_CHANGED_FLAG;
		if ((events & READ) && !(old_events & READ)) {
			it->flags |= REVIVE_READ_EVENT_FLAG;
		}
	}
}

Reactor::IOEvents
WFMOReactor::getEvents(ReactorHandlerId const& id) const
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	return HandlerRepository::iterFromId(id)->events;
}

ReactorTimerId
WFMOReactor::registerTimer(EventHandlerPtr const& handler, ACE_Time_Value const* timeout)
{
	assert(handler);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	TimerQueue* tq = m_ptrTimers.get();
	ACE_Time_Value was_earliest_time = tq->getEarliestTime();
	ReactorTimerId id = tq->add(handler, timeout);
	assert(id);
	
	if (m_isInsideDemux && tq->getEarliestTime() < was_earliest_time) {
		wakeup();
	}
	
	return id;
}

void
WFMOReactor::rescheduleTimer(ReactorTimerId const id, ACE_Time_Value const* timeout)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	TimerQueue* tq = m_ptrTimers.get();
	ACE_Time_Value was_earliest_time = tq->getEarliestTime();
	tq->reschedule(id, timeout);
	
	if (m_isInsideDemux && tq->getEarliestTime() < was_earliest_time) {
		wakeup();
	}
}

void
WFMOReactor::unregisterTimer(ReactorTimerId const& id)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	m_ptrTimers->erase(TimerQueue::iterFromId(id));
}

void
WFMOReactor::unregisterAllTimers()
{
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	m_ptrTimers->clear();
}

Reactor::Status
WFMOReactor::handleEvents()
{
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	if (m_isStopped) {
		m_isStopped = AtomicOps::add(&m_isStoppedAtomic, 0);
		if (m_isStopped) {
			return STOPPED;
		}
	}
	
	unsigned const old_dispatch_wave = m_dispatchWave;
	continueIODispatching();
	m_ptrTimers->continueDispatching(*m_ptrMutex, m_isStopped);
	if (m_isStopped) {
		return STOPPED;
	}
	if (old_dispatch_wave != m_dispatchWave) {
		// This means a recursive call to handleEvents() has
		// already called WaitForMultipleObjects() and did event
		// dispatching, so there is no reason to do that again.
		return SUCCESS;
	}

	{
		MtAntiGuard<ACE_Lock> anti_guard(*m_ptrMutex);
		m_beforeSleepSignal.emit(); // may add / remove handlers
	}
	if (m_isStopped) {
		return STOPPED;
	}
	
	unsigned const ws_size = updateWaitSet();
	DWORD res = 0;
	{
		DemultiplexerGuard guard(*m_ptrMutex, *m_ptrDemuxMutex, m_isInsideDemux);
		res = WaitForMultipleObjects(ws_size, m_events, FALSE, getDemuxTimeout());
	}
	if (res == WAIT_FAILED || (res >= WAIT_ABANDONED_0 && res < WAIT_ABANDONED_0 + ws_size)) {
		return FAILURE;
	}
	
	++m_dispatchWave;
	
	bool const woken_up = (res == WAIT_OBJECT_0);
	bool const io_events = (res >= WAIT_OBJECT_0 + 1 && res < WAIT_OBJECT_0 + ws_size);
	if (io_events) {
		m_ioIter.reset(res - WAIT_OBJECT_0, ws_size);
	} else {
		m_ioIter.reset(0, 0);
	}
	m_ptrTimers->updateDispatchThreshold();
	
	if (woken_up) {
		::WSAResetEvent(m_events[0]);
		m_isStopped = AtomicOps::add(&m_isStoppedAtomic, 0);
		if (m_isStopped) {
			return STOPPED;
		}
	}
	
	if (io_events) {
		continueIODispatching();
	}
	m_ptrTimers->continueDispatching(*m_ptrMutex, m_isStopped);
	
	return m_isStopped ? STOPPED : SUCCESS;
}

Reactor::Status
WFMOReactor::runEventLoop()
{
	Status status = SUCCESS;
	do {
		status = handleEvents();
	} while (status == SUCCESS);
	return status;
}

void
WFMOReactor::wakeup()
{
	::WSASetEvent(m_events[0]);
}

void
WFMOReactor::stop()
{
	if (AtomicOps::set(&m_isStoppedAtomic, 1) == 0) {
		wakeup();
	}
}

void
WFMOReactor::restart()
{
	AtomicOps::set(&m_isStoppedAtomic, 0);
}

sigc::signal<void>&
WFMOReactor::beforeSleepSignal()
{
	return m_beforeSleepSignal;
}

unsigned
WFMOReactor::updateWaitSet()
{
	unsigned pos = 1; // m_events[0] is the wakeup event
	HandlerRepository::iterator it = m_ptrHandlers->begin();
	HandlerRepository::iterator const end = m_ptrHandlers->first_inactive;
	for (; it != end; ++it, ++pos) {
		WSAEVENT evt = (WSAEVENT)it->opaque;
		unsigned flags = it->flags;
		if (flags & EVENTS_CHANGED_FLAG) {
			long const wfmo_events = translateEventsToWFMO(it->events);
			int res = ::WSAEventSelect((SOCKET)it->handle, evt, wfmo_events);
			assert(res == 0);
		}
		
		if (it->events & READ) {
			if ((flags & (REVIVE_READ_EVENT_FLAG|EOF_REACHED_FLAG)) == REVIVE_READ_EVENT_FLAG) {
				if (reviveReadEvent(it->handle)) {
					flags |= EOF_REACHED_FLAG;
				}
			}
			if (flags & EOF_REACHED_FLAG) {
				BOOL res = ::WSASetEvent(evt);
				assert(res);
			}
		}
		
		m_events[pos] = evt;
		m_handlers[pos] = &*it;
		it->flags = (flags & EOF_REACHED_FLAG) | pos;
	}
	return pos;
}

DWORD
WFMOReactor::getDemuxTimeout() const
{
	ACE_Time_Value const target_time = m_ptrTimers->getEarliestTime();
	if (target_time == ACE_Time_Value::max_time) {
		return INFINITE;
	}
	
	ACE_Time_Value const cur_time = m_ptrTimers->getCurrentTime();
	if (cur_time + ACE_Time_Value(0, 2) >= target_time) {
		/*
		These 2 microseconds compensate the effects of
		TimerQueue::updateDispatchThreshold() and
		TimerQueue::ensureTimeAfterThreshold().
		*/
		return 0;
	}
	
	/*
	We round the timeout upwards to a millisecond, to prevent a situation where
	WaitForMultipleObjects() returns on timeout, but no timers are dispatched.
	*/
	ACE_Time_Value timeout = target_time - cur_time;
	timeout.usec(timeout.usec() + (999 - timeout.usec() % 1000));
	DWORD const max_to = INFINITE - 1;
	ACE_Time_Value const max_timeout(max_to / 1000, (max_to % 1000) * 1000);
	if (timeout >= max_timeout) {
		return max_to;
	} else {
		return timeout.msec();
	}
}

void
WFMOReactor::continueIODispatching()
{
	while (!m_isStopped && m_ioIter.next(*this)) {
		Handler const * handler = m_handlers[m_ioIter.pos];
		// The handler may have been removed by unregisterHandler(),
		// but m_ioIter doesn't know about that.
		if (handler) {
			dispatchIO(*handler);
		}
	}
}

void
WFMOReactor::dispatchIO(Handler const& handler)
{
	/*
	Unfortunately we can't dispatch all events in one go, because
	event handlers may erase the current handler, may advance m_ioIter,
	and may even indirectly call updateWaitSet().
	*/
	
	ACE_HANDLE handle = handler.handle;
	EventHandlerBase* eh = handler.handler.get();
	if (m_ioIter.phase == 0 && (handler.events & WRITE)) {
		dispatchIOEvent(handle, eh, &EventHandlerBase::handleWrite);
	} else if (m_ioIter.phase == 1 && (handler.events & EXCEPT)) {
		dispatchIOEvent(handle, eh, &EventHandlerBase::handleExcept);
	} else if (m_ioIter.phase == 2 && (handler.events & READ)) {
		dispatchIOEvent(handle, eh, &EventHandlerBase::handleRead);
	}
}

void
WFMOReactor::dispatchIOEvent(
	ACE_HANDLE handle, EventHandlerBase* eh, HandlerFuncPtr func)
{
	MtAntiGuard<ACE_Lock> anti_guard(*m_ptrMutex);
	(eh->*func)(handle);
}

long
WFMOReactor::queryEvent(unsigned pos)
{
	Handler const* h = m_handlers[pos];
	if (!h || h->events == NO_EVENTS) {
		// handler have been unregistered or disabled
		return 0;
	}
	
	WSAEVENT evt = m_events[pos];
	assert(evt != WSA_INVALID_EVENT);
	WSANETWORKEVENTS evts;
	int res = ::WSAEnumNetworkEvents((SOCKET)h->handle, evt, &evts);
	assert(res == 0);
	long revents = evts.lNetworkEvents;
	if (revents & FD_CLOSE) {
		h->flags |= EOF_REACHED_FLAG;
	} else if (h->flags & EOF_REACHED_FLAG) {
		revents |= FD_CLOSE;
	}
	if (revents & FD_READ) {
		if (reviveReadEvent(h->handle)) {
			h->flags |= EOF_REACHED_FLAG;
			revents |= FD_CLOSE;
		}
	}
	return revents;
}

void
WFMOReactor::destroyEvent(Handler const& handler)
{
	WSAEVENT evt = (WSAEVENT)handler.opaque;
	int res1 = ::WSAEventSelect((SOCKET)handler.handle, evt, 0);
	assert(res1 == 0);
	BOOL res2 = ::WSACloseEvent(evt);
	assert(res2);
	unsigned const idx = handler.flags & EVENT_IDX_MASK;
	if (idx != 0) {
		m_events[idx] = WSA_INVALID_EVENT;
		m_handlers[idx] = 0;
	}
}

// returns true if EOF have been reached
bool reviveReadEvent(ACE_HANDLE handle)
{
	char buf[1];
	int res = ::recv((SOCKET)handle, buf, sizeof(buf), MSG_PEEK);
	if (res != SOCKET_ERROR) {
		return res == 0;
	}
	int err = ::WSAGetLastError();
	return err != WSAEWOULDBLOCK && err != WSAENOTCONN;
}

long translateEventsToWFMO(Reactor::IOEvents events)
{
	long res = 0;
	if (events & Reactor::READ) {
		res |= WFMO_READ;
	}
	if (events & Reactor::WRITE) {
		res |= WFMO_WRITE;
	}
	if (events & Reactor::EXCEPT) {
		res |= WFMO_EXCEPT;
	}
	return res;
}


/*===================== WFMOReactor::IODispatchIter =====================*/

bool
WFMOReactor::IODispatchIter::next(WFMOReactor& reactor)
{
	if (pos == end) {
		return false;
	}
	assert(pos < end);
	
	switch (phase) {
		case -1:
		case_m1:
		phase = 0;
		revents = reactor.queryEvent(pos);
		if (revents & WFMO_WRITE) {
			break;
		}
		// fall through
		case 0:
		phase = 1;
		if (revents & WFMO_EXCEPT) {
			break;
		}
		// fall through
		case 1:
		phase = 2;
		if (revents & WFMO_READ) {
			break;
		}
		// fall through
		case 2:
		phase = 0;
		if (++pos == end) {
			return false;
		}
		goto case_m1;
	}
	
	return true;
}

#endif // HAVE_WFMO_REACTOR
