//////////////////////////////////////////////////////////////////
//
// GkStatus.h thread for external interface
//
// $Id: GkStatus.cxx,v 1.16.2.34 2004/05/25 19:54:01 zvision Exp $
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
// 	990924	initial version (Jan Willamowius)
//	991025	Added command thread (Ashley Unitt)
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4800 ) // one performance warning off
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4101 ) // warning unused locals off
#endif

#include <ptlib.h>
#ifndef WIN32
#include <signal.h>
#endif
#include <ptlib/sockets.h>
#include <ptclib/telnet.h>
#include <h225.h>
#include "gk_const.h"
#include "Toolkit.h"
#include "SoftPBX.h"
#include "RasTbl.h"
#include "RasSrv.h"
#include "GkStatus.h"



static const char *authsec="GkStatus::Auth";

void ReloadHandler(); // avoid to include...

const int GkStatus::NumberOfCommandStrings = 47;
const static PStringToOrdinal::Initialiser GkStatusClientCommands[GkStatus::NumberOfCommandStrings] =
{
	{"printallregistrations",    GkStatus::e_PrintAllRegistrations},
	{"r",                        GkStatus::e_PrintAllRegistrations},
	{"?",                        GkStatus::e_PrintAllRegistrations},
	{"printallregistrationsverbose", GkStatus::e_PrintAllRegistrationsVerbose},
	{"rv",                       GkStatus::e_PrintAllRegistrationsVerbose},
	{"??",                       GkStatus::e_PrintAllRegistrationsVerbose},
	{"printallcached",           GkStatus::e_PrintAllCached},
	{"rc",                       GkStatus::e_PrintAllCached},
	{"printcurrentcalls",        GkStatus::e_PrintCurrentCalls},
	{"c",                        GkStatus::e_PrintCurrentCalls},
	{"!",                        GkStatus::e_PrintCurrentCalls},
	{"printcurrentcallsverbose", GkStatus::e_PrintCurrentCallsVerbose},
	{"!!",                       GkStatus::e_PrintCurrentCallsVerbose},
	{"cv",                       GkStatus::e_PrintCurrentCallsVerbose},
	{"find",                     GkStatus::e_Find},
	{"f",                        GkStatus::e_Find},
	{"findverbose",              GkStatus::e_FindVerbose},
	{"fv",                       GkStatus::e_FindVerbose},
	{"disconnectip",             GkStatus::e_DisconnectIp},
	{"disconnectcall",           GkStatus::e_DisconnectCall},
	{"disconnectalias",          GkStatus::e_DisconnectAlias},
	{"disconnectendpoint",       GkStatus::e_DisconnectEndpoint},
	{"disconnectsession",        GkStatus::e_DisconnectSession},
	{"clearcalls",               GkStatus::e_ClearCalls},
	{"unregisterallendpoints",   GkStatus::e_UnregisterAllEndpoints},
	{"unregisteralias",          GkStatus::e_UnregisterAlias},
	{"unregisterip",             GkStatus::e_UnregisterIp},
	{"transfercall",             GkStatus::e_TransferCall},
	{"makecall",                 GkStatus::e_MakeCall},
	{"yell",                     GkStatus::e_Yell},
	{"who",                      GkStatus::e_Who},
	{"gk",                       GkStatus::e_GK},
	{"help",                     GkStatus::e_Help},
	{"h",                        GkStatus::e_Help},
	{"version",                  GkStatus::e_Version},
	{"v",                        GkStatus::e_Version},
	{"debug",                    GkStatus::e_Debug},
	{"statistics",               GkStatus::e_Statistics},
	{"s",                        GkStatus::e_Statistics},
	{"reload",                   GkStatus::e_Reload},
#if HAS_WAITARQ
	{"RouteToAlias",             GkStatus::e_RouteToAlias},
	{"rta",	                     GkStatus::e_RouteToAlias},
	{"RouteReject",              GkStatus::e_RouteReject},
#endif
	{"shutdown",                 GkStatus::e_Shutdown},
	{"exit",                     GkStatus::e_Exit},
	{"quit",                     GkStatus::e_Exit},
	{"q",                        GkStatus::e_Exit}
};

int GkStatus::Client::StaticInstanceNo = 0;


GkStatus::GkStatus() : PThread(1000, NoAutoDeleteThread), m_IsDirty(false)
{
}

GkStatus::~GkStatus()
{
}

void GkStatus::Initialize(const PIPSocket::Address & _GKHome)
{
	GKHome = _GKHome;
	Resume();	// start the thread
}

void GkStatus::Main()
{
	unsigned qSize = GkConfig()->GetInteger("ListenQueueLength", GK_DEF_LISTEN_QUEUE_LENGTH);
	WORD sPort = (WORD)GkConfig()->GetInteger("StatusPort", GK_DEF_STATUS_PORT);
	StatusListener.Listen(GKHome, qSize, sPort, PSocket::CanReuseAddress);

	StatusListener.SetReadTimeout(GkConfig()->GetInteger("StatusReadTimeout", 3000));
	while (StatusListener.IsOpen()) {
		CleanupClients();

		PTelnetSocket * NewConnection = new PTelnetSocket;
		if (!NewConnection->Accept(StatusListener)) {
			delete NewConnection;
			continue;
		}
		// add new connection to connection list
		Client * NewClient = new Client( this, NewConnection );
		WriteLock lock(ClientSetLock);
		Clients.push_back(NewClient);
		// Socket will be deleted by the client thread when it closes...
	}
}

void GkStatus::Close()
{
	PTRACE(2, "GK\tClosing Status thread.");

	// close my listening socket
	StatusListener.Close();

	// close all connected clients
	ClientSetLock.StartRead();
	ForEachInContainer(Clients, mem_fun(&Client::Close));
	ClientSetLock.EndRead();

	while (!Clients.empty()) {
		Sleep(PTimeInterval(100)); // 1/10 sec
		CleanupClients();
	}

	PTRACE(2, "GK\tClosed Status thread.");
}

// function object used by for_each
class WriteWhoAmI {
  public:
	WriteWhoAmI(GkStatus::Client *c) : writeTo(c) {}
	void operator()(const GkStatus::Client *) const;

  private:
	GkStatus::Client *writeTo;
};

void WriteWhoAmI::operator()(const GkStatus::Client *pclient) const
{
	writeTo->WriteString("  " + pclient->WhoAmI() + "\r\n");
}

// function object used by for_each
class ClientSignalStatus {
  public:
	ClientSignalStatus(const PString &m, int l) : msg(m), level(l) {} 
	void operator()(GkStatus::Client *) const;

  private:
	const PString &msg;
	int level;
};

void ClientSignalStatus::operator()(GkStatus::Client *pclient) const
{
	if (!pclient->IsLoggedIn())
		return;

	int ctl = pclient->GetTraceLevel();
	if ((ctl<=10) && (ctl>=0) && (level >= ctl)) 
		pclient->WriteString(msg);
}


void GkStatus::SignalStatus(const PString &Message, int level)
{
	ReadLock lock(ClientSetLock);
	ForEachInContainer(Clients, ClientSignalStatus(Message, level));
}

bool GkStatus::DisconnectSession(int InstanceNo, Client *kicker)
{
	PTRACE(1, "Disconnect Session " << InstanceNo);
	ReadLock lock(ClientSetLock);
	std::list<Client *>::iterator ClientIter = Clients.begin();
	while (ClientIter != Clients.end()) {
		Client *c = *ClientIter++;
		if (c->GetInstanceNo() == InstanceNo) {
			c->WriteString("Disconnected by session " + kicker->WhoAmI());
			c->Close();
			return true;
		}
	}
	
	return false;
}

void GkStatus::ShowUsers(Client *c) const
{
	ReadLock lock(ClientSetLock);
	ForEachInContainer(Clients, WriteWhoAmI(c));
}

void GkStatus::CleanupClients()
{
	if (IsDirty()) {
		/* we will only delete one client per round */

		WriteLock lock(ClientSetLock);
		std::list<Client *>::iterator ClientIter = Clients.begin();
		while (ClientIter != Clients.end()) {
			Client *c = *ClientIter++;
			if (c->IsDeletable()) {
				Clients.remove(c);
				PTRACE(5, "GK\tRemoveClient " << c->GetInstanceNo());
				delete c;
				return;
			}
		}

		// no more cleanum rounds
		SetDirty(false);
	}
}


namespace {

PString PrintGkVersion()
{
	return PString("Version:\r\n") + Toolkit::GKVersion() +
#ifdef LARGE_FDSET
		PString(PString::Printf, "Large fd_set(%d) enabled\r\n", LARGE_FDSET) +
#endif
		"\r\nGkStatus: Version(1.0) Ext()\r\n"
		"Toolkit: Version(1.0) Ext(" + Toolkit::Instance()->GetName() +
		")\r\n" + SoftPBX::Uptime() + "\r\n;\r\n";
}

}

PStringToOrdinal GkStatus::Client::Commands(NumberOfCommandStrings, GkStatusClientCommands, TRUE);

GkStatus::Client::Client( GkStatus * _StatusThread, PTelnetSocket * _Socket )
	: PThread(1000, NoAutoDeleteThread),
	  InstanceNo(++StaticInstanceNo), TraceLevel(1), ErrorCnt(0),
	  PleaseDelete(false), LoggedIn(false),
	  Socket(_Socket),
	  StatusThread(_StatusThread)
{
	Socket->SetWriteTimeout(GkConfig()->GetInteger("StatusWriteTimeout", 5000));
	Resume();	// start the thread
}


GkStatus::Client::~Client()
{
	Close();
	int i = 0;
	while (i++ < 10)
		if (Mutex.WillBlock())
			PProcess::Sleep(1000);
		else
			break;
	PTRACE_IF(1, i > 10, "Warning: Mutex still blocked for client " << InstanceNo);
	delete Socket;
}


bool GkStatus::Client::ReadCommand(PString& cmd)
{
	PString Command;
	int	CharRead;
	
	while (true) {
		CharRead = Socket->ReadChar();
		if ( CharRead < 0 )
			return false;
		if ( CharRead == '\n' )
			break;
		if( CharRead == '\b' ) { // BackSpace
			Command.Delete(Command.GetLength()-1,1);
			WriteString(" \b"); // clear the last character on telnet connection
			continue;
		}
		if ( CharRead != '\r' )	// ignore carriage return
			Command += (char)CharRead;
		else {
			PTimeInterval timeOri = Socket->GetReadTimeout();
			Socket->SetReadTimeout(10);
			Socket->ReadChar();
			Socket->SetReadTimeout(timeOri);
			break;
		}
	}
	
	cmd = Command;
	return true;
}

void GkStatus::Client::Main()
{
#if PTRACING
	PIPSocket::Address PeerAddress;
	Socket->GetPeerAddress(PeerAddress);
	PTRACE(2, "GK\tGkStatus new status client: addr " << PeerAddress);
#endif

	if (!AuthenticateClient()) {
		WriteString("Access forbidden!\r\n", 1);
	} else {
		LoggedIn = true;
		TraceLevel = 0;
		// the welcome messages
		WriteString(PrintGkVersion());

		BOOL exit_and_out = FALSE;
		while (Socket->IsOpen() && !exit_and_out) {
			PString Line;

			if (!ReadCommand(Line))
				break;

			Line = Line.Trim(); // the 'Tokenise' seems not correct for leading spaces
			const PStringArray Args = Line.Tokenise(" ", FALSE);
			if (Args.GetSize() < 1) 
				continue;

			const PCaselessString &Command = Args[0];

			PTRACE(2, "GK\tGkStatus got command " << Command);
			if (Commands.Contains(Command.ToLower())) {
				PINDEX key = Commands[Command];
				switch(key)
				{
				case GkStatus::e_DisconnectIp:
					// disconnect call on this IP number
					if (Args.GetSize() == 2)
						SoftPBX::DisconnectIp(Args[1]);
					else
						WriteString("Syntax Error: DisconnectIp <ip address>\r\n");
					break;
				case GkStatus::e_DisconnectAlias:
					// disconnect call on this alias
					if (Args.GetSize() == 2)
						SoftPBX::DisconnectAlias(Args[1]);
					else
						WriteString("Syntax Error: DisconnectAlias <h.323 alias>\r\n");
					break;
				case GkStatus::e_DisconnectCall:
					// disconnect call with this call number
					if (Args.GetSize() >= 2)
						for (PINDEX p=1; p < Args.GetSize(); ++p)
							SoftPBX::DisconnectCall(Args[p].AsInteger());
					else
						WriteString("Syntax Error: DisconnectCall <call number> ...\r\n");
					break;
				case GkStatus::e_DisconnectEndpoint:
					// disconnect call on this alias
					if (Args.GetSize() == 2)
						SoftPBX::DisconnectEndpoint(Args[1]);
					else
						WriteString("Syntax Error: DisconnectEndpoint ID\r\n");
					break;
				case GkStatus::e_DisconnectSession:
					// disconnect a user from status port
					if (Args.GetSize() == 2) {
						if (StatusThread->DisconnectSession(Args[1].AsInteger(), this))
							WriteString("Session " + Args[1] + " disconnected\r\n");
						else
							WriteString("Session " + Args[1] + " not found\r\n");
					}
					else
						WriteString("Syntax Error: DisconnectSession SessionID\r\n");
					break;
				case GkStatus::e_ClearCalls:
					CallTable::Instance()->ClearTable();
					break;
				case GkStatus::e_PrintAllRegistrations:
					// print list of all registered endpoints
					SoftPBX::PrintAllRegistrations(*this);
					break;
				case GkStatus::e_PrintAllRegistrationsVerbose:
					// print list of all registered endpoints verbose
					SoftPBX::PrintAllRegistrations(*this, TRUE);
					break;
				case GkStatus::e_PrintAllCached:
					// print list of all cached outer-zone endpoints
					SoftPBX::PrintAllCached(*this, (Args.GetSize() > 1));
					break;
				case GkStatus::e_PrintCurrentCalls:
					// print list of currently ongoing calls
					SoftPBX::PrintCurrentCalls(*this);
					break;
				case GkStatus::e_PrintCurrentCallsVerbose:
					// print list of currently ongoing calls
					SoftPBX::PrintCurrentCalls(*this, TRUE);
					break;
				case GkStatus::e_Statistics:
					SoftPBX::PrintStatistics(*this, TRUE);
					break;
				case GkStatus::e_Find:
					if (Args.GetSize() == 2)
						SoftPBX::PrintEndpoint(Args[1], *this, FALSE);
					else
						WriteString("Syntax Error: Find alias\r\n");
					break;
				case GkStatus::e_FindVerbose:
					if (Args.GetSize() == 2)
						SoftPBX::PrintEndpoint(Args[1], *this, TRUE);
					else
						WriteString("Syntax Error: FindVerbose alias\r\n");
					break;
				case GkStatus::e_Yell:
					StatusThread->SignalStatus(PString("  "+WhoAmI() + ": " + Line + "\r\n"));
					break;
				case GkStatus::e_Who:
					StatusThread->ShowUsers(this);
					WriteString(";\r\n");
					break;
				case GkStatus::e_GK:
					WriteString(RasThread->GetParent() + "\r\n;\r\n");
					break;
				case GkStatus::e_Help:
					PrintHelp();
					break;
				case GkStatus::e_Debug:
					DoDebug(Args);
					break;
				case GkStatus::e_Version:
					WriteString(PrintGkVersion());
					break;
				case GkStatus::e_Exit:
					Close();
					exit_and_out = TRUE;
					break;
				case GkStatus::e_UnregisterAllEndpoints:
					SoftPBX::UnregisterAllEndpoints();
					WriteString("Done\n;\n");
					break;
					case GkStatus::e_UnregisterAlias:
					// unregister this alias
					if (Args.GetSize() == 2)
						SoftPBX::UnregisterAlias(Args[1]);
					else
						WriteString("Syntax Error: UnregisterAlias Alias\r\n");
					break;
				case GkStatus::e_UnregisterIp:
					// unregister this IP
					if (Args.GetSize() == 2)
						SoftPBX::UnregisterIp(Args[1]);
					else
						WriteString("Syntax Error: UnregisterIp <ip addr>\r\n");
					break;
				case GkStatus::e_TransferCall:
					if (Args.GetSize() == 3)
						SoftPBX::TransferCall(Args[1], Args[2]);
					else
						WriteString("Syntax Error: TransferCall Source Destination\r\n");
					break;
				case GkStatus::e_MakeCall:
					if (Args.GetSize() == 3)
						SoftPBX::MakeCall(Args[1], Args[2]);
					else
						WriteString("Syntax Error: MakeCall Source Destination\r\n");
					break;
				case GkStatus::e_Reload:
					ReloadHandler();
					break;
				case GkStatus::e_Shutdown:
					if (!Toolkit::AsBool(GkConfig()->GetString(authsec, "Shutdown", "1"))) {
						WriteString("Not allowed!\r\n");
						break;
					}
					PTRACE(1, "Shutting down the GK");
					SoftPBX::PrintStatistics(*this, TRUE);
					// raise signal, so ShutdownHandler
					// can cleanly terminate gatekeeper
#ifdef WIN32
					GenerateConsoleCtrlEvent(CTRL_C_EVENT, 0);
#else
					raise (SIGINT);
#endif


					break;
#if HAS_WAITARQ
				case GkStatus::e_RouteToAlias:
					if (Args.GetSize() == 4) {
						SoftPBX::RouteToAlias(Args[1], Args[2], Args[3]);
					}
					else
						WriteString("Syntax Error: RouteToAlias <target agent> <calling endpoint ID> <callRef>\r\n");
		  		break;
				case GkStatus::e_RouteReject:
					if (Args.GetSize() == 3) {
						SoftPBX::RouteReject(Args[1], Args[2]);
					}
					else
						WriteString("Syntax Error: RouteReject <calling endpoint ID> <callRef>\r\n");
		  		break;
#endif
 
				default:
					PTRACE(3, "WRONG COMMANDS TABLE ENTRY. PLEASE LOOK AT THE CODE.");
					WriteString("Error: Internal Error.\r\n");
					break;
				}
			} else {
				// commmand not recognized
				PTRACE(3, "Gk\tUnknown Command.");
				WriteString("Error: Unknown Command.\r\n");
			}
		}
	}
	
	PTRACE(2, "GK\tGkStatus client " << InstanceNo << " (" << PeerAddress.AsString() << ") has disconnected");
	PleaseDelete = true;
	StatusThread->SetDirty(true);

	// returning from main will delete this thread
}


void GkStatus::Client::DoDebug(const PStringArray & Args)
{
	if (Args.GetSize() <= 1) {
		WriteString("Debug options:\r\n"
			    "  trc [+|-|n]       Show/modify trace level\r\n"
			    "  cfg SEC PAR       Read and print a config PARameter in a SECtion\r\n"
			    "  set SEC PAR VAL   Write a config VALue PARameter in a SECtion\r\n"
			    "  remove SEC PAR    Remove a config VALue PARameter in a SECtion\r\n"
			    "  remove SEC        Remove a SECtion\r\n"
			    "  printrm VERBOSE   Print all removed endpoint records\r\n");
	} else {
		if (Args[1] *= "trc") {
			if(Args.GetSize() >= 3) {
				if((Args[2] == "-") && (PTrace::GetLevel() > 0)) 
					PTrace::SetLevel(PTrace::GetLevel()-1);
				else if(Args[2] == "+") 
					PTrace::SetLevel(PTrace::GetLevel()+1);
				else PTrace::SetLevel(Args[2].AsInteger());
			}
			WriteString(PString(PString::Printf, "Trace Level is now %d\r\n", PTrace::GetLevel()));
		} else if (Args[1] *= "cfg") {
			if (Args.GetSize()>=4)
				WriteString(GkConfig()->GetString(Args[2],Args[3],"") + "\r\n;\r\n");
			else if (Args.GetSize()>=3) {
				PStringList cfgs(GkConfig()->GetKeys(Args[2]));
				PString result = "Section [" + Args[2] + "]\r\n";
				for (PINDEX i=0; i < cfgs.GetSize(); ++i) {
					PString v(GkConfig()->GetString(Args[2], cfgs[i], ""));
					result += cfgs[i] + "=" + v + "\r\n";
				}
				WriteString(result + ";\r\n");
			}
		} else if ((Args[1] *= "set") && (Args.GetSize()>=5)) {
			Toolkit::Instance()->SetConfig(1, Args[2], Args[3], Args[4]);
			WriteString(GkConfig()->GetString(Args[2],Args[3],"") + "\r\n");
		} else if (Args[1] *= "remove") {
			if (Args.GetSize()>=4) {
				Toolkit::Instance()->SetConfig(2, Args[2], Args[3]);
				WriteString("Remove " + Args[3] + " in section " + Args[2] + "\r\n");
			} else if (Args.GetSize()>=3) {
				Toolkit::Instance()->SetConfig(3, Args[2]);
				WriteString("Remove section " + Args[2] + "\r\n");
			}
		} else if ((Args[1] *= "printrm")) {
			SoftPBX::PrintRemoved(*this, (Args.GetSize() >= 3));
		} else {
			WriteString("Unknown debug command!\r\n");
		}
	}
}


void GkStatus::Client::PrintHelp()
{
	WriteString("Commands:\r\n");
	for (PINDEX i = 0; i < Commands.GetSize(); ++i)
		WriteString(Commands.GetKeyAt(i) + "\r\n");
	WriteString(";\r\n");
	return;
}



bool GkStatus::Client::WriteString(const PString & Message, int level) // level defaults to 0
{
	if (level < TraceLevel)
		return true;
	if (!Socket->IsOpen())
		return false;

	PWaitAndSignal lock(Mutex);
	bool result = Socket->WriteString(Message);
	if (!result) {
		PTRACE(2, "GkStatus can't write to client " << InstanceNo);
		if (++ErrorCnt > 3) {
			PTRACE(1, "GkStatus client " << InstanceNo << " dead, close it!");
			Close();
		}
	} else if (ErrorCnt > 0)
		--ErrorCnt;
	return result;
}

bool GkStatus::Client::Close()
{
	return Socket->IsOpen() ? Socket->Close() : false;
}

PString GkStatus::Client::WhoAmI() const
{
	return PString(InstanceNo) + '\t' + Socket->GetName() + '\t' + UserLoggedIn;
}

bool GkStatus::Client::AuthenticateClient()
{
	bool result = false, logical_or;

	PIPSocket::Address PeerAddress;
	Socket->GetPeerAddress(PeerAddress);
	PTRACE(4, "Auth client from " << PeerAddress);

	PINDEX p = 0;
	const PString rules = GkConfig()->GetString(authsec, "rule", "forbid");
	while (true) {
		PINDEX q = rules.FindOneOf("&|", p);
		result = CheckRule(rules(p, q-1).Trim());
		if (q == P_MAX_INDEX)
			break;
		logical_or = (rules[q] == '|') ;
		if ((logical_or && result) || !(logical_or || result))
			break;
		p = q + 1;
	}
	
	return result;
}

bool GkStatus::Client::CheckRule(const PString & rule)
{
	PIPSocket::Address PeerAddress;
	Socket->GetPeerAddress(PeerAddress);
	const PString peer = PeerAddress.AsString();

	PTRACE(5, "Auth client rule=" << rule);
	bool result = false;
	if (rule *= "forbid") { // "*=": case insensitive
		result =  false;
	} else if (rule *= "allow") {
		result =  true;
	} else if (rule *= "explicit") {
		PString val = GkConfig()->GetString(authsec, peer, "");
		if (val.IsEmpty()) { // use "default" entry
			PTRACE(5,"Auth client rule=explicit, ip-param not found, using default");
			result = Toolkit::AsBool(GkConfig()->GetString(authsec, "default", "FALSE"));
		} else 
			result = Toolkit::AsBool(val);
	} else if (rule *= "regex") {
		PString val = GkConfig()->GetString(authsec, peer, "");
		if (val.IsEmpty()) {
			PTRACE(5,"Auth client rule=regex, ip-param not found, using regex");
			result = Toolkit::MatchRegex(peer, GkConfig()->GetString(authsec, "regex", ""));
		} else
			result = Toolkit::AsBool(val);
	} else if (rule *= "password") {
		result = AuthenticateUser();
	} else {
		PTRACE(1, "Warning: Invalid [GkStatus::Auth].rule");
	}
	return result;
}

bool GkStatus::Client::AuthenticateUser()
{
	int retries = 0;
	PString UserName, Password;
	PString pass;
	int Delay = GkConfig()->GetInteger(authsec, "DelayReject", 0);

	while ((retries < 3) && (!LoggedIn)) {
		Socket->SendWont(PTelnetSocket::EchoOption);
		WriteString("\r\n" + Toolkit::GKName() + " login: ", 1);
		if (!ReadCommand(UserName)) {
			++retries;
			break;
		}
		UserName = UserName.Trim();

		WriteString("\r\nPassword: ", 1);

		Socket->SendWill(PTelnetSocket::EchoOption);

		if (!ReadCommand(Password)) {
			retries++;
			break;
		}
		
		Password = Password.Trim();

		pass = GetPassword(UserName);
		if ((!pass.IsEmpty()) && (pass == Password)) {
			LoggedIn = true;
			UserLoggedIn = UserName;
			PTRACE(1, "Auth: user " << UserName << " logged in");
		} else {
			Sleep(Delay * 1000);
		}

		WriteString("\r\n", 1);
		++retries;
	}

	Socket->SendWont(PTelnetSocket::EchoOption);
	return LoggedIn;
}

PString GkStatus::Client::GetPassword(const PString & UserName) const
{
	int filled = GkConfig()->GetInteger(authsec, "KeyFilled", 0);
	return Toolkit::CypherDecode(UserName, GkConfig()->GetString(authsec, UserName, ""), filled);
}
