//////////////////////////////////////////////////////////////////
//
// ProxyChannel.h
//
// Copyright (c) Citron Network Inc. 2001-2002
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// initial author: Chin-Wei Huang <cwhuang@linux.org.tw>
// initial version: 12/7/2001
//
//////////////////////////////////////////////////////////////////

#ifndef PROXYCHANNEL_H
#define PROXYCHANNEL_H "#(@) $Id: ProxyChannel.h,v 1.12.2.20 2004/05/12 17:46:35 zvision Exp $"

#include <map>
#include "RasTbl.h"
#include "ProxyThread.h"

class H245Handler;
class H245ProxyHandler;
class NATHandler;
class CallSignalSocket;
class H245Socket;
class UDPProxySocket;
class T120ProxySocket;
class LogicalChannel;
class RTPLogicalChannel;
class T120LogicalChannel;

class Q931;
class H245_RequestMessage;
class H245_ResponseMessage;
class H245_CommandMessage;
class H245_IndicationMessage;
class H245_H2250LogicalChannelParameters;
class H245_OpenLogicalChannel;
class H245_OpenLogicalChannelAck;
class H245_OpenLogicalChannelReject;
class H245_CloseLogicalChannel;
class H225_CallTerminationCause;

class H245Handler {
// This class handles H.245 messages which can either be transmitted on their
// own TCP connection or can be tunneled in the Q.931 connection
public:
	H245Handler(const PIPSocket::Address & local, const PIPSocket::Address & remote);
	virtual ~H245Handler();

	virtual void OnH245Address(H225_TransportAddress &);
	virtual bool HandleMesg(PPER_Stream &);
	virtual bool HandleFastStartSetup(H245_OpenLogicalChannel &);
	virtual bool HandleFastStartResponse(H245_OpenLogicalChannel &);
	typedef bool (H245Handler::*pMem)(H245_OpenLogicalChannel &);

	PIPSocket::Address GetLocalAddr() const { return localAddr; }
	void SetLocalAddr(const PIPSocket::Address & local) { localAddr = local; }
	bool IsSessionEnded() const { return isH245ended; }

protected:
	virtual bool HandleRequest(H245_RequestMessage &);
	virtual bool HandleResponse(H245_ResponseMessage &);
	virtual bool HandleCommand(H245_CommandMessage &);
	virtual bool HandleIndication(H245_IndicationMessage &);

	NATHandler *hnat;

private:
	PIPSocket::Address localAddr, remoteAddr;
	bool isH245ended;
};

class H245ProxyHandler : public H245Handler {
public:
	typedef std::map<WORD, LogicalChannel *>::iterator iterator;
	typedef std::map<WORD, LogicalChannel *>::const_iterator const_iterator;
	typedef std::map<WORD, RTPLogicalChannel *>::iterator siterator;
	typedef std::map<WORD, RTPLogicalChannel *>::const_iterator const_siterator;

	H245ProxyHandler(CallSignalSocket *, const PIPSocket::Address &, const PIPSocket::Address &, H245ProxyHandler * = 0);
	virtual ~H245ProxyHandler();

	// override from class H245Handler
	virtual bool HandleFastStartSetup(H245_OpenLogicalChannel &);
	virtual bool HandleFastStartResponse(H245_OpenLogicalChannel &);

	LogicalChannel *FindLogicalChannel(WORD);
	RTPLogicalChannel *FindRTPLogicalChannelBySessionID(WORD);
	
private:
	// override from class H245Handler
	virtual bool HandleRequest(H245_RequestMessage &);
	virtual bool HandleResponse(H245_ResponseMessage &);

	bool OnLogicalChannelParameters(H245_H2250LogicalChannelParameters *, WORD);
	bool HandleOpenLogicalChannel(H245_OpenLogicalChannel &);
	bool HandleOpenLogicalChannelAck(H245_OpenLogicalChannelAck &);
	bool HandleOpenLogicalChannelReject(H245_OpenLogicalChannelReject &);
	bool HandleCloseLogicalChannel(H245_CloseLogicalChannel &);

	RTPLogicalChannel *CreateRTPLogicalChannel(WORD, WORD);
	RTPLogicalChannel *CreateFastStartLogicalChannel(WORD);
	T120LogicalChannel *CreateT120LogicalChannel(WORD);
	bool RemoveLogicalChannel(WORD flcn);

	ProxyHandleThread *handler;
	std::map<WORD, LogicalChannel *> logicalChannels;
	std::map<WORD, RTPLogicalChannel *> sessionIDs;
	std::map<WORD, RTPLogicalChannel *> fastStartLCs;
	H245ProxyHandler *peer;
};

class CallSignalSocket : public TCPProxySocket {
public:
#ifndef LARGE_FDSET
	PCLASSINFO ( CallSignalSocket, TCPProxySocket )
#endif
	CallSignalSocket();
	CallSignalSocket(CallSignalSocket *, WORD);
	~CallSignalSocket();

	// override from class ProxySocket
        virtual Result ReceiveData();
	virtual bool EndSession();

	// override from class TCPProxySocket
	virtual TCPProxySocket *ConnectTo();

	// override from class PTCPSocket
	virtual BOOL Connect(const Address &);

	void SendReleaseComplete(const H225_CallTerminationCause * = 0);
	void SendReleaseComplete(H225_ReleaseCompleteReason::Choices);

	bool HandleH245Mesg(PPER_Stream &);
	void OnH245ChannelClosed() { m_h245socket = 0; }
	void BuildFacilityPDU(Q931 &, int, const PObject * = 0);
	void SetPeerAddress(const Address &, WORD);
	bool IsNATSocket() const { return m_isnatsocket; }

protected:
//	CallSignalSocket(CallSignalSocket *);
	void SetRemote(CallSignalSocket *);
	TCPProxySocket *InternalConnectTo();
	TCPProxySocket *ForwardCall();

	bool OnSetup(H225_Setup_UUIE &);
	bool OnCallProceeding(H225_CallProceeding_UUIE &);
	bool OnConnect(H225_Connect_UUIE &);
	bool OnAlerting(H225_Alerting_UUIE &);
	bool OnInformation(H225_Information_UUIE &);
	bool OnReleaseComplete(H225_ReleaseComplete_UUIE &);
	bool OnFacility(H225_Facility_UUIE &);
	bool OnProgress(H225_Progress_UUIE &);
	bool OnEmpty(H225_H323_UU_PDU_h323_message_body &);
	bool OnStatus(H225_Status_UUIE &);
	bool OnStatusInquiry(H225_StatusInquiry_UUIE &);
	bool OnSetupAcknowledge(H225_SetupAcknowledge_UUIE &);
	bool OnNotify(H225_Notify_UUIE &);
//	bool OnNonStandardData(PASN_OctetString &);
	bool OnTunneledH245(H225_ArrayOf_PASN_OctetString &);
	bool OnFastStart(H225_ArrayOf_PASN_OctetString &, bool);

	bool OnInformationMsg(Q931 &);

	template<class UUIE> bool HandleH245Address(UUIE & uu)
	{
		if (uu.HasOptionalField(UUIE::e_h245Address)) {
			if (!SetH245Address(uu.m_h245Address)) {
				uu.RemoveOptionalField(UUIE::e_h245Address);
				return true;
			}
			return (m_h245handler != 0);
		}
		return false;
	}
	template<class UUIE> bool HandleFastStart(UUIE & uu, bool fromCaller)
	{
		return (m_h245handler && uu.HasOptionalField(UUIE::e_fastStart)) ?
			OnFastStart(uu.m_fastStart, fromCaller) : false;
	}

	callptr m_call;

	// localAddr is NOT the local address the socket bind to,
	// but the local address that remote socket bind to
	// they may be different in multi-homed environment
	Address localAddr, peerAddr;
	WORD peerPort;

private:
	void InternalInit();
	void BuildReleasePDU(Q931 &, const H225_CallTerminationCause *) const;
	// if return false, the h245Address field will be removed
	bool SetH245Address(H225_TransportAddress &);

	WORD m_crv;
	H245Handler *m_h245handler;
	H245Socket *m_h245socket;
	bool m_h245Tunneling, m_isnatsocket;
	Result m_result;
	Q931 *m_lastQ931;
	H225_H323_UserInformation *m_setupUUIE;
};

inline void CallSignalSocket::SetPeerAddress(const Address & ip, WORD pt)
{
	peerAddr = ip, peerPort = pt;
}

inline bool CallSignalSocket::HandleH245Mesg(PPER_Stream & strm)
{
	return m_h245handler->HandleMesg(strm);
}

#endif // __proxychannel_h__

