//////////////////////////////////////////////////////////////////
//
//	WaitingARQ.h
//	
//	Waiting ARQ is used in the following way.
//  An ARQ is received in the Gk where the alias is a virtual queue ( CallCenter ).
//  The request is linked to a waiting queue ( WaitingARQlist ) while a GFA message  ( Get First Available Agent )
//  is sent via the status channel to a CallCenter queue system. An agent is selected and the call is presented to this agent. 
//  The call is answered by the agent ( pickup call) , a message GFC is sent back to the Gk with information 
//  callId , ip-address , alias.
//	The waiting ARQ list is scanned for this callId and the onARQ is called with a registred endpoint (the agent endpoint)  
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
// 	020300	initial version (Dennis Lazreg )
//
//////////////////////////////////////////////////////////////////

#ifndef WAITINGARQ_H
#define WAITINGARQ_H "#(@) $Id: WaitingARQ.h,v 1.1.2.12 2004/05/12 17:46:39 zvision Exp $"

#if HAS_WAITARQ

#include <list>
#include "singleton.h"
#include "stl_supp.h"
#include "RasSrv.h"

class WaitingARQlist : public Singleton<WaitingARQlist> {
		
		class WaitingARQ {

			private:
				
			int						m_requestSeqNumber; // Request Unique number
  			H225_AdmissionRequest	mm_ARQ; // Admission Request Record
			long					m_enterTime; // Enter time in this queue.
			PString					m_vAgent; // Virtual Agent name..
  			endptr					m_reqEP;


			public:

				PString               m_epId;							// Endpoint Identifier
				H225_RasMessage       m_ARQ;						    // Admission Request Record
				PIPSocket::Address    m_rx_addr;					    // Sender IP address
				int                   m_callReference;     // Call Reference number  

				int DoARQ (void);
				int DoARJ (void) const;

				bool CompReqId (int req_ID) const;
				bool CompCallRef( int callRef ) const; 
				bool CompEndpointId (PString epId ) const;  
				bool IsTimeOut (long) const;
				
				PString getEndPointIdentifier () const;
				PString getAgentQueue() const ; 
				
				WaitingARQ(int seqNum, const H225_RasMessage & obj_arq, const PIPSocket::Address & rx_addr,
									 const PString  &vAgent, const endptr & ep)
									: m_requestSeqNumber(seqNum), m_vAgent(vAgent), m_reqEP (ep), m_ARQ(obj_arq), m_rx_addr(rx_addr)
				{
					m_enterTime     = time(NULL);
					mm_ARQ          = m_ARQ ; 
					m_callReference = mm_ARQ.m_callReferenceValue ;
					m_epId          = mm_ARQ.m_endpointIdentifier.GetValue();
				};
			};

		private:
			/// a list of pending ARQs (RouteRequests)
			std::list<WaitingARQ *> wArqList;
			/// a mutex protecting ARQ list and config settings
			mutable PReadWriteMutex m_listMutex;
			/// virtual queues enabled/disabled
			bool m_active;
			/// a sequence number generator for WaitingARQs
			int m_sequenceNumber;
			/// a timeout (in seconds) for RouteRequests to be answered
			long m_requestTimeout;
			/// an array of aliases for virtual queues
			PStringArray m_callCenterQueues;
			/// an array of prefixes for virtual queues
			PStringArray m_callCenterPrefixes;
			/// a regular expression for virtual queues
			PString m_callCenterRegex;

	public:
		typedef std::list<WaitingARQ *>::iterator iterator;
		typedef std::list<WaitingARQ *>::const_iterator const_iterator;


		WaitingARQlist ();
		~WaitingARQlist();
		
		void LoadConfig();
		bool IsActive(void) const;

		bool InsertWaitingARQ(const H225_RasMessage & obj_arq, const PIPSocket::Address & rx_addr, const PString &ccQueue, const endptr & reqEP);
		bool IsWaitingARQ(const H225_AdmissionRequest& obj_arq);
		void CheckWaitingARQ();
		
		bool IsDestinationVirtualQueue(
			const PString& destination
			) const;
		bool RouteToAlias (
			const PString& TargetAlias, 
			const PString& SourceEpId, 
			const PString& CallRef
			);
		bool RouteReject (
			const PString& SourceEpId, 
			const PString& CallRef
			);

protected:
		iterator FindByRequestSequenceNumber (int);
		iterator FindByEndpointIdAndCallRef (const PString& epId , int callRef) ;
		
		iterator FindByEndpointId (PString epid);
		iterator FindByCallRefNumber (int);
};


inline PString WaitingARQlist::WaitingARQ::getEndPointIdentifier () const {
	 return (m_epId )  ;
}

inline PString WaitingARQlist::WaitingARQ::getAgentQueue() const {
	return (m_vAgent);
}


inline int WaitingARQlist::WaitingARQ::DoARQ(void)
{
	RasThread->DoARQ(m_rx_addr, m_ARQ);
	return 0;	// avoid VC++ 6.0 STL bug, must return something to be usable in mem_fun
}


inline int WaitingARQlist::WaitingARQ::DoARJ(void) const
{
	RasThread->ReplyARQ (m_reqEP, endptr(NULL), m_ARQ);
	return 0;	// avoid VC++ 6.0 STL bug, must return something to be usable in mem_fun
}


inline bool WaitingARQlist::WaitingARQ::CompReqId(int req_ID) const
{
		return (m_requestSeqNumber == req_ID);
}

inline bool WaitingARQlist::WaitingARQ::CompCallRef(int callRef) const
{
		return (m_callReference == callRef);
}

inline bool WaitingARQlist::WaitingARQ::CompEndpointId(PString ep) const
{
	return (m_epId == ep);
}

inline bool WaitingARQlist::WaitingARQ::IsTimeOut(long sec) const
{
	return (time(NULL) - m_enterTime) > sec;
}

inline WaitingARQlist::~WaitingARQlist()
{
	WriteLock lock(m_listMutex);
	DeleteObjectsInContainer<WaitingARQ>(wArqList);
}

inline bool WaitingARQlist::IsActive(void) const
{
	return m_active;
}

inline WaitingARQlist::iterator WaitingARQlist::FindByRequestSequenceNumber (int reqId)
{
	return find_if(wArqList.begin(), wArqList.end(), bind2nd(mem_fun(&WaitingARQ::CompReqId), reqId ));
}

inline WaitingARQlist::iterator WaitingARQlist::FindByCallRefNumber (int callRef)
{

		return find_if(wArqList.begin(), wArqList.end(), bind2nd(mem_fun(&WaitingARQ::CompCallRef), callRef  ));
}

inline WaitingARQlist::iterator WaitingARQlist::FindByEndpointId (PString epId)
{
		return find_if(wArqList.begin(), wArqList.end(), bind2nd(mem_fun(&WaitingARQ::CompEndpointId), epId  ));
}

inline WaitingARQlist::iterator WaitingARQlist::FindByEndpointIdAndCallRef (const PString& epId , int callRef)
{
	iterator iter = wArqList.begin();
	while (iter != wArqList.end()) {
		bool ep = (*iter)->m_epId == epId ;
		bool rf = (*iter)-> m_callReference==callRef ;
		if ( ep && rf ) { 
			return iter;
		}	
		++iter;
	}
	return iter ;
	
}


#endif // HAS_WAITARQ
#endif // WAITINGARQ_H
