//////////////////////////////////////////////////////////////////
//
// gk_wldap.cxx
//
// $Id: gk_wldap.cxx,v 1.1.2.3 2004/05/12 17:46:40 zvision Exp $
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
//      2003/10/01      initial version (Franz J Ehrengruber)
//                      based on gkldap.cxx (Markus Muehlenbernd)
//
//////////////////////////////////////////////////////////////////

// LDAP authentification
#if defined(HAS_WLDAP)

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include "gk_wldap.h"

// init file section name
const char *ldap_attr_name_sec = "GkLDAP::LDAPAttributeNames";
const char *ldap_auth_sec = "GkLDAP::Settings";

// constructor
GkLDAP::GkLDAP()
{
  LDAPConn=NULL;
  // LDAP Initialisation must be done with method "Initialize"!
} // GkLDAP constructor

// destructor
GkLDAP::~GkLDAP()
{
	Destroy();
  // LDAP Disconnection/unbind must be done with methode "Close"! 
} // GkLDAP destructor

void 
GkLDAP::Initialize(PConfig &cfg) { // 'real', private constructor 
  
  if(NULL!=LDAPConn)
	return;

  LDAP_TIMEVAL default_timeout;
  default_timeout.tv_sec = 10;
  default_timeout.tv_usec = 0;

  using namespace lctn;		// LDAP config tags and names
  // The defaults are given by the constructor of LDAPAttributeNamesClass
  AN.insert(LDAPANValuePair(LDAPAttrTags[CommonName],
 		                    cfg.GetString(ldap_attr_name_sec, 
					        LDAPAttrTags[CommonName],
					        "cn")));
  AN.insert(LDAPANValuePair(LDAPAttrTags[AccountStatus],
		                    cfg.GetString(ldap_attr_name_sec, 
					        LDAPAttrTags[AccountStatus],
					        "h323status")));
  AN.insert(LDAPANValuePair(LDAPAttrTags[H323ID],
		                    cfg.GetString(ldap_attr_name_sec, 
					        LDAPAttrTags[H323ID],
					        "H323ID")));
  AN.insert(LDAPANValuePair(LDAPAttrTags[TelephonNo],
			                cfg.GetString(ldap_attr_name_sec, 
					        LDAPAttrTags[TelephonNo],
					        "telephoneNumber")));
  AN.insert(LDAPANValuePair(LDAPAttrTags[H235PassWord],
			                cfg.GetString(ldap_attr_name_sec, 
					        LDAPAttrTags[H235PassWord],
					        "userPassword")));

  PString ServerName     = cfg.GetString(ldap_auth_sec, "ServerName", "ldap");
  int ServerPort         = cfg.GetString(ldap_auth_sec, "ServerPort", "389").AsInteger();
  PString SearchBaseDN   = cfg.GetString(ldap_auth_sec, "SearchBaseDN", 
					       "o=University of Michigan, c=US");
  PString BindUserDN     = cfg.GetString(ldap_auth_sec, "BindUserDN", 
					       "cn=Babs Jensen,o=University of Michigan, c=US");
  PString BindUserPW     = cfg.GetString(ldap_auth_sec, "BindUserPW", "RealySecretPassword");
  unsigned int timelimit = cfg.GetString(ldap_auth_sec, "timelimit", "0").AsUnsigned();


  LDAPConn = new LDAPCtrl(&AN, 
	                      default_timeout, 
						  ServerName, 
			              SearchBaseDN, 
						  BindUserDN, BindUserPW, 
			              timelimit, 
						  ServerPort);

} // Initialize

void 
GkLDAP::Close(void) {
	if(NULL!=LDAPConn)
		LDAPConn->Close();   // abandon all LDAP operations/unbind
} // Close

void 
GkLDAP::Destroy() { // 'real', private destructor
	if(NULL!=LDAPConn)
		delete LDAPConn;     // delete the LDAPCtrl object
} // Destroy

PString 
GkLDAP::convertE123ToDialedDigits(PString e123) {
  e123.Replace("+","");
  // remove all whitespaces
  e123.Replace(" ","", TRUE);
  // remove all "."
  e123.Replace(".","", TRUE);
  return e123;
}

int 
GkLDAP::getAttributes(const PString & common_name, const PStringArray & attr_names, LDAPAnswer * answer) {
	PWaitAndSignal lock(m_usedLock);

	int attribute_count = 0;
	
	// initiate LDAP search
	int num_entries = LDAPConn->LdapUserLookup(common_name, attr_names, answer); 

	//success - collect attributes names and values
	if(num_entries > 0) {
		attribute_count = LDAPConn->LdapCollectAttributes(answer);
	}
	return attribute_count;
}

int 
GkLDAP::getAttributeValues(LDAPAnswer * answer, const int & attribute_name, PStringList & attrib_values) { 
	int attribvalue_count = 0;
	using namespace lctn;
	if (answer->LDAPec.size()) {
		LDAPEntryClass::iterator pFirstDN = answer->LDAPec.begin();
		if((pFirstDN->second).count(AN[LDAPAttrTags[attribute_name]])) {
			attrib_values = (pFirstDN->second)[AN[LDAPAttrTags[attribute_name]]];
			if(!attrib_values.IsEmpty()) {
				for (PINDEX j = 0; j < attrib_values.GetSize(); j++) {
					PString attrvalue = attrib_values[j];
					if(!attrvalue.IsEmpty()) {
						PTRACE(5, "LDAP\tGkLDAP::getAttributeValues() attrvalue '" << attrvalue << "' for attribute '" << AN[LDAPAttrTags[attribute_name]] << "'");
						attribvalue_count++;
					}
				}
			}
		}
	}
	return attribvalue_count;
}

bool 
GkLDAP::validateAliases(const H225_ArrayOf_AliasAddress & aliases, PString & H323ID_attrvalue, PStringList & TelephonNo_values) {
	PString aliasStrE164;
	PString aliasStrH323ID;
	bool found = false;

	// check H323ID in 'H225_ArrayOf_AliasAddress' match the one returned by LDAP
	for (PINDEX i = 0; i < aliases.GetSize() && !found; i++) {
		// h323id
		if (aliases[i].GetTag() == H225_AliasAddress::e_h323_ID) {
			aliasStrH323ID = H323GetAliasAddressString(aliases[i]);
			PTRACE(5, "LDAP\tGkLDAP::validateAliases() aliasStrH323ID '" << aliasStrH323ID << "' H323ID_attrvalue '" << H323ID_attrvalue << "'");
			if(!H323ID_attrvalue.IsEmpty()) {
				if(aliasStrH323ID != H323ID_attrvalue) {
					PTRACE(4, "LDAP\tGkLDAP::validateAliases '" << aliasStrH323ID << "' not found in LDAP directory");
					return false;
				}
				else
					found = true;
			}
		}

	} // for

	// H323ID required
	if(!found)
		return found;

	// check the e164(s) in 'H225_ArrayOf_AliasAddress' match the ones returned by LDAP
	//
	// the rules:
	// 1) if we have a single e164 alias in 'H225_AliasAddress::e_dialedDigits':
	//    we return TRUE on the first match found in PString list 'telephoneNumbers[]'.
	//
	// 2) if we have MORE then ONE  e164 alias in 'H225_AliasAddress::e_dialedDigits':
	//    we return TRUE, if EACH e164 alias has a matching entry in PString list 'telephoneNumbers[]'.
	found = false;
	if(aliases.GetSize() > 0) {
		i = 0;
		do {
			if (aliases[i].GetTag() == H225_AliasAddress::e_dialedDigits) {
				aliasStrE164 = H323GetAliasAddressString(aliases[i]);
				PTRACE(5, "LDAP\tGkLDAP::validateAliases() aliasStrE164 '" << aliasStrE164 << "'");
				// check if the supplied alias[i] exists in 
				// the list of telephoneNumbers from LDAP attribute
				found = false;
				for (PINDEX j = 0; j < TelephonNo_values.GetSize(); j++) {
					PTRACE(5, "LDAP\tGkLDAP::validateAliases() TelephonNo_values["<< j <<"] '" << TelephonNo_values[j] << "' aliasStrE164 '" << aliasStrE164 << "'");
					// match - get the next e164 if any
					if(aliasStrE164 == convertE123ToDialedDigits(TelephonNo_values[j])) {
						found = true;
						continue;
					}
			
				} // for
			
				// no match - dont waist your time
				if(!found)
					return found;
			
			} //if .GetTag()

			i++;
		} while(i < (aliases.GetSize()));
	} // if

	// return success/failure
	return found;
}

#endif // HAS_WLDAP
