/*
 * sqlacct.h
 *
 * SQL accounting module for GNU Gatekeeper
 *
 * Copyright (c) 2004, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: sqlacct.h,v $
 * Revision 1.1.2.1  2004/04/23 16:01:16  zvision
 * New direct SQL accounting module (SQLAcct)
 *
 */
#ifdef HAS_ACCT

#ifndef SQLACCT_H
#define SQLACCT_H "@(#) $Id: sqlacct.h,v 1.1.2.1 2004/04/23 16:01:16 zvision Exp $"

#include "gkacct.h"

/** This accounting module stores call information directly to an SQL database.
    It uses generic SQL interface, so different SQL backends are supported.
    Queries to store accounting information are parametrized using named 
    parameters. Currently, the following are supported:
	%g - gatekeeper name
	%n - call number (not unique after gatekeeper restart)
	%d - call duration (seconds)
	%c - Q.931 disconnect cause (decimal integer)
	%s - unique (for this gatekeeper) call (Acct-Session-Id)
	%u - H.323 ID of the calling party
	%{CallId} - H.323 call identifier (16 hex 8-bit digits)
	%{ConfId} - H.323 conference identifier (16 hex 8-bit digits)
	%{setup-time} - timestamp string for Q.931 Setup message
	%{connect-time} - timestamp string for a call connected event
	%{disconnect-time} - timestamp string for a call disconnect event
	%{caller-ip} - signaling IP addres of the caller
	%{caller-port} - signaling port of the caller
	%{callee-ip} - signaling IP addres of the called party
	%{callee-port} - signaling port of the called party
	%{src-info} - a colon separated list of source aliases
	%{dest-info} - a colon separated list of destination aliases
	%{Calling-Station-Id} - calling party number
	%{Called-Station-Id} - called party number
*/
class GkSQLConnection;
class SQLAcct : public GkAcctLogger
{
	PCLASSINFO(SQLAcct,GkAcctLogger)
public:
	enum Constants {
		/// events recognized by this module
		SQLAcctEvents = AcctStart | AcctUpdate | AcctStop
	};
	
	/// Create a logger that sends accounting to an SQL database
	SQLAcct( 
		/// config to be read for module settings
		PConfig& cfg, 
		/// name from Gatekeeper::Acct section
		const PString& moduleName,
		/// name for a config section with logger settings
		/// pass NULL to use the moduleName as the section name
		const char* cfgSecName = NULL
		);
		
	/// Destroy the accounting logger
	virtual ~SQLAcct();

	/** Log accounting event.
	
		@return
		Status of this logging operation (see #Status enum#)
	*/
	virtual Status LogAcctEvent( 
		AcctEvent evt, /// accounting event to log
		callptr& call /// additional data for the event
		);

protected:
	/** Fill the map with query parameters (name => value associations).
	    Derived classes may override this method to append some custom
	    parameters. Do not forget to call the base class implementation then
	    to setup default parameters!
	*/
	virtual void SetupQueryParams(
		/// query parameters (name => value) associations
		std::map<PString, PString>& params,
		/// call (if any) associated with an accounting event being logged
		callptr& call
		) const;
			
private:
	/* No copy constructor allowed */
	SQLAcct(const SQLAcct&);
	/* No operator= allowed */
	SQLAcct& operator=(const SQLAcct&);

private:
	/// connection to the SQL database
	GkSQLConnection* m_sqlConn;
	/// parametrized query string for the call start event
	PString m_startQuery;
	/// parametrized alternative query string for the call start event
	PString m_startQueryAlt;
	/// parametrized query string for the call update event
	PString m_updateQuery;
	/// parametrized query string for the call stop event
	PString m_stopQuery;
	/// parametrized alternative query string for the call stop event
	PString m_stopQueryAlt;
	/// for fast access
	PString m_gkName;
};

#endif /* SQLACCT_H */
#endif /* HAS_ACCT */
