/*
 * ipaudit_util.c
 *
 * Copyright 2001,2002 Chris Green <cmg@uab.edu>
 *
 * $Id: ipaudit_util.c,v 1.47 2002/02/06 03:32:20 cmg Exp $
 *
 * Some taken from ipstrings.c
 * By Jon Rifkin <jon.rifkin@uconn.edu>
 * Copyright 1999,2000 Jonathan Rifkin
 * 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ipaudit_util.h"
#include "ipaudits.h"
#include "str_util.h"
#include <assert.h>
#include <math.h>

extern int line_count;
static time_t now = 0;

/* returns 1 if we are in dst, 0 otherwise */
int is_dst() {
    static int dst = -1;
    struct tm *t;
    time_t t_temp;
    if(dst == -1) {
	t_temp = time(NULL);
	t = localtime(&t_temp);
	dst = t->tm_isdst;
    }
    return dst;
}

/*
  returns a pointer to a static buffer representing the date for the
  start range 
*/
char *time2file_max(time_t when) {
    static char ret[256];
    size_t s;
    struct tm *tm;
    
    tm = localtime(&when);

    if(tm->tm_min < 30) {
	tm->tm_min = 0;
    } else {
	tm->tm_min = 30;
    }

    s = strftime(ret, sizeof(ret) - 1, "%Y-%m-%d-%H:%M.txt.gz", tm);
    
    if(!s) {
	fprintf(stderr,"something's fishing in time to file for %d\n", (int) when);
	exit(1);
    }
    return ret;
}

/* returns a pointer to a static buffer representing the date for the
   end range  */
char *time2file_min(time_t when) {
    static char ret[256];
    size_t s;
    struct tm *tm;

    /* when minus a half hour and floor to the lowest 1/2 hour */
    when = when - (when % 1800);
    tm = localtime(&when);

    if(tm->tm_min < 30) {
	tm->tm_min = 0;
    } else {
	tm->tm_min = 30;
    }

    s = strftime(ret, sizeof(ret) - 1, "%Y-%m-%d-%H:%M.txt.gz", tm);
    
    if(!s) {
	fprintf(stderr,"something's fishing in time to file for %d\n", (int) when);
	exit(1);
    }
    return ret;
}


/*
  converts dates like 2001-05-04-10:30 to seconds

  also handles filenames ;-)
*/
time_t str2time(const char *datestr) {
    static struct tm when;
    char *date_format = "%Y-%m-%d-%H:%M";
    char *date_format2 = "%Y-%m-%d";

    char *t;
#ifdef DEBUG
    /* printf("str2time got %s", datestr); */
#endif /*DEBUG*/

    if(datestr == NULL) {
	return time(NULL);
    }
   
    memset(&when, 0, sizeof(struct tm));
    when.tm_isdst = is_dst();
    
    t =	strptime(datestr, date_format, &when);
    
    if(t == NULL) {
	t = strptime(datestr, date_format2, &when);
	if(t == NULL) {
	    fprintf(stderr, "Invalid Date: %s\n", datestr);
	    exit(1);
	}
    }

    return mktime(&when);
}


/* convert time in ipaudit time format to size_t ints
   14:00:00.8094

   returns 0 if something goes wrong, 1 if everything went ok
*/
int ia2time(char *tstr, iptm *when) {
    char *ret, *tmp;
    char *msec_string = NULL;
    
    /* find the first . in the string set it to null ( so the tstr is
       now pointing to a null terminated string ) and increment it 
    */

    /* since index crashes on a NULL, we need to check it first */
    if(tstr == NULL)
	return 0;

    tmp = Strdup(tstr);
    
    msec_string = index(tmp,'.');

    if(msec_string == NULL) {
	fprintf(stderr, "msec_string is NULL! for %s\n", tstr);
	free(tmp);
	return 0;
    } else {
	*msec_string = '\0';
    }

    /* make sure we aren't at the end of the string */
    if( *(msec_string + 1) != '\0') {
	msec_string = msec_string + 1;
    } else {
	fprintf(stderr, "msec_string+1 is NULL for %s\n", tstr);
	free(tmp);
	return 0;
    }

    when->msec = atoi(msec_string);

    ret = strptime(tstr, "%H:%M:%S", &when->date_struct);

    free(tmp);
    
    if(ret==NULL) {
	return 0;
    }
    return 1;	
}

/* compare two timeformat things

returns -1 if a < b
return  0 if a == b
return 1 if a > b

*/
int ia_cmp(iptm *a, iptm *b) {
    int ret;

    /* years */
    ret = a->date_struct.tm_year - b->date_struct.tm_year;
    if (ret < 0) {
	return -1;
    } else if ( ret > 0 ) {
	return 1;
    }

    ret = a->date_struct.tm_mon - b->date_struct.tm_mon;
    if (ret < 0) {
	return -1;
    } else if ( ret > 0 ) {
	return 1;
    }

    ret = a->date_struct.tm_hour - b->date_struct.tm_hour;
    if (ret < 0) {
	return -1;
    } else if ( ret > 0 ) {
	return 1;
    }

    ret = a->date_struct.tm_min - b->date_struct.tm_min;
    if (ret < 0) {
	return -1;
    } else if ( ret > 0 ) {
	return 1;
    }
    
    ret = a->date_struct.tm_sec - b->date_struct.tm_sec;
    if (ret < 0) {
	return -1;
    } else if ( ret > 0 ) {
	return 1;
    }

    ret = a->msec - b->msec;
    if (ret < 0) {
	return -1;
    } else if ( ret > 0 ) {
	return 1;
    }
    
    /* they are equal */
    return 0;    
}


/*
  Converts a human readable talker form to an integer
  1 local
  2 remote
  
  return -1 on error.
*/
int get_talker(char *talker) {
    if(!talker || strlen(talker) != 1) {
	return -1;
    }
    
    if (! strncasecmp("L", talker, 1) ||
	! strncasecmp("1", talker, 1)) {
        return 1;
    } else if (! strncasecmp("R", talker, 1) ||
	       ! strncasecmp("2", talker, 1)) {
	return 2;
    }

    return -1;
}

char int2talker(int t) {
    switch(t) {
    case 1:
	return 'L';
    case 2:
	return 'R';
    default:
	return '-';
	
    }
}


/*  
    Read and format ip address :
    for example,  "112.51.1" -> "112.051.001"
*/
char *get_localstr(char *instr) {
    /*  Room for string containing formated ip address  */
    char *buffer = (char *) calloc(16,sizeof(char)); 
    char *b = buffer;
    char triplet[3];
    char *p = instr;
    int  i,n;
    
    /*  Search over all bytes (things between ..)  */
    while (1) {
	n = 0;
	while ((*p>='0' && *p<='9') && n<3) 
	    triplet[n++] = *p++;
	/*  Either non-digit or two many digits - return NULL */
	if (*p!='.' && *p!=0) return NULL;
	for (i=n;i<3;i++) *b++ = '0';
	for (i=0;i<n;i++) *b++ = triplet[i];
	*b++ = *p;
	if (*p==0) {
	    return buffer;
	}
	p++;
    }
}

/* returns a malloced buffer for the name */
char *Getprotobynumber(int pnum, ProtoName *pname) {
    struct protoent *p;
    p = getprotobynumber(pnum);

    if(p == NULL) {
	snprintf(pname->value,  PNAME_MAX, "%d", pnum);
    } else {
	snprintf(pname->value, PNAME_MAX, "%s", p->p_name);
    }

    return pname->value;
}

/* returns a "human readable"  form of the number of bytes */
char *bytes2human(double bytes, ByteString *b) {
    double converted;
    char unit = 0;
    memset(b->value,0,BYTESTRING_MAX);
    
    if(bytes < KBYTE) {
	converted = bytes; /* nothing to do - already in bytes */
    } else if (bytes < MBYTE) {
	converted = bytes/ (float) KBYTE;
	unit = 'K';
    } else if (bytes < GBYTE) {
	converted = bytes / (float) MBYTE;
	unit = 'M';
    } else {
	converted = bytes / (float) GBYTE;
	unit = 'G';
    }

    if(unit) {
	snprintf(b->value, BYTESTRING_MAX, "%.2f%c", converted, unit);
    } else {
	snprintf(b->value, BYTESTRING_MAX, "%d", (int) converted);
    }
    return b->value;
}

/* returns a byte count from a "human readable"  form */
int human2bytes(const char *str) {
    char *cp;
    double bytefloat;
    int conversion = 1;
    char *mystr = Strdup(str);
    int found_decimal = 0;     /* allow a single decimal place in our string */

    cp = mystr;

    while(*cp != '\0' && (isdigit(*cp) || *cp == '.')) {
	if(*cp == '.') {
	    if (!found_decimal) {
		found_decimal = 1;
	    } else {
		fprintf(stderr,
			"human2bytes: Found multiple decimal places in %s\n",
			str);
		exit(1);
	    }
	}
	cp++;
    }

    /* process the unit if there is one */
    if(*cp != '\0') {
	switch(toupper(*cp)) {
	case 'B':
	    conversion = 1;
	    break;
	case 'K':
	    conversion = KBYTE;
	    break;
	case 'M':
	    conversion = MBYTE;
	    break;
	case 'G':
	    conversion = GBYTE;
	    break;
	default:
	    fprintf(stderr, "Unknown conversion factor %c found in %s\n", *cp, str);
	    exit(1);
	}
	/* where units start, the string ends */
	cp = '\0';
    }

    bytefloat = strtod(mystr, NULL);
    free(mystr);

    bytefloat = bytefloat * conversion;
    return (int) floor(bytefloat);
}


/*  
    convert a ip in 001.001.001.001 format to
    1.1.1.10
*/
char *demunge_ip(char *instr, IPString* ret) {
    /*  Room for string containing formated ip address  */
    char *buffer = ret->value;
    char *b = buffer;
    char triplet[3];
    char *p = instr;
    int  i,n;
    int seen_nonzero = 0; 

    i = strlen(instr);
    /* make sure that we atleast have a chance of being a valid IP
       1.1.1.1 (7)     111.111.111.111 (15)  */
    if(i < 7 || i > 15) return NULL;
	
    /*  Search over all bytes (things between ..)  */
    while (1) {
	n = 0;
	seen_nonzero = 0;
	/* makes sure that we get up to 3 things that are a digit */
	while ((*p>='0' && *p<='9') && n<3) {
	    triplet[n++] = *p++;
	}

	/*  Either non-digit or two many digits - return NULL */
	if (*p!='.' && *p!='\0') return NULL;

	/* Walk through the triplet of chars and as long as we haven't
	   seen '0', skip 0's in the output buffer
	*/
	for (i=0;i<n;i++) {
	    if ((triplet[i] == '0') && !seen_nonzero) {
		continue;
	    } else {
		seen_nonzero = 1;
	    }
	    *b++ = triplet[i];
	}

	/* make sure we put a 0 atleast */
	if(!seen_nonzero) *b++ = '0';
	
	*b++ = *p; /* maps either the . or the \0' */
	if (*p=='\0')	return buffer;
	p++;
    }
}

/*  
    Read and format ip address :
    for example,  "112.51.1" -> "112.051.001"
*/

char *munge_ip(char *old, char *new) {
    /*  Room for string containing formated ip address  */
    char *b = new;
    char triplet[3];
    char *p = old;
    int  i,n;
    
    /*  Search over all bytes (things between ..)  */
    while (1) {
	n = 0;
	while ((*p>='0' && *p<='9') && n<3) 
	    triplet[n++] = *p++;
	/*  Either non-digit or two many digits - return NULL */
	if (*p!='.' && *p!=0) return NULL;
	for (i=n;i<3;i++) *b++ = '0';
	for (i=0;i<n;i++) *b++ = triplet[i];
	*b++ = *p;
	if (*p==0)	return new;
	p++;
    }
    return new;
}

/*
  Given a line in ipaudit format, parse it into a IAD struct.
  
  138.026.155.028 216.033.236.250 6 1064 80  54 54 1 1 14:00:00.8094 14:00:00.8908 1 2
  0               1               2  3    4   5 6  7 9  9                  10     11 12

  Implementation:

  parse it into an a parms array and then strdup,atoi,kick,scream into
  the supplied IAD struct

  depth is the number of fields in we need to parse
*/

#define FIELD_COUNT 13
int parse_ipaudit_line(char *line, int depth, IAD *data, iptm *curdate, IPString *ip_local, IPString *ip_remote)
{
    int fc;
    char *sp, *cp; /* start and current */
    char *parms[FIELD_COUNT];

    bzero(parms, FIELD_COUNT);

    assert(*line != '\0'); /* should fix to handle gracefully */

    sp = cp = line;  /* everyone starts out the same */

    
    fc = 0;

    if(depth > IAD_NONE) {
	while(fc <= depth && fc < FIELD_COUNT) {
	    sp = cp;

	    /* keep incrementing until we see a space */
	    while(!isspace(*cp) && *cp != '\0') {
		cp++;
	    }
	
	    *cp = '\0';
	    /* make sure we set the rest of the string after we've
	       gone through and set the "rest field"
	       
	       
	    */
	    cp++;

	    parms[fc] = sp;
	
	    while(isspace(*cp)) {
		if(*cp == '\0' && (fc != depth)) {
		    fprintf(stderr, "parse_ipaudit_line: null at before my time");
		    fprintf(stderr,
			    "parse_ipaudit_line: fc: %d: depth: %d line: %s",
			    fc, depth, line);
		    exit(1);
		}
		cp++;
	    }
	
	    switch(fc) {
	    case IAD_LOCAL_IP:
		data->local_ip = parms[IAD_LOCAL_IP];
		break;
	    case IAD_REMOTE_IP:
		data->remote_ip = parms[IAD_REMOTE_IP];
		break;
	    case IAD_PROTOCOL:
		data->protocol = atoi(parms[IAD_PROTOCOL]);
		break;
	    case IAD_LPORT:
		data->lport = atoi(parms[IAD_LPORT]);
		break;
	    case IAD_RPORT:
		data->rport = atoi(parms[IAD_RPORT]);
		break;
	    case IAD_BYTES_RECV:
		data->bytes_recv = atoi(parms[IAD_BYTES_RECV]);
		break;
	    case IAD_BYTES_SENT:
		data->bytes_sent = atoi(parms[IAD_BYTES_SENT]);
		break;
	    case IAD_PACKETS_RECV:
		data->packets_recv = atoi(parms[IAD_PACKETS_RECV]);
		break;
	    case IAD_PACKETS_SENT:
		data->packets_sent = atoi(parms[IAD_PACKETS_SENT]);
		break;
	    case IAD_FIRST_TALKER:
		data->first_talker = atoi(parms[IAD_FIRST_TALKER]);
		break;
	    case IAD_LAST_TALKER:
		data->last_talker = atoi(parms[IAD_LAST_TALKER]);
		break;
	    case IAD_FIRST_STR:
		data->first_timestr = parms[IAD_FIRST_STR];
		break;
	    case IAD_LAST_STR:
		data->last_timestr = parms[IAD_LAST_STR];
		break;
	    case IAD_FIRST_TIME:
	    case IAD_LAST_TIME:
	    case IAD_FILL_ALL:
	    case IAD_NONE:
	    default:
		printf("should never get here %d: %s\n", fc, line);
		exit(1);
	    }
	    fc++;
	}
    } else { 
	data->rest = line;
	return 0;
    }

    /* When we parse the entire string, there is no more to parse
       after the LAST_TALKER field, there should be a way to rewrite
       this into a cleaner automaton
    */

    if(fc >= IAD_LAST_TALKER) {
	data->rest = NULL;
    } else {
	data->rest = cp;
    }

    data->filled = depth;
    
    return 0;
}
    

int debug_print_iad(IAD *data) {
    char s[256];
    fprintf(stdout, "Local IP: %s\n", data->local_ip);
    fprintf(stdout, "Remote IP: %s\n", data->remote_ip);
    fprintf(stdout, "Protocol: %d\n", data->protocol);
    fprintf(stdout, "Local Port: %d\n", data->lport);
    fprintf(stdout, "Remote Port: %d\n", data->rport);
    fprintf(stdout, "Bytes Recv: %d\n", data->bytes_recv);
    fprintf(stdout, "Bytes Sent: %d\n", data->bytes_sent);
    fprintf(stdout, "Packets Recv: %d\n", data->packets_recv);
    fprintf(stdout, "Packets Sent: %d\n", data->packets_sent);

    strftime(s, 255, "%Y-%m-%d-%H:%M:%S", &data->first_time.date_struct);
    fprintf(stdout, "First Time: %s.%d\n", s, data->first_time.msec);

    strftime(s, 255, "%Y-%m-%d-%H:%M:%S", &data->last_time.date_struct);
    fprintf(stdout, "Last Time: %s.%d\n", s, data->last_time.msec);
    fprintf(stdout, "First Talker: %d\n", data->first_talker);
    fprintf(stdout, "Last Talker: %d\n", data->first_talker);
    return 0;
}

#define TIME_LEN 256
int pretty_print_iad(IAD *data, int displaylinecount, const char *dateprefix) {
    int depth = data->filled;
    int fc = 0;

    ProtoName pname;
    ByteString inb;
    IPString ip;

    if(displaylinecount) {
	printf("%5d: ", line_count);
    }
	

    if(depth > IAD_NONE && data->local_ip) {
	for(fc=0;fc <= depth && fc<=IAD_LAST_TALKER;fc++) {
#ifdef DEBUG
	    printf("fc is %d , depth is %d\n", fc, depth);
#endif
	    switch(fc) {
	    case IAD_LOCAL_IP:
		printf("%-15s ",demunge_ip(data->local_ip,&ip));
		break;
	    case IAD_REMOTE_IP:
		printf("%-15s ", demunge_ip(data->remote_ip,&ip));
		break;
	    case IAD_PROTOCOL:
		Getprotobynumber(data->protocol, &pname);
		printf("%4s ",Getprotobynumber(data->protocol, &pname));
		break;
	    case IAD_LPORT:
		printf("%5d ", data->lport);
		break;
	    case IAD_RPORT:
		printf("%5d ", data->rport);
		break;
	    case IAD_BYTES_RECV:
		printf("%6s ",  bytes2human(data->bytes_recv, &inb));
		break;
	    case IAD_BYTES_SENT:
		printf("%6s ",  bytes2human(data->bytes_sent, &inb));
		break;
	    case IAD_PACKETS_RECV:
		printf("%5d ",  data->packets_recv);
		break;
	    case IAD_PACKETS_SENT:
		printf("%5d ",  data->packets_sent);
		break;
	    case IAD_FIRST_STR:
		if(dateprefix) {
		    printf("%s%s ", dateprefix, data->first_timestr);
		} else {
		    printf("%s ", data->first_timestr);
		}
		break;
	    case IAD_LAST_STR:
		if(dateprefix) {
		    printf("%s%s ", dateprefix, data->last_timestr);
		} else {
		    printf("%s ", data->last_timestr);
		}
		break;
	    case IAD_FIRST_TALKER:
		printf("%c ", int2talker(data->first_talker));
		break;
	    case IAD_LAST_TALKER:
		printf("%c ", int2talker(data->last_talker));
		break;
	    case IAD_FILL_ALL:
	    case IAD_NONE:
	    default:
		printf("should never get here %d\n", fc);
		exit(1);

	    }
	}
    }

    if(data->rest) {
	printf(" %s\n", data->rest);
    } else {
	printf("\n");
    }

    return 0;
}

int raw_print_iad(IAD *data,  int displaylinecount, const char *dateprefix) {
    const int depth = data->filled;
    int fc = 0;

    if(displaylinecount) {
	printf("%5d: ", line_count);
    }
	

    if(depth > IAD_NONE && data->local_ip) {
	for(fc=0;fc <= depth && fc<=IAD_LAST_STR;fc++) {
	    switch(fc) {
	    case IAD_LOCAL_IP:
		printf("%-15s ",data->local_ip);
		break;
	    case IAD_REMOTE_IP:
		printf("%-15s ", data->remote_ip);
		break;
	    case IAD_PROTOCOL:
		printf("%2d ",data->protocol);
		break;
	    case IAD_LPORT:
		printf("%5d ", data->lport);
		break;
	    case IAD_RPORT:
		printf("%5d ", data->rport);
		break;
	    case IAD_BYTES_RECV:
		printf("%8d ",  data->bytes_recv);
		break;
	    case IAD_BYTES_SENT:
		printf("%8d ",  data->bytes_sent);
		break;
	    case IAD_PACKETS_RECV:
		printf("%5d ",  data->packets_recv);
		break;
	    case IAD_PACKETS_SENT:
		printf("%5d ",  data->packets_sent);
		break;
	    case IAD_FIRST_STR:
		if(dateprefix) {
		    printf("%s%s ", dateprefix, data->first_timestr);
		} else {
		    printf("%s ", data->first_timestr);
		}
		break;
	    case IAD_LAST_STR:
		if(dateprefix) {
		    printf("%s%s ", dateprefix, data->last_timestr);
		} else {
		    printf("%s ", data->last_timestr);
		}
		break;
	    case IAD_FIRST_TALKER:
		printf("%d ", data->first_talker);
		break;
	    case IAD_LAST_TALKER:
		printf("%d ", data->last_talker);
		break;
	    case IAD_FILL_ALL:
	    case IAD_NONE:
	    default:
		printf("should never get here %d\n", fc);
		exit(1);

	    }
	}
    }

    if(data->rest) {
	printf(" %s\n", data->rest);
    } else {
	printf("\n");
    }

    return 0;
}

/* set the static variable now to the current time */
void init_now(time_t n) {
    if(!now && !n) {
	now = time(NULL);
	now -= (1800 + (now & 1800));
    } else if (n) {
	now = n;	
    }
}

void init_portnode(PortNode *node) {
    if(node) {
	node->operator = PORT_NOP;
	node->min_port = PORT_INIT;
	node->max_port = PORT_INIT;
	node->next = NULL;	
    } else {
	fprintf(stderr, "init_portnode: node was NULL!\n");
    }
}

void init_iptm(time_t t, iptm *iptime) {
    struct tm *tp;

    static int init = 1;
    static iptm tsave;

    /* 1 time initialization */
    if(init) {
	init = 0;
	init_now(0);
	tp = localtime(&t);
	tsave.date_struct.tm_sec = tp->tm_sec;
	tsave.date_struct.tm_min = tp->tm_min;
	tsave.date_struct.tm_hour= tp->tm_hour;
	tsave.date_struct.tm_mday = tp->tm_mday;
	tsave.date_struct.tm_mon = tp->tm_mon;
	tsave.date_struct.tm_year = tp->tm_year;
	tsave.date_struct.tm_wday = tp->tm_wday;
	tsave.date_struct.tm_yday = tp->tm_yday;
	tsave.date_struct.tm_isdst = tp->tm_isdst;
    }
    
    memcpy(iptime, &tsave, sizeof(iptm));

    
}
/* given a IAD structure, initialize the int fields to -1 and the
   pointers to NULL;
*/
void init_search(Search *data) {
    struct passwd *pent;

    init_now(0);
    
    data->local_ip = NULL;
    data->remote_ip = NULL;
    data->protocol = -1;
    data->lport = NULL;
    data->rport = NULL;
    data->min_bytes = -1;
    data->max_bytes = -1;
    data->min_packets = -1;
    data->max_packets = -1;

    data->start_time = 0;
    data->end_time = 0;

   
    init_iptm(now, &data->iptm_start_time);
    init_iptm(now, &data->iptm_end_time);
    
    data->start_time = now;

    data->first_talker = -1;
    data->last_talker = -1;
    data->match = MATCH_ANY;
    data->limit = LIMIT;


    while((pent = getpwent())!=NULL) {
	if (!(strncmp(pent->pw_name, "ipaudit", strlen(pent->pw_name))))
	    break;
    }

    if(pent==NULL) {
	fprintf(stderr, "Warning: no ipaudit user found\n");
	data->ipaudit_30min = Strdup("/home/ipaudit/data/30min");
    } else {
	data->ipaudit_30min = Strconcat(pent->pw_dir, "/data/30min");
    }
}


/* given a IAD structure, initialize the int fields to -1 and the
   pointers to NULL;
*/
void init_iad(IAD *data) {
    int init = 1;
    static IAD iadsave;

    if(init) {
	init = 0;
	iadsave.rest = NULL;
	iadsave.local_ip = NULL;
	iadsave.remote_ip = NULL;	
	iadsave.protocol = -1;
	iadsave.lport = -1;
	iadsave.rport = -1;
	iadsave.bytes_sent = -1;
	iadsave.bytes_recv = -1;
	iadsave.packets_sent = -1;
	iadsave.packets_recv = -1;
	init_iptm(now, &iadsave.first_time);
	init_iptm(now, &iadsave.last_time);
	iadsave.first_talker = -1;
	iadsave.last_talker = -1;
    }

    memcpy(data, &iadsave, sizeof(IAD));
}

int check_iad(IAD *data, Search *search, const char *prefix) {
    int byte_sum = 0;
    int packet_sum = 0;
    time_t i_first, i_last;

#ifdef DEBUG
    //printf("data->filled is %d search->need is %d\n", data->filled, search->need);
#endif

    if(search->need == IAD_NONE) {
	return 1;
    }

    if(search->local_ip) {
	if(strncmp(data->local_ip, search->local_ip, strlen(search->local_ip))) {
	    return 0; 
	}
    }

    if(search->remote_ip) {
	if(strncmp(data->remote_ip, search->remote_ip, strlen(search->remote_ip))) {
	    return 0; 
	}
    }

    if(search->match != MATCH_ANY) {
	switch(data->protocol) {
	case ICMP:
	    if(!(search->match & MATCH_ICMP)) {
		return 0;
	    }
	    break;
	case TCP:
	    if(!(search->match & MATCH_TCP)) {
		return 0;
	    }
	    break;
	case UDP:
	    if(!(search->match & MATCH_UDP)) {
		return 0;
	    }
	    break;
	default:
	    return 0;
	}
    }

    if(search->lport) {
	if(!check_portnode(search->lport, data->lport)) {
	    return 0;
	}
    }

    if(search->rport) {
	if(!check_portnode(search->rport, data->rport)) {
	    return 0;
	}
    }
    
    if(search->first_talker != -1) {
	if(data->first_talker != search->first_talker) {
	    return 0;
	}
    }

    if(search->last_talker != -1) {
	if(data->last_talker != search->last_talker) {
	    return 0;
	}
    }

    if((search->min_bytes != -1) || (search->max_bytes != -1)) {
	byte_sum  = data->bytes_recv + data->bytes_sent;

	if(search->min_bytes != -1) {
	    if(byte_sum < search->min_bytes) {
		return 0;
	    }
	}

	if(search->max_bytes != -1) {
	    if(byte_sum > search->max_bytes) {
		return 0;
	    }
	}
    }

    
    if((search->min_packets != -1) || (search->max_packets != -1)) {
	packet_sum  = data->packets_recv + data->packets_sent;

	if(search->min_packets != -1) {
	    if(packet_sum < search->min_packets) {
		return 0;
	    }
	}

	if(search->max_packets != -1) {
	    if(packet_sum > search->max_packets) {
		return 0;
	    }
	}
    }

    if (search->need >= IAD_FIRST_TIME) {
	printf("%d == search->need\n", search->need);
	i_first = find_time(prefix, data->first_timestr);
	i_last = find_time(prefix, data->last_timestr);
	    mktime(&data->last_time.date_struct); 
	
	if(search->end_time < i_first) { 
	    return 0; 
	} 

	if(search->start_time > i_last) { 
	    return 0; 
	} 
	i_first = mktime(&data->first_time.date_struct);
    }

    return 1;
}

/* given a prefix of like "2001-01-01" and  timestr of
   "20:21:12",  return the time_t it represents */
   
time_t find_time(const char *prefix, const char *timestr) {
    char *total_time;
    time_t ret;

    if(!timestr) {
	fprintf(stderr, "find_time: Must have and timestr\n");
	exit(1);
    }
    
    total_time = Strconcat(prefix, timestr);
    ret = str2time(total_time);
    free(total_time);
    return ret;
}


/*
  Determine ports to accept from string of format
  "42:214,!43,6667" means ports 42-214, not port 43 and port 6667
  
*/
int parse_portstr(char *str, Search *search, int which) {
    char *p,*wp;
    PortNode *portnode;
    PortNode *head;

    head = malloc(sizeof(PortNode));
    init_portnode(head);
    if(which == IAD_LPORT) {
	search->lport = head;
    } else {
	search->rport = head;;
    }

    p = str;
    
    while (*str!='\0') {
	portnode = malloc(sizeof(PortNode));
	init_portnode(portnode);
	portnode->operator = PORT_EQ;
	
	/*  Find next : , '\0'  */
	while (*p!=',' && *p!='\0') 
	    p++;

	if(*p!='\0') {
	    *p = '\0';
	    p++;
	}

	/* parse this "word" apart */
	wp = str;
	
	if(*str == '!') {
	    portnode->operator = PORT_NE;
	    str++;
	}
	
	while (*wp!=':' && *wp!='\0') 
	    wp++;
	
	if(*wp == '\0') {
	    portnode->min_port = atoi(str);
	} else {
	    *wp++ = '\0';
	    portnode->min_port = atoi(str);
	    portnode->max_port = atoi(wp);
	}
	
	add_portnode(head, portnode);
	
	/* we're either starting a word or at the end of processing */

	str = p;
    }
    
    return 0;
}
			      

void add_portnode(PortNode *head, PortNode *portnode) {
    PortNode *p;
    
    if(!head) {
	fprintf(stderr, "add_portnode: head is NULL!\n");
	exit(1);
    }

    if(!portnode) {
	fprintf(stderr, "add_portnode: portnode is NULL!\n");
	exit(1);
    }


    for(p=head;p->next!=NULL; p = p->next);

    p->next = portnode;
	
}

/* returns 1 if the port matches the port list check */
int check_portnode(PortNode *portnode, int port) {
    PortNode *p;
    int ret = 0;

    p = portnode;

    /* NULL means match everything */
    if(!p) return 1;
    
    for(p=portnode;p!=NULL; p = p->next) {
	switch(p->operator) {
	case PORT_NOP:
	    break;
	case PORT_EQ:
	    /* is it a range */
	    if(p->max_port != PORT_INIT) {
		if(port <= p->max_port && port >= p->min_port) {
		    ret = 1;
		}
	    } else if(port == p->min_port) {
		ret = 1;
	    }
	    break;
	case PORT_NE:
	    if(p->max_port != PORT_INIT) {
		if(port <= p->max_port && port >= p->min_port) {
		    return 0;
		}
	    } else if(port == p->min_port) {
		return 0;
	    } else {
		ret = 1;
	    }
	}
    }

    return ret;
}
