#ifndef _IPAUDIT_UTIL_H
#define _IPAUDIT_UTIL_H

#define _GNU_SOURCE

/* needed for strptime */
#define _XOPEN_SOURCE 

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <netdb.h>
#include <ctype.h>
#include <math.h>
#include <pwd.h>
#include <sys/types.h>

#define ICMP 1
#define TCP  6
#define UDP  17

#define LOCAL  1
#define REMOTE 2

#define LIMIT 100

#define KBYTE 1024
#define MBYTE 1048576
#define GBYTE 1073741824

#define MATCH_TCP  0x0001
#define MATCH_UDP  0x0002
#define MATCH_ICMP 0x0004
#define MATCH_ANY  0xFFFF

#define PNAME_MAX 16

#define MIN30 1800

#define PORT_INIT -1
#define PORT_NOP 0
#define PORT_EQ 1
#define PORT_NE 2

typedef struct _PortNode {
    int operator;
    int min_port;
    int max_port;
    struct _PortNode *next;
} PortNode;

typedef struct _ProtoName {
    char value[PNAME_MAX]; 
} ProtoName;


#define BYTESTRING_MAX 16
typedef struct _ByteString {
    char value[BYTESTRING_MAX];
} ByteString;

typedef struct _IPString {
    char value[16];
} IPString;

/* just an superset of the tm struct with msec defined as well */
typedef struct _iptm {
    struct tm date_struct;
    int msec;
} iptm;

/* data structure to represent the search options */

typedef struct _ipauditsearch {
    int need; /* how much of our IAD we must fill in */
    char *local_ip;   /* list of source ips */ 
    char *remote_ip;   /* list of dest ips */
    PortNode *lport; /* list of dest ports */
    PortNode *rport;  /* list of source ports */
    int protocol; /* list of protocols */
    
    int first_talker;
    int last_talker;

    int min_bytes;
    int max_bytes;
    int min_packets;
    int max_packets;
    
    int limit; /* max number of lines to return */
    char *ipaudit_30min;
    int match;
        
    time_t start_time;
    time_t end_time;

    iptm iptm_start_time;
    iptm iptm_end_time;
} Search;

/* constants to know how much to break apart the string */
#define IAD_NONE  -1
#define IAD_LOCAL_IP 0
#define IAD_REMOTE_IP 1
#define IAD_PROTOCOL 2
#define IAD_LPORT 3
#define IAD_RPORT 4
#define IAD_BYTES_RECV 5
#define IAD_BYTES_SENT 6
#define IAD_PACKETS_RECV 7 
#define IAD_PACKETS_SENT 8
#define IAD_FIRST_STR 9
#define IAD_LAST_STR 10
#define IAD_FIRST_TALKER 11
#define IAD_LAST_TALKER 12
#define IAD_FIRST_TIME 13
#define IAD_LAST_TIME 14
#define IAD_FILL_ALL 15

/* data structure to represent a single line from IPaudit */
typedef struct _ipauditdata {
    int filled; /* how much of our IAD we've filled in */
    char *rest;  /* unparsed end of the string */
    char *remote_ip;
    char *local_ip;
    int protocol;
    int lport;
    int rport;
    int bytes_sent;
    int bytes_recv;
    int packets_sent;
    int packets_recv;
    char *first_timestr; /*  '13:24:23.1234\0' */
    char *last_timestr; /*  '13:24:23.1234\0' */
    int first_talker;
    int last_talker;
    iptm first_time;
    iptm last_time;
} IAD;

char *time2file_max(time_t when);
char *time2file_min(time_t when);

time_t find_time(const char *prefix, const char *timestr);

time_t str2time(const char *datestr);
char *get_localstr(char *instr);
char *munge_ip(char *old, char *new);
char *demunge_ip(char *instr, IPString *ipstring);

char int2talker(int t);
int get_talker(char *talker);

int ia2time(char *tstr,iptm *when);
int ia_cmp(iptm *a, iptm *b);

int parse_ipaudit_line(char *line,
		       int depth,
		       IAD *data,
		       iptm *curdate,
		       IPString *ip_local,
		       IPString *ip_remote);

int debug_print_iad(IAD *data);
int pretty_print_iad(IAD *data, int displaylinecount, const char *dateprefix);
int raw_print_iad(IAD *data, int displaylinecount, const char *dateprefix);



void init_now(time_t n);
void init_iptm(time_t t, iptm *i);
void init_search(Search *data);
void init_iad(IAD *data);
void init_portnode(PortNode *node);

int check_iad(IAD *data, Search *search, const char* prefix);
int check_portnode(PortNode *portnode, int port);

int human2bytes(const char *str);

char *Getprotobynumber(int pnum, ProtoName *pname);
char *bytes2human(double bytes, ByteString *b);

int parse_portstr(char *str, Search *search, int which);
void add_portnode(PortNode *head, PortNode *portnode);
    
#endif _IPAUDIT_UTIL_H 
