/*
 * ipaudits.c
 *
 * Copyright 2001,2002 Chris Green <cmg@uab.edu>
 *
 * $Id: ipaudits.c,v 1.45 2002/02/05 22:41:46 cmg Exp $
 *
 * PV insprired by snort.c by Marty Roesch <roesch@sourcefire.com.
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <zlib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <limits.h>

#include "ipaudit_util.h"
#include "ipaudits.h"

/* globals */
/* program variables */
PV pv;

extern char *optarg;                /* for getopt */
extern int optind;                /* for getopt */
extern int errno;

int line_count = 1;

/* dump the command line */

/* returns true if we have passed the max number of lines we should do */
int max_lines() {
    if (!pv.search.limit) {
	return 1;
    }

    if(line_count > pv.search.limit) {
	return 0;
    } else {
	return 1;
    }
}

void print_pv() {
    printf("Local IP: %s\n", pv.search.local_ip);
    printf("Remote IP: %s\n", pv.search.remote_ip);

    //printf("Locate port: %d\n",pv.search.lport);
    //printf("Remote port: %d\n", pv.search.rport);

    printf("Protocol: %d\n", pv.search.protocol);

    printf("First Talker: %d\n", pv.search.first_talker);
    printf("Last Talker: %d\n", pv.search.last_talker);

    printf("Min Bytes: %d\n", pv.search.min_bytes);
    printf("Max Bytes: %d\n", pv.search.max_bytes);

    printf("Min Packets: %d\n", pv.search.min_packets);
    printf("Max Packets: %d\n", pv.search.max_packets);

    printf("Line Limit: %d\n", pv.search.limit);

    printf("Data Dir: %s\n", pv.search.ipaudit_30min);
    printf("Orig Dir: %s\n", pv.orig_dir); 
    printf("Match -- ICMP(%d) TCP(%d) UDP(%d)\n",
	   (pv.search.match & MATCH_ICMP),
	   (pv.search.match & MATCH_TCP),
	   (pv.search.match & MATCH_UDP));

    printf("Start time: %d End Time: %d\n",
	   (int) pv.search.start_time,
	   (int) pv.search.end_time);
    printf("Search->need: %d\n", pv.search.need);
}

void basic_tests() {
    iptm iptime;
    int ret;
    IAD iad1, iad2;
    ProtoName pname;
    ByteString b;
    IPString ips,ips1;
    
    char *l1 = Strdup("138.026.155.028 216.033.236.250 6 1064 80  54 54 1 1 14:00:00.8094 14:00:00.8908 1 2\n");
    char *l2 = Strdup("138.026.155.028 216.033.236.250 6 1064 80  54 54 1 1 14:00:00.8094 14:00:00.8908 1 2");
	
    init_iptm(time(NULL),&iptime);

    init_iad(&iad1);
    init_iad(&iad2);
	
    printf("Demunge IP tests\n");
    printf("%s\n", demunge_ip("001.001.001.001", &ips));
    printf("%s\n", demunge_ip("111.101.01.001", &ips));
    printf("%s\n", demunge_ip("111.101.01.000", &ips));

    printf("GetProto(6) == %s\n", Getprotobynumber(6, &pname));
    printf("GetProto(17) == %s\n", Getprotobynumber(17, &pname));
    printf("GetProto(1) == %s\n", Getprotobynumber(1, &pname));
    printf("GetProto(57) == %s\n",  Getprotobynumber(57, &pname));
    
    printf("bytes2human(1) == %s\n", bytes2human(1, &b));
    printf("bytes2human(1000) == %s\n", bytes2human(1000, &b));
    printf("bytes2human(100000) == %s\n", bytes2human(100000, &b));
    printf("bytes2human(10000000) == %s\n", bytes2human(10000000, &b));
    printf("bytes2human(1000000000) == %s\n", bytes2human(1000000000, &b));

    ret = ia2time("14:00:00.8094", &iptime);
    if(ret) {
	printf("run 14:00:00.8094\n");
	printf("%d\n", iptime.msec);
	printf("%d\n", iptime.date_struct.tm_sec);
	printf("%d\n", iptime.date_struct.tm_min);
	printf("%d\n", iptime.date_struct.tm_hour);
	printf("%d\n", iptime.date_struct.tm_mday);
    }
    
    ret = ia2time("14:00:00.", &iptime);
    printf("1 | 0 = %d\n", 1 | 0);
    if(ret) {
	printf("run 14:00:00.\n");
	printf("%d\n", iptime.msec);
	printf("%d\n", iptime.date_struct.tm_sec);
	printf("%d\n", iptime.date_struct.tm_min);
	printf("%d\n", iptime.date_struct.tm_hour);
	printf("%d\n", iptime.date_struct.tm_mday);
    }

    ret = ia2time("234", &iptime);
    if(ret) {
	printf("run 234\n");
	printf("%d\n", iptime.msec);
	printf("%d\n", iptime.date_struct.tm_sec);
	printf("%d\n", iptime.date_struct.tm_min);
	printf("%d\n", iptime.date_struct.tm_hour);
	printf("%d\n", iptime.date_struct.tm_mday);
    }

    ret = ia2time(NULL, &iptime);
    if(ret) {
	printf("run NULL\n");
	printf("%d\n", iptime.msec);
	printf("%d\n", iptime.date_struct.tm_sec);
	printf("%d\n", iptime.date_struct.tm_min);
	printf("%d\n", iptime.date_struct.tm_hour);
	printf("%d\n", iptime.date_struct.tm_mday);
    }


    parse_ipaudit_line(l1, IAD_FILL_ALL, &iad1, NULL, &ips,&ips1);
    debug_print_iad(&iad1);
    parse_ipaudit_line(l2, IAD_FILL_ALL, &iad2, NULL, &ips,&ips1);
    debug_print_iad(&iad2);

    printf("munge(138.26.1.2) == %s\n", munge_ip("138.26.1.2", (char *) &ips.value));
    printf("munge(138.216.111.211) == %s\n", munge_ip("138.216.111.211", (char *) &ips.value));
    printf("munge(0.0.0.0) == %s\n", munge_ip("0.0.0.0", (char *) &ips.value));

}

void usage() {
    printf(
	   "ipaudits [-sdSDmtTbBpPlHhnVRFL] [search_start_date [search_end_date]]\n\n
-s local ip  (eg: 192.168.1.1 )
-d remote ip
-S local port (eg: 21,23,137:139,!138 )
-D remote port
-m protocol (eg: tcp)
-t first talker (eg: R)
-T last talker (eg: R)
-h help
-H data directory ( eg: ipaudit/data/30min  -- defaults to ~ipaudit/data/30min )
-b min bytes (eg: 10)
-B max bytes
-p min packets
-P max packets
-l limit ( line count - default: %d 0=unlimited )
-R raw  ( print out with out the fancy stuff - zgrep replacement)
-V debug command line parameters
The fancyollowing options are very slow:
-F don't show sessions that ended before search_start_date 
-L don't show sessions that started before search_end_date
Note: end_date defaults to start_date + 30 min

The dates pick streams that had some packets during said time range
", LIMIT);
}

void parse_cmd_line(int argc, char *argv[]) {
    int ch;
    long int protocol;
    char *dp = malloc(PATH_MAX);
    IPString *ip;
    
    pv.pretty_print = 1;
    pv.showlinecount = 1;
    pv.orig_dir = dp;

    if(!getcwd(dp, PATH_MAX)) {
	fprintf(stderr, "Can't get current directory: %s", strerror(errno));
    }
    
    if(argc > 2) {
	pv.normal = 0;
    }
    
    while((ch = getopt(argc, argv, "s:d:S:D:m:t:T:b:B:p:P:l:H:hn:RVFL")) != -1 ) {
	switch(ch) {
	case 's':
	    /* local ip */
	    pv.search.local_ip = Strdup(optarg);
	    ip = malloc(sizeof(IPString));
	    pv.search.local_ip = munge_ip(optarg, ip->value);
	    if(pv.search.need < IAD_LOCAL_IP) {
		pv.search.need = IAD_LOCAL_IP;
	    }

	    if(!pv.search.local_ip) {
		fprintf(stderr,
			"Warning - invalid Local Ip: %s\n", optarg);
		exit(1);
	    }

	    break;
	case 'd':
	    /* remote ip */
	    ip = malloc(sizeof(IPString));
	    pv.search.remote_ip = munge_ip(optarg, ip->value);
	    if(pv.search.need < IAD_REMOTE_IP) {
		pv.search.need = IAD_REMOTE_IP;
	    }
	    if(!pv.search.remote_ip) {
		fprintf(stderr, "Warning - invalid Remote Ip: %s\n", optarg);
		exit(1);
	    }
	    break;
      	case 'S':
	    /* local port */
	    parse_portstr(optarg, &pv.search, IAD_LPORT);

	    if(pv.search.need < IAD_LPORT) {
		pv.search.need = IAD_LPORT;
	    } 

	    break;
	case 'D':
	    /* remote port */
	    parse_portstr(optarg, &pv.search, IAD_RPORT);

	    if(pv.search.need < IAD_RPORT) {
		pv.search.need = IAD_RPORT;
	    }
	    break;
	case 'm':
	    /* protocols to match */
	    if(!strcasecmp(optarg,"icmp")) {
		pv.search.protocol = ICMP;
		pv.search.match = MATCH_ICMP;
	    } else if (!strcasecmp(optarg,"tcp")) {
		pv.search.protocol = TCP;
		pv.search.match = MATCH_TCP;
	    } else if (!strcasecmp(optarg,"udp")) {
		pv.search.protocol = UDP;
		pv.search.match = MATCH_UDP;
	    } else {
		protocol = strtol(optarg, (char **) NULL, 10);
		if(protocol > 128) {
		    fprintf(stderr, "unknown protocol %s", optarg);
		}
		pv.search.protocol = (int) protocol;
	    }
	    if(pv.search.need < IAD_PROTOCOL) {
		pv.search.need = IAD_PROTOCOL;
	    }
	    break;
	case 't':
	    /* first talker */
	    pv.search.first_talker = get_talker(optarg);
	    if (pv.search.first_talker < 0) {
		fprintf(stderr, "Unknown First Talker: %s\n",optarg);
		exit(1);
	    }
	    if(pv.search.need < IAD_FIRST_TALKER) {
		pv.search.need = IAD_FIRST_TALKER;
	    } 
	    break;
	case 'T':
	    pv.search.last_talker = get_talker(optarg);
	    if (pv.search.last_talker < 0) {
		fprintf(stderr, "Unknown Last Talker: %s\n", optarg);
		exit(1);
	    }
	    if(pv.search.need < IAD_LAST_TALKER) {
		pv.search.need = IAD_LAST_TALKER;
	    }
	    break;
	case 'b':
	    pv.search.min_bytes = human2bytes(optarg);
	    if(pv.search.need < IAD_BYTES_SENT) {
		pv.search.need = IAD_BYTES_SENT;
	    }
	    break;
	case 'B':
	    pv.search.max_bytes = human2bytes(optarg);
	    if(pv.search.need < IAD_BYTES_SENT) {
		pv.search.need = IAD_BYTES_SENT;
	    }
	    break;
	case 'p':
	    pv.search.min_packets = human2bytes(optarg);
	    if(pv.search.need < IAD_PACKETS_SENT) {
		pv.search.need = IAD_PACKETS_SENT;
	    } 	    
	    break;
	case 'P':
	    pv.search.max_packets = human2bytes(optarg);
	    if(pv.search.need < IAD_PACKETS_SENT) {
		pv.search.need = IAD_PACKETS_SENT;
	    } 	    
	    break;
	case 'l':
	    pv.search.limit = atoi(optarg);
	    break;
	case 'H':
	    pv.search.ipaudit_30min = Strdup(optarg);
	    break;
	case 'R':
	    pv.pretty_print = 0;
	    break;
	case 'V':
	    pv.debug_pv = 1;
	    break;
	case 'F':
	case 'L':
	    pv.search.need = IAD_FILL_ALL;
	    break;
	case 'h':
	    usage();
	    exit(0);

	default:
	    printf("Should exit\n");
	}
    }

    /* if we are to populate the talker fields enough to make pretty output,
       we need to break the fields down completely but not parse the time :-) */
    
    if(pv.pretty_print) {
	if(pv.search.need < IAD_LAST_TALKER) {
	    pv.search.need = IAD_LAST_TALKER;
	}
    }

    /* parse the start date */
    if(optind < argc) {
	pv.search.start_time = str2time(argv[optind]);
#ifdef DEBUG
	printf("changed start time to %s\n", argv[optind]);
#endif /*DEBUG*/
    }
    
    optind++;

    /* parse the end date */
    if(optind < argc) {
	pv.search.end_time = str2time(argv[optind]);
#ifdef DEBUG
	printf("changed end time to %s\n", argv[optind]);
#endif /*DEBUG*/
    }

    optind++;

    /* that should have been the last argument */
    if(optind < argc) {
	usage();
	exit(1);
    }

    if(pv.search.end_time == 0) {
	pv.search.end_time = pv.search.start_time + MIN30 - 1;
    }
} 

void process_line(char *line, const char *prefix) {
    IAD iad;
    iptm iptime;
    //time_t ftime = str2time(fname);
    IPString l, r;
    
#ifdef DEBUG
    //    printf("process_line: prefix is %s\n", prefix);
#endif /*DEBUG*/
    init_now(0);
    init_iptm(0,&iptime);
    init_iad(&iad);

    parse_ipaudit_line(line, pv.search.need, &iad, &iptime, &l, &r);
    // see if we should print or not 
    if(check_iad(&iad, &pv.search, prefix)) {
	if(pv.pretty_print) {
	    pretty_print_iad(&iad, pv.showlinecount,prefix);
	} else {
	    //debug_print_iad(&iad);
	    raw_print_iad(&iad, pv.showlinecount, prefix);
	}
	line_count++;

	/* when we are done, we are done. */
	if(!max_lines()) return;

    }
}

void process_buf(char *buf, const char *fname) {
    /* saved_pointer will remain the same between
       invocations to process the remaining
       chars of the old buffer */
    char *sp, *cp, *end;
    static char *saved_pointer = NULL;
    char *line = NULL;
    int len = strlen(buf);
    end = buf + len;
    sp = cp = buf;

    /* need to handle saved_pointer here */
    
    while(end != cp) {
	sp = cp;
	
	while(*cp != '\n' && *cp != '\0') {
	    cp++;
	}
	switch(*cp) {
	case '\n':
	    *cp = '\0';
	    cp++;
	    if(saved_pointer) {
		/* allocate enough space for the size of the line that is in the old buffer
		   plus the size of the line in the new buffer + 1 for the nul */
		line = Strconcat(saved_pointer, sp);
#ifdef DEBUG
/* 		printf("line: *%s*\nsaved_pointer: *%s*\nsp: *%s*\n", */
/* 		       line, */
/* 		       saved_pointer, */
/* 		       sp); */
#endif /*DEBUG*/
		process_line(line, fname);
		saved_pointer = NULL;

		free(line);
	    } else {
		line = sp;
#ifdef DEBUG
		// printf("entering process line; no saved_pointer; line: %s\n", line);
#endif /*DEBUG*/

		process_line(line, fname);
	    }

	    /* when we are done, we are done. */
	    if(!max_lines()) return;
	    
	    break;
	case '\0':
	    /* got a null, setting saved_pointer to NULL */
	    //printf("setting saved_pointer to NULL\n");
	    if(*sp != '\0') {
#ifdef DEBUG		
		//		printf("setting saved_pointer to %s\n",sp);
#endif
		saved_pointer = sp; 
	    } else {
#ifdef DEBUG
		//printf("setting saved_pointer to NULL\n");
#endif
		saved_pointer = NULL;
	    }
	    
	    return;
	default:
	    printf("boggle!2\n");
	}
    }
}

/* strlen("00:00") */
#define TIMELEN 5
/* process the actual file */
void process_file(const char *fname) {
    gzFile *file;
    char buf[2][BUFSIZE];
    int len = 0;
    int init = 1; /* starting a new file */
    int whichBuf = 0;
    char *prefix = Strdup(fname);
    char *firstdot;
    firstdot = index(prefix,'.');

    /* if prefix is a file name, strip off the 00:00 if it exists and
       atleast the .txt.gz
    */
    if(firstdot != NULL) {
	
	if((firstdot - TIMELEN) >  prefix) {
	    *(firstdot - TIMELEN) = '\0';
	} else {
	    *firstdot = '\0';
	}

    } 
    
    file = gzopen(fname,"rb");
    
    if (!file) {
	fprintf(stderr, "Can't open file %s: %s\n", fname, strerror(errno));
	// exit(1);
	return;
    }
    /* read into 2 separate buffers
       read all the data out of one before we start reading the seconds
    */
    while(init || len) {
	init = 0;
	len = gzread(file,buf[whichBuf],BUFSIZE - 1);
	buf[whichBuf][len] = '\0';

	if(!len) break;
	
	process_buf(buf[whichBuf], prefix);
	
	/* when we are done, we are done. -- break to close file.*/
	if(!max_lines())
	    break;

	whichBuf ^= 1;
    }

    free(prefix);
    gzclose(file);
}

/* process all the files across the time range */
void process_batch() {
    time_t curtime = pv.search.start_time;
    int check;
#ifdef DEBUG
    printf("Min File: %s\n",time2file_min(pv.search.start_time));
    printf("Max File: %s\n",time2file_max(pv.search.end_time));
    printf("Start Time: %d\n", (int) pv.search.start_time);
    printf("Min 30 : %d\n", MIN30);
    
#endif /* DEBUG */
    if((check = chdir(pv.search.ipaudit_30min)) < 0) {
	fprintf(stderr,
		"ipaudits: Can't chdir(%s): %s\n",
		pv.search.ipaudit_30min,
		strerror(errno));
#ifndef DEBUG
	exit(1); /* my development machine doesn't have ~ipaudit */
#endif /*DEBUG*/
    }
		
    for(curtime = pv.search.start_time; curtime <= pv.search.end_time; curtime+=MIN30) {
	process_file(time2file_min(curtime));
	if(!max_lines()) {
	    return;
	}
    }
}

int main(int argc, char *argv[]) {
    /* just in case */
    memset(&pv,0,sizeof(PV));
    init_search(&pv.search);
    //pv.search.need = IAD_FILL_ALL;
    pv.search.need = IAD_NONE;
    /* set the start date to the most recent 30 minute interval */

    parse_cmd_line(argc, argv);
    
    if(pv.debug_pv)
	print_pv();
    
    //basic_tests();
    
    
    /* go ahead and search through all the files */
    process_batch();
    chdir(pv.orig_dir);
    return 0;
}

