#ifndef TCP_SERVER_H
#define TCP_SERVER_H

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#define _USE_BSD

#include <netdb.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/fcntl.h>
#include <errno.h>
#include <sys/signal.h>
#include <sys/wait.h>
#include <arpa/inet.h>

#include <stdarg.h>

#include "tcp_ip_ruleset.h"


class TcpServer
 {
  protected:
  
    int socket_fd;
    struct sockaddr_in  serv_addr;
    int status;
    int debug_level;
    FILE* log;
    TcpIpRuleSet ip_rules;
    int access_control;

    virtual void handle_client(int client_socked_fd) = 0;
    
    virtual void log_access_denied(char *ip_str)
     {
      info("access denied for %s", ip_str);
     }

    void start_server(int port, int listen_queue_size, char* log_file);
    int access_allowed(int client_socket_fd);
    
    void cleanup();
    char* get_peer_addr(int client_socket_fd);
    void print_time(FILE* fp)
    {
     time_t now;
     time(&now);
     char* time_str = ctime(&now);
     time_str[strlen(time_str) - 1] = 0;
     fprintf(fp, "[%s] ", time_str);
    }

  public:
   
   TcpServer(int port, int listen_queue_size = 50, char* log_file = NULL);
   ~TcpServer();
   void run_server();
   void enable_access_control() { access_control  = 1; } 
   void disable_access_control() { access_control  = 0; } 
   void add_ip_rule(char* netnum, char* netmask) 
    {
     ip_rules.add_rule(netnum, netmask);
    }

   void set_debug_level(int new_debug_level) { debug_level = new_debug_level;}

   virtual void error(char* fmt, ...)
    {
     va_list ap;
     va_start( ap, fmt);
     print_time(log);
     fprintf(log, "TCP Server error:");
     vfprintf(log, fmt, ap);
     fprintf(log, ": errno = '%s'\n", strerror(errno));
     status = 0;
    }
  
   virtual void fatal_error(char* fmt, ...)
    {
     va_list ap;
     va_start( ap, fmt);
     fprintf(stderr, "TCP Server fatal error:");
     vfprintf(stderr, fmt, ap);
     fprintf(stderr, ": errno = '%s'\n", strerror(errno));
     exit(1);
    }

   virtual void warn(char* fmt, ...)
    {
     va_list ap;
     va_start( ap, fmt);
     print_time(log);
     fprintf(log, "TCP Server warning:");
     vfprintf(log, fmt, ap);
     fprintf(log, ": errno = '%s'\n", strerror(errno));
    }
  
   virtual void info(char* fmt, ...)
    {
     va_list ap;
     va_start( ap, fmt);
     print_time(log);
     fprintf(log, "TCP Server info:");
     vfprintf(log, fmt, ap);
     fprintf(log, "\n");
    }
 
   virtual void debug(int msg_level, char* fmt, ...)
    {
     va_list ap;
     va_start( ap, fmt);
   
     if(msg_level < debug_level)
      {
       print_time(log);
       fprintf(log, "TCP Server debug info:");
       vfprintf(log, fmt, ap);
       fprintf(log, "\n");
      }
    }

 };

#endif
