/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: common.c,v 1.101.4.3 2005/02/25 17:30:03 anton Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////////////////////
static char *help[] = {

"?",  "top level commands",
	"auth", "authenticates a user to open corresponding unit",
	"data", "send raw data to processor service",
	"debug", "turning on or off debugging on some action",
	"exit", "finish this client session and disconnect",
	"kill", "kill the daemon very quickly, with last data loss",
	"mode", "switch the client operation mode",	
  	"monitor", "look and store specified unit activity",						   
	"oid", "generate new oid",
	"read", "get raw data from processor service",
	"reload", "gracefully shutdown and restart the daemon",
	"save", "write the running config to a file",
	"schedule", "schedule an action to do it intime",
	"service", "set up service",
	"show", "shows various system parameters", 
	"shutdown", "gracefully shut down the daemon",
	"user", "users management",
	NULL, NULL,

"show", "shows various system parameters",
	"config", "running configuration",
	"connections", "active connections to server",
	"list", "NetUnits list with applied policy list",
	"monitor", "monitor information", 
	"policy", "policy list",
	"processor", "processor service status",
	"schedule", "active scheduled actions",
	"units", "NetUnits list",
	"users", "users list",
	"ds", "data-source services status",
	"version", "software version and current status",
	NULL, NULL,

"show config", "running configuration",
   "<cr>", "running configuration",
   "brief", "the same but without NetUnits",
   NULL, NULL,

"show list", "NetUnits list with applied policy list",
   "<cr>", "NetUnits list with applied policy list",
   "full", "the same with flow counters",
   NULL, NULL,

"mode", "switch the client operation mode",
   "data-source", "data source mode",
   "human", "user i/o mode (default)",
   "storage", "storage mode",
   NULL, NULL,

"oid", "generate new oid",
	"host", "for unit host",
    "net", "for unit net",
    "group", "for unit group",
    "cluster", "for unit cluster",
    NULL, NULL,

"data", "send raw data to processor service \ndata {PREFIX} {NETUNIT_OID} {AP_OID} {FROM} {TO} {IN} {OUT}",
   "PREFIX", "'F','M','W','D','H' - record prefix", 
   "NETUNIT_OID", "NetUnit OID, for which this record is entered", 
   "AP_OID", "Accounting Policy OID", 
   "FROM", "UNIXTIME, record start time", 
   "TO", "UNIXTIME, record stop time", 
   "IN", "LONG LONG, bytes in", 
   "OUT", "LONG LONG, bytes out",
   NULL, NULL,

"read", "get raw data from processor service \nread {PREFIX} {NETUNIT_OID} {AP_OID} {FROM} {TO} {TIMEOUT}",
   "PREFIX", "'F','M','W','D','H' - record prefix", 
   "NETUNIT_OID", "NetUnit OID, for which this record is entered", 
   "AP_OID", "Accounting Policy OID", 
   "FROM", "UNIXTIME, record start time", 
   "TO", "UNIXTIME, record stop time", 
   "TIMEOUT", "Query timeout, in milliseconds",
   NULL, NULL,

"schedule", "schedule an action to do it intime\nschedule oid {OID} time {TIME} action {ACTION}",
	"TIME", "time to schedule this action",
	"ACTION", "action to perform",
	NULL, NULL,

NULL,
	NULL
 } ;

char UNKNOWN_REFERENCE[]="<..>";

//////////////////////////////////////////////////////////////////////////////////////////
// Logging Management ///////////////////////////////////////////////////////////////////
#ifdef DEBUG
#define NO_INIT_PARSE 0
#else
#define NO_INIT_PARSE (conn==cInternal)
#endif

pthread_mutex_t parse_lock = PTHREAD_MUTEX_INITIALIZER;
void aParse(Connection *conn, char const *fmt, ...){
	if (!conn->stream_w || flag_quiet || NO_INIT_PARSE) return;
	
	va_list ap;
	pthread_mutex_lock(&parse_lock);

	fprintf(conn->stream_w, "parse: ");
	va_start(ap, fmt);
	vfprintf(conn->stream_w, fmt, ap);
	va_end(ap);
	
	pthread_mutex_unlock(&parse_lock);
}

pthread_mutex_t log_lock = PTHREAD_MUTEX_INITIALIZER;
void aLog(u_char level, char const *fmt, ...){

	time_t t;
	struct timeval tv;
	char t_T[30];
	char *where,*state;
	u_char id=0;
	Service *s;
	Connection *conn;
	va_list ap;
	int facility;
	static char str2log2[256];
	static char str2log1[256+64];
	
	switch(level){
				case D_INFO:    state="INFO"; facility=LOG_INFO; break;
				case D_ERR:     state="ERR"; facility=LOG_ERR; break;
				case D_CRIT:    state="CRIT"; facility=LOG_CRIT; break;
				case D_WARN:    state="WARN"; facility=LOG_WARNING; break;
                case D_DEBUG:   
#ifndef DEBUG
				return;
#endif 
				state="DEBUG"; 
				facility=LOG_DEBUG; 
				break;
                default:
                state=UNKNOWN_REFERENCE; 
                facility=LOG_NOTICE;
                break;
	}
	
	pthread_mutex_lock(&log_lock);

	if((s=Services.getServiceByThr(pthread_self()))) {
		where=s->getName();
		id=s->instance;
	} else if((conn=Connections.getConnectionByThr(pthread_self()))) {
		where=conn->getName();
		id=conn->conn_id;
	} else 	
		where=UNKNOWN_REFERENCE; 

	time(&t); timeU2T(t, t_T);
	gettimeofday(&tv, NULL);

	if (flag_nodaemon && !flag_quiet) 
		fprintf(stdout, "%s.%04u %s:%u [%s]: ", t_T, (unsigned)((double)tv.tv_usec/100), where, id, state);
	if (flag_log) 
		fprintf(LOGFILE, "%s.%04u %s:%u [%s]: ", t_T, (unsigned)((double)tv.tv_usec/100), where, id, state);
  
	if (flag_syslog) { bzero(str2log2, 256); bzero(str2log1, 256+64); }

	va_start(ap, fmt);
	if (flag_nodaemon && !flag_quiet) vfprintf(stdout, fmt, ap);
	if (flag_log) vfprintf(LOGFILE, fmt, ap);
	if (flag_syslog) vsnprintf(str2log2,256, fmt, ap);
	va_end(ap);

	if (flag_syslog) {
	    snprintf(str2log1,256+64,"%s:%u %s", where, id, str2log2);
	    syslog(facility,str2log1);
	}

	pthread_mutex_unlock(&log_lock);
	if (level==D_ERR) termination(SIGTERM);
	if (level==D_CRIT) termination(SIGTERM);
}
//////////////////////////////////////////////////////////////////////////////////////////
void logrotate(int signal){
	if(flag_log) {
		fclose(LOGFILE);
		LOGFILE=fopen(path_to_log, "at");
	}
}
//////////////////////////////////////////////////////////////////////////////////////////
void termination(int signal){
	global_return_code=signal;
	sleep(1); //we should wait until sMain is sleeping, this is not guarantee this
	printf("\n");
	if(signal==PARSE_KILL || signal==SIGKILL) {
		aMemoryRelease();
		char buf[32];
		aLog(D_WARN, "KILL signal received at %s\n", timeU2T(time(NULL), buf));
		unlink("/var/run/netams.pid");
		exit(0);
	}
	sMain->Wakeup();
}
/////////////////////////////////////////////////////////////////////////////////////////////
char *timeU2T(time_t time, char *buf){
	struct tm tm;
	if (buf==NULL) buf=(char *)aMalloc(25);
	localtime_r(&time, &tm);
	sprintf(buf, "%02d.%02d.%4d %02d:%02d:%02d", tm.tm_mday, tm.tm_mon+1, tm.tm_year+1900, tm.tm_hour, tm.tm_min, tm.tm_sec);
	return buf;
}
///////////////////////////////////////////////////////////////////////////
time_t timeT2U(char *buf, time_t *time){
	struct tm tm;
	time_t t;
	sscanf(buf, "%02d.%02d.%4d %02d:%02d:%02d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year, &tm.tm_hour, &tm.tm_min, &tm.tm_sec);
	tm.tm_mon = tm.tm_mon-1;
	tm.tm_year = tm.tm_year-1900;
	tm.tm_isdst = -1;
	t=mktime(&tm);
	if (time!=NULL) memcpy(time, &t, sizeof(time_t));
	return t;
}
/////////////////////////////////////////////////////////////////////////////////////////////
char *bytesQ2T(unsigned long long bytes, char *buf){
if (buf==NULL) buf=(char*)aMalloc(12);
if (bytes<KILOBYTE) sprintf(buf, "%llu", bytes); 
else if (bytes>=KILOBYTE && bytes<MEGABYTE) sprintf(buf, "%.3fK", (double)bytes/(KILOBYTE));
else if (bytes>=MEGABYTE && bytes <GIGABYTE) sprintf(buf, "%.3fM", (double)bytes/(MEGABYTE));
else if (bytes>=GIGABYTE /*&& bytes <GIGABYTE*KILOBYTE */) sprintf(buf, "%.3LfG", (long  double)bytes/(GIGABYTE));
else aLog(D_WARN, "bytesQ2T called with enormous value: %llu\n", bytes);
return buf;
}
//////////////////////////////////////////////////////////////////////////////////////////
unsigned long long bytesT2Q(char *buf){
if (buf==NULL) { return 0; }
long double b;
sscanf(buf, "%Lf", &b);
if (strchr(buf, 'K')!=NULL) b*=KILOBYTE;
if (strchr(buf, 'M')!=NULL) b*=MEGABYTE;
if (strchr(buf, 'G')!=NULL) b*=GIGABYTE;

return (unsigned long long)b;
} 
//////////////////////////////////////////////////////////////////////////////////////////
pthread_mutex_t oid_lock = PTHREAD_MUTEX_INITIALIZER;
#define MAX_OID		0x0FFFFF	//max oid
#define CELL_SIZE	32
#define CELL_MASK	(CELL_SIZE-1)
u_int32_t oids_hash[(MAX_OID+1)>>5];		//it's important sizeof(u_int)=32
unsigned oids_used;

void aOidsInit() {
	oids_used=0;
#if defined (LINUX) || defined (SOLARIS)
	srandom(time(NULL));
#else
	srandomdev(); // true random numbers at each run
#endif
	bzero(&oids_hash, (MAX_OID+1)>>5);
}

oid newOid(oid newid){
	newid&=MAX_OID; //restrict oids we work

	oid result=newid;
	u_short poz;
	u_char bit;
	
	pthread_mutex_lock(&oid_lock);
	
	if(newid) {
		poz=(unsigned)(result>>5);
		bit=result&CELL_MASK;
		goto OK;
	}
	
	for(u_char j=0;j<25;j++) {
		result=random()&MAX_OID;	
		poz=(unsigned)(result>>5);
		bit=result&CELL_MASK;
		if(!(oids_hash[poz]&(1<<bit)))  goto OK;
	}
	
	for(result=1; result<MAX_OID; result++) { // no oids with 0000 in the end
		poz=(unsigned)(result>>5);
		bit=result&CELL_MASK;
		if(!(oids_hash[poz]&(1<<bit))) goto OK;
	}

	pthread_mutex_unlock(&oid_lock);
	aLog(D_WARN, "newOid cannot construct new id\n");
	return 0;
OK:
	oids_used++;
	oids_hash[poz]|=(1<<bit);
	pthread_mutex_unlock(&oid_lock);
	return result;

}
//////////////////////////////////////////////////////////////////////////////////////////
void PrepareTimeCounters(struct time_counters *tc, time_t t1) {
        struct tm tm1;

        localtime_r(&t1, &tm1);
        tm1.tm_sec=0; tm1.tm_min=0; tm1.tm_isdst=-1;
        tc->ht=mktime(&tm1);

        tm1.tm_hour=0; tm1.tm_isdst=-1;
        tc->dt=mktime(&tm1);

        if (tm1.tm_wday==0) tm1.tm_wday=7;
        tm1.tm_mday+=(8-tm1.tm_wday)-7;
        tc->wt=mktime(&tm1);

        localtime_r(&t1, &tm1);
        tm1.tm_sec=0; tm1.tm_min=0; tm1.tm_hour=0; tm1.tm_isdst=-1;
        tm1.tm_mday=1; // tm1.tm_mon-=1; if (tm1.tm_mon<0) {         tm1.tm_year--; tm1.tm_mon+=12; }
        tc->mt=mktime(&tm1);
	
}
//////////////////////////////////////////////////////////////////////////////////////////
void FillTimeCounters(policy_data *np, struct time_counters *tc){

	time_t t2;

	t2=tc->ht;
	if (np->h.from!=t2) {
		np->h.from=t2;
		np->h.in=np->h.out=0;
	}

	t2=tc->dt;
	if (np->d.from!=t2) {
		np->d.from=t2;
		np->d.in=np->d.out=0;
	}

	t2=tc->wt;
	if (np->w.from!=t2) {
		np->w.from=t2;
		np->w.in=np->w.out=0;
	}

	t2=tc->mt;
	if (np->m.from!=t2) {
		np->m.from=t2;
		np->m.in= np->m.out=0;
	}

}

//////////////////////////////////////////////////////////////////////////////////////////
// Debugging management
char *debug_list[64];
char *debug_list_help[64]; 
pthread_mutex_t debug_lock = PTHREAD_MUTEX_INITIALIZER;

#if defined(DEBUG) || defined(OLD_GCC)
void aDebug(u_char d, char const *fmt, ...){
#ifdef DEBUG
	pthread_mutex_lock(&debug_lock);
	Connection *conn;
	va_list ap;

	for (conn=Connections.root; conn!=NULL; conn=conn->next) 
		if (debug_list[d] && aDebugIsSet(conn->debug, d)) {
			if (conn->stream_w && !(flag_quiet&&(conn==cInternal))) 
				fprintf(conn->stream_w, "|%s: ", debug_list[d]);
			#ifdef DEBUG_TO_LOG
			if (flag_log) fprintf(LOGFILE, "|%s: ", debug_list[d]);
			#endif
    	
			va_start(ap, fmt);
			if (conn->stream_w && !(flag_quiet&&(conn==cInternal))) 
				vfprintf(conn->stream_w, fmt, ap);
			conn->Flush();
			#ifdef DEBUG_TO_LOG
			if (flag_log) vfprintf(LOGFILE, fmt, ap);
			#endif
    			va_end(ap);
		}

	pthread_mutex_unlock(&debug_lock);
	return;
#endif
}
#endif

u_char aDebugAdd(Debug &d, char *str){
	for (u_char i=0; i<64; i++){
		if (debug_list[i] && !strcmp(str, debug_list[i])) {
			if (i==DEBUG_NONE) d=0L;
			else if (i==DEBUG_ALL) d=~0;
			else d |= (Debug)(1<<i);
			return 1;
		}
	}
	return 0;
}

u_char aDebugRm(Debug &d, char *str){
	for (u_char i=0; i<64; i++){
		if (debug_list[i] && !strcmp(str, debug_list[i])) {
			if (i==DEBUG_NONE) d=~0;
			else if (i==DEBUG_ALL) d=0;
			else d &=~ (Debug)(1<<i);
			return 1;
		}
	}
	return 0;
}

u_char aDebugIsSet(Debug d, u_char u){
	if (u==DEBUG_NONE && d==0) return 1;
	else if (u==DEBUG_ALL && d==~0) return 1;
	else if (u!=DEBUG_NONE && (d&((Debug)1<<u))) return 1; 
	return 0;
}

void aDebugInit(){
	for (u_char i=0; i<=63; i++) { debug_list[i]=NULL;	debug_list_help[i]=NULL; }
	debug_list[DEBUG_NONE]="none"; debug_list_help[DEBUG_NONE]="turn off all debugging info";             
	debug_list[DEBUG_COMMAND]="command"; debug_list_help[DEBUG_COMMAND]="entered command processing";         
	debug_list[DEBUG_PARSE]="parse"; debug_list_help[DEBUG_PARSE]="entered command parse";
	debug_list[DEBUG_SLEEP]="sleep"; debug_list_help[DEBUG_SLEEP]="services going in/out sleep mode";
	debug_list[DEBUG_SERVER]="server"; debug_list_help[DEBUG_SERVER]="commands server activity";
	debug_list[DEBUG_PROC_MUX]="proc_mux"; debug_list_help[DEBUG_PROC_MUX]="processor multiplexer activity";
	debug_list[DEBUG_DS_IP]="ds_ip"; debug_list_help[DEBUG_DS_IP]="data sources, IP-packets based";
	debug_list[DEBUG_STORAGE]="storage"; debug_list_help[DEBUG_STORAGE]="storage activity";
	debug_list[DEBUG_ALERT]="alert"; debug_list_help[DEBUG_ALERT]="system alerts";
	debug_list[DEBUG_SCHED]="scheduler"; debug_list_help[DEBUG_SCHED]="scheduled actions";
	debug_list[DEBUG_HTML]="html"; debug_list_help[DEBUG_HTML]="html static pages generation";
	debug_list[DEBUG_MONITOR]="monitor"; debug_list_help[DEBUG_MONITOR]="specified units packet headers";
	debug_list[DEBUG_LOGIN]="login"; debug_list_help[DEBUG_LOGIN]="user login events";
	debug_list[DEBUG_QUOTA]="quota"; debug_list_help[DEBUG_QUOTA]="quota control events";
	debug_list[DEBUG_IPTREE]="iptree"; debug_list_help[DEBUG_IPTREE]="iptree engine events";
	debug_list[DEBUG_FLOW]="flow"; debug_list_help[DEBUG_FLOW]="ip flow engine events";
	debug_list[DEBUG_DS_MUX]="ds_mux"; debug_list_help[DEBUG_DS_MUX]="data source multiplexer activity";
	debug_list[DEBUG_MEMORY]="memory"; debug_list_help[DEBUG_MEMORY]="memory usage events";
	debug_list[DEBUG_POLICY]="policy"; debug_list_help[DEBUG_POLICY]="policy checking actions";
	debug_list[DEBUG_BILLING]="billing"; debug_list_help[DEBUG_BILLING]="billing events";
	debug_list[DEBUG_PVM]="pvm"; debug_list_help[DEBUG_PVM]="PVM transport events";
	debug_list[DEBUG_ALL]="all"; debug_list_help[DEBUG_ALL]="turn on all debugging info (be careful!)";     
}
//////////////////////////////////////////////////////////////////////////////////////////
void cHelp(Connection *conn, u_char no_flag, char *str){
	
	aDebug(DEBUG_PARSE, "help %s%s\n", no_flag?"NO ":"", str);

	u_char printed=0;

	if (!strcasecmp(str, "debug")) {
		fprintf(conn->stream_w, "turning %s debugging on following action:\n", no_flag?"OFF":"ON");
		for (u_char i=0; i<=63; i++) if (debug_list[i])
			fprintf(conn->stream_w, "   %12s | %s\n", debug_list[i], debug_list_help[i]);		
		fprintf(conn->stream_w, "\n"); printed=1;
	}
	else {
		u_char i=0;
		while(help[i]) {
			if (!strcasecmp(str, help[i])) {
				fprintf(conn->stream_w, "%s\n", help[i+1]);	i+=2;
				while(help[i]) {
					fprintf(conn->stream_w, "   %12s | %s\n", help[i], help[i+1]);	i+=2;
				}
			fprintf(conn->stream_w, "\n"); printed=1;
			}
			else { while(help[i]) i+=2; }
			i+=2;
		}
	}
	if (!printed) fprintf(conn->stream_w, "no help is available on: %s\n", str);
	return;
}
//////////////////////////////////////////////////////////////////////////////////////////
void aShowVersion(FILE *f) {
	fprintf(f, "%s version %d.%d.%d (build %d", aaa_fw_software_name, aaa_fw_major_version, aaa_fw_minor_version, aaa_fw_subversion, aaa_fw_build_version);
	if (aaa_fw_build_version_local) fprintf(f, ".%d", aaa_fw_build_version_local);
	if (buildforstring) fprintf(f, "-%s", buildforstring);
	fprintf(f, ") %s / %s\n", aaa_fw_build_person, aaa_fw_build_time);
}

//////////////////////////////////////////////////////////////////////////////////////////
void cShowVersion(Connection *c){
	aShowVersion(c->stream_w);

	struct rusage res;
	struct timeval now;
	unsigned long days, hours, mins, secs;
	time_t t_secs;
	double mksecs;

	gettimeofday(&now, NULL);
	if (0!=getrusage(RUSAGE_SELF, &res)) { aParse(c, "resourse usage retrival failed: %s\n", strerror(errno)); return; }

	t_secs=(now.tv_sec - program_start.tv_sec);
	days=t_secs/(60*60*24);
	hours=(t_secs-days*60*60*24)/(60*60);
	mins=(t_secs-(days*60*60*24)-(hours*60*60))/60;
	secs=t_secs-(days*60*60*24)-(hours*60*60)-mins*60;
	mksecs=(double)(now.tv_usec - program_start.tv_usec)/1000000;
	fprintf(c->stream_w, "Run time:");
	if (days>0) fprintf(c->stream_w, " %ld days", days);
	if (hours>0) fprintf(c->stream_w, " %ld hours", hours);
	if (mins>0) fprintf(c->stream_w, " %ld mins", mins);
	fprintf(c->stream_w, " %.4f secs\n", fabs((double)secs+mksecs));

	t_secs=(res.ru_stime.tv_sec);
	days=t_secs/(60*60*24);
	hours=(t_secs-days*60*60*24)/(60*60);
	mins=(t_secs-(days*60*60*24)-(hours*60*60))/60;
	secs=t_secs-(days*60*60*24)-(hours*60*60)-mins*60;
	mksecs=(double)(res.ru_stime.tv_usec)/1000000;
	fprintf(c->stream_w, "System time:");
	if (days>0) fprintf(c->stream_w, " %ld days", days);
	if (hours>0) fprintf(c->stream_w, " %ld hours", hours);
	if (mins>0) fprintf(c->stream_w, " %ld mins", mins);
	fprintf(c->stream_w, " %.4f secs\n", (double)secs+mksecs);

	fprintf(c->stream_w, "Average CPU/system load: %3.2f%%\n", 100*((float)(res.ru_stime.tv_sec)+(float)(res.ru_stime.tv_usec)/1000000)/((float)(now.tv_sec - program_start.tv_sec)+(float)(now.tv_usec - program_start.tv_usec)/1000000)); 
	fprintf(c->stream_w, "Process ID: %u RES: %luK\n", getpid(), res.ru_maxrss); 
	fprintf(c->stream_w, "Memory allocated: %llu (%lu), freed (%lu) (%u NULL) [%lu used]\n", aGetBytesAllocated(), aGetTimesAllocated(), aGetTimesFreed(), aGetTimesFreedNull(), aGetTimesAllocated() - aGetTimesFreed());
	fprintf(c->stream_w, "\n");

	fprintf(c->stream_w, "Total objects:\n");
	
	fprintf(c->stream_w, "   Oids used: %u\n", oids_used);
	fprintf(c->stream_w, "   NetUnits: %u\n", Units.num_units);
	fprintf(c->stream_w, "   Policies: %u\n", PolicyL.num_policies);
	fprintf(c->stream_w, "   Services: %u\n", Services.getServices());
	fprintf(c->stream_w, "   Users: %u\n", Users.getUsers());
	fprintf(c->stream_w, "   Connections: %u active, %u total\n", Connections.getConnectionsNum(), Connections.getConnectionsLastId());
	fprintf(c->stream_w, "   Scheduled tasks: %u\n", Sched.num_tasks);
	{	
	        Service *s=NULL;
        	while((s=Services.getServiceNextByType(SERVICE_ALERTER,s))) {
                	ServiceAlerter_cfg *cfg=(ServiceAlerter_cfg*)s->cfg;
                	fprintf(c->stream_w, "Alerter %u queue max: %u, current: %u\n", s->instance, cfg->queue->max_items, cfg->queue->num_items);
        	}

	}
	fprintf(c->stream_w, "\nServices info:\n");
	
	Services.listInfo(c->stream_w);
}
//////////////////////////////////////////////////////////////////////////////////////////
time_t aGetActual(char prefix, time_t t){
	struct tm tm;
	time_t actual=0;

	localtime_r(&t, &tm);
	
	//next in special order most often we check for H, then for M and so on
	switch(prefix) {
		case 'H': {
			tm.tm_sec=0; tm.tm_min=0; tm.tm_isdst=-1;
			actual=mktime(&tm);
			break;
		}
		case 'M': {
			tm.tm_sec=0; tm.tm_min=0; tm.tm_hour=0; tm.tm_isdst=-1;
			tm.tm_mday=1;  tm.tm_mon++; //if (tm.tm_mon<0) { 	tm.tm_year--; tm.tm_mon+=12; }
			actual=mktime(&tm);
			break;
		}
		case 'W': {
			tm.tm_sec=0; tm.tm_min=0; tm.tm_hour=0; tm.tm_isdst=-1; 
			if (tm.tm_wday==0) tm.tm_wday=7;
			tm.tm_mday+=(8-tm.tm_wday)-7;
			actual=mktime(&tm);
			break;
		}
		case 'D': {
			tm.tm_sec=0; tm.tm_min=0; tm.tm_hour=0; tm.tm_isdst=-1;
			actual=mktime(&tm);
			break;
		}
		case 'F':
			actual=t;
			break;
	}

	return actual;
}
//////////////////////////////////////////////////////////////////////////////////////////
void print_to_string(char **string, char const *fmt, ...){

	va_list ap;
	char *result;
	char *ret;

   	va_start(ap, fmt);
	if (-1==vasprintf(&ret, fmt, ap)) { va_end(ap); return; }
   	va_end(ap);

	if (*string) {
		result=(char*)aMalloc(strlen(*string)+strlen(ret)+1);
		sprintf(result, "%s%s", *string, ret);
		aFree(*string); 
		(*string)=result;
	}
	else 
		(*string)=set_string(ret);
	
	free(ret);
}
//////////////////////////////////////////////////////////////////////////////////////////
char *set_string(char *string) {
	char *res=NULL;
	if(string) {
		res=(char*)aMalloc(strlen(string)+1);
		strcpy(res, string);
	}
	return res;
}
//////////////////////////////////////////////////////////////////////////////////////////
// command execution
// if buffer==NULL, we don't need in command output; else print_to_string onto it
void cExec(char *cmd, char **buffer){

	Connection *conn = new Connection("exec", CONN_FD_VIRT);
	conn->permissions=UPERM_ALL;
	Connections.Insert(conn);

	aCommand(conn, cmd);
	
	fflush(conn->stream_w);

//	aDebug(DEBUG_PARSE, "cExec res: \"%s\"\n", conn->buffer?conn->buffer:"??"); 

	if (conn->buffer && buffer) print_to_string(buffer, "%s", conn->buffer);
	Connections.Delete(conn);
	delete conn;
}
//////////////////////////////////////////////////////////////////////////////////////////
int cNewOid(Connection *c, char *type) {
	fprintf(c->stream_w, "%06X\n", newOid(0));
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////////////////////
void cAccessScriptCall(restrict_type action, NetUnit *u, char *p){
	if (!ProcessorCfg.access_script) return;
	char *buffer;
	int i;

	char *param=NULL;
	if (u->type==NETUNIT_HOST) {
		print_to_string(&param, "%s %s", inet_ntoa(((NetUnit_host*)u)->ip), p);
	}
	else if (u->type==NETUNIT_USER) {
		print_to_string(&param, "%s %s", inet_ntoa(((NetUnit_user*)u)->ip), p);
	}
	else {
		param=set_string(p);
	}

	unsigned len = strlen(ProcessorCfg.access_script) + 15 + strlen(param);
	buffer = (char*)aMalloc(len);
	oid oid=u->id;

	switch (action) {
		case DROP:
			snprintf(buffer, len, "%s DENY %06X %s", ProcessorCfg.access_script, oid, param?param:"");
			break;
		case PASS:	
			snprintf(buffer, len, "%s ALLOW %06X %s", ProcessorCfg.access_script, oid, param?param:"");
			break;
		case PASS_LOCAL:
			snprintf(buffer, len, "%s LOCAL %06X %s", ProcessorCfg.access_script, oid, param?param:"");
			break;
		default:
			break;
	}

	i=system(buffer);
	aLog(D_INFO, "cAccessScriptCall oid %06X action %d system:%d\n", oid, action, i);
	aFree(param);
	aFree(buffer);
}
//////////////////////////////////////////////////////////////////////////////////////////
int cShowPerf(Connection *conn, char *param[]) {
	char *filename=NULL;
	FILE *out;

	if (param[2]!=empty) filename=param[2];
	
	if (!strcasecmp(param[3], "header") || !filename) {
		if(filename) 
			out=fopen(filename, "wt");
		else
			out=conn->stream_w;

		if (!out) return 1;
		aShowVersion(out);
		/*
		TOD		Time-Of-Day
		RTM		Run Time
		STM		System Time
		LOAD		System Load
		RES		Resources Usage
		*/
		fprintf(out, "%19s  %7s  %7s  %5s  %5s  ","TOD","RTM","STM","LOAD","RES");
		
		Services.showPerf(out, 1);
		fprintf(out, "\n");

	} else {
		if(filename)
			out=fopen(filename, "at");
		else
			out=conn->stream_w;

		if (!out) return 1;
	}

	struct rusage res;
	struct timeval now;
	time_t t_secs;
	double mksecs;
	static char t_T[32];
	time_t t;

	time(&t); timeU2T(t, t_T);
	gettimeofday(&now, NULL);

	//TOD
//	fprintf(out, "%s.%04u  ", t_T, (unsigned)((double)now.tv_usec/100));
	fprintf(out, "%-19s  ", t_T);

	//RTM
	t_secs=(now.tv_sec - program_start.tv_sec);
	fprintf(out, "%7lu  ", t_secs);

	//STM
	if (0!=getrusage(RUSAGE_SELF, &res)) { fclose(out); return 2; }
	t_secs=(res.ru_stime.tv_sec);
	mksecs=(double)(res.ru_stime.tv_usec)/1000000;
	fprintf(out, "%7f  ", t_secs+mksecs); 

	//LOAD
	fprintf(out, "%3.2f  ", 100*((float)(res.ru_stime.tv_sec)+(float)(res.ru_stime.tv_usec)/1000000)/((float)(now.tv_sec - program_start.tv_sec)+(float)(now.tv_usec - program_start.tv_usec)/1000000)); 

	//RES
	fprintf(out, "%5lu  ", res.ru_maxrss);

	//show performance counters for all services which provides such an info
	Services.showPerf(out);

	fprintf(out, "\n");
	if(filename) fclose(out);
	
	return 0;
}
//////////////////////////////////////////////////////////////////////////////////////////
int cShowSystemHealth(Connection *conn) {
	int health=0;
	
	double loadavg[3];
	if (getloadavg(loadavg, 3)!=-1) { 
		fprintf(conn->stream_w, "Current system load: %2.0f%%", 100*loadavg[1]);
		if (loadavg[1]>=0.8) health++;
	}
	
	struct statvfs_name_local vfs;
	Service *sh = Services.getServiceNextByType(SERVICE_HTML, NULL);
	if (sh) { 
		Html_cfg *sh_cfg = (Html_cfg*)sh->cfg; 
		if (!statvfs_name_local(sh_cfg->path, &vfs)) fprintf(conn->stream_w, ", HTML files folder free disk space: %2.0f%%", 100*(double)vfs.f_bfree/vfs.f_blocks);
		if ((double)vfs.f_bfree/vfs.f_blocks<=0.1) health++;
	}

	ServiceStorage_cfg *ss_cfg=NULL;
	if (ProcessorCfg.st_root) ss_cfg=(ServiceStorage_cfg*)ProcessorCfg.st_root->s->cfg;
	if (ss_cfg && ss_cfg->db_path) {
		if (!statvfs_name_local(ss_cfg->db_path, &vfs)) fprintf(conn->stream_w, ", Primary storage folder free disk space: %2.0f%%", 100*(double)vfs.f_bfree/vfs.f_blocks);
		if ((double)vfs.f_bfree/vfs.f_blocks<=0.1) health++;
	}

	fprintf(conn->stream_w, "\n");
	return health;
}
//////////////////////////////////////////////////////////////////////////////////////////
u_char getAddr(char *str, in_addr *addr, in_addr *mask) {
	u_char mlen=32;
	char *p=strchr(str, '/');
	if (p) {
		*p=0;
		inet_aton(str, addr);
		p++;
		mlen=atoi(p);
		if(mlen>32) {
			in_addr ia;
			inet_aton(p, &ia);
			mlen=MASK2MLEN(&ia);
		}
		if(mask) {
			mask->s_addr=htonl(MLEN2MASK(mlen));
			addr->s_addr&=mask->s_addr; //auto-correction
		}
	} else 
		inet_aton(str, addr);

	return mlen;
}

u_char mask2mlen(in_addr *mask) {
	unsigned tmp=mask->s_addr;
        u_char m=0;

        for(u_char i=0;i<32;i++) {
                if(tmp&1) m++;
                tmp=tmp>>1;
        }
	
	return m;
}
//////////////////////////////////////////////////////////////////////////////////////////
int getdayofweek(char *str){
	int res=-1;
	if (!strncasecmp(str, "mon", 3)) res=0;
	else if (!strncasecmp(str, "tue", 3)) res=1;
	else if (!strncasecmp(str, "wed", 3)) res=2;
	else if (!strncasecmp(str, "thr", 3)) res=3;
	else if (!strncasecmp(str, "fri", 3)) res=4;
	else if (!strncasecmp(str, "sat", 3)) res=5;
	else if (!strncasecmp(str, "sun", 3)) res=6;
	return res;
}

char *daysofweek[7] = { "Mon", "Tue", "Wed", "Thr", "Fri", "Sat", "Sun" };
//////////////////////////////////////////////////////////////////////////////////////////
#ifndef HAVE_BILLING
#ifndef MAKE_FOR
char *buildforstring=NULL;
#endif
#endif

	
