/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: memory.c,v 1.22 2005/01/17 13:13:21 jura Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////////////////////
// Memory management ///////////////////////////////////////////////////////////////////
static unsigned long long bytes_allocated=0;
static unsigned long times_allocated=0;
static unsigned long times_freed=0;
static unsigned times_freed_null=0;
pthread_mutex_t mem_lock = PTHREAD_MUTEX_INITIALIZER;

#if defined(DEBUG) && defined(MEMORY_DEBUG)
static u_char MemoryInitialized=0;
u_char AllocMemory(void *ptr, size_t size);
u_char FreeMemory(void *ptr);
typedef struct mem_unit {
	void *ptr;
	size_t size;
	pthread_t allocated_by;
	mem_unit *next;
} mem_unit;
// mem hash
mem_unit *mem_ready=NULL;
mem_unit **mem_hash;  //this is hash where we store information about allocated memory
//i've choosed another hash then standart and belive it's faster
//because of this card(0...0xFFFF)=0x10000 size used
#define MEM_HASH_SIZE	0x10000
#define MEM_HASH_MASK	(MEM_HASH_SIZE-1)
#define MEM_HASH(ptr)	((unsigned)ptr>>4)&MEM_HASH_MASK
//#define MEM_HASH(ptr)    (unsigned)ptr % MEM_HASH_MASK  //this is standart 

//thread list for cShowMem()
typedef struct thread_unit {
        pthread_t id;
        thread_unit *next;
        unsigned errors;
	unsigned long bytes_allocated;
	unsigned long times_allocated;
	unsigned long times_freed;
} thread_unit;
thread_unit *thread_root=NULL;
thread_unit *getThread(pthread_t id);
#else
void aMemoryRelease() {}
void cShowMemory(Connection *conn) {
	aParse(conn, "This command enabled only if compiled with -DDEBUG and -DMEMORY_DEBUG\n");
} 
#endif

void *aMalloc(size_t size, int critical){
        pthread_mutex_lock(&mem_lock);
        void *res;
	//we seriously depend that res are zeroed on allocation
        res=calloc(1,size);
        if (res==NULL) {
                if (critical)
                        aLog(D_CRIT, "malloc of %d bytes failed!\n", size);  //will never returns
                else
                        aLog(D_WARN, "malloc of %d bytes failed!\n", size);
                pthread_mutex_unlock(&mem_lock);
                return NULL;
        }
        times_allocated++;
        bytes_allocated+=size;
#if defined(DEBUG) && defined(MEMORY_DEBUG)
	if(MemoryInitialized) AllocMemory(res,size);
#endif
        pthread_mutex_unlock(&mem_lock);
        return res;
}

void aFree(void *ptr){
        pthread_mutex_lock(&mem_lock);
#if defined(DEBUG) && defined(MEMORY_DEBUG)
        if(MemoryInitialized && !FreeMemory(ptr)) {
		if(!ptr) times_freed_null++;
		pthread_mutex_unlock(&mem_lock);
		return;
	}
#endif
        if (!ptr)
                times_freed_null++;
        else {
                times_freed++;
                free(ptr);
        }
	ptr=NULL;
        pthread_mutex_unlock(&mem_lock);
}

unsigned long long aGetBytesAllocated() { return bytes_allocated; }
unsigned long aGetTimesAllocated() { return times_allocated; }
unsigned long aGetTimesFreed() { return times_freed; }
unsigned aGetTimesFreedNull() { return times_freed_null; }
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
void aMemoryInit() {
#if defined(DEBUG) && defined(MEMORY_DEBUG)
	mem_hash=(mem_unit**)calloc(MEM_HASH_SIZE,sizeof(mem_unit*));
	MemoryInitialized=1;
	aLog(D_INFO, "Memory Debugging Initialized\n");
#endif
	
	//do some work for Processor
	bzero(&ProcessorCfg, sizeof(ServiceProcessor_cfg));
}
//////////////////////////////////////////////////////////////////////////////////////////
#if defined(DEBUG) && defined(MEMORY_DEBUG)

void aMemoryRelease() {
	// produce statistic
	thread_unit *t;
	Service *s;
	Connection *c;
	char *where;
	unsigned id=0;

	aDebug(DEBUG_MEMORY, " Thread  |    Service   |bytes allocated|times allocated|times freed|errors\n");
	for(t=thread_root;t!=NULL;t=t->next){
		if((s=Services.getServiceByThr(t->id))){
                        where=s->getName();
                        id=s->instance;
                } else if((c=Connections.getConnectionByThr(t->id))) {
                        where="conn";
                        id=c->conn_id;
                } else
                        where=UNKNOWN_REFERENCE;

                aDebug(DEBUG_MEMORY, "%p %-11s:%u %15lu %15lu %11lu %6u\n",t->id,where,id,t->bytes_allocated,t->times_allocated,t->times_freed,t->errors);
	}

	//clear all
	MemoryInitialized=0;
	aLog(D_INFO, "Memory Debugging Stopped\n");
	mem_unit *tmp,*ptr;

	//clear hash
	for(unsigned i=0;i<MEM_HASH_SIZE;i++) {
		for(ptr=mem_hash[i];ptr!=NULL;) {
			tmp=ptr;
			ptr=ptr->next;
			free(tmp);
		}
	}
	free(mem_hash);
	
	//clear ready
	while(mem_ready) {
		ptr=mem_ready;
		mem_ready=mem_ready->next;
		free(ptr);
	}

	//clear thread info
	while(thread_root) {
		t=thread_root;
		thread_root=thread_root->next;
		free(t);
	}
}

u_char AllocMemory(void *ptr, size_t size) {
        Service *s;
	Connection *conn;
	char *where;
        unsigned id=0;
        pthread_t caller=pthread_self();
	mem_unit *m,*p;	
	thread_unit *t;

        if((s=Services.getServiceByThr(caller))){
                where=s->getName();
                id=s->instance;
        } else if((conn=Connections.getConnectionByThr(caller))) {
                where="conn";
                id=conn->conn_id;
        }else
                where=UNKNOWN_REFERENCE;
	
	for(p=mem_hash[MEM_HASH(ptr)];p!=NULL;p=p->next)
                if(p->ptr==ptr) {
			aLog(D_WARN,"thread %p (%s:%u) tries to use used pointer %p\n",caller,where,id,ptr);
			//rememer error in thread info
			t=getThread(p->allocated_by);
			t->errors++;
			t=getThread(caller);
			t->errors++;
                        return 0;
                }

	//fill thread info
	t=getThread(caller);
	t->bytes_allocated+=size;
	t->times_allocated++;

	//look for mem_unit
	if(mem_ready) {
		m=mem_ready;
		mem_ready=mem_ready->next;
	} else 
		m=(mem_unit*)calloc(1,sizeof(mem_unit));

	//fill mem_unit
	m->ptr=ptr;
	m->size=size;
	m->allocated_by=caller;
	
	//insert into hash	
	m->next=mem_hash[MEM_HASH(ptr)];
	mem_hash[MEM_HASH(ptr)]=m;

        aDebug(DEBUG_MEMORY, "%p - %lu bytes allocated from thread %p %s:%u\n",ptr,size,caller,where,id);

	return 1;
}

u_char FreeMemory(void *ptr) {
        Service *s;
	Connection *conn;
	mem_unit *m,*p=NULL;
	char *where;
	unsigned id=0;
	pthread_t caller=pthread_self();
	thread_unit *t;

	if((s=Services.getServiceByThr(caller))){
		where=s->getName();
		id=s->instance;
        } else if((conn=Connections.getConnectionByThr(caller))) {
        	where="conn";
		id=conn->conn_id;
	}else 
                where=UNKNOWN_REFERENCE;
	
	unsigned start=MEM_HASH(ptr);
	for(m=mem_hash[start];m!=NULL;m=m->next) {
                if(m->ptr==ptr) break;
		p=m;
	}

	if(!m) {
		aLog(D_WARN,"thread %p (%s:%u) tries to free unused pointer %p\n",caller,where,id,ptr);
		t=getThread(caller);
		t->errors++;
		return 0;
	}
	
	size_t size=m->size;

	//fill thread info
        t=getThread(m->allocated_by);
        t->bytes_allocated-=size;
        t->times_freed++;

        aDebug(DEBUG_MEMORY, "%p - %lu bytes memory freed from thread %p %s:%u\n",ptr,size,caller,where,id);
	
	//remove from hash
	if(m==mem_hash[start]) mem_hash[start]=m->next;
	else p->next=m->next;
	//put to ready
	m->next=mem_ready;
	mem_ready=m;
	
	return 1;
}
//////////////////////////////////////////////////////////////////////////////////////////
thread_unit *getThread(pthread_t id) {
	thread_unit *t=NULL;
	
	for(t=thread_root;t!=NULL;t=t->next)
		if(t->id==id) break;

	if(!t) {
		t=(thread_unit*)calloc(1,sizeof(thread_unit));
		t->id=id;
		t->next=thread_root;
		thread_root=t;
	}
	return t;
}
//////////////////////////////////////////////////////////////////////////////////////////
//"show memory"
void cShowMemory(Connection *conn) {
	thread_unit *t;
	Service *s;
	Connection *c;
	char *where;
	unsigned id=0;

	pthread_mutex_lock(&mem_lock);

	fprintf(conn->stream_w, " Thread  |    Service   |bytes allocated|times allocated|times freed|errors\n"); 
	for(t=thread_root;t!=NULL;t=t->next){
		if((s=Services.getServiceByThr(t->id))){
                	where=s->getName();
                	id=s->instance;
        	} else if((c=Connections.getConnectionByThr(t->id))) {
                	where="conn";
                	id=c->conn_id;
        	} else
                	where=UNKNOWN_REFERENCE;
		
		fprintf(conn->stream_w, "%p %-11s:%u %15lu %15lu %11lu %6u\n",t->id,where,id,t->bytes_allocated,t->times_allocated,t->times_freed,t->errors);
	}
	
        //gather memory hash statistic
        mem_unit *ptr;
        unsigned used=0;
        unsigned max_chain=0;
        unsigned tmp;
        unsigned long total=0;

        for(unsigned i=0;i<MEM_HASH_SIZE;i++) {
                if(!(ptr=mem_hash[i])) continue;
                used++;
                tmp=0;
                for(;ptr!=NULL;ptr=ptr->next) {
                        tmp++;
                }
                total+=tmp;
                if(max_chain<tmp) max_chain=tmp;
        }
	
        fprintf(conn->stream_w, "\nMemory HASH: size=%u, %lu pointers hashed, %u nodes used, max chain=%u\n",MEM_HASH_SIZE+1,total,used,max_chain);


	pthread_mutex_unlock(&mem_lock);
}
#endif
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
