/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: netams.h,v 1.152.4.7 2005/03/17 18:05:38 anton Exp $ */

#ifndef _NETAMS_H_
#define _NETAMS_H_

#include "config.h"

#define RUN_PATH 	"/var/netams"

// Includes of local definitions
class Connection;
class Service;
class User;
class ConnectionsList;
class ServicesList;
class UsersList;
class Policy;
class PdList;
class NetUnit;
class FIFO;
class DS_FIFO;
class ScheduleList;
class PrefixTree;
class IPTree;
class Flow;
class Account;
struct policy_data;
struct alert;
struct ServiceAlerter_cfg;
struct sLoginData;
struct sQuotaData;
struct Monitor_cfg;
struct Quota_cfg;
struct Login_cfg;
extern char UNKNOWN_REFERENCE[];
extern char *buildforstring;

#ifndef MAX_PARAM
#define MAX_PARAM	64	//number of parameters in line
#endif

#ifndef MAX_UNITS
#define MAX_UNITS	2000	//number of units we definitely will server, if you want more - increase it
#endif

#define CHECK_ACCT	0
#define CHECK_FW	1  

//match flags
typedef u_char match;

#define MATCH_NONE	0
#define MATCH_SRC	1
#define MATCH_DST	2
#define MATCH_BOTH	3

typedef struct Html_cfg;

#include "common.h"
#include "connections.h"
#include "services.h"
#include "security.h"
#include "users.h"
#include "bw.h"
#include "flow.h"
#include "policy_prefix.h"
#include "policy.h"
#include "net_units.h"
#include "iptree.h"
#include "alert_report.h"
#include "messages_fifo.h"
#include "st_any.h"
#include "memory.h"
#include "lib.h"

#ifdef HAVE_BILLING
#include "billing.h"
#endif

#define ADD	1
#define REMOVE	0

#define DEBUG_TO_LOG	1
#define DEFAULT_READ_TIMEOUT 1000
#define SCHED_MIN_SLEEP_TIME 999

//default delays for different services
#define PROCESSOR_DELAY 30 //secs
#define PROCESSOR_LIFETIME 300  //secs
#define QUOTACTL_DELAY 55  //secs
#define WEBLOGIN_DELAY 110 //secs

#define KILOBYTE 1024L
#define MEGABYTE (KILOBYTE*KILOBYTE)
#define GIGABYTE (KILOBYTE*KILOBYTE*KILOBYTE)

#define PARSE_KILL	-1
#define PARSE_EXIT	-2
#define PARSE_RELOAD	-3
#define PARSE_SHUTDOWN	-4
#define PARSE_OK	1

#define HOST_MASK	0xFFFFFFFF
/////////////////////////////////////////////////////////////////////////////////////////
// Versions Management //////////////////////////////////////////////////////////////////
extern const char aaa_fw_software_name[];
extern const unsigned int aaa_fw_major_version;
extern const unsigned int aaa_fw_minor_version;
extern const unsigned int aaa_fw_subversion;
extern const unsigned int aaa_fw_build_version;
extern const unsigned int aaa_fw_build_version_local;
extern const char aaa_fw_build_time[];
extern const char aaa_fw_build_person[];

//////////////////////////////////////////////////////////////////////////////////////////
extern int flag_log, flag_quiet, flag_nodaemon, sql_config, flag_syslog;
extern FILE *LOGFILE;
extern char *config_file_name;
extern const char path_to_config[];
extern const char path_to_log[];
extern char *empty;
extern struct timeval program_start;

extern ConnectionsList Connections;
extern ServicesList Services;
extern UsersList Users;
extern ScheduleList Sched;
extern NetUnitsList Units;
extern PolicyList PolicyL;
extern Connection *cInternal;
extern Service *sMain;
extern Service *sSched;
extern User *uInternal;
extern MessageManager MsgMgr; 
extern FIFO Mux_in;
extern int global_return_code;

//////////////////////////////////////////////////////////////////////////////////////////
int aCommand(Connection *conn, char *cmd);
void cSave(Connection *conn);
int cLoad(Connection *conn, char *filename);
void cDebug(Connection *conn, char *param[], u_char no_flag);
void cShowConfig(FILE *f, Connection *conn, int nopasswords=0);
void cRotate(Connection *conn,char *param[]);
//////////////////////////////////////////////////////////////////////////////////////////
// server service functions
void sServerInit(Service *s);
void sServerProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sServerListCfg(Service *s, FILE *f);
//////////////////////////////////////////////////////////////////////////////////////////
// data-source service functions
void sDSInit(Service *s);
void sDSProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sDSListCfg(Service *s, FILE *f);
void cShowDS(Connection *conn);

#define  MAX_PKT_SIZE    0xffff

//we need this for netflow time syncronization
enum ds_flag 	{ DS_NONE=0, DS_CLOCK_REMOTE, DS_DIVERT, DS_TEE, DS_PROMISC, DS_IPQ, DS_ULOG };

typedef struct NFSource {
	in_addr_t src_addr;
	unsigned long seq_id;
	unsigned long long flows;
	unsigned long long packets;
	unsigned errors;
	long timestamp;
	NFSource *next;
} NFSource;

typedef struct ds_rule {
	u_short number;
	char *rule_str;
	ds_rule *next;
} ds_rule;

typedef struct ServiceDS_cfg {
	oid id;
	ds_flag flags;

	policy_target_type type;
	unsigned long long total_flows;
	unsigned long long total_packets;
	unsigned total_errors;
	long timestamp;
	
	char *src_cmd;
	ds_rule *root;

	unsigned long delay; //cumulative delay in microseconds accumulated for all packets
	unsigned skewdelay; // measured in microseconds and based on previous value
	int pc[10];
	long bc[10];
	
	//netflow
	struct in_addr l_addr; 	//address from we listen for netflow packets
	NFSource *nfroot; 	//list of netflow sources we are serving
	
	//libpcap
	void *pcap_data;
	
	union {
		unsigned nlmask; //for ulog
		unsigned port;
	};

	class IPTree *IPtree;
	class Flow *flow;
	class DS_FIFO *ds_fifo;
	IPStatMsg *msg;
	
	u_char *packet;
	
	Service *child;
	} ServiceDS_cfg;

unsigned sDSMeasure(ServiceDS_cfg* cfg, struct timeval *start, unsigned len);
//////////////////////////////////////////////////////////////////////////////////////////
// storage service functions
void sStorageInit(Service *s);
void sStorageProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sStorageListCfg(Service *s, FILE *f, int nopasswords=0);
typedef struct st_unit {
	oid id;
	storage_category cat;
	FIFO *in;
	FIFO *out;
	Service *s;
	st_unit *next;
	} st_unit;
typedef struct ServiceStorage_cfg {
	storage_type type;
	char *hostname;
	char *username;
	char *password;
	char *dbname; // 03.11.2002, default to 'netams'
	char *socket; 
	unsigned short port;
	FIFO *in;
	FIFO *out;
	char *db_path; // where our database are? (to check for HDD space)
	
	char *oids; //filename of temporary oids file
	char *raw;  //filename of temporary raw file
	char *summary; //filename of temporary summary file

	void *fd[ST_CONN_TYPES_NUM]; //connection to database
} ServiceStorage_cfg;

//////////////////////////////////////////////////////////////////////////////////////////
// processor service functions
extern Service *Processor;

void sProcessorProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sProcessorInit(Service *s);
u_char cShowStat(Connection *conn, char *param[]);
void sProcessorListCfg(Service *s, FILE *f, u_char nopasswords=0);
void cShowProcessor(Connection *conn);
void cShowVersion(Connection *conn);

enum restrict_type { DROP=1, PASS=0, PASS_LOCAL=2 , BRK=3};

typedef struct AutoAssignEntry {
	in_addr start;
	in_addr stop;
	AutoAssignEntry *next;
	} AutoAssignEntry;

enum au_type_enum { AU_TYPE_HOST=0, AU_TYPE_USER };  
enum au_naming_enum { AU_NAMING_BY_DNS=0, AU_NAMING_PREFIX1=1, AU_NAMING_PREFIX2=2 };
typedef struct AutoUnitsEntry {
	u_char id;
	au_type_enum type;
	au_naming_enum naming;
	char *prefix;
	char *put_to_group;
	AutoUnitsEntry *next;
	} AutoUnitsEntry;

void CreateAutoUnit(oid id, in_addr addr);

typedef struct ServiceProcessor_cfg {
	struct st_unit *st_root;
	NetUnit *def;
	unsigned delay; // in seconds
	unsigned lifetime; // in seconds
	restrict_type restrict_local, restrict_all;
	char *access_script;
	FIFO *fifo;  //this is for billing
	AutoAssignEntry *auto_assign;
	AutoUnitsEntry *auto_units;
	} ServiceProcessor_cfg;

extern ServiceProcessor_cfg ProcessorCfg;
//////////////////////////////////////////////////////////////////////////////////////////
// scheduler service classes and functions
class Task {
	public:
		oid id;
		time_t when;
		char *action;
		char *str_interval;
		Task *next;
		u_char visible;
		Task();
		~Task();
		};

class ScheduleList {
	public:
		Task *root;
		Task *last;
		unsigned short num_tasks;
		pthread_mutex_t *lock;
		ScheduleList();
		~ScheduleList();
		void Insert(Task *t);
		void Delete(oid id);
		void Delete(Task *t);
		Task *getTask(char *action);
		Task *getTaskById(oid id);
		void listTasks(Connection *conn);
		void listTasksCfg(FILE *f);
		unsigned short getTasks() { return num_tasks; }
		};

void sSchedulerInit(Service *s);
int cTask(Connection *conn, char *param[], u_char no_flag);
int cShowScheduleList(Connection *conn);
time_t getInterval(char *s);
//////////////////////////////////////////////////////////////////////////////////////////
// html service functionality
enum html_lang { HTML_LANG_EN, HTML_LANG_RU };
enum cpages_type { CPAGES_ALL, CPAGES_GROUPS, CPAGES_NONE, CPAGES_GROUPLIST }; // client pages - unit user&host
enum apages_type { APAGES_ALL, APAGES_NONE }; // accounts pages for billing service

extern Service *Html;
typedef struct Html_grouplist_item {
	oid groupid;
	struct Html_grouplist_item *next;
	} Html_grouplist_item;

typedef struct Html_cfg {
	char *path;
	html_lang lang;
	char *run_period;
	u_char is_distributed;
	cpages_type cpages;
	apages_type apages;
	char *url;
	char *servlet_url;
	u_char is_htaccess;
	Connection *conn;
	Html_grouplist_item *grouproot;
	} Html_cfg;
void sHtmlInit(Service *s);
void sHtmlProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sHtmlListCfg(Service *s, FILE *f);
void cHtml(Connection *conn, char *param[], u_char no_flag);
extern char mon_name[][12];
void sHtmlAction(Html_cfg *cfg);

//////////////////////////////////////////////////////////////////////////////////////////
// monitor virtual service functionality
void sMonitorProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sMonitorInit(Service *s);
void sMonitorCancel(void *v);
void cShowMonitor(Connection *conn);
void sMonitorListCfg(Service *s, FILE *out);
enum monitor_to { MON_FILE, MON_STORAGE, MON_NONE };
typedef struct Monitor_cfg {
	monitor_to to;
	char *name;
	unsigned long total;  //total items
	unsigned commit;	//items in tmp file
	unsigned long time;	//last time saving data
	u_char storage_no;
	char *query;
	FILE *fd;
	} Monitor_cfg;

void aMonitor(NetUnit *u, IPFlowStat *flow);
//////////////////////////////////////////////////////////////////////////////////////////
// web login service functionality
typedef struct WLdata {
	WLdata *next;
	User *user;
	time_t timeout;
	time_t inact;
	char *timeout_s;
	char *inact_s;
	SysPolicy pol;
	oid pol_perm;
	wlUlist *root;
	unsigned totalunits;
	u_char multi;
	} WLdata;

extern Service *Weblogin;

typedef struct Weblogin_cfg {
	WLdata *WLroot;
	unsigned delay;
} Weblogin_cfg;

void sWLInit(Service *s);
void sWLProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sWLListCfg(Service *s, FILE *f);
void cShowTimeouts(Connection *conn);
//////////////////////////////////////////////////////////////////////////////////////////
// PVM gateway service functionality
typedef struct PVM_cfg {
	FIFO *queue;
	unsigned long msg_send;
	unsigned long msg_send_ok;
	char *host;
	char *path;
	FILE *io;
	//nothing yet
	} PVM_cfg;
void sPVMInit(Service *s);
void sPVMProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sPVMListCfg(Service *s, FILE *f);
void *sPVM(void *s);
void sPVMCancel(void *);
void sPvmSend(Account*, char *);
//////////////////////////////////////////////////////////////////////////////////////////
// login service functionality
extern Service *Login;

#define LOGIN_ACTIVE		0x01
#define LOGIN_DEFAULT_ACTIVE	0x02 //currently not used
#define LOGIN_STRICT		0x04
#define LOGIN_UPDATE		0x08

typedef struct sLoginData {
	u_char flags;
	time_t opened; //time when opened
	time_t inact; // unit-specific inactivity timeout
	time_t abs; // unit-specific absolute timeout
	struct in_addr ip_from;
	struct in_addr last_ip_from;
	struct ether_addr mac_from;
	struct ether_addr last_mac_from;
	char *password;
	} sLoginData;

typedef struct Login_cfg {
	time_t default_inact;
	time_t min_inact;
	time_t max_inact;
	time_t default_abs;
	time_t min_abs;
	time_t max_abs;
	u_char min_passwd_length;
	u_char storage;
	u_char set_user_ip;
	unsigned delay;
	Service *st;
	char *file;
	unsigned relogin; // =1 (default) if we allow relogin for active, otherwise =0
	pthread_rwlock_t *rwlock; // for multiple threads access same config data
	} Login_cfg;

#define S_LOGIN_DEF_default_inact		0
#define S_LOGIN_DEF_default_abs 		0
#define S_LOGIN_DEF_max_inact			12*60*60
#define S_LOGIN_DEF_min_inact			60
#define S_LOGIN_DEF_max_abs				24*12*60*60
#define S_LOGIN_DEF_min_abs				60
#define S_LOGIN_DEF_min_passwd_length	3
#define S_LOGIN_DEF_delay				60

void sLgInit(Service *s);
void sLgProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sLgListCfg(Service *s, FILE *f);
void cShowLogin(Connection *conn, char *param[]);
//////////////////////////////////////////////////////////////////////////////////////////
void cAccessScriptCall(restrict_type action, NetUnit *u, char *param=NULL);
//////////////////////////////////////////////////////////////////////////////////////////
// quota control service functionality
extern Service *Quota;

typedef struct qstat {
	unsigned long long in;
	unsigned long long out;
	unsigned long long sum;
	unsigned long long softin;
	unsigned long long softout;
	unsigned long long softsum;
	} qstat;

#define QUOTA_NONE		0
#define QUOTA_ACTIVE		0x01
#define QUOTA_SOFTBLOCKED	0x02
#define QUOTA_BLOCKED		0x04
#define QUOTA_UPDATE		0x08
#define QUOTA_NSS		0x10
#define QUOTA_NHS		0x20
#define QUOTA_NRS		0x40

typedef struct sQuotaData {
	Policy *policy;
	qstat h, d, w, m;
	u_char soft_treshold;
	u_char flags;
	time_t blocked_time;
	oid nso, nho, nro;
	Policy *fw_block_policy; //this policy will be added to this unit's fw list when blocked
	policy_flag fw_block_policy_flags;
	} sQuotactlData;

#define S_QUOTA_DEF_arraysize 2
typedef struct Quota_cfg {
	Policy *default_policy;
	Policy *default_fw_block_policy; //this policy will be added to all units fw list when blocked
	policy_flag default_fw_block_policy_flags;

	u_char default_soft_treshold;
	u_char storage;
	unsigned short delay;
	Service *st;
	char *file;

	pthread_rwlock_t *rwlock; // for multiple threads access same config data 

	oid notify_soft[S_QUOTA_DEF_arraysize];
	oid notify_hard[S_QUOTA_DEF_arraysize];
	oid notify_return[S_QUOTA_DEF_arraysize];
} Quota_cfg;

#define S_QUOTA_DEF_soft_treshold 80
#define S_QUOTA_DEF_delay 60
#define S_QUOTA_DEF_notify_soft 1
#define S_QUOTA_DEF_notify_hard 1
#define S_QUOTA_DEF_notify_return 1

void sQuInit(Service *s);
void cShowQuota(Connection *conn, char *param[]);
void sQuListCfg(Service *s, FILE *f);
void sQuProcessCfg(char *param[], Connection *conn, u_char no_flag);
//////////////////////////////////////////////////////////////////////////////////////////
// quotactl control service functionality

typedef struct Qdef {
        char *name;
        Policy *policy;
        qstat h, d, w, m;
        qUlist *root;
        Qdef *next;

        SysPolicy sys_policy;
        oid sys_policy_perm;

        } Qdef;

typedef struct Quotactl_cfg {
        Qdef *root;
        pthread_rwlock_t *rwlock;
        User *soft_reach; User *hard_reach; User *reset_back;
        u_char user_soft_reach, user_hard_reach, user_reset_back;
        unsigned short delay; // in seconds
        } Quotactl_cfg;

void sQuotactlInit(Service *s);
void sQuotactlProcessCfg(char *param[], Connection *conn, u_char no_flag);
void sQuotactlListCfg(Service *s, FILE *f);
void cShowQuotactl(Connection *conn, char *param[]);
//////////////////////////////////////////////////////////////////////////////////////////
// events functionality
void LogEvent(event_type event, oid uid, oid usid, oid aid, char const *fmt, ...);
//////////////////////////////////////////////////////////////////////////////////////////
#endif /* !_NETAMS_H_ */
