/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: netflow.c,v 1.32 2005/01/17 13:13:21 jura Exp $ */

#include "config.h"
#include "lib.h"

#define NETFLOW_CLASS
#include "netflow.h"

void Debug(char *msg,...);

#define aMalloc(x) calloc(1,x)
#define aFree(x)   free(x)

//////////////////////////////////////////////////////////////////////////
NetFlow::NetFlow(char *dst_host, unsigned short dst_port) {
	bzero(&udp_dest, sizeof(udp_dest));
    	udp_dest.sin_family=AF_INET;
    	udp_dest.sin_port=htons(dst_port);
    	inet_aton(dst_host, &udp_dest.sin_addr);
    	udp_socket_fd=socket(AF_INET, SOCK_DGRAM, 0);
    	if (udp_socket_fd==-1)
    	{
        	fprintf(stderr, "failed to create UDP socket: %s\n", strerror(errno));
        	exit(5);
    	}
	
	int status=bigsockbuf(udp_socket_fd, SO_SNDBUF, FT_SO_RCV_BUFSIZE);
	if (status<0) {
		fprintf(stderr, "failed to setsockopt send buffer: %s", strerror(errno));
		exit(5);
	} else {
		Debug("send bufer set to %u\n", status);
	}
	
	t_active=60;
	t_inactive=60;
	t_expired=7;
	t_msg_expired=300;

	e_used=0;
	e_total=0;
	t_now=t=t_msg=time(NULL);
	table=NULL;
	ready=NULL;
	last_id=0; 
	expired_id=0;
	sent_flows=0;
	sent_packets=0;
	table=(entry**)aMalloc(IPV4_HASH_SIZE*sizeof(entry*));
	message=(IPStat5Msg*)aMalloc(sizeof(IPStat5Msg));
        message->header.version=htons(FLOW_VERSION_5);
        message->header.count=0;
	message->header.unix_secs=t_now; //see PR 65
	message->header.SysUptime=0; //see PR 65
        message->header.engine_id=0;
        message->header.engine_type=htons(31);
        message->header.flow_sequence=htonl(sent_flows);
}
//////////////////////////////////////////////////////////////////////////
NetFlow::~NetFlow() {
	FlushAll();
	entry *ptr=ready;
	while(ready) {
		ptr=ready;
		ready=ready->next;
	aFree(ptr);
	}
	aFree(table);
	aFree(message);
	close(udp_socket_fd);
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::SetTimeouts(unsigned active,unsigned inactive,unsigned expired, unsigned msg_expired) {
	if(active) t_active=active;
	if(inactive) t_inactive=inactive;
	if(expired) t_expired=expired;
	if(msg_expired) t_msg_expired=msg_expired;
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::CheckExpire(entry *e){
    	if ((unsigned)e->last - (unsigned)e->start > t_active) e->expired=1;
    	else if (t_now - (unsigned)e->last > t_inactive) e->expired=1;
#ifdef DEBUG 
	if(e->expired) Debug("%u expired\n", e->id);
#endif
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::Processpacket(struct ip *ip) {
    	unsigned hash;
	u_short sp=0, dp=0;
	in_addr_t src,dst;
	u_char proto;
	u_char tos;
	entry *e;
	u_short chunk=0;

	dst = ip->ip_dst.s_addr;
	src = ip->ip_src.s_addr;
	proto = ip->ip_p;
	tos = ip->ip_tos;

	if (ntohs(ip->ip_off) & (IP_MF | IP_OFFMASK)) {
		//packet fragmented. check obtained from /usr/src/sys/netinet/ip_fastfwd.c
		sp=dp=0;
		hash=IPV4_ADDR_HASH(src, dst);
        } else 
	if (ip->ip_p==IPPROTO_TCP || ip->ip_p==IPPROTO_UDP) {
		struct tcphdr *th;
		th=(struct tcphdr *)((unsigned char *)ip + ip->ip_hl*4);
		sp=th->th_sport;
		dp=th->th_dport;
		hash=IPV4_FULL_HASH(src, dst, sp, dp);
	} else {
		hash=IPV4_ADDR_HASH(src, dst);
	}
	
	for(e=table[hash]; e!=NULL; e=e->next){
		if (e->ip_dst==dst && e->ip_src==src &&
		e->ip_proto==proto && e->ip_tos==tos &&
		e->src_port==sp && e->dst_port==dp) {
			e->ip_len+=ntohs(ip->ip_len);
			if(!e->count) e->start=t_now;
                	e->count++;
                	e->last=t_now;
                	break;
        	}
		chunk++;
    	}
	
	if (e==NULL) {
		if(chunk > MAX_CHUNK) FlushAll(); //protection against DoS
		if(ready) { //we'll use ready empty entry
			e=ready;
			ready=ready->next;
		} else { // we must create a new flow record
        		e=(entry*)aMalloc(sizeof(entry));
			e_total++;
		}
		e_used++;
        	e->ip_dst=dst;
        	e->ip_src=src;
        	e->dst_port=dp;
		e->src_port=sp;
		e->ip_len=ntohs(ip->ip_len);
        	e->ip_proto=proto;
        	e->ip_tos=tos;
        	e->start=e->last=t_now;
        	e->expired=0;
        	e->id=last_id;
        	e->count=1;
        	last_id++;
        	
		e->next=table[hash];
		table[hash]=e;
		
#ifdef DEBUG
        	char buf1[32], buf2[32];
        	strcpy(buf1, inet_ntoa(ip->ip_src));
        	strcpy(buf2, inet_ntoa(ip->ip_dst));

      		Debug("[%u(%u)]-%p %u s:%s:%u d:%s:%u %u\n", hash, e->id, e, proto, buf1, ntohs(sp), buf2, ntohs(dp), ntohs(ip->ip_len));
#endif
	}
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::Expiresearch(){
    	unsigned i;
    	entry *e, *prev;
	entry *tmp;
	
	t_now=time(NULL);
	if((unsigned)(t_now-t)<t_expired) return;
	t=t_now;
		
    	Debug("ExpireSearch t:[%u] a:[%u] at %lu: \n", last_id,last_id-expired_id, t_now);
    	
	for (i=0; i<IPV4_HASH_SIZE; i++) {
        	prev=NULL;
		e=table[i];
		while(e) {	
			tmp=e->next;
            		CheckExpire(e);
            		if (e->expired) {
				// with high probability used packet might be used again thou ...
				if(e->count) {
					ToSend(e);
					e->count=0;
					e->ip_len=0;	
					e->expired=0;
					e->start=e->last=t_now;
//					e->id=last_id;
//                			last_id++;
					prev=e;
                		} else {
					expired_id++;
					if (e==table[i])
                                        	table[i]=(entry *)e->next;
                                	else {
                                        	prev->next=e->next;
                                	}
					//this means we have flow that was unaccessible one round
					//move it ready 
					e->next=ready;
					ready=e;
					e_used--;
				}
            		} else {
            			prev=e;
			}
			e=tmp;
        	}
    	}
	if(unsigned(t-t_msg) > t_msg_expired) DoSend();
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::FlushAll() {
    	unsigned i;
    	entry *e;
	entry *tmp;

    	Debug("FlushAll::begin\n");
        Debug("FlushAll t:[%u] a:[%u] at %lu: ", last_id, last_id-expired_id, t_now);

    	for (i=0; i<IPV4_HASH_SIZE; i++) {
		e=table[i];
		while(e) {
                	Debug("%u{%u} ", e->id, i);
            		expired_id++;
			tmp=e->next;
            		if(e->count) ToSend(e);
			e->next=ready;
			ready=e;
			e_used--;
            		e=tmp;
		}
		table[i]=NULL;
    	}
	
	// it's time to check if there is stalled unsent data in message
	DoSend();

    	Debug("FlushAll::end\n");
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::ToSend(entry *e) {

	IPFlow5Stat  *record=&message->records[message->header.count];

        record->dOctets=htonl(e->ip_len);
        record->dPkts=htonl(e->count);
        record->input=0;
        record->output=0;
        record->nexthop.s_addr=INADDR_ANY;

        record->tos=e->ip_tos;
        record->prot=e->ip_proto;
        record->First=htonl((e->start-message->header.unix_secs)*1000);   //trick here, see PR 65 
        record->Last=htonl((e->last-message->header.unix_secs)*1000);   //trick here, see PR 65

        record->dst_as=0;
        record->dst_mask=0;
        record->dstaddr.s_addr=e->ip_dst;
        record->dstport=e->dst_port;

        record->src_as=0;
        record->src_mask=0;
        record->srcaddr.s_addr=e->ip_src;
        record->srcport=e->src_port;

        message->header.count++;
        Debug("record %u in packet %llu prepared, %u\n", message->header.count-1, sent_packets+1, e->ip_len);
        if (message->header.count==V5FLOWS_PER_PAK) DoSend();
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::DoSend() {
	if (!message || !message->header.count) return;
    	Debug("DoSend::begin\n");
    	sent_packets++;
    	sent_flows=sent_flows+message->header.count;
        Debug("sending NetFlow v5 packet with %u flows\n", message->header.count);
	
	unsigned count=message->header.count;
    	message->header.count=htons(message->header.count);
	message->header.unix_secs=htonl(message->header.unix_secs);
    	if (sendto(udp_socket_fd, message, sizeof(struct Flow5StatHdr)+count*sizeof(struct IPFlow5Stat), 0, (struct sockaddr *)&udp_dest, sizeof(udp_dest)) <0)
        	Debug("UDP send failed[%u]: %s", errno, strerror(errno));

    	Debug("DoSend::end\n");
	t_msg=t_now;
//prepare new header
	message->header.unix_secs=t_now;
	message->header.count=0;
	message->header.flow_sequence=htonl(sent_flows);
}
//////////////////////////////////////////////////////////////////////////
void NetFlow::Status() {
	Debug("\nlast_id=%u, active=%u\n", last_id, last_id-expired_id);
        Debug("sent %llu flows in %llu packets\n", sent_flows, sent_packets);
}
//////////////////////////////////////////////////////////////////////////
u_char debug=0, quiet=0;

void Debug(char *msg,...){
#ifdef DEBUG
	va_list args;
	char *params;

	if (!debug || quiet) return;
	params=(char *)malloc(2049);
	va_start(args, msg);
	vsprintf(params, msg, args);
	va_end(args);

	fprintf(stdout, "[%lu]: %s", time(NULL), params);
	fflush(stdout);

	free(params);
#endif
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
