/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: p_prefix.c,v 1.28 2005/01/17 13:13:21 jura Exp $ */

#include "netams.h"

/////////////////////////////////////////////////////////////////////////////////////////////
PrefixFile::PrefixFile(){
	list=NULL;
	filename=NULL;
	num_prefixes=0;
	descriptor=NULL;
	ptree=new PrefixTree;
	rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
	pthread_rwlock_init(rwlock, NULL);
	//	printf("PrefixFile constructor: this=%p\n", this);
}
/////////////////////////////////////////////////////////////////////////////////////////////
PrefixFile::~PrefixFile(){
	if(list) this->Unload();
        delete ptree;
	pthread_rwlock_destroy(rwlock);
	aFree(rwlock);
	aFree(filename);
}
/////////////////////////////////////////////////////////////////////////////////////////////
int PrefixFile::Load(char *f){
	char *tmp=set_string(f);
	if (filename) aFree(filename);
	filename=tmp;
	descriptor=fopen(filename, "r");
	if (!descriptor) { aLog(D_WARN, "unable to open prefix file: %s\n", strerror(errno)); return -1; }

	char buf[64], *addr;
	u_char mask;
	struct in_addr ia;
	unsigned i=0;

	pthread_rwlock_wrlock(rwlock);
	
	while (!feof(descriptor)){
		addr=fgets(buf, 63, descriptor);
		if(!addr) continue;	
		if(addr[0]=='#') addr++;
		if(addr[0]=='!') addr++;
		if(buf[0]!='#' && !isdigit(addr[0])) continue;
		num_prefixes++;
	}
	rewind(descriptor);
	
	list=(prefix*)aMalloc(sizeof(prefix)*(num_prefixes));

	while (!feof(descriptor) && i<num_prefixes){
		addr=fgets(buf, 63, descriptor);
		if(!addr) continue;
		
		list[i].flags=0;

		if(addr[0]=='#') {
			list[i].flags|=PREFIX_COMMENTED;
			addr++;
		}
		if(addr[0]=='!') {
			list[i].flags|=PREFIX_INVERTED;
			addr++;
		}
		if(buf[0]!='#' && !isdigit(addr[0])) continue;

		mask=getAddr(addr, &ia);

		list[i].network.s_addr=ia.s_addr;
		list[i].mask=mask;
		list[i].match=0;
		if(!(list[i].flags&PREFIX_COMMENTED)) ptree->prefix2tree(&list[i]);

		aDebug(DEBUG_IPTREE, "i=%4u buf=%s%s%s/%u\n", i, 
				(list[i].flags&PREFIX_COMMENTED)?"#":"",
				(list[i].flags&PREFIX_INVERTED)?"!":"",
				inet_ntoa(ia), mask);
		i++;	
	}
	pthread_rwlock_unlock(rwlock);
	aLog(D_INFO, "%u prefixes from %s read ok, starting %p\n", i, filename, list); 
	aLog(D_INFO, "PrefixTree: %lu nodes [%u] + %lu dynamic links [%u] allocated, %lu bytes used\n",ptree->nodes,sizeof(IPTree_node),ptree->dlink,256*sizeof(IPTree_node*),ptree->nodes*sizeof(IPTree_node)+ptree->dlink*256*sizeof(IPTree_node*));
	fclose(descriptor);
	descriptor=NULL;
	return 0;
}
/////////////////////////////////////////////////////////////////////////////////////////////
void PrefixFile::Unload(){

	aLog(D_INFO, "%u prefixes from %s closed\n", num_prefixes, filename); 
	pthread_rwlock_wrlock(rwlock);
	ptree->freetree(ptree->root);
	aFree(list);
	list=NULL;
	pthread_rwlock_unlock(rwlock);
	num_prefixes=0;
}
/////////////////////////////////////////////////////////////////////////////////////////////
u_char PrefixFile::Check(unsigned addr){
	u_char mf=0;

	if(!ptree->root) return 0;
	pthread_rwlock_rdlock(rwlock);
	if(ptree->checktree(addr))  mf=1;
	pthread_rwlock_unlock(rwlock);
	return mf;
}
/////////////////////////////////////////////////////////////////////////////////////////////
void PrefixFile::Dump(Connection *c){
	prefix *p;

	aParse(c, "prefix file %s, total %u entries\n", filename, num_prefixes);

	pthread_rwlock_rdlock(rwlock);
	for (unsigned i=0; i<num_prefixes; i++) {
		p=&list[i];
		aParse(c, "prefix %05u %s%s%s/%u %llu\n", i, 
			(p->flags&PREFIX_COMMENTED)?"#":"",
			(p->flags&PREFIX_INVERTED)?"!":"",
			inet_ntoa(p->network), p->mask, p->match);
	}
	pthread_rwlock_unlock(rwlock);
}
/////////////////////////////////////////////////////////////////////////////////////////////
void PrefixFile::Do(Connection *conn, char *action, char *action2){
	if (!strcasecmp(action, "reload")) { 
		Unload(); 
		Load(filename); 
	}
	else if (!strcasecmp(action, "save")) Save(conn);
	else if (!strcasecmp(action, "dump")) Dump(conn); 
	else if (!strcasecmp(action, "unload")) Unload(); 
	else if (!strcasecmp(action, "zeromatch")) {
		pthread_rwlock_wrlock(rwlock);
		for (unsigned i=0; i<num_prefixes; i++) list[i].match=0;
		pthread_rwlock_unlock(rwlock);
		aParse(conn, "all prefixes match set to zero\n");
	}
	else aParse(conn, "unknown 'do' action requested: '%s'\n", action);
}
/////////////////////////////////////////////////////////////////////////////////////////////
void PrefixFile::Save(Connection *conn){
	if(!ptree->root) {aParse(conn, "nothing to save: ip tree is empty\n"); return;}
	descriptor=fopen(filename, "w");
	if (!descriptor) {
		aLog(D_WARN, "unable to open prefix file %s: %s!\n",filename,strerror(errno)); 
		return;
	}
	rewind(descriptor);
	prefix *p;

	pthread_rwlock_rdlock(rwlock);
	for (unsigned i=0; i<num_prefixes; i++){
		p=&list[i];
		fprintf(descriptor, "%s%s%s/%u\n", 
			(p->flags&PREFIX_COMMENTED)?"#":"",
			(p->flags&PREFIX_INVERTED)?"!":"",
			inet_ntoa(p->network), p->mask);
	}

	pthread_rwlock_unlock(rwlock);
	fflush(descriptor);
	fclose(descriptor);
	aParse(conn, "%u prefixes saved to %s(%p)\n", num_prefixes, filename, descriptor);
}
/////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////
