/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: services.c,v 1.72 2005/01/17 13:13:22 jura Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////
ServicesList Services;
//////////////////////////////////////////////////////////////////////////
char *service_type_name[SERVICE_TYPES_NUM]= { "undef", "storage", "processor", "data-source", "monitor", "html", "quota", "quotactl", "login", "alerter", "scheduler", "billing", "weblogin", "pvmgate", "server", "main", "ds_ipq", "ds_ipfw", "ds_libpcap", "ds_ulog" };

//////////////////////////////////////////////////////////////////////////
Service::Service(service_type s_type, u_char i){
	type=s_type;
	
	instance=i;
	next=NULL;
	t_id=(pthread_t)0;
	sleep_mutex=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t)); pthread_mutex_init(sleep_mutex, NULL);
	sleep_cond=(pthread_cond_t*)aMalloc(sizeof (pthread_cond_t)); pthread_cond_init(sleep_cond, NULL);
	sleep_state=0;
	cfg=(void*)NULL;
	flags=0;
}

Service::~Service(){
// netbsd has ugly pthread_mutex_destroy implementation: it cannot destroy already locked mutex 
// we have no chances to unlock it from here since destructor thread is not owner
#ifndef NETBSD
	pthread_mutex_destroy(sleep_mutex); aFree(sleep_mutex);
	pthread_cond_destroy(sleep_cond); aFree(sleep_cond);
#endif
}

char *Service::getName(){ return service_type_name[(u_char)type]; }

void Service::setTID() { t_id = pthread_self(); }

void Service::Shutdown(){
	
	flags&=~SERVICE_RUN; //mark service as not RUNning

	if(type==SERVICE_MONITOR) sMonitorCancel(this);
	else if (t_id) {
		pthread_cancel(t_id);
		pthread_join(t_id, NULL);  //it's necessary here because there is delete after it 
	}
}
	
u_char Service::Sleep(unsigned sec){	 // parameter in seconds!!!!
	int cancel_state;
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &cancel_state);
	sleep_state=1;
	
	aDebug(DEBUG_SLEEP, "going to sleep service %s for %d secs\n", this->getName(), sec);
	pthread_mutex_lock(sleep_mutex);

	if (sec) {
		struct timespec ts;
		struct timeval tv;
		gettimeofday(&tv, NULL);

		TIMEVAL_TO_TIMESPEC(&tv, &ts);
		ts.tv_sec+=sec;

		int i=pthread_cond_timedwait(sleep_cond, sleep_mutex, &ts);
		pthread_mutex_unlock(sleep_mutex);
		sleep_state=0;
		pthread_setcancelstate(cancel_state, NULL);
		if (i && i!=ETIMEDOUT) aLog(D_WARN, "sleeping of %s failed: %s\n", this->getName(), strerror(i));
		else if (i && i==ETIMEDOUT) return 1;
		return 0;
	}
	else { 
		pthread_cond_wait(sleep_cond, sleep_mutex); 
		pthread_mutex_unlock(sleep_mutex);
		sleep_state=0;
	}
	pthread_setcancelstate(cancel_state, NULL);
	return 0;
}

u_char Service::Wakeup(){
#ifdef DEBUG
	Service *s=Services.getServiceByThr(pthread_self());
	if(s)
		aDebug(DEBUG_SLEEP, "(%s:%u) waking up service %s:%u, %s\n", s->getName(),s->instance, this->getName(), this->instance, !sleep_state?"WAS_UP":"WAS_DOWN");
	else 
		aDebug(DEBUG_SLEEP, "(thread %u) waking up service %s:%u, %s\n", pthread_self(), this->getName(), this->instance, !sleep_state?"WAS_UP":"WAS_DOWN");
#endif
	if (sleep_state==0) return 1;
	sleep_state=0;
	pthread_cond_signal(sleep_cond);
	return 0;
}

void Service::Start(){
	if (type==SERVICE_SERVER) sServerInit(this);
	else if (type==SERVICE_PROCESSOR) sProcessorInit(this);
	else if (type==SERVICE_STORAGE) sStorageInit(this);
	else if (type==SERVICE_DATASOURCE) sDSInit(this);
	else if (type==SERVICE_ALERTER) sAlerterInit(this);
	else if (type==SERVICE_QUOTACTL) sQuotactlInit(this);
	else if (type==SERVICE_QUOTA) sQuInit(this);
#ifdef HAVE_BILLING
	else if (type==SERVICE_BILLING) sBiInit(this);
#endif
	else if (type==SERVICE_WEBLOGIN) sWLInit(this);
	else if (type==SERVICE_HTML) sHtmlInit(this);
	else if (type==SERVICE_SCHEDULER) sSchedulerInit(this);
	else if (type==SERVICE_PVMGATE) sPVMInit(this);
	else if (type==SERVICE_LOGIN) sLgInit(this);
	else if (type==SERVICE_MONITOR) sMonitorInit(this);
	else { aLog(D_WARN, "service %s undefined\n", getName()); return; }
	aLog(D_INFO, "service %s starting thread...\n", getName());
}

void Service::ProcessCfg(char *param[], Connection *conn, u_char no_flag){
	if (type==SERVICE_SERVER) sServerProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_PROCESSOR) sProcessorProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_DATASOURCE) sDSProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_STORAGE) sStorageProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_ALERTER) sAlerterProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_QUOTACTL) sQuotactlProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_QUOTA) sQuProcessCfg(param, conn, no_flag);
#ifdef HAVE_BILLING	
	else if (type==SERVICE_BILLING) sBiProcessCfg(param, conn, no_flag);
#endif
	else if (type==SERVICE_WEBLOGIN) sWLProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_PVMGATE) sPVMProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_LOGIN) sLgProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_HTML) sHtmlProcessCfg(param, conn, no_flag);
	else if (type==SERVICE_MONITOR) sMonitorProcessCfg(param, conn, no_flag);
}

void Service::listCfg(FILE *f, int nopasswords){
	
	//strict service we want to see in show config
	if(!cfg || type==SERVICE_SCHEDULER) return;

	//do not show instance for single instance service
	if(type==SERVICE_PROCESSOR || type==SERVICE_SCHEDULER || type==SERVICE_QUOTA \
	|| type==SERVICE_LOGIN || type==SERVICE_BILLING || type==SERVICE_HTML) {
		fprintf(f, "service %s\n", getName());
	} else
		fprintf(f, "service %s %u\n",getName(), instance);

	if (type==SERVICE_SERVER) sServerListCfg(this, f);
	else if (type==SERVICE_PROCESSOR) sProcessorListCfg(this, f, nopasswords);
	else if (type==SERVICE_DATASOURCE) sDSListCfg(this, f);
	else if (type==SERVICE_STORAGE) sStorageListCfg(this, f, nopasswords);
	else if (type==SERVICE_ALERTER) sAlerterListCfg(this, f);
	else if (type==SERVICE_QUOTACTL) sQuotactlListCfg(this, f);
	else if (type==SERVICE_QUOTA) sQuListCfg(this, f);
#ifdef HAVE_BILLING	
	else if (type==SERVICE_BILLING) sBiListCfg(this, f);
#endif
	else if (type==SERVICE_WEBLOGIN) sWLListCfg(this, f);
	else if (type==SERVICE_PVMGATE) sPVMListCfg(this, f);
	else if (type==SERVICE_LOGIN) sLgListCfg(this, f);
	else if (type==SERVICE_HTML) sHtmlListCfg(this, f);
	else if (type==SERVICE_MONITOR) sMonitorListCfg(this, f);
}

void Service::listInfo(FILE *f) {
	if (type==SERVICE_DATASOURCE) {
		ServiceDS_cfg *c=(ServiceDS_cfg*)cfg;
		char *pt;
		switch (c->type){
			case PT_IP_TRAFFIC: pt="IP_FILT"; break;
			case PT_NETFLOW_TRAFFIC: pt="NETFLOW"; break;
			case PT_LIBPCAP_TRAFFIC: pt="LIBPCAP"; break;
			default : pt="<\?\?>"; break;
		}
		fprintf(f, " Data-source ID=%d type %s source %s:%u loop %llu average %d mcsec\n", instance,  pt, c->src_cmd?c->src_cmd:"--", c->port, c->total_packets, (c->total_packets)?(unsigned)(c->delay/c->total_packets):0);
		int avg_pps=0; for (int i=0; i<10; i++) avg_pps+=c->pc[i]; avg_pps=avg_pps/9;
		long avg_bps=0; for (int i=0; i<10; i++) avg_bps+=c->bc[i]; avg_bps=avg_bps/9;
		fprintf(f, "    Perf: average skew delay %lu mcsec, PPS: %u, BPS: %lu\n", c->skewdelay, avg_pps, avg_bps);
	}
	else if (type==SERVICE_STORAGE) {
		ServiceStorage_cfg *c=(ServiceStorage_cfg*)cfg;
		char *pt;
		switch (c->type){
			case HASH: pt="HASH"; break;
			case MY_SQL: pt="MYSQL"; break;
			case POSTGRES: pt="POSTGRES"; break;
			case ORACLE: pt="ORACLE"; break;
			default : pt="<\?\?>"; break;
		}
		fprintf(f, " Storage ID=%d type %s wr_q %u/%lu rd_q %u/%lu\n", instance, pt, c->in->num_items, c->in->total_items, c->out->num_items, c->out->total_items); 
	}
}

void Service::showPerf(FILE *f, u_char isheader) {
	if (type==SERVICE_DATASOURCE) {
		ServiceDS_cfg *c=(ServiceDS_cfg*)cfg;
		int avg_pps=0; for (int i=0; i<10; i++) avg_pps+=c->pc[i]; avg_pps=avg_pps/9;
		long avg_bps=0; for (int i=0; i<10; i++) avg_bps+=c->bc[i]; avg_bps=avg_bps/9;
		if (isheader) fprintf(f, "DsLOOP:%u DsAVG:%u DsDly:%u DsPps:%u DsBps:%u ", instance, instance, instance, instance, instance);
		else fprintf(f, "%8llu %7u %7lu %7u %7lu\t", c->total_packets, (c->total_packets)?(unsigned)(c->delay/c->total_packets):0, c->skewdelay, avg_pps, avg_bps);
	}
	else if (type==SERVICE_STORAGE) {
		ServiceStorage_cfg *c=(ServiceStorage_cfg*)cfg;
		if (isheader) fprintf(f, "StWRITE:%u ", instance);
		else fprintf(f, "%7lu ", c->in->total_items);
	}
}
//////////////////////////////////////////////////////////////////////////
ServicesList::ServicesList(){
	root=NULL;
	num_services=0;
	tries_lock=tries_lock_failed=0;
	unknown_service=UNKNOWN_REFERENCE;
        rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
        pthread_rwlock_init(rwlock, NULL);
}

ServicesList::~ServicesList(){
        pthread_rwlock_destroy(rwlock);
        aFree(rwlock);
}

void ServicesList::Insert(Service *s){
	tries_lock++;
        int err=pthread_rwlock_trywrlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	Service *d,*p=NULL;
	for(d=root; d!=NULL; d=d->next) {
		if (d==s) {
		        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
			return;
		}
		p=d;
	}
	
	if(root) p->next=s; else root=s;
	num_services++;
//	aDebug(DEBUG_COMMAND, "service %s:%u inserted, %u services\n", s->getName(), s->instance, num_services);
        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}	

void ServicesList::Delete(Service *s){
	tries_lock++;
        int err=pthread_rwlock_trywrlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	Service *d, *p=NULL;
	for(d=root; d!=NULL; d=d->next)	{
		if (d==s) {
			if (s==root) root=s->next;
			else p->next=s->next;

			num_services--;
			break;
		}
		p=d; 
	}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

Service *ServicesList::getService(service_type type, u_char i){
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
	for(d=root; d!=NULL; d=d->next)
		if (d->type==type && d->instance==i) break;
	
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return d;
}

Service *ServicesList::getServiceNextByType(service_type type, Service *s) {
	tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	
	Service *d;
	for(s?d=s->next:d=root; d!=NULL; d=d->next)
		if(d->type==type) break;
			
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
        return d;
}

Service *ServicesList::getServiceByThr(pthread_t t){
	Service *d;
//	printf("ServicesList::getServiceByThr locked by %d\n", t);
        pthread_rwlock_rdlock(rwlock); 

	for(d=root; d!=NULL; d=d->next)
		if (pthread_equal(t, d->t_id)) break;
	
	pthread_rwlock_unlock(rwlock);
	return d;
}

void ServicesList::StartAll(Service *except){
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
	
        for (d=root; d!=NULL; d=d->next) {
		d->flags|=SERVICE_RUN;
		if (except!=d) d->Wakeup();
	}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void ServicesList::ShutdownAll(Service *except){
	Service *s=NULL;
	
	//shutdown server first
	if((s=getServiceNextByType(SERVICE_SERVER))) {
		aLog(D_INFO, "shutting down %s:%u service\n", s->getName(), s->instance);
		s->Shutdown();
		Delete(s);
		delete s;
        }

	//shutdown data-source, flush data to processor
	while((s=getServiceNextByType(SERVICE_DATASOURCE, NULL))) {
		aLog(D_INFO, "shutting down %s:%u service\n", s->getName(), s->instance);
		s->Shutdown();
		Delete(s);
		delete s;
	}
	
	//shutdown processor and flush data to storages
	if((s=Processor)) {
		aLog(D_INFO, "shutting down processor service\n");
		s->Shutdown();
		Delete(s);
		delete s;
	}

#ifdef HAVE_BILLING
	//shutdown billing and flush data to storages
	if((s=Billing)) {
		aLog(D_INFO, "shutting down billing service\n");
		s->Shutdown();
		Delete(s);
		delete s;
	}
#endif

	//shutdown quota and flush data to storages
        if((s=Quota)) {
                aLog(D_INFO, "shutting down quota service\n");
                s->Shutdown();
                Delete(s);
                delete s;
        }
	
	//shutdown login and flush data to storages
        if((s=Login)) {
                aLog(D_INFO, "shutting down login service\n");
                s->Shutdown();
                Delete(s);
                delete s;
        }

	//shutdown storages and flush data to disk
	s=NULL;
	while((s=getServiceNextByType(SERVICE_STORAGE, NULL))) {
		aLog(D_INFO, "shutting down %s:%u service\n", s->getName(), s->instance);
		s->Shutdown();
		Delete(s);
		delete s;
	}
	
	for(Service *tmp=root; tmp!=NULL;) {
		s=tmp;
		tmp=tmp->next;
		if (except!=s) {
			aLog(D_INFO, "shutting down %s:%u service\n", s->getName(), s->instance);
			s->Shutdown();
			if(s==root) root=root->next;
			else root->next=s->next;
			delete s;
		} 
	}
}

void ServicesList::listCfg(FILE *f, int nopasswords) {
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	
	Service *d;
	for (d=root; d!=NULL; d=d->next) d->listCfg(f, nopasswords); 
        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void ServicesList::listInfo(FILE *f) {
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
	for (d=root; d!=NULL; d=d->next) d->listInfo(f);

        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void ServicesList::showPerf(FILE *f, u_char isheader) {
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
	for (d=root; d!=NULL; d=d->next) d->showPerf(f, isheader);

        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}
//////////////////////////////////////////////////////////////////////////
void cService(Connection *conn, char *param[], u_char no_flag){
	Service *s;
	u_char instance;
	service_type type=SERVICE_UNDEF;
	char *name=NULL;
        
	if (!strcasecmp(param[1], "?")) { cHelp(conn, no_flag, "service"); return; }
        
	for(u_char i=1;i<SERVICE_TYPES_NUM;i++) {
		if(!strcasecmp(service_type_name[i],param[1])) {
			type=(service_type)i;
			name=service_type_name[i];
			break;
		}
	}

	if(type==SERVICE_UNDEF) {
		aParse(conn, "service %s undefined\n", param[1]);
		return;
	}
	
	instance = strtol(param[2], NULL, 10);
	
	//check for services that might be only in single instance
	if(type==SERVICE_PROCESSOR || type==SERVICE_SCHEDULER || type==SERVICE_QUOTA \
	|| type==SERVICE_LOGIN || type==SERVICE_BILLING || type==SERVICE_WEBLOGIN \
	|| type==SERVICE_HTML) {
		if(instance) aParse(conn,"Service might be only in one instance. Using %s:0 instead\n",name);
		instance=0;
	} 

	s=Services.getService(type, instance);
	if (s) {
		if(no_flag) {
			aParse(conn,"uninitialing service %s:%u\n", name, instance);
			Services.Delete(s);
			s->Shutdown();
			delete s;
			return;
		} else {
			conn->service=s;
			aParse(conn, "switching to service %s:%u for configuring\n", name, instance);
		}
	} else {
		if(no_flag) {
			aParse(conn, "service %s not exist\n", param[1]);
			return;
		}
		s=new Service(type, instance);
		s->Start();
		if(s->cfg) {
			aParse(conn,"creating service %s:%u\n", name, instance);
			aLog(D_INFO, "service %s:%u initialized\n", name, instance);
			Services.Insert(s);
			conn->service=s;
		} else  {
			aParse(conn, "service %s undefined\n", param[1]);
			delete s;
		}
	}
}
//////////////////////////////////////////////////////////////////////////
