/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2002 Anton Vinokurov <anton@netams.com>
***	(c) 2002-2005 NeTAMS Development Team
***	All rights reserved. See 'Copying' file included in distribution
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: st_hash.c,v 1.18 2005/01/17 13:13:22 jura Exp $ */

#include "netams.h"

#ifdef USE_HASH

#ifndef LINUX
#include <db.h>
#else
#include <db1/db.h>
#endif

DB* stOpenHash(Service *s, u_char type);
void stCloseHash(Service *s, DB *db);

int stSaveHashRaw(DB *db, char *key, char *value);
u_char stLoadHashRaw(DB *db, char *key, char *value);

//////////////////////////////////////////////////////////////////////////
DB* stOpenHash(Service *s, u_char type){
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg *)s->cfg;
	// first, we will determine the path to hash files
	// two ways - the directory where config file is, or what is specified in 'path' variable
	
	int fd;
	struct stat sb;
	DB *db;
	char *hash_file;

	switch(type) {
		case ST_CONN_RAW:
			hash_file=cfg->raw;
			break;
		case ST_CONN_SUMMARY:
			hash_file=cfg->summary;
			break;
		default:
			return NULL;
	}	

	fd=open(hash_file, O_RDWR, S_IRWXU);
	if (fd!=-1) { 
		fstat(fd, &sb); 
		close(fd); 
		if (!sb.st_size) { 
			unlink(hash_file); 
			aLog(D_INFO, "stOpenHash(%s)%d: old archive truncated\n", hash_file, s->instance);
		} 
	}

	db=dbopen(hash_file, O_CREAT|O_RDWR, S_IRWXU, DB_HASH, NULL);
	if (db==NULL) 
		aLog(D_WARN, "stOpenHash:%u(%s) %s , continue closed\n", s->instance, hash_file, strerror(errno));
	else 
		aDebug(DEBUG_STORAGE, "hash-%s:%u database opened at %s\n", st_table_name[type], s->instance, hash_file); 

	return db;
}	
//////////////////////////////////////////////////////////////////////////
void stCloseHash(Service *s, DB *db){
	if(db->close(db)<0)
		aLog(D_WARN, "HASH:%u can't close: %s\n",s->instance, strerror(errno));
	else
		aDebug(DEBUG_STORAGE, "HASH:%u closed\n", s->instance); 
}
//////////////////////////////////////////////////////////////////////////
void stSaveHash(Service *s){
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg *)s->cfg;
	char key[80], value[80];
	Message *msg;
	Message_Store *smsg;
	DB *raw=NULL;
	DB *summary=NULL;

	while((msg=cfg->in->TryPop())) {
		if (msg->type==MSG_STORE) {
			smsg=(Message_Store*)msg;
			aDebug(DEBUG_STORAGE, "ST->HASH wr st:%u unit:%06X acct:%06X from:%u to:%u\n", s->instance, smsg->netunit, smsg->ap, smsg->data->from, smsg->ts);
		
			switch (smsg->prefix){
			case 'F':
				if(!raw) {
					raw=stOpenHash(s,ST_CONN_RAW);
					if(!raw) {
						aLog(D_WARN, "Can't open HASH:%u for raw: %s\n",s->instance,strerror(errno));
						return;
					}
				}
				snprintf(key, 80, "%06X-%06X-%u-%u ", smsg->netunit, smsg->ap, smsg->data->from, smsg->ts);
				snprintf(value, 80, "%llu %llu ", smsg->data->in, smsg->data->out);
				stSaveHashRaw(raw, key, value);
				break;
			case 'M':
			case 'W':
			case 'D':
			case 'H':
				if(!summary) {
					summary=stOpenHash(s,ST_CONN_SUMMARY);
					if(!summary) {
						aLog(D_WARN, "Can't open HASH:%u for summary: %s\n",s->instance,strerror(errno));	
						return;
					}
				}
				snprintf(key, 80, "%c-%06X-%06X-%u ", smsg->prefix, smsg->netunit, smsg->ap, smsg->data->from);
				snprintf(value, 80, "%llu %llu ", smsg->data->in, smsg->data->out);
				stSaveHashRaw(summary, key, value);
				break;
			}
		} else {
			aDebug(DEBUG_STORAGE, "message type %d not STORE - discarded\n",msg->type);
		}
		msg=cfg->in->Pop();
		if(!msg->Active()) MsgMgr.Delete(msg);
	}
	
	if(raw) stCloseHash(s,raw);
	if(summary) stCloseHash(s,summary); 
}				 
//////////////////////////////////////////////////////////////////////////
int stSaveHashRaw(DB *db, char *key, char *value){
	DBT k, v;
	int i;

	k.size=strlen(key); 
	k.data=key; 
	v.size=strlen(value); 
	v.data=value;

	i=db->put(db, &k, &v, 0);

	aDebug(DEBUG_STORAGE, "HASH->HDD '%s'.'%s'%s\n", key, value, !i?"":strerror(errno));

	return i;
}
//////////////////////////////////////////////////////////////////////////
u_char stLoadHash(Service *s){
	ServiceStorage_cfg *cfg=(ServiceStorage_cfg *)s->cfg;
	char key[80], value[80];
	DB *db;
	Message *msg;
	Message_Read *rmsg;
	pstat *cps=NULL;
	u_char res=0;
	NetUnit *u;
	
	if(!cfg->out->TryPop()) return 1;
	
	if(!(db=stOpenHash(s,ST_CONN_SUMMARY))) {
		return 0;
	}

	while((msg=cfg->out->TryPop())) {
		if (msg->type==MSG_READ) {
			rmsg=(Message_Read*)msg;
			if(!(u=Units.getUnitById(rmsg->netunit))) goto CLEAN;
			aDebug(DEBUG_STORAGE, "ST<-HASH rd st:%u unit:%06X acct:%06X\n", s->instance, rmsg->netunit, rmsg->ap);

			if(!u->ap) goto CLEAN; //somehow unit do not have account data
			
			//protect counters
			pthread_rwlock_wrlock(u->ap->rwlock);

			//actually there is no "F"low read requests
			switch (rmsg->prefix){
				case 'M': cps=&(rmsg->pdata->m); break;
				case 'W': cps=&(rmsg->pdata->w); break;
				case 'D': cps=&(rmsg->pdata->d); break;
				case 'H': cps=&(rmsg->pdata->h); break;
			}

			snprintf(key, 80, "%c-%06X-%06X-%u ", rmsg->prefix, rmsg->netunit, rmsg->ap, cps->from);
			if(stLoadHashRaw(db, key, value)) {
				unsigned long long in;
				unsigned long long out;
				sscanf(value, "%llu %llu", &in, &out);
		
				//update counters
				cps->in+=in;
				cps->out+=out;
				res=1;
			} else {
				aLog(D_WARN, "Can't read data from HASH:%u\n", s->instance);
				res=0;
			}
			pthread_rwlock_unlock(u->ap->rwlock);
			if(!res) return 0;
		} else {
			aDebug(DEBUG_STORAGE, "message type %d not READ - discarded\n",msg->type);
		}
CLEAN:
		msg=cfg->out->Pop();
		if(!msg->Active()) MsgMgr.Delete(msg); 
	}
	return 1;
}
//////////////////////////////////////////////////////////////////////////
u_char stLoadHashRaw(DB *db, char *key, char *value){
	DBT k, v;
	int i; 

	k.size=strlen(key); 
	k.data=key; 
	v.size=strlen(value); 
	i=db->get(db, &k, &v, 0);
	if (i==0) { 
		memcpy(value, v.data, v.size);
		aDebug(DEBUG_STORAGE, "HASH<-HDD '%s'.'%s'\n", key, value);
	}
	else if (i==1) { 
		strcpy(value, "0 0 0 ");
		aDebug(DEBUG_STORAGE, "HASH<-HDD '%s'.notfound\n", key);
	}
	else {
		aDebug(DEBUG_STORAGE, "HASH<-HDD %s\n", strerror(errno));
		return 0;
	}
	return 1;
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
#endif
