/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "callsign.h"

extern int errno;

/* callsign entry */
static struct callsign_ent {
	int slen;	/* sign length */
	char *sign;	/* short name */
	int nlen;	/* name length */
	char *name;	/* full name */
} *call_sign = NULL;

static char *signfpath = NULL;
static time_t signmtime = 0;

static void
freesign()
{
	register i;

	if (call_sign == NULL) return;
	for (i = 0; call_sign[i].sign != NULL; i++) {
		free(call_sign[i].sign);
		free(call_sign[i].name);
	}
	free(call_sign);
	call_sign = NULL;
}

static void
refreshsign()
{
	struct stat st;

	if (signfpath != NULL) {
		if (stat(signfpath, &st) == 0 && st.st_mtime > signmtime)
			initcallsign(NULL);
	}
}

int
initcallsign(orgdir)
	char *orgdir;
{
	register i;
	register char *ptr;
	char *line, *sign;
	struct stat st;

	if (orgdir != NULL) {
		char fpath[1024];

		if (signfpath != NULL) free(signfpath);
		sprintf(fpath, "%s/%s", orgdir, CALLSIGNLIST);
		signfpath = strdup(fpath);
	}
	closesignent();
	if (signfpath == NULL) return 0;
	if (stat(signfpath, &st) < 0 || !st.st_size) return 0;
	if ((ptr = (char *)loadfile(signfpath)) == NULL) return 0;
	signmtime = st.st_mtime + 1;
	freesign(call_sign);

	for (i = 0; (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		sign = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; *ptr && (u_char)*ptr <= 0x20; ptr++);
		if (!*ptr) continue;
		call_sign = realloc(call_sign, (i + 1) * sizeof(struct callsign_ent));
		if (call_sign == NULL) break;
		call_sign[i].slen = strlen(sign);
		call_sign[i].nlen = strlen(ptr);
		if ((call_sign[i].sign = strdup(sign)) == NULL ||
		    (call_sign[i].name = strdup(ptr)) == NULL) return i;
		i++;
	}
	if (i) {
		call_sign = realloc(call_sign, (i + 1) * sizeof(struct callsign_ent));
		if (call_sign == NULL) i--;
		call_sign[i].slen = call_sign[i].nlen = 0;
		call_sign[i].sign = call_sign[i].name = NULL;
	}
	return i;
}

char *
getnamebysign(sign)
	register char *sign;
{
	register i, len;

	refreshsign();
	if (call_sign == NULL || sign == NULL || *sign == '\0') return NULL;
	len = strlen(sign);
	for (i = 0; call_sign[i].sign != NULL; i++)
		if (call_sign[i].slen == len &&
		    !strcasecmp(call_sign[i].sign, sign)) break;
	return call_sign[i].name;
}

char *
getsignbyname(name)
	register char *name;
{
	register i, len;

	refreshsign();
	if (call_sign == NULL || name == NULL || *name == '\0') return NULL;
	len = strlen(name);
	for (i = 0; call_sign[i].name != NULL; i++)
		if (call_sign[i].nlen == len && call_sign[i].slen > 0 &&
		    !strcasecmp(call_sign[i].name, name)) break;
	return call_sign[i].sign;
}

static int
appendsignent(sign, name)
	char *sign, *name;
{
	int fd;
	char buf[80];

	if (*sign == '\0') return 0;
	if (signfpath == NULL) return -1;
	if ((fd = open(signfpath, O_WRONLY|O_APPEND, 0644)) < 0)
		return -1;
	snprintf(buf, sizeof(buf), "%-.16s\t\t%s\n", sign, name);
	while (write(fd, buf, strlen(buf)) < 0 && errno == EINTR);
	close(fd);
	return 0;
}

static int
rewritesignent(idx, sign)
	int idx;
	char *sign;
{
	int i, fd;
	char *ptr, buf[80], tmp[1024];

	if (signfpath == NULL) return -1;
	sprintf(buf, ".callsign.%d", (int)getpid());
	strcpy(tmp, signfpath);
	if ((ptr = strrchr(tmp, '/')) != NULL) *++ptr = '\0';
	strcat(tmp, buf);
	if ((fd = open(tmp, O_WRONLY|O_CREAT|O_TRUNC|O_APPEND, 0644)) < 0)
		return -1;
	free(call_sign[idx].sign);
	call_sign[idx].sign = strdup(sign);
	call_sign[idx].slen = strlen(sign);
	for (i = 0; call_sign[i].sign != NULL &&
	     call_sign[i].name != NULL; i++) if (call_sign[i].slen > 0) {
		snprintf(buf, sizeof(buf), "%-.16s\t\t%s\n",
			 call_sign[i].sign, call_sign[i].name);
		while (write(fd, buf, strlen(buf)) < 0 && errno == EINTR);
	}
	close(fd);
	if ((fd = unlink(signfpath)) == 0) link(tmp, signfpath);
	unlink(tmp);
	return 0;
}

int
addsignent(sign, name)
	char *sign, *name;
{
	register i, len;

	refreshsign();
	if (call_sign == NULL) return appendsignent(sign, name);
	len = strlen(name);
	for (i = 0; call_sign[i].name != NULL; i++)
		if (call_sign[i].nlen == len &&
		    !strcasecmp(call_sign[i].name, name)) break;
	if (call_sign[i].sign == NULL) return appendsignent(sign, name);
	if (strcmp(call_sign[i].sign, sign))
		return rewritesignent(i, sign);
	return 0;
}

static int signentry = 0;

char *
getsignent()
{
	static char buf[80];

	if (signentry == 0) refreshsign();
	if (call_sign == NULL) return NULL;

	if (call_sign[signentry].sign != NULL &&
	    call_sign[signentry].name != NULL &&
	    call_sign[signentry].slen > 0) {
		snprintf(buf, sizeof(buf), "%-16.16s %s",
			 call_sign[signentry].sign, call_sign[signentry].name);
		signentry++;
		return buf;
	}
	signentry = 0;
	return NULL;
}

void
closesignent()
{
	signentry = 0;
}
