/*
 *	Copyright (c) 1994-1997 The CAD lab of the
 *	Siberian State Academy of Telecommunications
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include "compat.h"
#include "drive.h"
#include "getconf.h"
#include "variables.h"

#define	BBSMAIL_ID	"bbsmail"
#define	BBSMAIL_VER	"v1.3"
#define	BUFLEN		256
#define	NAMELEN		50

extern int errno;

FILE *fdin;
char *progname, *bbsdir, *bbsgecos, *to, *subj, *folder;
char tmpbuf[1024], tempfile[1024], from[BUFLEN], host[BUFLEN];
uid_t bbsuid;
long body;

long mail_from(void);
char *mail_forward(char *);
void enter_mail(char *, char *, char *);
void forward_mail(char *, char *, char *);
void usage(void);

main(argc, argv)
	int argc;
	char **argv;
{
	int i, op, mbox, restrict;
	struct passwd *pwd;
	char *p, *userdir, username[NAMELEN], addr[BUFLEN], name[BUFLEN];
	extern int optind, opterr;
	extern char *optarg;

	progname = strippath(argv[0]);

	if ((pwd = getpwnam(BBS)) == NULL &&
	    (pwd = getpwnam("tns")) == NULL) {
		fprintf(stderr, "%s: no BBS in this system\n", progname);
		exit(1);
	}
	bbsuid = pwd->pw_uid;
	bbsdir = strdup(pwd->pw_dir);
	if ((p = strchr(pwd->pw_gecos, ',')) != NULL) *p = '\0';
	bbsgecos = strdup(pwd->pw_gecos);
	gethostname(host, sizeof(host));

	restrict = FALSE;
	mbox = TRUE;
	fdin = NULL;
	subj = NULL;
	folder = NULL;
	opterr = 0;
	while ((op = getopt(argc, argv, "i:f:os:r?")) != EOF)
		switch (op) {
			case 'i':	/* input mail from file */
				if ((fdin = fopen(optarg, "r")) == NULL) {
					fprintf(stderr, "%s: can't open \"%s\": %s\n",
						progname, optarg, strerror(errno));
					exit(1);
				}
				break;
			case 'f':	/* X-Folder: name */
				folder = strdup(optarg);
				break;
			case 'o':	/* output to stdout */
				mbox = FALSE;
				break;
			case 's':	/* subject */
				subj = strdup(optarg);
				break;
			case 'r':	/* restricted, from users bbs only */
				restrict = TRUE;
				break;
			case '?':
			default:
				usage();
		}
	if (argv[optind] == NULL) usage();

	if (mbox && getuid() == geteuid()) { /* try to set euid to bbs */
		if (seteuid(bbsuid) < 0) {
			fprintf(stderr, "%s: Permission denied\n", progname);
			exit(1);
		}
	}
	initcallsign(bbsdir);
	initgetconf(bbsdir);

	if (fdin == NULL) {	/* message from stdin -- make temp copy */
		mktemp(strcpy(tempfile, TMPFILEMASK));
		if ((fdin = fopen(tempfile, "w+")) == NULL) {
			fprintf(stderr, "%s: can't open \"%s\": %s\n",
				progname, tempfile, strerror(errno));
			exit(1);
		}
		while (fgets(tmpbuf, sizeof(tmpbuf), stdin) != NULL) {
			tmpbuf[sizeof(tmpbuf)-1] = '\0';
			if (fputs(tmpbuf, fdin) == EOF) {
				fprintf(stderr, "%s: can't write \"%s\": %s\n",
					progname, tempfile, strerror(errno));
				unlink(tempfile);
				exit(1);
			}
		}
	} else tempfile[0] = '\0';

	body = mail_from();
	parsefrom(from, addr, name);

	if (name[0] == '\0') {
		for (i = 0; addr[i] != '\0' && addr[i] != '@'; i++) {
			if (addr[i] == '.') name[i] = ' ';
			else name[i] = addr[i];
		}
		if (!i)	{
			fprintf(stderr, "%s: bad sender address \"%s\"\n",
				progname, addr);
			if (*tempfile) unlink(tempfile);
			exit(1);
		}
		name[i] = '\0';
	}
	if (restrict && getuserdir(name) == NULL) {
		fprintf(stderr, "%s: mail from unknown BBS user \"%s\" <%s> rejected\n",
			progname, name, addr);
		if (*tempfile) unlink(tempfile);
		exit(1);
	}
	op = optind;
	while ((p = argv[optind]) != NULL) {
		for (i = 0; *p && *p != '@' && i < NAMELEN-1; i++, p++) {
			if (*p == '.')	username[i] = ' ';
			else		username[i] = *p;
		}
		if (!i) {
			fprintf(stderr, "%s: bad BBS username format \"%s\"\n",
				progname, argv[optind++]);
			continue;
		}
		username[i] = '\0';
		if ((userdir = getuserdir(username)) == NULL) {
			fprintf(stderr, "%s: mail for unknown BBS user \"%s\" <%s> rejected\n",
				progname, username, argv[optind++]);
			continue;
		}
		to = argv[optind];
		if (mbox) {
			if ((p = mail_forward(userdir)) == NULL) {
				(void)strcat(userdir, "/");
				(void)strcat(userdir, MAILBOX);
				enter_mail(addr, name, userdir);
			} else forward_mail(addr, name, p);
		} else enter_mail(addr, name, NULL);

		optind++;
		op++;
	}
	if (*tempfile) unlink(tempfile);
	exit(op != optind);
}

char *
mail_forward(userhome)
	char *userhome;
{
	FILE *fp;
	char *p;
	static char buf[1024];

	sprintf(buf, "%s/%s", userhome, USERFORWARDFILE);
	if ((fp = fopen(buf, "r")) == NULL) return NULL;
	buf[0] = '\0';
	fgets(buf, sizeof(buf), fp);
	fclose(fp);
	buf[sizeof(buf)-1] = '\0';
	if ((p = strchr(buf, '\n')) != NULL) *p = '\0';
	return *buf ? buf : NULL;
}

void
strcopy(s, t)
	char *s, *t;
{
	int i;
	for (i = 0; *s; s++) {
		if (!i && *s == ' ') continue;
		if (i++ < BUFLEN) *t++ = *s;
		else break;
	}
	*t = '\0';
}

long
mail_from()
{
	char *p;
	int hline;

	hline = 0;
	from[0] = '\0';

	rewind(fdin);
	while (fgets(tmpbuf, sizeof(tmpbuf), fdin) != NULL) {
		tmpbuf[sizeof(tmpbuf)-1] = '\0';
		for (p = tmpbuf; *p && *p != '\n'; p++)
			if (((*p)&0x7F) < 32) *p = ' ';
		if (*p != '\n') continue;	/* too long line */
		*p = '\0';
		if (*tmpbuf == '\0') break;
		if (!strncmp(tmpbuf, "From ", 5)) {
			if (hline == 0) hline = 1;
			else hline = -1;
		} else if (hline == 1 && !strncmp(tmpbuf, "From: ", 6))
			strcopy(&tmpbuf[6], from);
	}
	if (hline != 1) from[0] = '\0';
	if (from[0] == '\0') {
		struct passwd *pwd;
		if ((pwd = getpwuid(getuid())) == NULL) {
			fprintf(stderr, "%s: can't getpwuid(%d): %s\n",
				progname, getuid(), strerror(errno));
			if (*tempfile) unlink(tempfile);
			exit(1);
		}
		if ((p = strchr(pwd->pw_gecos, ',')) != NULL) *p = '\0';
		p = pwd->pw_gecos;
		if (*p != '\0') 
			sprintf(from, "\"%s\" <%s@%s>", p, pwd->pw_name, host);
		else	sprintf(from, "%s@%s", pwd->pw_name, host);
	}
	return (hline == 1 ? ftell(fdin) : (long)0);
}

void
fprint_rcv(fp)
	FILE *fp;
{
	time_t tm;
	char buf[50], bbsname[BUFLEN];
	struct passwd *pwd;

	if ((pwd = getpwuid(geteuid())) == NULL) return;
	tm = time(NULL);
	strcpy(bbsname, getsysconf(SYSTEMNAME, bbsgecos));
	bbsname[20] = '\0';
	strftime(buf, sizeof(buf), "%a, %e %b %Y %H:%M:%S", localtime(&tm));
	fprintf(fp, "Received: (%s@%s) by %s (%s) with %s %s; %s %s\n",
		pwd->pw_name, host, getsysconf(MYHOSTNAME, host),
		bbsname, BBSMAIL_ID, BBSMAIL_VER,
		buf, gettzone());
}

void
enter_mail(addr, name, mboxfile)
	char *addr, *name, *mboxfile;
{
	FILE *fp;
	char *p, *xf;
	int emptyline, header, lines;

	if (mboxfile != NULL) {
		xf = "E-mail";
		if ((fp = fopen(mboxfile, "a")) == NULL) {
			fprintf(stderr, "%s: can't append to \"%s\": %s\n",
				progname, mboxfile, strerror(errno));
			if (*tempfile) unlink(tempfile);
			exit(1);
		}
		flock(fileno(fp), LOCK_EX);
	} else {
		xf = "stdout";
		fp = stdout;
	}
	if (folder != NULL) xf = folder;

	if (!body) {	/* file, not mail? */
		time_t tm = time(NULL);
		if ((p = getlogin()) == NULL) p = BBSMAIL_ID;
		fprintf(fp, "From %s@%s %s", p, host, ctime(&tm));
		fprint_rcv(fp);
		fprint_msgid(fp, host);
		fprintf(fp, "From: %s\n", from);
		fprintf(fp, "To: %s\n", to);
		if (subj) fprintf(fp, "Subject: %s\n", subj);
		fprintf(fp, "X-Folder: %s\n", xf);
		fprintf(fp, "X-Mailer: %s %s\n\n", BBSMAIL_ID, BBSMAIL_VER);
		emptyline = 1;
		header = 0;
	} else {
		emptyline = 0;
		header = 1;
	}
	lines = 0;
	rewind(fdin);
	while (fgets(tmpbuf, sizeof(tmpbuf), fdin) != NULL) {
		tmpbuf[sizeof(tmpbuf)-1] = '\0';
		for (p = tmpbuf; *p && *p != '\n'; p++)
			if (((*p)&0x7F) < 32) *p = ' ';
		if (*p != '\n') continue;	/* too long line */
		if (header) {
			if (lines == 1) {
				fprint_rcv(fp);
				lines++;
			}
			if (subj && !strncmp(tmpbuf, "Subject: ", 9)) {
				fprintf(fp, "Subject: %s\n", subj);
				lines++;
				continue;
			}
		}
		if (emptyline && !strncmp(tmpbuf, "From ", 5))
			fputc('>', fp);
		emptyline = (*tmpbuf == '\n');
		if (header && emptyline) {
			header = 0;
			fprintf(fp, "X-Folder: %s\n", xf);
			lines++;
		}
		fputs(tmpbuf, fp);
		lines++;
	}
	if (*tmpbuf != '\n') fputc('\n', fp);

	if (mboxfile != NULL) {
		flock(fileno(fp), LOCK_UN);
		fclose(fp);
	}
}

void
forward_mail(addr, name, to)
	char *addr, *name, *to;
{
	char *mailer = getsysconf(MAILER, DEFMAILER);
	sprintf(tmpbuf, "USER=%s", name);
	putenv(tmpbuf);
	sprintf(tmpbuf, "EMAIL=%s", addr);
	putenv(tmpbuf);
	tmpbuf[0] = '\0';
	if (*mailer != '/') {
		strcpy(tmpbuf, bbsdir);
		strcat(tmpbuf, "/");
	}
	strcat(tmpbuf, mailer);
	strcat(tmpbuf, " ");
	strcat(tmpbuf, to);
	strcat(tmpbuf, " ");
	strcat(tmpbuf, tempfile);
	if (system(tmpbuf)) {
		fprintf(stderr, "%s: \"%s\": system() failed\n", progname, tmpbuf);
		if (*tempfile) unlink(tempfile);
		exit(1);
	}
}

void
usage()
{
	fprintf(stderr, "%s %s\n", BBSMAIL_ID, BBSMAIL_VER);
	fprintf(stderr, "\
usage: %s [-r] [-o] [-i file] [-f name] [-s subj] to-addr...\n", progname);
	fprintf(stderr, "\
where:\n\
  -r       Restricted mode, allow messages only from users of this bbs\n\
  -o       Output message to stdout; default mailbox of bbs user\n\
  -i file  Input message from file; default stdin\n\
  -f name  Use folder name for message header; default E-mail or stdout\n\
  -s subj  Subject of this message\n\
  to-addr  Addresses of bbs users\n");
	fprintf(stderr, "\
NOTE: This program intended to use by root or be sure it have RIGHT set-uid!\n");
	exit(1);
}
