/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <termios.h>
#include <stdio.h>
#include <string.h>
#ifdef	__STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <errno.h>
#include <unistd.h>
#include <ctype.h>

#include "drive.h"
#include "loginfo.h"

#ifndef	EXTERN_APPS
#include <signal.h>
#include "sysmsg.h"
#endif

#ifdef BUFSIZ
#undef BUFSIZ
#endif
#define BUFSIZ		256
#define EMPTY		'\0'

extern int errno;
extern struct loginfo process;

static int nb = 0;
static u_char cbuf = EMPTY;
static struct termios tbufsave;
static u_char strbuf[BUFSIZ];
u_char *strptr = strbuf;
int nchr = 0;

#ifndef	EXTERN_APPS
extern int termflags;
extern int acttimer;
extern unsigned char table[256][2];
#else
static int termflags = 0;
#endif

#ifndef	EXTERN_APPS
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>

static	int monfd = -1;			/* output monitor file */

static	int sockin = -1;		/* input monitor socket */
static	int sockout = -1;		/* output monitor socket */
static	struct sockaddr_un namein;	/* input socket-file name */
static  struct sockaddr_un nameout;	/* output socket-file name */

int
monfile(buf, len)
	register void *buf;
	register len;
{
	register rval;
	if (monfd < 0) return 0;
	rval = write(monfd, buf, len);
	if (rval < 0) {
		close(monfd);
		monfd = -1;
	}
	return rval;
}

static	int (*monitor)() = monfile;

int
monsock(buf, len)
	register void *buf;
	register len;
{
	register rval;

	if (sockout < 0) {
		monitor = monfile;
		return 0;
	}
	rval = sendto(sockout, buf, len, 0, (struct sockaddr *)&nameout,
		      sizeof(struct sockaddr_un));
	if (rval < 0 && errno != ENOBUFS) {
#ifdef	DEBUG
		perror("sendto");
#endif
		close(sockout);
		monitor = monfile;
		close(sockin);
		sockin = -1;
		unlink(namein.sun_path);
	}
	return rval;
}

void
onusr1()
{
	register sock;

	sig_set(SIGUSR1, onusr1);
	if ((sock = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0) return;
	nameout.sun_family = AF_UNIX;
	sprintf(nameout.sun_path, "%s/%s.IN", MONITORDIR, fntty);
	sockout = sock;
	monitor = monsock;
	if ((sock = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0) return;
	namein.sun_family = AF_UNIX;
	sprintf(namein.sun_path, "%s/%s.OUT", MONITORDIR, fntty);
	unlink(namein.sun_path);
	if (bind(sock, (struct sockaddr *)&namein, sizeof(struct sockaddr_un)) == 0)
		sockin = sock;
}

int syschatinactive = 0;

static void
endchat()
{
	syschatinactive = 0;
}

void
onusr2()
{
	register ch;
	sigfunc intsave;
	extern void sigcatch();

	syschatinactive = 1;
	intsave = signal(SIGINT, SIG_IGN);
	sig_set(SIGUSR1, endchat);
	signal(SIGUSR2, SIG_IGN);
	tcflush(0, TCIOFLUSH);
	nchr = 0;
	putstr("\007\n\n*** Begin CHAT with SysAdm ***\n\n");

	while (syschatinactive) {
		ch = getchr(0);
		switch (ch) {
			case '\b':
			case 0x7f:
				putchr('\b');
				putchr(' ');
				putchr('\b');
				break;
			case '\r':
			case '\n':
				putstr("\r\n");
				break;
			default:
				putchr(ch);
		}
	}

	putstr("\007\n*** End CHAT ***\n\n");
	tcflush(0, TCIFLUSH);
	signal(SIGINT, intsave);
	sig_set(SIGUSR1, onusr1);
	sig_set(SIGUSR2, onusr2);
}

int
openmonitor()
{
	char *ptr;

	if ((ptr = getsysconf(MONITOR, NULL)) != NULL) {
		char buf[256], *ptr2;
		if ((ptr2 = strchr(ptr, '%')) != NULL && ptr2[1] == 's') {
			snprintf(buf, sizeof(buf), ptr, fntty);
			ptr = buf;
		}
		monfd = open(ptr, O_WRONLY|O_CREAT|O_TRUNC, 0640);
	}
	return 0;
}

#endif	/* EXTERN_APPS */

int
ttysave()
{
	int speed;
	static int speed_tab[] = {
	0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800, 9600,
	19200, 38400, 57600, 76800, 115200, 153600, 230400, 307200, 460800 };

#ifndef	EXTERN_APPS
	deftable();
#endif
	tcgetattr(0, &tbufsave);
	speed = (int)cfgetospeed(&tbufsave);
	if (speed > 0 && speed < 23) speed = speed_tab[speed];
	return speed;
}

int
ttysetraw()
{
	struct termios tbuf;

	tbuf = tbufsave;
	cfmakeraw(&tbuf);
	tbuf.c_iflag = (IXANY | IGNPAR | IXON | ICRNL);
	tbuf.c_lflag = ISIG;
	tbuf.c_oflag = (OPOST | ONLCR);
	tbuf.c_cc[VINTR] = 0x03;    /* ^C - interrupt */
	tbuf.c_cc[VQUIT] = 0x04;    /* ^D - disconnect */
	tbuf.c_cc[VMIN] = 1;
	tbuf.c_cc[VTIME] = 0;
	return tcsetattr(0, TCSAFLUSH, &tbuf);
}

ttyrestore(flush)
	int flush;
{
	if (!flush) {
		/*
		 * Discard any data writen but not transmited.
		 */
		tcflush(0, TCIOFLUSH);
		cfsetospeed(&tbufsave, 0);	/* terminate connection */
		tcsetattr(0, TCSANOW, &tbufsave);
	} else	tcsetattr(0, TCSAFLUSH, &tbufsave);
}

int
putchr(ch)
	u_char ch;
{
	if ((termflags & RIPTERM) || (process.flags & (INCONF | INTALK)))
		nchr = 0;
	else {
		if (nchr || inqueue(0)) return ch;
#ifndef	EXTERN_APPS
		if (sockin > 0 && inqueue(sockin)) return ch;
	}
	(*monitor)(&ch, 1);
	ch = table[ch & 0377][1];
#else
	}
#endif
	write(1, &ch, 1);
	return ch;
}

int
#ifdef	__STDC__
putstr(const char *fmt, ...)
#else
putstr(fmt, va_alist)
	const char *fmt;
	va_dcl
#endif
{
	register i, len;
	unsigned char sb[2048];
	va_list ap;

#ifdef	__STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	len = vsnprintf(sb, sizeof(sb), fmt, ap);
	va_end(ap);
	if ((termflags & RIPTERM) || (process.flags & (INCONF | INTALK)))
		nchr = 0;
	else {
		if (nchr || inqueue(0)) return len;
#ifndef	EXTERN_APPS
		if (sockin > 0 && inqueue(sockin)) return len;
	}
	(*monitor)(sb, len);
	for (i = 0; i < len; i++) sb[i] = table[sb[i] & 0377][1];
#else
	}
#endif
	return write(1, sb, len);
}

int
#ifdef	__STDC__
putrawstr(char *fmt, ...)
#else
putrawstr(fmt, va_alist)
	char *fmt;
	va_dcl
#endif
{
	register len;
	char sb[2048];
	va_list ap;

#ifdef	__STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	len = vsnprintf(sb, sizeof(sb), fmt, ap);
	va_end(ap);
	if ((termflags & RIPTERM) || (process.flags & (INCONF | INTALK)))
		nchr = 0;
	else {
		if (nchr || inqueue(0)) return len;
#ifndef	EXTERN_APPS
		if (sockin > 0 && inqueue(sockin)) return len;
	}
	(*monitor)(sb, len);
#else
	}
#endif
	return write(1, sb, len);
}

char *
putstrmax(ptr, max)
	char *ptr;
	int max;
{
	register i;
	register char *ptr2;

	if ((i = strlen(ptr)) > max) {
		for (ptr2 = &ptr[max-1]; *ptr2 != '/' && *ptr2 != '-' &&
		     *ptr2 != '.' && *ptr2 != ',' &&
		     (unsigned char)*ptr2 > 0x20 && ptr2 != ptr; ptr2--);
		if (ptr2 == ptr) ptr2 = &ptr[max-1];
		i = ++ptr2 - ptr;
	} else	ptr2 = NULL;
	putstr("%.*s\n", i, ptr);
	return ptr2;
}

int
bufread()
{
	static u_char stdbuf[BUFSIZ];
	static u_char *stdptr = stdbuf;

	if (!nb) {
		register n = 0;
		/* tcflush(0, TCIFLUSH); */		/* ??? */
#ifndef	EXTERN_APPS
		if (sockin > 0) {
			fd_set ready;
			do {
				FD_ZERO(&ready);
				FD_SET(0, &ready);
				FD_SET(sockin, &ready);
			} while ((n = select(sockin + 1, &ready, NULL, NULL, NULL)) < 0 &&
				 errno == EINTR);
			if (n > 0) {
				n = 0;
				if (FD_ISSET(0, &ready))
					n = read(0, stdbuf, BUFSIZ);
				if (FD_ISSET(sockin, &ready))
					n += read(sockin, &stdbuf[n], BUFSIZ - n);
			}
		}
#endif
		if (n < 1) while ((n = read(0, stdbuf, BUFSIZ)) == 0);
		if (n < 0) return n;
		nb = n;
		stdptr = stdbuf;
	}
	--nb;
	return *stdptr++;
}

int
cget()
{
	int c = (int)cbuf;

	if (c != EMPTY) cbuf = (u_char)EMPTY;
	else {
		while ((c = bufread()) < 0)
			if (errno != EINTR) {
				perror("read");
				c = '\n';
				break;
			}
	}
#ifndef	EXTERN_APPS
	c = table[c & 0377][0];
#endif
	return c & 0377;
}

int
getchr(flag)		/*	get one byte from the port		*/
	int flag;	/* if flag = 0 then any char acceptable		*/
			/* if flag = 1 then only char < 0x80 acceptable */
{
	register ch;

	globalflags |= INACTIVITY;
	while ((ch = cget()) & ((flag != 0) * 0x80));
	if (ch == '\033') {
		ch = cget();
		if (ch == '\033') ch = 'Q';
		else if (ch == '[') {
			ch = cget();
			switch (ch) {
				case 'A': ch = 'U';
					break;
				case 'B': ch = 'D';
					break;
				case 'C': ch = '>';
					break;
				case 'D': ch = '<';
					break;
			}
		}
	}
	globalflags &= ~INACTIVITY;
#ifndef	EXTERN_APPS
	acttimer = INACTTIMER * 60;
#endif
	return ch;
}

eraseline(i, echo)
	int i, echo;
{
	if (i) while (i) {
		i--;
		if (echo) putstr("\b \b");
	}
	return i;
}

u_char *
bufstr(nc, max, flag, echo)
	int nc, max, flag;
	register echo;
{
	register i;
	register u_char ch;
	int touch;

	if (!max) max = sizeof(strbuf)-1;
	else nchr = 0;
	if (nchr) return strptr;
	i = nc;
	strbuf[i] = '\0';
	if (flag < 0) {
		touch = TRUE;
		flag = 0;
	} else	touch = FALSE;
	while (i < max) {
		while((ch = getchr(flag)) < ' ' || ch == 127) {
			if (ch == '\r' || ch == '\n') goto entered;
			touch = TRUE;
			if (ch == '\b' || ch == 127) { /* Backspace */
				if (i) {
					i--;
					strbuf[i] = '\0';
					if (echo) putstr("\b \b");
				} else putchr('\007');
			} else if (ch == 21) {	/* Control-U, erase line */
				if (i) {
					i = eraseline(i, echo);
					strbuf[i] = '\0';
				} else putchr('\007');
			}
/*	Illegal nonprintable chars. Very noise, commented it.	*/
/*			else putchr('\007');	*/
		}
		if (ch == '%' || (flag == 2 && !(ch >= '0' && ch <= '9')))
			putchr('\007');
		else {
			if (!touch && flag != 2) {
				i = eraseline(i, echo);
				strbuf[i] = '\0';
				touch = TRUE;
			}
			strbuf[i++] = ch;
			strbuf[i] = '\0';
			if (echo) putchr(ch);
		}
	}
	tcflush(0, TCIFLUSH);
	nb = 0;
entered:
	strbuf[i] = '\0';
	nchr = i;
	strptr = strbuf;
	return strptr;
}

int
getkey(echo)
	register echo;
{
	register ch;
	while ((ch = *bufstr(0, 0, 0, echo)) == ' ') --nchr, ++strptr;
	if (ch) --nchr, ++strptr;
	else ch = '\r';
	return ch;
}

int
getask(echo)
	int echo;
{
	register ch;

	if (termflags & RIPTERM) nchr = 0;
	if (nchr) {
		--nchr;
		ch = *strptr++;
		return ch;
	}
	ch = getchr(1);
	if (echo && (unsigned char)ch > 0x20 && ch != 127) putchr(ch);
	return ch;
}

char *
getstr(max, flag, echo)
	register max, flag, echo;
{
	register char *ptr;

	ptr = (char *)bufstr(0, max, flag, echo);
	while (*ptr == ' ' || *ptr == '\t') ptr++;
	nchr = 0;
	strptr = strbuf;
	return ptr;	
}

char *
editstr(str, max, flag)
	char *str;
	int max, flag;
{
	register char *ptr;
	int nc;

	nc = strlen(str);
	if (max && nc >= max) nc = max - 1;
	strncpy((char*)strbuf, str, nc);
	strbuf[nc] = '\0';
	nchr = 0;
	putstr(strbuf);
	ptr = (char *)bufstr(nc, max, flag, 1);
	while (*ptr == ' ' || *ptr == '\t') ptr++;
	nchr = 0;
	strptr = strbuf;
	return ptr;	
}

int
flushinput()
{
	nchr = 0;
	strbuf[0] = '\0';
}

char *
chatinput(prompt)
	char *prompt;
{
	char *ptr;
	int max;
	static char buf[BUFSIZ];

	nchr = 0;
	if (globalflags & BEGINCHAT) {
		globalflags &= ~BEGINCHAT;
		strbuf[0] = '\0';
#ifdef	EXTERN_APPS
		putstr("\n\007CHAT Started. Type text or /h - help, /q - quit\n\n");
	}
	putstr("\r%s: ", prompt);
#else
		LOGIT(LOG_INFO, "Begin talk");
		putstr("\n\007%s\n\n", sysmsg(MSG_CHATSTARTED));
	}
	acttimer = INACTTIMER * 60;
	if (termflags & (ANSITERM | RIPTERM))
		putstr("\r\033[1;36m%s\033[5;37m:\033[0m ", prompt);
	else	putstr("\r%s: ", prompt);
#endif
	max = 77 - strlen(prompt);
	if (max < 0) max = 0;
	ptr = (char *)bufstr(putstr(strbuf), max, -1, 1);
	while (*ptr == ' ' || *ptr == '\t') ptr++;
	strcpy(buf, ptr);
	flushinput();
	return buf;
}

char *
gettok(max, flag, echo)
	register max, flag, echo;
{
	register char *ptr, *ptr2;

/*	do { */
		ptr = (char *)bufstr(0, max, flag, echo);
		while (*ptr == ' ') ptr++, nchr--;
/*	} while (!*ptr); */

	if ((ptr2 = strchr(ptr, ' ')) != NULL) {
		*ptr2++ = 0;
		while (*ptr2 == ' ') ptr2++;
		if (*ptr2) {
			nchr -= (ptr2 - ptr);
			strptr = (u_char *)ptr2;
			return ptr;
		}
	}
	nchr = 0;
	strptr = strbuf;
	return ptr;
}
