/*
 *	Copyright (c) 1996,1997 The CAD lab of the
 *	Siberian State Academy of Telecommunications
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include <ctype.h>
#include <utime.h>

#include "drive.h"
#include "scandir.h"
#ifdef	HAVE_LIBMSGAPI
#include "msgapi.h"
#endif

#define	MSGPACK_ID	"msgpack"
#define	MSGPACK_VER	"v1.2"

#define	BUFSIZE			4096
#define	MSGTYPE_FILE		0
#define	BIG_BUF_SIZE		51200	/* 50kb */

int cnt, num_msgs, bflag, cflag, mflag, qflag;
time_t time_old;
char msgbase[1024];

#define	STAT_READ	1
#define	STAT_OLD	2
#define	STAT_DEL	4
#define	STAT_STATUS	(STAT_READ | STAT_OLD | STAT_DEL)
#define	STAT_MARK	8

struct msg_ent {
	int status;	/* message status */
	long head;	/* offset of header */
	long eof;	/* offset of end */
	time_t date;	/* date/time */
};

main(argc, argv)
	int argc;
	char *argv[];
{
	int op, type;
	char *p;
	extern int optind, opterr;
	extern char *optarg;

	num_msgs = time_old = bflag = cflag = mflag = qflag = 0;
	opterr = 0;
	while ((op = getopt(argc, argv, "bcd:qmn:?")) != EOF)
		switch (op) {
			case 'b':	/* make backup files */
				bflag++;
				break;
			case 'c':	/* convert msg base to text file */
				cflag++;
				break;
			case 'd':	/* days old to expire messages */
				time_old = (time_t)atoi(optarg);
				if (time_old < 1) usage();
				time_old = time(NULL) - time_old * 86400;
				break;
			case 'q':	/* quiet mode */
				qflag++;
				break;
			case 'm':	/* pack mailboxes */
				mflag++;
				break;
			case 'n':	/* max number of messages */
				num_msgs = atoi(optarg);
				if (num_msgs < 1) usage();
				break;
			case '?':
			default:
				usage();
		}
	if (argv[optind] == NULL) usage();

	if (getuid() == geteuid()) { /* try to set euid to bbs */
		struct passwd *pwd;
		if ((pwd = getpwnam(BBS)) != NULL ||
		    (pwd = getpwnam("tns")) != NULL) {
			if (seteuid(pwd->pw_uid) < 0) {
				fprintf(stderr, "%s: Permission denied\n", MSGPACK_ID);
				exit(1);
			}
		}
	}

	for (cnt = 0, op = optind; (p = argv[optind]) != NULL; optind++) {
		if (mflag) {
			if (!qflag) {
				printf("Scaning userdir in %s...\n", p);
				fflush(stdout);
			}
			if (!packuserbase(p)) op++;
		} else {
			if ((type = getmsgbase(p)) != -1) {
				if ((p = strrchr(msgbase, '/')) != NULL) p++;
				else p = msgbase;
				if (type == MSGTYPE_FILE) {
					if (!mailpack(p)) op++, cnt++;
				}
#ifdef	HAVE_LIBMSGAPI
				else	if (!msgpack(p, type)) op++, cnt++;
#endif
			}
		}
	}
	if (qflag < 3)
		printf("%d %s scaned and packed\n", cnt,
		       mflag ? "Mail Boxes" : "Message Bases");
	exit(op != optind);
}

int
packuserbase(usrdir)
	char *usrdir;
{
	register i, j;
	int nf, nl, type;
	struct dir_ent *fn, *ln;
	char buf[1024];

	if ((nf = scan_dir(usrdir, &fn, dirs_only, SCAN_SORTBYNAME)) < 0) {
		perror(usrdir);
		return -1;
	}
	for (i = 0; i < nf; i++) {
		sprintf(buf, "%s/%s", usrdir, fn[i].name);
		if ((nl = scan_dir(buf, &ln, dirs_only, SCAN_SORTBYNAME)) < 0)
			continue;
		for (j = 0; j < nl; j++) {
			sprintf(buf, "%s/%s/%s/%s", usrdir, fn[i].name, ln[j].name, MAILBOX);
			if ((type = getmsgbase(buf)) != -1) {
				sprintf(buf, "%s %s", fn[i].name, ln[j].name);
				if (type == MSGTYPE_FILE)
					if (!mailpack(buf)) cnt++;
#ifdef	HAVE_LIBMSGAPI
				else	if (!msgpack(buf, type)) cnt++;
#endif
			}
		}
		free_dir(&ln, nl);
	}
	free_dir(&fn, nf);
	return 0;
}

#ifndef	HAVE_LIBMSGAPI

int
getmsgbase(fname)
	char *fname;
{
	char *p;
	struct stat st;

	(void)strcpy(msgbase, fname);

	if (stat(msgbase, &st) == 0 && (st.st_mode & S_IFMT) == S_IFREG) {
		if ((p = strrchr(msgbase, '.')) == NULL ||
		    strcmp(p, ".sqd") || strcmp(p, ".sqi"))
			return MSGTYPE_FILE;
	}
	return -1;
}

#else	/* HAVE_LIBMSGAPI */

static int
findmsg(file)
	register struct dirent *file;

{
	register char *p;

	if (!isdigit(file->d_name[0])) return 0;
	if ((p = strchr(file->d_name, '.')) == NULL) return 0;
	if (strcmp(p, ".msg")) return 0;
	return 1;
}

int
getmsgbase(fname)
	char *fname;
{
	int type;
	char *p;
	struct stat st;

	if (*fname == '$') {
		fname++;
		type = MSGTYPE_SQUISH;
	} else	type = MSGTYPE_SDM;

	strcpy(msgbase, fname);

	if (stat(msgbase, &st) < 0) {
		strcat(msgbase, ".sqd");
		if (stat(msgbase, &st) < 0 ||
		    (st.st_mode & S_IFMT) != S_IFREG) {
			perror(fname);
			return -1;
		}
		type = MSGTYPE_SQUISH;
	} else if ((st.st_mode & S_IFMT) == S_IFREG) {
		if ((p = strrchr(msgbase, '.')) == NULL || strcmp(p, ".sqd"))
			type = MSGTYPE_FILE;
		else	type = MSGTYPE_SQUISH;
	} else if ((st.st_mode & S_IFMT) == S_IFDIR) {
		struct dir_ent *msgs;
		if ((type = scan_dir(msgbase, &msgs, findmsg, 0)) <= 0) {
			if (type < 0) perror(fname);
			else fprintf(stderr, "%s: Not SDM or empty MsgBase\n", fname);
			return -1;
		}
		free_dir(&msgs, type);
		type = MSGTYPE_SDM;
	} else {
		fprintf(stderr, "%s: Not a Squish or SDM MsgBase\n", fname);
		return -1;
	}
	if (type == MSGTYPE_SQUISH && (p = strrchr(msgbase, '.')) != NULL)
		*p = '\0';
	return type;
}

time_t
MsgTime(dosdate)
	union stamp_combo *dosdate;
{
	struct tm tmmsg;
	DosDate_to_TmDate(dosdate, &tmmsg);
	return (mktime(&tmmsg));
}

int
msgpack(area, type)
	char *area;
	int type;
{
	FILE *fp;
	time_t date;
	XMSG msg;
	MSG *in_area, *out_area;
	MSGH *in_msg, *out_msg;
	char *what, *ctrl, *p;
	int in_msgn, out_msgn, ctrllen, offs, got;
	struct _minf mi;
	char tmpname[1024], bakname[1024], buf[BUFSIZE];

	mi.req_version = 0;
	mi.def_zone = DEFMSGZONE;
	MsgOpenApi(&mi);
	if ((in_area = MsgOpenArea(msgbase, MSGAREA_NORMAL, type)) == NULL) {
		fprintf(stderr, "can't read \"%s\"\n", msgbase);
		MsgCloseApi();
		return -1;
	}
	MsgLock(in_area);
	strcpy(tmpname, msgbase);
	if (!cflag) {
		strcat(tmpname, "~tmp");
		if ((out_area = MsgOpenArea(tmpname, MSGAREA_CREATE, type)) == NULL) {
			fprintf(stderr, "can't create \"%s\"\n", tmpname);
			MsgCloseArea(in_area);
			MsgCloseApi();
			return -1;
		}
		MsgLock(out_area);
	} else {
		if (type == MSGTYPE_SDM) strcat(tmpname, ".msg");
		if ((fp = fopen(tmpname, "w")) == NULL) {
			perror(tmpname);
			MsgCloseArea(in_area);
			MsgCloseApi();
			return -1;
		}
	}
	if (mflag) what = "Mbox";
	else what = "Area";
	for (in_msgn = out_msgn = 1; in_msgn <= MsgHighMsg(in_area); in_msgn++) {
		if (!qflag && (in_msgn % 5) == 0) {
			printf("%s: %-25.25s Msg: %d\r", what, area, in_msgn);
			fflush(stdout);
		}
		if (num_msgs && in_msgn <= (int)MsgGetNumMsg(in_area) - num_msgs)
			continue;

		if ((in_msg = MsgOpenMsg(in_area, MOPEN_READ, in_msgn)) == NULL)
			continue;
		ctrllen = MsgGetCtrlLen(in_msg);
		if ((ctrl = malloc(ctrllen)) == NULL) ctrllen = 0;
		if (MsgReadMsg(in_msg, &msg, 0, 0, NULL, ctrllen, (unsigned char *)ctrl)) {
			fprintf(stderr, "%s: %-25.25s Msg: %d - can't read (error %d)!\n",
				what, area, in_msgn, msgapierr);
			if (ctrl) free(ctrl);
			MsgCloseMsg(in_msg);
			continue;
		}
		date = MsgTime(&msg.date_written);
		if (time_old && date < time_old) {
			if (ctrl) free(ctrl);
			MsgCloseMsg(in_msg);
			continue;
		}
		msg.attr |= MSGSCANNED;
		msg.replyto = 0;
		memset(msg.replies, '\0', sizeof(msg.replies));
		if (!cflag) {
			if ((out_msg = MsgOpenMsg(out_area, MOPEN_CREATE, 0)) == NULL) {
				fprintf(stderr, "%s: %-25.25s Msg: %d - can't write (error %d)!\n",
					what, area, in_msgn, msgapierr);
				if (ctrl) free(ctrl);
				MsgCloseMsg(in_msg);
				continue;
			}
			MsgWriteMsg(out_msg, FALSE, &msg, NULL, 0,
				    MsgGetTextLen(in_msg), ctrllen, (unsigned char *)ctrl);
		} else {
			time_t now = time(NULL);
			fprintf(fp, "From %s %s", MSGPACK_ID, ctime(&now));
			strftime(bakname, sizeof(bakname),
				 "%a, %e %b %Y %H:%M:%S", localtime(&date)); 
			fprintf(fp, "Date: %s %s\n", bakname, (char *)gettzone());
			fprintf(fp, "Message-Id: <%d@bbs>\n",
				MsgMsgnToUid(in_area, in_msgn));
			if (!mflag) {
				strcpy(bakname, (char *)msg.from);
				if ((p = strchr(bakname, ' ')) != NULL) *p = '.';
				fprintf(fp, "From: \"%s\" <%s@bbs>\n", msg.from, bakname);
			} else	fprintf(fp, "From: %s\n", msg.from);
			if (!mflag) {
				if (strcasecmp((char *)msg.to, "All")) {
					strcpy(bakname, (char *)msg.to);
					if ((p = strchr(bakname, ' ')) != NULL)	*p = '.';
					fprintf(fp, "To: \"%s\" <%s@bbs>\n", msg.to, bakname);
				}
			} else	fprintf(fp, "To: %s\n", msg.to);
			fprintf(fp, "Subject: %s\n", msg.subj);
			if ((p = strrchr(msgbase, '/')) != NULL) p++;
			else p = msgbase;
			fprintf(fp, "X-Folder: %s\n", p);
			fprintf(fp, "\n");
		}
		for (offs = 0; offs < MsgGetTextLen(in_msg); offs += got) {
			got = MsgReadMsg(in_msg, NULL, offs, BUFSIZE,
					 (unsigned char *)buf, 0, NULL);
			if (got < 1) break;
			if (!cflag) {
				MsgWriteMsg(out_msg, TRUE, NULL, (unsigned char *)buf,
					    got, MsgGetTextLen(in_msg), 0, NULL);
			} else {
				for (p = buf; *p && p < buf+got; p++)
					if (*p == '\r') *p = '\n';
				fwrite(buf, sizeof(char), got-1, fp);
			}
		}
		if (ctrl) free(ctrl);
		if (!cflag) MsgCloseMsg(out_msg);
		else fputc('\n', fp);
		MsgCloseMsg(in_msg);
		out_msgn++;
	}
	if (!cflag) MsgCloseArea(out_area);
	else fclose(fp);
	MsgCloseArea(in_area);
	MsgCloseApi();

	if (!cflag) {
		if (type == MSGTYPE_SQUISH) {
			strcpy(tmpname, msgbase);
			strcat(tmpname, ".sqd");
			if (bflag) {
				strcpy(bakname, msgbase);
				strcat(bakname, ".sqd.bak");
				unlink(bakname);
				link(tmpname, bakname);
			}
			unlink(tmpname);
			strcpy(bakname, msgbase);
			strcat(bakname, "~tmp.sqd");
			link(bakname, tmpname);
			unlink(bakname);

			strcpy(tmpname, msgbase);
			strcat(tmpname, ".sqi");
			if (bflag) {
				strcpy(bakname, msgbase);
				strcat(bakname, ".sqi.bak");
				unlink(bakname);
				link(tmpname, bakname);
			}
			unlink(tmpname);
			strcpy(bakname, msgbase);
			strcat(bakname, "~tmp.sqi");
			link(bakname, tmpname);
			unlink(bakname);
		} else {
			if (bflag) {
				strcpy(bakname, msgbase);
				strcat(bakname, ".bak");
				deldirtree(bakname);
				sprintf(buf, "%s %s %s", SYS_MV, msgbase, bakname);
				system(buf);
			} else	deldirtree(msgbase);
			sprintf(buf, "%s %s %s", SYS_MV, tmpname, msgbase);
			system(buf);
		}
	} else if (mflag) chmod(tmpname, 0660);
	if (cflag > 1) {
		if (type == MSGTYPE_SQUISH) {
			strcpy(tmpname, msgbase);
			strcat(tmpname, ".sqd");
			unlink(tmpname);
			strcpy(tmpname, msgbase);
			strcat(tmpname, ".sqi");
			unlink(tmpname);
		} else {
			deldirtree(msgbase);
			if (rename(tmpname, msgbase) < 0) perror("rename");
		}
	}
	if (qflag < 2)
		printf("%s: %-25.25s Msg: %d -> %d\n", what, area, in_msgn, out_msgn);
	return 0;
}

#endif	/* HAVE_LIBMSGAPI */

struct msg_ent *
open_mbox(fp, eof, cnt)
	FILE *fp;
	long eof;
	int *cnt;
{
	register char *p;
	int nmsg, status, flag;
	long offs;
	char buf[1024], from[256], date[40];
	struct msg_ent *mbox;

	*from = '\0';
	flag = 1;
	mbox = NULL;
	nmsg = 0;
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		for (p = buf; *p && *p != '\n'; p++)
			if ((unsigned char)*p < 0x20) *p = ' ';
		if (*p == '\0') {	/* too long line */
			flag = 0;
			continue;
		}
		*p++ = '\0';
		if (flag && !strncmp(buf, "From ", 5)) {
			offs = ftell(fp) - (p - buf);
			/* Save the original copy of From field */
			strncpy(from, &buf[5], sizeof(from));
			from[sizeof(from)-1] = '\0';
			*date = '\0';
			status = 0;
			flag = 0;
			if (nmsg && !mbox[nmsg-1].eof) mbox[nmsg-1].eof = offs;
			continue;
		}
		flag = (*buf == '\0');
		if (*from == '\0') continue;
		if (*buf == '\0') {	/* message body follow if any */
			if ((p = strchr(from, ' ')) != NULL) *p++ = '\0';
			else p = "";
			mbox = realloc(mbox, (nmsg+1) * sizeof(struct msg_ent));
			if (mbox == NULL) {
				nmsg = 0;
				break;
			}
			mbox[nmsg].status = status;
			mbox[nmsg].head = offs;
			mbox[nmsg].eof = 0; /* assign later */
			mbox[nmsg].date = parsedate(*date ? date : p, NULL);
			nmsg++;
			*from = '\0';
		} else if (!strncmp(buf, "Date: ", 6)) {
			for (p = &buf[6]; *p && *p == ' '; p++);
			strncpy(date, p, sizeof(date));
			date[sizeof(date)-1] = '\0';
		} else if (!strncmp(buf, "Status: ", 8)) {
			for (p = &buf[8]; *p; p++) switch (*p) {
				case 'R': status |= STAT_READ; break;
				case 'O': status |= STAT_OLD; break;
				case 'D': status |= STAT_DEL; break;
				case 'K': status |= STAT_DEL; break;
			}
		}
	}
	if (nmsg && !mbox[nmsg-1].eof) mbox[nmsg-1].eof = eof;
	*cnt = nmsg;
	return mbox;
}

int
fast_copy_fp(sfp, tfp, len)
	FILE *sfp, *tfp;
	int len;
{
	int n;
	static char *bigbuf = NULL;

	if (len < 1) return 0;
	if (bigbuf == NULL && (bigbuf = malloc(BIG_BUF_SIZE)) == NULL)
		return -1;
	while (len > 0) {
		if (len > BIG_BUF_SIZE) n = BIG_BUF_SIZE;
		else n = len;
		if (fread(bigbuf, sizeof(char), n, sfp) != n) break;
		if (fwrite(bigbuf, sizeof(char), n, tfp) != n) break;
		len = len - n;
	}
	return (len ? -1 : 0);
}

int
mailpack(area)
	char *area;
{
	int in_msg, out_msg, nmsg;
	FILE *in_fp, *out_fp;
	long eof;
	struct stat st;
	struct utimbuf tm;
	char *what, tmpname[1024];
	struct msg_ent *mbox;

	if (cflag) {
		fprintf(stderr, "%s: Can't convert to itself\n", msgbase);
		return -1;
	}
	if (mflag) what = "Mbox";
	else what = "Area";

	if (stat(msgbase, &st) < 0 || st.st_size < 5) {
		if (qflag < 2)
			printf("%s: %-25.25s Msg: 0 -> 0\n", what, area);
		return 0;
	}
	tm.actime = st.st_atime;
	tm.modtime = st.st_mtime;
	eof = (long)st.st_size;

	if ((in_fp = fopen(msgbase, "r")) == NULL) {
		perror(msgbase);
		return -1;
	}
	strcpy(tmpname, msgbase);
	strcat(tmpname, "~tmp");
	if ((out_fp = fopen(tmpname, "w")) == NULL) {
		perror(tmpname);
		fclose(in_fp);
		return -1;
	}
	mbox = open_mbox(in_fp, eof, &nmsg);

	for (in_msg = out_msg = 0; in_msg < nmsg; in_msg++) {
		/* sanity check: skip deleted message */
		if (mbox[in_msg].status & STAT_DEL) continue;
		if (!qflag && (in_msg % 5) == 0) {
			printf("%s: %-25.25s Msg: %d\r", what, area, in_msg+1);
			fflush(stdout);
		}
		if (num_msgs && in_msg < nmsg - num_msgs) continue;
		if (time_old && mbox[in_msg].date < time_old) continue;
		fseek(in_fp, mbox[in_msg].head, SEEK_SET);
		fast_copy_fp(in_fp, out_fp, mbox[in_msg].eof - mbox[in_msg].head);
		out_msg++;
	}
	if (mbox != NULL) free(mbox);

	if (in_msg != out_msg) {	/* check for new messages */
		if (stat(msgbase, &st) == 0 && (long)st.st_size > eof) {
			/* append new messages */
			if ((in_fp = freopen(msgbase, "r", in_fp)) != NULL) {
				fseek(in_fp, eof, SEEK_SET);
				fast_copy_fp(in_fp, out_fp, (long)st.st_size - eof);
			}
		}
	}
	if (in_fp != NULL) fclose(in_fp);
	fclose(out_fp);
	if (in_msg != out_msg) {
		if (rename(tmpname, msgbase) == 0)
			utime(msgbase, &tm);
		else	perror("rename");
	} else unlink(tmpname);
	if (mflag) chmod(msgbase, 0660);

	if (qflag < 2)
		printf("%s: %-25.25s Msg: %d -> %d\n", what, area, in_msg, out_msg);
	return 0;
}

usage()
{
	fprintf(stderr, "%s %s\n", MSGPACK_ID, MSGPACK_VER);
	fprintf(stderr, "usage: %s [-b] ", MSGPACK_ID);
#ifdef	HAVE_LIBMSGAPI
	fprintf(stderr, "[-c] ");
#endif
	fprintf(stderr, "[-q] [-m] [-d days_old] [-n num_msgs] msgbases...\n");
	fprintf(stderr, "where:\n\
  -b       Make backup copy of message base\n");
#ifdef	HAVE_LIBMSGAPI
	fprintf(stderr, "\
  -c       Convert message base to text file (-cc = convert & delete src)\n");
#endif
	fprintf(stderr, "\
  -q       Quiet mode, no progress metter (-qq = most quieter)\n\
  -m       Scan for MailBoxes in userbase tree rooted in `msgbase'\n\
  days_old Days old to expire messages\n\
  num_msgs Number of messages to keep in message base\n");
	fprintf(stderr, "\
example1: msgpack -n 500 /usr/bbs/msg/*  -- to pack MsgBases\n\
example2: msgpack -d 30 -m /usr/bbs/usr  -- to pack MailBoxes\n");
	fprintf(stderr, "\
NOTE: This program intended to use by root or be sure it have RIGHT set-uid!\n");
	exit(1);
}
