/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <signal.h>
#include <setjmp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include <errno.h>

#include "drive.h"
#include "loginfo.h"
#include "rematch.h"
#include "udpserv.h"

#define	SU		"SetUser v1.2"
#define	MAXUDPWAIT	3	/* sec. */

extern int errno;

int setvalue(int port, ...);
char *getvalue(int port, ...);


main(argc, argv)
	int argc;
	char **argv;
{
	int i, set, ct, op, fileflag, findmask, port[MAXBBSUSERS];
	uid_t uid;
	char *p, pattern[100], settings[1000];
	struct loginfo *info;
	char *tty_str, *un_str;
	regex *tty_rgc, *un_rgc = NULL;
	FILE *fp;
	extern int optind, opterr;
	extern char *optarg;

	/*
	* Is effective user ID missmatch with the real user ID.
	*/
	if ((uid = getuid()) != geteuid()) {
		fprintf(stderr, "Permission missmatch, do not use set-user-ID for these program\n");
		exit(1);
	}

	/*
	* Check if I'm on the control terminal.
	* Prepare the tty pattern for default.
	*/
	ct = isatty(0);
	if (ct)	/* Set the default tty */
		tty_str = strippath(ttyname(0));
	else	/* No control terminal (under cron?), match all tty */
		tty_str = "tty*";
	if ((tty_rgc = recomp(tty_str)) == NULL) {
		fprintf(stderr, "Can't regcomp \"%s\"\n", p);
		exit(1);
	}

	/*
	* Parse command line arguments.
	*/
	opterr = 0;
	fileflag = findmask = 0;
	while ((op = getopt(argc, argv, "nrst:u:")) != EOF) {
		switch (op) {
			case 'n': /* print number of users */
				for (i = 0; getinfo(0) != NULL; i++);
				printf("%d\n", i);
				exit(0);
			case 's': /* Save cmds to .setuser flag file */
				/*
				* Append commands to .setuser file, it must
				* be processed latter when drive get started.
				*/
				fileflag |= 1;
				break;
			case 'r': /* Remove .setuser flag file */
				fileflag |= 2;
				break;
			case 't': /* Set terminal pattern */
				tty_str = strippath(optarg);
				refree(tty_rgc);
				if ((tty_rgc = recomp(tty_str)) == NULL) {
					fprintf(stderr, "Can't regcomp \"%s\"\n", tty_str);
					exit(1);
				}
				findmask |= 1;
				break;
			case 'u': /* Set username pattern */
				un_str = optarg;
				if (un_rgc != NULL) refree(un_rgc);
				if ((un_rgc = recomp(un_str)) == NULL) {
					fprintf(stderr, "Can't regcomp \"%s\"\n", un_str);
					exit(1);
				}
				findmask |= 2;
				break;
			case '?':
			default:
				usage();
		}
	}
	if (fileflag && uid) {
		fprintf(stderr, "The -r and -s options for root only\n");
		exit(1);
	}

	/*
	* Construct string of settings from argv array.
	*/
	settings[0] = '\0';
	for (set = i = 0; argv[optind] != NULL; i++, optind++) {
		if (strchr(argv[optind], '=') != NULL) set++;
		if (i) strcat(settings, " ");
		strcat(settings, argv[optind]);
	}
	if (set) {
		if (set != i) {
			fprintf(stderr, "Use only SET or GET variables at a time\n");
			exit(1);
		}
		if (uid) {
			struct passwd *pwd;
			/*
			* We are not root -- get bbs user ID.
			*/
			if ((pwd = getpwnam(BBS)) == NULL &&
			    (pwd = getpwnam("tns")) == NULL) {
				fprintf(stderr, "No %s in this system\n", BBS);
				exit(1);
			}
			/*
			* Check if I'm a bbs.
			*/
			if (uid != pwd->pw_uid) {
				fprintf(stderr, "The SET variables function for root or %s only\n", BBS);
				exit(1);
			}
		}
	}

	if (fileflag & 2) { /* Remove switch has been used */
		if (unlink(SETUSERFILE) < 0 && ct) {
			/* If have control terminal then warning report */
			fprintf(stderr, "Warning: Can't remove %s: %s\n",
				SETUSERFILE, strerror(errno));
		}
	}

	/*
	* Is we have settings for users in command line.
	*/
	if (!i) {
		/* If no control terminal then nothing to do */
		if (!ct) exit(0);
		findmask = -1;	/* Do list of users */
	}

	/*
	* Scan loginfo records to get users on-line.
	* Get the udp ports of matched users and fill in port[] array.
	*/
	op = 0;
	while (op < MAXBBSUSERS && (info = getinfo(0)) != NULL)
		switch (findmask) {
			case 0: /* Try to match for tty pattern */
			case 1:
				if (reexec(tty_rgc, info->tty))
					port[op++] = info->port;
				break;
			case 2: /* Try to match for username pattern */
				if (reexec(un_rgc, info->name))
					port[op++] = info->port;
				break;
			case 3: /* Match tty and username simultaneously */
				if (reexec(tty_rgc, info->tty) &&
				    reexec(un_rgc, info->name))
					port[op++] = info->port;
				break;
			default: /* List of users on-line */
				printf("%-6.6s %.15s  %-24.24s %-6d   pid:%d\n",
				       info->tty,
				       ctime(&info->ltime)+4,
				       info->name, info->baud, info->pid);
		}
	closeinfo();
	refree(tty_rgc);
	if (un_rgc != NULL) refree(un_rgc);

	/*
	* Construct pattern string.
	*/
	switch (findmask) {
		case 0:
		case 1:
			sprintf(pattern, "T:%s", tty_str);
			break;
		case 2:
			sprintf(pattern, "U:%s", un_str);
			break;
		case 3:
			sprintf(pattern, "T:%s U:%s", tty_str, un_str);
			break;
		default: /* List of users? All done */
			exit(0);
	}

	/*
	* Is settings for off-line users needed.
	*/
	if (fileflag & 1) {
		if ((fp = fopen(SETUSERFILE, "a")) == NULL) {
			fprintf(stderr, "Can't write to %s\n", SETUSERFILE);
			exit(1);
		}
		fprintf(fp, "%s %s\n", pattern, settings); /* Save settings */
		fclose(fp);
	}

	if (!op) { /* No one user matched */
		/* Under cron? This is normal exit */
		if (ct) {
			if (fileflag & 1) fprintf(stderr, "Settings saved. ");
			fprintf(stderr, "No match for users on-line\n");
		}
		exit(0);
	}

	/*
	* Prepare own udp socket to send settings through.
	*/
	if (initnet() < 0) exit(1);

	/*
	* Drop settings to users on-line.
	*/
	for (findmask = i = 0; i < op; i++) {
		if (!set) {
			if ((p = getvalue(port[i], settings)) != NULL) {
				puts(p);
				findmask++;
			}
		} else findmask += !setvalue(port[i], settings);
	}

	exit(findmask != op);
}

static int sock;

int
initnet()
{
	int on = 1;
	struct sockaddr_in sin;

	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		return -1;
	}
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on)) < 0) {
		perror("setsockopt SO_REUSEADDR");
		return -1;
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);
	sin.sin_port = 0;
	if (bind(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		perror("bind");
		return -1;
	}
	return 0;
}

static jmp_buf dontanswer;

static void
waitforanswer()
{
	alarm(0);
	signal(SIGALRM, SIG_DFL);
	longjmp(dontanswer, 1);
}

char *
getvalue(int port, ...)
{
	register i;
	int len;
	char *cp;
	va_list ap;
	struct sockaddr_in to;
	struct sockaddr_in from;
	static char buf[1024];

	va_start(ap, port);
	cp = va_arg(ap, char *);
	strcpy(buf, GETINFOMARKER);
	len = strlen(buf);
	(void)vsprintf(&buf[len], cp, ap);
	va_end(ap);
	len = strlen(buf);

	to.sin_family = AF_INET;
	to.sin_addr.s_addr = inet_addr(LOOPBACK);
	to.sin_port = port;

	if (setjmp(dontanswer)) {
		fprintf(stderr, "recvfrom: Operation timed out\n");
		return NULL;
	}
	signal(SIGALRM, waitforanswer);
	alarm(MAXUDPWAIT);
	if (sendto(sock, buf, len, 0, (struct sockaddr *)&to, sizeof(to)) < 0) {
		perror("sendto");
		return NULL;
	}
	len = sizeof(from);
	while ((i = recvfrom(sock, buf, sizeof(buf)-1, 0,
			(struct sockaddr *)&from, &len)) < 1)
		if (i < 0 && errno != EINTR) {
			perror("recvfrom");
			alarm(0);
			signal(SIGALRM, SIG_DFL);
			return NULL;
		}
	alarm(0);
	signal(SIGALRM, SIG_DFL);
	buf[i] = '\0';
	return buf;
}

int
setvalue(int port, ...)
{
	register len;
	char *cp;
	va_list ap;
	struct sockaddr_in to;
	char buf[1024];

	va_start(ap, port);
	cp = va_arg(ap, char *);
	strcpy(buf, CMDSERVMARKER);
	len = strlen(buf);
	(void)vsprintf(&buf[len], cp, ap);
	va_end(ap);
	len = strlen(buf);

	to.sin_family = AF_INET;
	to.sin_addr.s_addr = inet_addr(LOOPBACK);
	to.sin_port = port;

	len = sendto(sock, buf, len, 0, (struct sockaddr *)&to, sizeof(to));
	if (len < 0) perror("sendto");
	return len;
}

usage()
{
	fprintf(stderr, "%s\n", SU);
	fprintf(stderr, "usage: setuser [-n] [-r] [-s] [-t tty] [-u user] [settings...]\n");
	fprintf(stderr, "where:\n\
  -n        print number of users currently on-line\n\
  -r        remove %s file\n\
  -s        save (append) settings to %s\n\
  -t tty    match on-line user by tty pattern, default current tty\n\
  -u user   match on-line user by name pattern\n\
  settings  list of settings for this user\n\n\
By default it display the list of on-line users\n",
		SETUSERFILE, SETUSERFILE);
	fprintf(stderr, "example:\n\
SET variables mode: setuser TIMELEFT=60 MENUPATH=Main_menu/Profile_menu\n\
GET variables mode: setuser USER TTY CALLNUMBER ONLINETIME TIMELEFT\n");

	exit(1);
}
