#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "rematch.h"


#ifdef	HAVE_V8_REGEXP
/*
 * V8 regexp(3) routines. It much faster then POSIX regex(3)
 */

#include <ctype.h>

void
makelowercase(s, t)
	register char *s, *t;
{
	while (*s) {
		*t = tolower(*s);
		s++, t++;
	}
	*t = 0;
}

regexp *
recomp(pat)
	char *pat;
{
	register char *p;
	regexp *rgc;

	if ((p = malloc(strlen(pat)+1)) == NULL) return NULL;
	(void)makelowercase(pat, p);
	rgc = regcomp(p);
	free(p);
	return rgc;
}

int
reexec(rgc, str)
	regexp *rgc;
	char *str;
{
	register char *s;
	int match;

	if ((s = malloc(strlen(str)+1)) == NULL) return 0;
	(void)makelowercase(str, s);
	match = regexec(rgc, s);
	free(s);
	return match;
}

void
refree(rgc)
	regexp *rgc;
{
	free(rgc);
}

int
rematch(pat, str)
	char *pat, *str;
{
	register char *p, *s;
	int match;
	regexp *rgc;

	if ((p = malloc(strlen(pat)+1)) == NULL ||
	    (s = malloc(strlen(str)+1)) == NULL) return 0;
	(void)makelowercase(pat, p);
	(void)makelowercase(str, s);
	match = 0;
	if ((rgc = regcomp(p)) != NULL) {
		match = regexec(rgc, s);
		free(rgc);
	}
	free(p);
	free(s);
	return match;
}

#else
/*
 * POSIX regex(3) routines
 */

regex_t *
recomp(pat)
	char *pat;
{
	regex_t *rgc;

	if ((rgc = malloc(sizeof(regex_t))) == NULL) return NULL;
	if (!regcomp(rgc, pat, REG_ICASE|REG_NOSUB)) return rgc;
	free(rgc);
	return NULL;
}

int
reexec(rgc, str)
	regex_t *rgc;
	char *str;
{
	return !regexec(rgc, str, 0, NULL, 0);
}

void
refree(rgc)
	regex_t *rgc;
{
	regfree(rgc);
	free(rgc);
}

int
rematch(pat, str)
	char *pat, *str;
{
	int match;
	regex_t rgc;

	match = 0;
	if (!regcomp(&rgc, pat, REG_ICASE|REG_NOSUB)) {
		match = !regexec(&rgc, str, 0, NULL, 0);
		regfree(&rgc);
	}
	return match;
}

#endif	/* HAVE_V8_REGEXP */
