/* upscmd - simple "client" to test instant commands via upsd

   Copyright (C) 2000  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "upsclient.h"

static void usage(char *prog)
{
	printf("Network UPS Tools upscmd %s\n\n", UPS_VERSION);
	printf("usage: %s [-h]\n", prog);
	printf("       %s [-l <ups>]\n", prog);
	printf("       %s [-u <username>] [-p <password>] <ups> <command>\n\n", prog);
	printf("Administration program to initiate instant commands on UPS hardware.\n");
	printf("\n");
	printf("  -h		display this help text\n");
	printf("  -l <ups>	show available commands on UPS <ups>\n");
	printf("  -u <username>	set username for command authentication\n");
	printf("  -p <password>	set password for command authentication\n");
	printf("\n");
	printf("  <ups>		UPS identifier - myups@localhost, etc.\n");
	printf("  <command>	Valid instant command - FPTEST, BTEST1, etc.\n");

	exit(0);
}

static void clean_exit(UPSCONN *ups, char *upsname, char *hostname, int code)
{
	if (upsname)
		free(upsname);

	if (hostname)
		free(hostname);

	upscli_disconnect(ups);

	exit(code);
}

static void listvars(char *rawname)
{
	char	buf[SMALLBUF], buf2[SMALLBUF], *ptr, *tmp, *q, *desc;

	int	port;
	char	*upsname, *hostname;
	UPSCONN	ups;

	upscli_splitname(rawname, &upsname, &hostname, &port);

	if (upscli_connect(&ups, hostname, port, 0) < 0) {
		fprintf(stderr, "Error: %s\n", upscli_strerror(&ups));
		clean_exit(&ups, upsname, hostname, 1);
	}

	if (upscli_getlist(&ups, upsname, UPSCLI_LIST_CMDS, buf, 
		sizeof(buf)) < 0) {

		fprintf(stderr, "Can't list instant commands: %s\n",
			upscli_strerror(&ups));

		clean_exit(&ups, upsname, hostname, 1);
	}

	if (strlen(buf) == 0) {
		fprintf(stderr, "No instant commands are supported on UPS [%s]\n",
		       rawname);
		clean_exit(&ups, upsname, hostname, 1);
	}

	printf("Instant commands supported on UPS [%s]:\n\n", rawname);

	ptr = buf;

	while (ptr) {
		tmp = ptr;
		ptr = strchr(tmp, ' ');
		if (ptr) {
			*ptr = '\0';
			ptr++;
		}

		printf("%s - ", tmp);

		/* now get description for this command */

		snprintf(buf2, sizeof(buf2), "INSTCMDDESC %s\n", tmp);

		if (upscli_sendline(&ups, buf2, strlen(buf2)) < 0) {
			fprintf(stderr, "Description fetch failed: %s\n", 
				upscli_strerror(&ups));
			continue;
		}

		if (upscli_readline(&ups, buf2, sizeof(buf2)) < 0) {
			fprintf(stderr, "Description fetch failed: %s\n", 
				upscli_strerror(&ups));
			continue;
		}

		q = strchr(buf2, '"');
		if (!q) {
			printf("Invalid description received from server (1)\n");
			continue;
		}

		desc = ++q;
		q = strchr(desc, '"');
		if (!q) {
			printf("Invalid description received from server (2)\n");
			continue;
		}

		*q = '\0';
		printf("%s\n", desc);
	}

	clean_exit(&ups, upsname, hostname, 0);
}

int main(int argc, char **argv)
{
	int	have_un, have_pw;
	char 	*prog, buf[SMALLBUF], username[SMALLBUF], password[SMALLBUF];
	int	i, ret;

	int	port;
	char	*upsname, *hostname;
	UPSCONN	ups;

	prog = argv[0];

	have_un = have_pw = 0;

	while ((i = getopt(argc, argv, "+l:hu:p:")) != EOF) {
		switch (i) {
			case 'l':
				listvars(optarg);
				break;

			case 'u':
				snprintf(username, sizeof(username), "%s", 
					optarg);
				have_un = 1;
				break;

			case 'p':
				snprintf(password, sizeof(password), "%s", 
					optarg);
				have_pw = 1;
				break;			

			case 'h':
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 2)
		usage(prog);

	upscli_splitname(argv[0], &upsname, &hostname, &port);

	if (upscli_connect(&ups, hostname, port, 0) < 0) {
		fprintf(stderr, "Error: %s\n", upscli_strerror(&ups));
		clean_exit(&ups, upsname, hostname, 1);
	}

	if (!have_un) {
		printf("Username: ");
		fgets(username, SMALLBUF, stdin);

		/* deal with that pesky newline */
		if (strlen(username) > 1)
			username[strlen(username) - 1] = '\0';
	}

	/* getpass leaks slightly - use -p when testing in valgrind */
	if (!have_pw) {
		snprintf(password, sizeof(password), "%s",
			getpass("Password: "));
	}

	snprintf(buf, sizeof(buf), "USERNAME %s\n", username);

	if (upscli_sendline(&ups, buf, strlen(buf)) < 0) {
		fprintf(stderr, "Can't set username: %s\n", 
			upscli_strerror(&ups));

		clean_exit(&ups, upsname, hostname, 1);
	}

	ret = upscli_readline(&ups, buf, sizeof(buf));

	if (ret < 0) {
		if (ups.upserror == UPSCLI_ERR_UNKCOMMAND) {
			fprintf(stderr, "Set username failed due to an "
				"unknown command.\n");

			fprintf(stderr, "You probably need to upgrade upsd.\n");
			clean_exit(&ups, upsname, hostname, 1);
		}

		fprintf(stderr, "Set username failed: %s\n", 
			upscli_strerror(&ups));

		clean_exit(&ups, upsname, hostname, 1);
	}

	snprintf(buf, sizeof(buf), "PASSWORD %s\n", password);

	if (upscli_sendline(&ups, buf, strlen(buf)) < 0) {
		fprintf(stderr, "Can't set password: %s\n", 
			upscli_strerror(&ups));
		clean_exit(&ups, upsname, hostname, 1);
	}

	if (upscli_readline(&ups, buf, sizeof(buf)) < 0) {
		fprintf(stderr, "Set password failed: %s\n", 
			upscli_strerror(&ups));

		clean_exit(&ups, upsname, hostname, 1);
	}

	if (upsname)
		snprintf(buf, sizeof(buf), "INSTCMD %s@%s\n", argv[1], 
		         upsname);
	else
		snprintf(buf, sizeof(buf), "INSTCMD %s\n", argv[1]);

	if (upscli_sendline(&ups, buf, strlen(buf)) < 0) {
		fprintf(stderr, "Can't send instant command: %s\n",
			upscli_strerror(&ups));

		clean_exit(&ups, upsname, hostname, 1);
	}

	if (upscli_readline(&ups, buf, sizeof(buf)) < 0) {
		fprintf(stderr, "Instant command failed: %s\n", 
			upscli_strerror(&ups));

		clean_exit(&ups, upsname, hostname, 1);
	}

	clean_exit(&ups, upsname, hostname, 0);

	/* NOTREACHED */
	return 0;
}
