/* upsmon - monitor power status over the 'net (talks to upsd via TCP)

   Copyright (C) 1998  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "common.h"

#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/socket.h>

#include "upsclient.h"
#include "upsmon.h"
#include "parseconf.h"
#include "timehead.h"
#
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

static	char	*shutdowncmd = NULL, *notifycmd = NULL;
static	char	*powerdownflag = NULL;

static	int	minsupplies = 1, sleepval = 5, deadtime = 15;

	/* default polling interval = 5 sec */
static	int	pollfreq = 5, pollfreqalert = 5;

	/* slave hosts are given 15 sec by default to logout from upsd */
static	int	hostsync = 15;  

	/* sum of all power values from config file */
static	int	totalpv = 0;

	/* should we go into an infinite loop upon shutdown? */
static	int	playdead = 0;

	/* default replace battery warning interval (seconds) */
static	int	rbwarntime = 43200;

	/* default "all communications down" warning interval (seconds) */
static	int	nocommwarntime = 300;

	/* default interval between the shutdown warning and the shutdown */
static	int	finaldelay = 5;

	/* set by SIGHUP handler, cleared after reload finishes */
static	int	reload = 0;

	/* userid for unprivileged process when using fork mode */
static	char	*run_as_user;

	/* SSL details - where to find certs, whether to use them */
static	char	*certpath = NULL;
static	int	certverify = 0;		/* don't verify by default */
static	int	forcessl = 0;		/* don't require ssl by default */

static	int	debuglevel = 0, userfsd = 0, use_pipe = 1, pipefd[2];

static	utype	*firstups = NULL;

	/* signal handling things */
static	struct sigaction sa;
static	sigset_t upsm_sigmask;

#ifdef SHUT_RDWR
#define	shutdown_how SHUT_RDWR
#else
#define	shutdown_how 2
#endif

static void debug(char *format, ...)
{
#ifdef HAVE_STDARG_H
	va_list	args;

	if (debuglevel < 1)
		return;

	va_start(args, format);
	vprintf(format, args);
	va_end(args);
#endif

	return;
}	

static void setflag(int *val, int flag)
{
	*val = (*val |= flag);
}
        
static void clearflag(int *val, int flag)  
{
	*val = (*val ^= (*val & flag));
}

static int flag_isset(int num, int flag)
{
	return ((num & flag) == flag);
}

static void wall(const char *text)
{
	FILE	*wf;

	wf = popen("wall", "w");

	if (!wf) {
		upslog(LOG_NOTICE, "Can't invoke wall");
		return;
	}

	fprintf(wf, "%s\n", text);
	pclose(wf);
} 

static void notify(const char *notice, int flags, const char *ntype, 
			const char *upsname)
{
	char	exec[LARGEBUF];
	int	ret;

	if (flag_isset(flags, NOTIFY_SYSLOG))
		upslogx(LOG_NOTICE, "%s", notice);

	/* fork here so upsmon doesn't get wedged if the notifier is slow */
	ret = fork();

	if (ret < 0) {
		upslog(LOG_ERR, "Can't fork to notify");
		return;
	}

	if (ret != 0)	/* parent */
		return;

	/* child continues and does all the work */

	if (flag_isset(flags, NOTIFY_WALL))
		wall(notice);

	if (flag_isset(flags, NOTIFY_EXEC)) {
		if (notifycmd != NULL) {
			snprintf(exec, sizeof(exec), "%s \"%s\"", notifycmd, notice);

			if (upsname)
				setenv("UPSNAME", upsname, 1);
			else
				setenv("UPSNAME", "", 1);

			setenv("NOTIFYTYPE", ntype, 1);
			system(exec);
		}
	}

	exit(0);
}

static void do_notify(const utype *ups, int ntype)
{
	int	i;
	char	msg[SMALLBUF], *upsname = NULL;

	/* grab this for later */
	if (ups)
		upsname = ups->sys;

	for (i = 0; notifylist[i].name != NULL; i++) {
		if (notifylist[i].type == ntype) {
			debug("do_notify: ntype 0x%04x (%s)\n", ntype, notifylist[i].name);

			snprintf(msg, sizeof(msg), notifylist[i].msg, ups ? ups->sys : "");

			notify(msg, notifylist[i].flags, notifylist[i].name, upsname);
			return;
		}
	}

	/* not found ?! */
}

/* check for master permissions on the server for this ups */
static void checkmaster(utype *ups)
{
	char	buf[SMALLBUF];

	/* don't bother if we're not configured as a master for this ups */
	if (!flag_isset(ups->status, ST_MASTER))
		return;

	if (ups->upsname)
		snprintf(buf, sizeof(buf), "MASTER %s\n", ups->upsname);
	else
		snprintf(buf, sizeof(buf), "MASTER\n");

	if (upscli_sendline(&ups->conn, buf, strlen(buf)) < 0) {
		upslogx(LOG_ALERT, "Can't set master mode on UPS [%s] - %s",
			ups->sys, upscli_strerror(&ups->conn));
		return;
	}

	if (upscli_readline(&ups->conn, buf, sizeof(buf)) == 0) {
		if (!strncmp(buf, "OK", 2))
			return;

		/* not ERR, but not caught by readline either? */

		upslogx(LOG_ALERT, "Master privileges unavailable on UPS [%s]", 
			ups->sys);
		upslogx(LOG_ALERT, "Response: [%s]", buf);
	}
	else {	/* something caught by readraw's parsing call */
		upslogx(LOG_ALERT, "Master privileges unavailable on UPS [%s]", 
			ups->sys);
		upslogx(LOG_ALERT, "Reason: %s", upscli_strerror(&ups->conn));
	}
}

/* authenticate to upsd and login */
static void do_login(utype *ups)
{
	char	buf[SMALLBUF];

	/* use it if we have it, but don't require it yet */

	if (ups->un) {
		snprintf(buf, sizeof(buf), "USERNAME %s\n", ups->un);
		if (upscli_sendline(&ups->conn, buf, strlen(buf)) < 0) {
			upslogx(LOG_ERR, "Can't set username on [%s]: %s",
				ups->sys, upscli_strerror(&ups->conn));
				return;
		}

		if (upscli_readline(&ups->conn, buf, sizeof(buf)) < 0) {
			upslogx(LOG_ERR, "Set username on [%s] failed (%s) -"
				" assuming ancient upsd",
				ups->sys, upscli_strerror(&ups->conn));

			/* just a warning for now - it's not required yet */
		}
	}

	/* authenticate first */
	snprintf(buf, sizeof(buf), "PASSWORD %s\n", ups->pw);

	if (upscli_sendline(&ups->conn, buf, strlen(buf)) < 0) {
		upslogx(LOG_ERR, "Can't set password on [%s]: %s",
			ups->sys, upscli_strerror(&ups->conn));
			return;
	}

	if (upscli_readline(&ups->conn, buf, sizeof(buf)) < 0) {
		upslogx(LOG_ERR, "Set password on [%s] failed: %s",
			ups->sys, upscli_strerror(&ups->conn));
		return;
	}

	/* catch insanity from the server - not ERR and not OK either */
	if (strncmp(buf, "OK", 2) != 0) {
		upslogx(LOG_ERR, "Set password on [%s] failed - got [%s]",
			ups->sys, buf);
		return;
	}

	/* password is set, let's login */

	if ((ups->upsname == NULL) || (strlen(ups->upsname) == 0))
		snprintf(buf, sizeof(buf), "LOGIN\n");
	else
		snprintf(buf, sizeof(buf), "LOGIN %s\n", ups->upsname);

	if (upscli_sendline(&ups->conn, buf, strlen(buf)) < 0) {
		upslogx(LOG_ERR, "Login to UPS [%s] failed: %s",
			ups->sys, upscli_strerror(&ups->conn));
		return;
	}

	if (upscli_readline(&ups->conn, buf, sizeof(buf)) < 0) {
		upslogx(LOG_ERR, "Can't login to UPS [%s]: %s",
			ups->sys, upscli_strerror(&ups->conn));
		return;
	}

	/* catch insanity from the server - not ERR and not OK either */
	if (strncmp(buf, "OK", 2) != 0) {
		upslogx(LOG_ERR, "Login on UPS [%s] failed - got [%s]",
			ups->sys, buf);
		return;
	}

	/* finally - everything is OK */
	debug("Logged into UPS %s\n", ups->sys);
	setflag(&ups->status, ST_LOGIN);

	/* now see if we also need to test master permissions */
	checkmaster(ups);
}

/* set flags and make announcements when a UPS has been checked successfully */
static void heardups(utype *ups)
{
	time_t	now;

	time(&now);
	ups->lastpoll = now;

	if (flag_isset(ups->status, ST_ALIVE))
		return;

	setflag(&ups->status, ST_ALIVE);

	/* only notify on subsequent reconnects */
	if (!flag_isset(ups->status, ST_FIRST))
		setflag(&ups->status, ST_FIRST);
	else
		do_notify(ups, NOTIFY_COMMOK);

	if (ups->pv == 0)	/* monitor only, no need to login */
		return;

	/* if not already logged in, do it now */
	if (!flag_isset(ups->status, ST_LOGIN))
		do_login(ups);
}

static void upsgone(utype *ups)
{
	/* if the UPS is already known to be gone, then do nothing here */
	if (!flag_isset(ups->status, ST_ALIVE))
		return;

	/* don't clear OL/OB/LB flags since we may use them later */
	clearflag(&ups->status, ST_ALIVE);
	clearflag(&ups->status, ST_LOGIN);

	do_notify(ups, NOTIFY_COMMBAD);
}

static void upsonbatt(utype *ups)
{
	sleepval = pollfreqalert;	/* bump up polling frequency */

	if (!flag_isset(ups->status, ST_ALIVE)) {
		debug("upsonbatt(%s) (first time)\n", ups->sys);
		heardups(ups);
		do_notify(ups, NOTIFY_ONBATT);
		setflag(&ups->status, ST_ONBATT);
		clearflag(&ups->status, ST_ONLINE);
		return;
	}

	debug("upsonbatt(%s) (normal)\n", ups->sys);
	
	heardups(ups);

	if (flag_isset(ups->status, ST_ONBATT)) 	/* no change */
		return;

	/* must have changed from OL to OB, so notify */

	do_notify(ups, NOTIFY_ONBATT);
	setflag(&ups->status, ST_ONBATT);
	clearflag(&ups->status, ST_ONLINE);
}

static void upsonline(utype *ups)
{
	if (!flag_isset(ups->status, ST_ALIVE)) {
		debug("upsonline(%s) (first time)\n", ups->sys);
		heardups(ups);
		setflag(&ups->status, ST_ONLINE);
		clearflag(&ups->status, ST_ONBATT);
		return;
	}

	debug("upsonline(%s) (normal)\n", ups->sys);

	heardups(ups);

	if (flag_isset(ups->status, ST_ONLINE)) 	/* no change */
		return;

	/* must have changed from OB to OL, so notify */

	do_notify(ups, NOTIFY_ONLINE);
	setflag(&ups->status, ST_ONLINE);
	clearflag(&ups->status, ST_ONBATT);
}

/* create the flag file if necessary */
static void setpdflag(void)
{
	FILE	*pdf;

	if (powerdownflag != NULL) {
		pdf = fopen(powerdownflag, "w");
		if (!pdf) {
			upslogx(LOG_ERR, "Failed to create power down flag!");
			return;
		}
		fprintf(pdf, "%s", SDMAGIC);
		fclose(pdf);
	}
}

/* the actual shutdown procedure */
static void doshutdown(void)
{
	int	ret;

	/* this should probably go away at some point */
	upslogx(LOG_CRIT, "Executing automatic power-fail shutdown");
	wall("Executing automatic power-fail shutdown\n");

	do_notify(NULL, NOTIFY_SHUTDOWN);

	sleep(finaldelay);

	/* in the pipe model, we let the parent do this for us */
	if (use_pipe) {
		char	ch;

		ch = 1;
		ret = write(pipefd[1], &ch, 1);
	} else {
		/* one process model = we do all the work here */

		if (geteuid() != 0)
			upslogx(LOG_WARNING, "Not root, shutdown may fail");

		setpdflag();

		ret = system(shutdowncmd);

		if (ret != 0)
			upslogx(LOG_ERR, "Unable to call shutdown command: %s\n",
		        	shutdowncmd);
	}

	/* if instructed to go into the infinite loop, then do so */
	if (playdead == 1)
		for (;;)
			sleep(100);

	/* hopefully not reached */
	exit(1);
}

/* set forced shutdown flag so other upsmons know what's going on here */
static void setfsd(utype *ups)
{
	char	buf[SMALLBUF];
	int	ret;

	debug("Setting FSD on UPS %s\n", ups->sys);

	if (ups->upsname)
		snprintf(buf, sizeof(buf), "FSD %s\n", ups->upsname);
	else
		snprintf(buf, sizeof(buf), "FSD\n");

	ret = upscli_sendline(&ups->conn, buf, strlen(buf));

	if (ret < 0) {
		upslogx(LOG_ERR, "FSD set on UPS %s failed: %s", ups->sys,
		        upscli_strerror(&ups->conn));
		return;
	}

	ret = upscli_readline(&ups->conn, buf, sizeof(buf));

	if (!strncmp(buf, "OK", 2))
		return;

	if (ret < 0)
		upslogx(LOG_ERR, "FSD set on UPS %s failed: %s", ups->sys,
		        upscli_strerror(&ups->conn));
	else
		upslogx(LOG_ERR, "FSD set on UPS %s failed: %s", ups->sys,
		        buf);
}

static void slavesync(void)
{
	utype	*ups;
	char	temp[SMALLBUF];
	time_t	start, now;
	int	maxlogins, logins;

	time(&start);

	for (;;) {
		maxlogins = 0;

		for (ups = firstups; ups != NULL; ups = ups->next) {

			/* only check login count on our master(s) */
			if (!flag_isset(ups->status, ST_MASTER))
				continue;

			if (upscli_getvar(&ups->conn, ups->upsname, "numlogins", 
			                temp, sizeof(temp)) >= 0) {
				logins = strtol(temp, (char **)NULL, 10);

				if (logins > maxlogins)
					maxlogins = logins;
			}
		}

		/* if no UPS has more than 1 login (us), then slaves are gone */
		if (maxlogins <= 1)
			return;

		/* after HOSTSYNC seconds, assume slaves are stuck and bail */
		time(&now);

		if ((now - start) > hostsync) {
			upslogx(LOG_INFO, "Host sync timer expired, forcing shutdown");
			return;
		}

		usleep(250000);
	}
}

static void forceshutdown(void)
{
	utype	*ups;
	int	isamaster = 0;

	debug("Shutting down any UPSes in MASTER mode...\n");

	/* set FSD on any "master" UPS entries (forced shutdown in progress) */
	for (ups = firstups; ups != NULL; ups = ups->next)
		if (flag_isset(ups->status, ST_MASTER)) {
			isamaster = 1;
			setfsd (ups);
		}

	/* if we're not a master on anything, we should shut down now */
	if (!isamaster)
		doshutdown();

	/* must be the master now */
	debug("This system is a master... waiting for slave logout...\n");

	/* wait up to HOSTSYNC seconds for slaves to logout */
	slavesync();

	/* time expired or all the slaves are gone, so shutdown */
	doshutdown();
}

static int is_ups_critical(utype *ups)
{
	time_t	now;

	/* FSD = the master is forcing a shutdown */
	if (flag_isset(ups->status, ST_FSD))
		return 1;

	/* not OB or not LB = not critical yet */
	if ((!flag_isset(ups->status, ST_ONBATT)) ||
		(!flag_isset(ups->status, ST_LOWBATT)))
		return 0;

	/* must be OB+LB now */

	/* if we're a master, declare it critical so we set FSD on it */
	if (flag_isset(ups->status, ST_MASTER))
		return 1;

	/* must be a slave now */

	/* FSD isn't set, so the master hasn't seen it yet */

	time(&now);

	/* give the master up to HOSTSYNC seconds before shutting down */
	if ((now - ups->lastnoncrit) > hostsync) {
		upslogx(LOG_WARNING, "Giving up on the master for UPS [%s]",
			ups->sys);
		return 1;
	}

	/* there's still time left */
	return 0;
}

/* recalculate the online power value and see if things are still OK */
static void recalc(void)
{
	utype	*ups;
	int	val_ol = 0;
	time_t	now;

	time(&now);
	ups = firstups;
	while (ups != NULL) {
		/* promote dead UPSes that were last known OB to OB+LB */
		if ((now - ups->lastpoll) > deadtime)
			if (flag_isset(ups->status, ST_ONBATT)) {
				debug ("Promoting dead UPS: %s\n", ups->sys);
				setflag(&ups->status, ST_LOWBATT);
			}

		/* note: we assume that a UPS that isn't critical must be OK *
		 *							     *
		 * this means a UPS we've never heard from is assumed OL     *
		 * whether this is really the best thing to do is undecided  */

		/* crit = (FSD) || (OB & LB) > HOSTSYNC seconds */
		if (is_ups_critical(ups))
			debug("Critical UPS: %s\n", ups->sys);
		else
			val_ol += ups->pv;

		ups = ups->next;
	}

	/* debug("Current power value: %d\n", val_ol);
	debug("Minimum power value: %d\n", minsupplies); */

	if (val_ol < minsupplies)
		forceshutdown();
}		

static void upslowbatt(utype *ups)
{
	if (!flag_isset(ups->status, ST_ALIVE)) {
		heardups(ups);
		setflag(&ups->status, ST_LOWBATT);
		return;
	}

	heardups(ups);

	if (flag_isset(ups->status, ST_LOWBATT)) 	/* no change */
		return;

	/* must have changed from !LB to LB, so notify */

	do_notify(ups, NOTIFY_LOWBATT);
	setflag(&ups->status, ST_LOWBATT);
}

static void upsreplbatt(utype *ups)
{
	time_t	now;

	time(&now);

	if ((now - ups->lastrbwarn) > rbwarntime) {
		do_notify(ups, NOTIFY_REPLBATT);
		ups->lastrbwarn = now;
	}
}

/* deal with a UPS that stays unavailable for awhile */
static void upsnocomm(utype *ups)
{
	time_t	now;

	time(&now);

	/* first only act if we're <nocommtime> seconds past the last poll */
	if ((now - ups->lastpoll) < nocommwarntime)
		return;

	/* now only complain if we haven't lately */
	if ((now - ups->lastncwarn) > nocommwarntime) {
		do_notify(ups, NOTIFY_NOCOMM);
		ups->lastncwarn = now;
	}
}

static void upsfsd(utype *ups)
{
	if (!flag_isset(ups->status, ST_ALIVE)) {
		heardups(ups);
		setflag(&ups->status, ST_FSD);
		return;
	}

	heardups(ups);

	if (flag_isset(ups->status, ST_FSD)) 	/* no change */
		return;

	/* must have changed from !FSD to FSD, so notify */

	do_notify(ups, NOTIFY_FSD);
	setflag(&ups->status, ST_FSD);
}

/* cleanly close the connection to a given UPS */
static void drop_connection(utype *ups)
{
	debug("Dropping connection to UPS [%s]\n", ups->sys);

	clearflag(&ups->status, ST_ALIVE);
	clearflag(&ups->status, ST_LOGIN);
	clearflag(&ups->status, ST_CONNECTED);

	upscli_disconnect(&ups->conn);
}

/* change some UPS parameters during reloading */
static void redefine_ups(utype *ups, int pv, const char *un, 
		const char *pw, const char *master)
{
	ups->retain = 1;

	if (ups->pv != pv) {
		upslogx(LOG_INFO, "UPS [%s]: redefined power value to %d", 
			ups->sys, pv);
		ups->pv = pv;
	}

	totalpv += ups->pv;

	if (ups->un) {
		if (strcmp(ups->un, un) != 0) {
			upslogx(LOG_INFO, "UPS [%s]: redefined username",
				ups->sys);

			free(ups->un);

			if (un)
				ups->un = xstrdup(un);
			else
				ups->un = NULL;

			/* 
			 * if not logged in force a reconnection since this
			 * may have been redefined to make a login work
			 */

			if (!flag_isset(ups->status, ST_LOGIN)) {
				upslogx(LOG_INFO, "UPS [%s]: retrying connection\n",
					ups->sys);	

				drop_connection(ups);
			}

		}	/* if (strcmp(ups->un, un) != 0) { */

	} else {

		/* adding a username? (going to new style MONITOR line) */

		if (un) {
			upslogx(LOG_INFO, "UPS [%s]: defined username",
				ups->sys);

			ups->un = xstrdup(un);

			/* possibly force reconnection - see above */

			if (!flag_isset(ups->status, ST_LOGIN)) {
				upslogx(LOG_INFO, "UPS [%s]: retrying connection\n",
					ups->sys);	

				drop_connection(ups);
			}

		}	/* if (un) */
	}

	/* paranoia */
	if (!ups->pw)
		ups->pw = xstrdup("");	/* give it a bogus, but non-NULL one */

	/* obviously don't put the new password in the syslog... */
	if (strcmp(ups->pw, pw) != 0) {
		upslogx(LOG_INFO, "UPS [%s]: redefined password", ups->sys);

		if (ups->pw)
			free(ups->pw);

		ups->pw = xstrdup(pw);

		/* possibly force reconnection - see above */

		if (!flag_isset(ups->status, ST_LOGIN)) {
			upslogx(LOG_INFO, "UPS [%s]: retrying connection\n",
				ups->sys);

			drop_connection(ups);
		}
	}

	/* slave -> master */
	if ((!strcasecmp(master, "master")) && (!flag_isset(ups->status, ST_MASTER))) {
		upslogx(LOG_INFO, "UPS [%s]: redefined as master", ups->sys);
		setflag(&ups->status, ST_MASTER);

		/* reset connection to ensure master mode gets checked */
		drop_connection(ups);
		return;
	}

	/* master -> slave */
	if ((!strcasecmp(master, "slave")) && (flag_isset(ups->status, ST_MASTER))) {
		upslogx(LOG_INFO, "UPS [%s]: redefined as slave", ups->sys);
		clearflag(&ups->status, ST_MASTER);
		return;
	}
}

static void addups(int reloading, const char *sys, const char *pvs, 
		const char *un, const char *pw, const char *master)
{
	int	pv;
	utype	*tmp, *last;

	/* un (username) may be NULL for now - old style host-based auth */

	if ((!sys) || (!pvs) || (!pw) || (!master)) {
		upslogx(LOG_WARNING, "Ignoring invalid MONITOR line in upsmon.conf!");
		upslogx(LOG_WARNING, "MONITOR configuration directives require four arguments.");
		return;
	}

	pv = strtol(pvs, (char **) NULL, 10);

	if (pv < 0) {
		upslogx(LOG_WARNING, "UPS [%s]: ignoring invalid power value [%s]", 
			sys, pvs);
		return;
	}

	last = tmp = firstups;

	while (tmp) {
		last = tmp;

		/* check for duplicates */
		if (!strcmp(tmp->sys, sys)) {
			if (reloading)
				redefine_ups(tmp, pv, un, pw, master);
			else
				upslogx(LOG_WARNING, "Warning: ignoring duplicate"
					" UPS [%s]", sys);
			return;
		}

		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(utype));
	tmp->sys = xstrdup(sys);
	tmp->pv = pv;

	/* build this up so the user doesn't run with bad settings */
	totalpv += tmp->pv;

	if (un)
		tmp->un = xstrdup(un);
	else
		tmp->un = NULL;

	tmp->pw = xstrdup(pw);
	tmp->status = 0;
	tmp->retain = 1;
	tmp->lastpoll = 0;
	tmp->lastnoncrit = 0;
	tmp->lastrbwarn = 0;
	tmp->lastncwarn = 0;

	if (!strcasecmp(master, "master"))
		setflag(&tmp->status, ST_MASTER);

	tmp->next = NULL;

	if (last)
		last->next = tmp;
	else
		firstups = tmp;

	if (tmp->pv)
		upslogx(LOG_INFO, "UPS: %s (%s) (power value %d)", tmp->sys, 
		        flag_isset(tmp->status, ST_MASTER) ? "master" : "slave",
			tmp->pv);
	else
		upslogx(LOG_INFO, "UPS: %s (monitoring only)", tmp->sys);

        upscli_splitname(tmp->sys, &tmp->upsname, &tmp->hostname, &tmp->port);
}		

static void set_notifymsg(const char *name, const char *msg)
{
	int	i;

	for (i = 0; notifylist[i].name != NULL; i++) {
		if (!strcasecmp(notifylist[i].name, name)) {

			/* only free if it's not the stock msg */
			if (notifylist[i].msg != notifylist[i].stockmsg)
				free(notifylist[i].msg);

			notifylist[i].msg = xstrdup(msg);
			return;
		}
	}

	upslogx(LOG_WARNING, "'%s' is not a valid notify event name\n", name);
}

static void set_notifyflag(const char *ntype, char *flags)
{
	int	i, pos;
	char	*ptr, *tmp;

	/* find ntype */

	pos = -1;
	for (i = 0; notifylist[i].name != NULL; i++) {
		if (!strcasecmp(notifylist[i].name, ntype)) {
			pos = i;
			break;
		}
	}

	if (pos == -1) {
		upslogx(LOG_WARNING, "Warning: invalid notify type [%s]\n", ntype);
		return;
	}

	ptr = flags;

	/* zero existing flags */
	notifylist[pos].flags = 0;

	while (ptr) {
		int	newflag;

		tmp = strchr(ptr, '+');
		if (tmp)
			*tmp++ = '\0';

		newflag = 0;

		if (!strcmp(ptr, "SYSLOG"))
			newflag = NOTIFY_SYSLOG;
		if (!strcmp(ptr, "WALL"))
			newflag = NOTIFY_WALL;
		if (!strcmp(ptr, "EXEC"))
			newflag = NOTIFY_EXEC;
		if (!strcmp(ptr, "IGNORE"))
			newflag = NOTIFY_IGNORE;

		if (newflag)
			notifylist[i].flags |= newflag;
		else
			upslogx(LOG_WARNING, "Invalid notify flag: [%s]\n", ptr);

		ptr = tmp;
	}
}

/* in split mode, the parent doesn't hear about reloads */
static void checkmode(char *cfgentry, char *oldvalue, char *newvalue, 
			int reloading)
{
	/* nothing to do if in "all as root" mode */
	if (use_pipe == 0)
		return;

	/* it's ok if we're not reloading yet */
	if (reloading == 0)
		return;

	/* also nothing to do if it didn't change */
	if ((oldvalue) && (newvalue)) {
		if (!strcmp(oldvalue, newvalue))
			return;
	}

	/* otherwise, yell at them */
	upslogx(LOG_WARNING, "Warning: %s redefined in split-process mode!",
		cfgentry);
	upslogx(LOG_WARNING, "You must restart upsmon for this change to work");
}

/* returns 1 if used, 0 if not, so conf_arg can complain about bogus configs */
static int parse_conf_arg(int numargs, char **arg)
{
	/* using up to arg[1] below */
	if (numargs < 2)
		return 0;

	/* SHUTDOWNCMD <cmd> */
	if (!strcmp(arg[0], "SHUTDOWNCMD")) {
		checkmode(arg[0], shutdowncmd, arg[1], reload);

		if (shutdowncmd)
			free(shutdowncmd);
		shutdowncmd = xstrdup(arg[1]);
		return 1;
	}

	/* POWERDOWNFLAG <fn> */
	if (!strcmp(arg[0], "POWERDOWNFLAG")) {
		checkmode(arg[0], powerdownflag, arg[1], reload);

		if (powerdownflag)
			free(powerdownflag);

		powerdownflag = xstrdup(arg[1]);

		if (!reload)
			upslogx(LOG_INFO, "Using power down flag file %s\n",
				arg[1]);

		return 1;
	}		

	/* NOTIFYCMD <cmd> */
	if (!strcmp(arg[0], "NOTIFYCMD")) {
		if (notifycmd)
			free(notifycmd);

		notifycmd = xstrdup(arg[1]);
		return 1;
	}

	/* POLLFREQ <num> */
	if (!strcmp(arg[0], "POLLFREQ")) {
		pollfreq = atoi(arg[1]);
		return 1;
	}

	/* POLLFREQALERT <num> */
	if (!strcmp(arg[0], "POLLFREQALERT")) {
		pollfreqalert = atoi(arg[1]);
		return 1;
	}

	/* HOSTSYNC <num> */
	if (!strcmp(arg[0], "HOSTSYNC")) {
		hostsync = atoi(arg[1]);
		return 1;
	}

	/* DEADTIME <num> */
	if (!strcmp(arg[0], "DEADTIME")) {
		deadtime = atoi(arg[1]);
		return 1;
	}

	/* MINSUPPLIES <num> */
	if (!strcmp(arg[0], "MINSUPPLIES")) {
		minsupplies = atoi(arg[1]);
		return 1;
	}

	/* RBWARNTIME <num> */
	if (!strcmp(arg[0], "RBWARNTIME")) {
		rbwarntime = atoi(arg[1]);
		return 1;
	}

	/* NOCOMMWARNTIME <num> */
	if (!strcmp(arg[0], "NOCOMMWARNTIME")) {
		nocommwarntime = atoi(arg[1]);
		return 1;
	}

	/* FINALDELAY <num> */
	if (!strcmp(arg[0], "FINALDELAY")) {
		finaldelay = atoi(arg[1]);
		return 1;
	}

	/* RUN_AS_USER <userid> */
 	if (!strcmp(arg[0], "RUN_AS_USER")) {
		if (run_as_user)
			free(run_as_user);

		run_as_user = xstrdup(arg[1]);
		return 1;
	}

	/* CERTPATH <path> */
	if (!strcmp(arg[0], "CERTPATH")) {
		if (certpath)
			free(certpath);

		certpath = xstrdup(arg[1]);
		return 1;
	}

	/* CERTVERIFY (0|1) */
	if (!strcmp(arg[0], "CERTVERIFY")) {
		certverify = atoi(arg[1]);
		return 1;
	}

	/* FORCESSL (0|1) */
	if (!strcmp(arg[0], "FORCESSL")) {
		forcessl = atoi(arg[1]);
		return 1;
	}

	/* using up to arg[2] below */
	if (numargs < 3)
		return 0;

	/* NOTIFYMSG <notify type> <replacement message> */
	if (!strcmp(arg[0], "NOTIFYMSG")) {
		set_notifymsg(arg[1], arg[2]);
		return 1;
	}

	/* NOTIFYFLAG <notify type> <flags> */
	if (!strcmp(arg[0], "NOTIFYFLAG")) {
		set_notifyflag(arg[1], arg[2]);
		return 1;
	}	

	/* using up to arg[4] below */
	if (numargs < 5)
		return 0;


	if (!strcmp(arg[0], "MONITOR")) {

		/* original style: no username (only 5 args) */

		if (numargs == 5) {

			/* <sys> <pwrval> <user> <pw> ("master" | "slave") */
			addups(reload, arg[1], arg[2], NULL, arg[3], arg[4]);
			return 1;
		}

		/* new style with username: 6 args */

		if (numargs == 6) {

			/* <sys> <pwrval> <user> <pw> ("master" | "slave") */
			addups(reload, arg[1], arg[2], arg[3], arg[4], arg[5]);
			return 1;
		}
	}

	/* didn't parse it at all */
	return 0;
}

/* wrap around the actual parser to detect things that aren't handled */
void conf_arg(int numargs, char **arg)
{
	if (numargs < 1)
		return;

	if (!parse_conf_arg(numargs, arg)) {
		int	i;
		char	errmsg[SMALLBUF];

		snprintf(errmsg, sizeof(errmsg), "upsmon.conf: invalid directive");

		for (i = 0; i < numargs; i++)
			snprintfcat(errmsg, sizeof(errmsg), " %s", arg[i]);

		upslogx(LOG_WARNING, errmsg);
	}
}

void conf_err(int linenum, char *errtext)
{
	if (linenum == 0)
		upslogx(LOG_ERR, "upsmon.conf: %s", errtext);
	else
		upslogx(LOG_ERR, "upsmon.conf line %d: %s", linenum, errtext);
}

static void loadconfig(void)
{
	int	ret;
	char	cfn[SMALLBUF];

	snprintf(cfn, sizeof(cfn), "%s/upsmon.conf", CONFPATH);

	ret = parseconf(cfn, conf_arg, conf_err);

	if (ret != 0) {
		if (reload == 1) {
			upslog(LOG_ERR, "Reload failed: can't open %s", cfn);
			return;
		}

		fatal("Can't parse %s", cfn);
	}
}

/* SIGPIPE handler */
static void sigpipe(int sig)
{
	debug("SIGPIPE: dazed and confused, but continuing...\n");
}

/* SIGQUIT, SIGTERM handler */
static void shut_myself_down(int sig)
{
	utype	*ups;

	upslogx(LOG_INFO, "Signal %d: exiting", sig);

	/* close all fds */
	for (ups = firstups; ups != NULL; ups = ups->next)
		upscli_disconnect(&ups->conn);

	exit(0);
}

static void user_fsd(int sig)
{
	upslogx(LOG_INFO, "User requested FSD!");

	userfsd = 1;
}

static void set_reload_flag(int sig)
{
	reload = 1;
}

/* install handlers for a few signals */
static void setupsignals(void)
{
	sigemptyset(&upsm_sigmask);
	sa.sa_mask = upsm_sigmask;
	sa.sa_flags = 0;

	sa.sa_handler = sigpipe;
	sigaction(SIGPIPE, &sa, NULL);

	sa.sa_handler = shut_myself_down;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	/* deal with the ones from userspace as well */

	sa.sa_handler = user_fsd;
	sigaction(SIGCMD_FSD, &sa, NULL);

	sa.sa_handler = set_reload_flag;
	sigaction(SIGCMD_RELOAD, &sa, NULL);
}

/* handler for alarm when getupsvarfd times out */
static void timeout(int sig)
{
	/* don't do anything here, just return */
}

/* remember the last time the ups was not critical (OB + LB) */
static void update_crittimer(utype *ups)
{
	/* if !OB or !LB, then it's not critical, so log the time */
	if ((!flag_isset(ups->status, ST_ONBATT)) || 
		(!flag_isset(ups->status, ST_LOWBATT))) {

		time(&ups->lastnoncrit);
		return;
	}

	/* fallthrough: let the timer age */
}

/* handle connecting to upsd, plus get SSL going too if possible */
static int try_connect(utype *ups)
{
	int	flags, ret;

	debug("Trying to connect to UPS [%s]\n", ups->sys);

	clearflag(&ups->status, ST_CONNECTED);

	/* force it if configured that way, just try it otherwise */
	if (forcessl == 1) 
		flags = UPSCLI_CONN_REQSSL;
	else
		flags = UPSCLI_CONN_TRYSSL;

	ret = upscli_connect(&ups->conn, ups->hostname, ups->port, flags);

	if (ret < 0) {
		upslogx(LOG_ERR, "UPS [%s]: connect failed: %s",
			ups->sys, upscli_strerror(&ups->conn));
			upsgone(ups);
		upsnocomm(ups);
		return 0;
	}

	setflag(&ups->status, ST_CONNECTED);

	/* if not doing SSL, we're done */
	if (!ups->conn.ssl)
		return 1;

	if (!certpath) {
		if (certverify == 1) {
			upslogx(LOG_ERR, "Configuration error: "
				"CERTVERIFY is set, but CERTPATH isn't");
			upslogx(LOG_ERR, "UPS [%s]: Connection impossible, "
				"dropping link", ups->sys);
			drop_connection(ups);
			upsgone(ups);
			upsnocomm(ups);
			return 0;	/* failed */
		}

		/* certverify is 0, so just warn them and return */
		upslogx(LOG_WARNING, "Certificate verification is disabled");
		return 1;
	}

	/* you REALLY should set CERTVERIFY to 1 if using SSL... */
	if (certverify == 0)
		upslogx(LOG_WARNING, "Certificate verification is disabled");

	ret = upscli_sslcert(&ups->conn, NULL, certpath, certverify);

	if (ret < 0) {
		upslogx(LOG_ERR, "UPS [%s]: SSL certificate set failed: %s",
			ups->sys, upscli_strerror(&ups->conn));
		drop_connection(ups);
		upsgone(ups);
		upsnocomm(ups);
		return 0;
	}

	setflag(&ups->status, ST_CONNECTED);

	return 1;
}

/* see what the status of the UPS is and handle any changes */
static void pollups(utype *ups)
{
	char	status[SMALLBUF], *stat, *ptr;
	struct	sigaction sa;
	sigset_t upsm_sigmask;

	/* try a reconnect here */
	if (!flag_isset(ups->status, ST_CONNECTED))
		if (try_connect(ups) != 1)
			return;

	if (ups->conn.ssl)
		debug("polling ups: %s [SSL]\n", ups->sys);
	else
		debug("polling ups: %s\n", ups->sys);

	sa.sa_handler = timeout;
	sigemptyset(&upsm_sigmask);
	sa.sa_mask = upsm_sigmask;
	sa.sa_flags = 0;
	sigaction(SIGALRM, &sa, NULL);

	alarm(10);

	if (upscli_getvar(&ups->conn, ups->upsname, "status", status, 
		sizeof(status)) == 0) {

		signal(SIGALRM, SIG_IGN);
		alarm(0);

		debug("     status: [%s]\n", status);

		/* empty response is the same as a dead ups */
		if (!strcmp(status, "")) {
			upsgone(ups);
			upsnocomm(ups);
			return;
		}

		/* clear these out early if they disappear */
		if (!strstr(status, "LB"))
			clearflag(&ups->status, ST_LOWBATT);
		if (!strstr(status, "FSD"))
			clearflag(&ups->status, ST_FSD);

		stat = status;

		while (stat != NULL) {
			ptr = strchr(stat, ' ');
			if (ptr)
				*ptr++ = '\0';

			debug("    parsing: [%s]: ", stat);

			if (!strcasecmp(stat, "OL"))
				upsonline(ups);
			if (!strcasecmp(stat, "OB"))
				upsonbatt(ups);
			if (!strcasecmp(stat, "LB"))
				upslowbatt(ups);
			if (!strcasecmp(stat, "RB"))
				upsreplbatt(ups);

			/* do it last to override any possible OL */
			if (!strcasecmp(stat, "FSD"))
				upsfsd(ups);

			update_crittimer(ups);

			stat = ptr;
		} 

		debug("\n");
		return;
	}

	/* fallthrough: no communications */
	signal(SIGALRM, SIG_IGN);
	alarm(0);

	/* try to make some of these a little friendlier */
	switch(ups->conn.upserror) {
		case UPSCLI_ERR_UNKNOWNUPS:
			if (!ups->upsname)
				upslogx(LOG_ERR, "Poll UPS [%s] failed - no "
				"UPS defined on server %s", ups->hostname);
			else
				upslogx(LOG_ERR, "Poll UPS [%s] failed - [%s] "
				"does not exist on server %s", 
				ups->sys, ups->upsname,	ups->hostname);
			break;
		default:
			upslogx(LOG_ERR, "Poll UPS [%s] failed - %s", 
				ups->sys, upscli_strerror(&ups->conn));
			break;
	}

	/* restart the connection if we had ever polled it */
	if (flag_isset(ups->status, ST_ALIVE)) {
		drop_connection(ups);
		upsgone(ups);
		upsnocomm(ups);
		return;
	}

	/* if anything worse than staleness/not defined, restart the link */
	if ((ups->conn.upserror != UPSCLI_ERR_DATASTALE) &&
		(ups->conn.upserror != UPSCLI_ERR_UNKNOWNUPS))
		drop_connection(ups);

	upsgone(ups);
	upsnocomm(ups);
}

/* remove the power down flag if it exists and is the proper form */
static void clearpdf(void)
{
	FILE	*pdf;
	char	buf[SMALLBUF];

	pdf = fopen(powerdownflag, "r");

	if (pdf == NULL)	/* no such file, nothing to do */
		return;

	/* if it exists, see if it has the right text in it */

	fgets(buf, sizeof(buf), pdf);
	fclose(pdf);

	/* reasoning: say upsmon.conf is world-writable (!) and some nasty
	 * user puts something "important" as the power flag file.  This 
	 * keeps upsmon from utterly trashing it when starting up or powering
	 * down at the expense of not shutting down the UPS.
	 *
	 * solution: don't let mere mortals edit that configuration file.
	 */

	if (!strncmp(buf, SDMAGIC, strlen(SDMAGIC)))	/* ok, it's from us */
		unlink(powerdownflag);
	else {
		upslogx(LOG_INFO, "%s doesn't seem to be a proper flag file.  Disabling.",
			  powerdownflag);
		powerdownflag = NULL;
	}
}

static void help(const char *progname)
{
	printf("Monitor UPS servers and possibly shutdown if necessary.\n\n");
	printf("usage: %s [-h] [-d] [-i] [-p] [-u <user>] [-c <cmd>]\n\n", progname);
	printf("\n");	
	printf("  -c <cmd>	send command to running process\n");	
	printf("		commands:\n");
	printf("		 - fsd: shutdown all master UPSes (use with caution)\n");
	printf("		 - reload: reread configuration\n");
	printf("		 - stop: stop monitoring and exit\n");
	printf("  -d		enable debugging code\n");
	printf("  -i		go into an infinite loop after calling shutdown\n");
	printf("  -p		always run privileged (disable privileged parent)\n");
	printf("  -u <user>	run child as user <user> (ignored when using -p)\n");
	exit(0);
}

/* set all the notify values to a default */
static void initnotify(void)
{
	int	i;

	for (i = 0; notifylist[i].name != NULL; i++) {
		notifylist[i].flags = NOTIFY_SYSLOG | NOTIFY_WALL;
		notifylist[i].msg = notifylist[i].stockmsg;
	}
}

static void runparent(int pipe)
{
	int	ret;
	char	ch;

	/* handling signals is the child's job */
	signal(SIGHUP, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	signal(SIGUSR2, SIG_IGN);

	ret = read(pipe, &ch, 1);

	if (ret < 1) {
		if (errno == ENOENT)
			fatalx("upsmon parent: exiting (child exited)");

		fatal("upsmon parent: read");
	}

	if (ch != 1)
		fatalx("upsmon parent: got bogus pipe command %c", ch);

	/* have to do this here - child is unprivileged */
	setpdflag();

	ret = system(shutdowncmd);

	if (ret != 0)
		upslogx(LOG_ERR, "parent: Unable to call shutdown command: %s\n",
	        	shutdowncmd);

	close(pipe);
	exit(0);
}

/* fire up the split parent/child scheme */
static void start_pipe(char *user)
{
	int	ret;

	ret = pipe(pipefd);

	if (ret)
		fatal("pipe creation failed");

	ret = fork();

	if (ret < 0)
		fatal("fork failed");

	/* start the privileged parent */
	if (ret != 0) {
		close(pipefd[1]);
		runparent(pipefd[0]);

		exit(1);	/* NOTREACHED */
	}

	close(pipefd[0]);

	/* write the pid file now, as we will soon lose root */
	writepid("upsmon");

	/* continuing from here is the unprivileged child, so let's drop root */
	if (user)
		switch_to_user(user, NULL);
	else
		droproot();
}

static void delete_ups(utype *target)
{
	utype	*ptr, *last;

	if (!target)
		return;

	ptr = last = firstups;

	while (ptr) {
		if (ptr == target) {
			upslogx(LOG_NOTICE, "No longer monitoring UPS [%s]\n",
				target->sys);

			/* disconnect cleanly */
			drop_connection(ptr);

			/* about to delete the first ups? */
			if (ptr == last)
				firstups = ptr->next;
			else
				last->next = ptr->next;

			/* release memory */

			if (ptr->sys)
				free(ptr->sys);

			if (ptr->upsname)
				free(ptr->upsname);
			if (ptr->hostname)
				free(ptr->hostname);
			if (ptr->pw)
				free(ptr->pw);
			free(ptr);

			return;
		}

		last = ptr;
		ptr = ptr->next;
	}

	/* shouldn't happen */
	upslogx(LOG_ERR, "delete_ups: UPS not found\n");
}	

/* see if we can open a file */
static int check_file(char *fn)
{
	char	chkfn[SMALLBUF];
	FILE	*f;

	snprintf(chkfn, sizeof(chkfn), "%s/%s", CONFPATH, fn);

	f = fopen(chkfn, "r");

	if (!f) {
		upslog(LOG_ERR, "Reload failed: can't open %s", chkfn);
		return 0;	/* failed */
	}

	fclose(f);
	return 1;	/* OK */
}

static void reload_conf(void)
{
	utype	*tmp, *next;

	upslogx(LOG_INFO, "Reloading configuration");

	/* sanity check */
	if (!check_file("upsmon.conf")) {
		reload = 0;
		return;
	}

	/* flip through ups list, clear retain value */
	tmp = firstups;

	while (tmp) {
		tmp->retain = 0;
		tmp = tmp->next;
	}

	/* reset paranoia checker */
	totalpv = 0;

	/* reread upsmon.conf */
	loadconfig();

	/* go through the utype struct again */
	tmp = firstups;

	while (tmp) {
		next = tmp->next;

		/* !retain means it wasn't in the .conf this time around */
		if (tmp->retain == 0)
			delete_ups(tmp);

		tmp = next;
	}

	/* see if the user just blew off a foot */
	if (totalpv < minsupplies) {
		upslogx(LOG_CRIT, "Fatal error: total power value (%d) less "
			"than MINSUPPLIES (%d)", totalpv, minsupplies);

		fatalx("Impossible power configuation, unable to continue");
	}

	/* finally clear the flag */
	reload = 0;
}

/* make sure the parent is still alive */
static void check_parent(void)
{
	int	ret;
	fd_set	rfds;
	struct	timeval	tv;
	time_t	now;
	static	time_t	lastwarn = 0;

	FD_ZERO(&rfds);
	FD_SET(pipefd[1], &rfds);

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	ret = select(pipefd[1] + 1, &rfds, NULL, NULL, &tv);

	if (ret == 0)
		return;

	/* this should never happen, but we MUST KNOW if it ever does */

	time(&now);

	/* complain every 2 minutes */
	if ((now - lastwarn) < 120) 
		return;

	lastwarn = now;
	do_notify(NULL, NOTIFY_NOPARENT);

	/* also do this in case the notifier isn't being effective */
	upslogx(LOG_ALERT, "Parent died - shutdown impossible");
}

int main(int argc, char *argv[])  
{
	int	i, cmd;

	cmd = 0;

	printf("Network UPS Tools upsmon %s\n", UPS_VERSION);

	while ((i = getopt(argc, argv, "+dhic:pu:")) != EOF) {
		switch (i) {
			case 'd':
				debuglevel = 1;
				break;
			case 'h':
				help(argv[0]);
				break;
			case 'i':
				playdead = 1;
				break;
			case 'p':
				use_pipe = 0;
				break;
			case 'c':
				if (!strncmp(optarg, "fsd", strlen(optarg)))
					cmd = SIGCMD_FSD;
				if (!strncmp(optarg, "stop", strlen(optarg)))
					cmd = SIGCMD_STOP;
				if (!strncmp(optarg, "reload", strlen(optarg)))
					cmd = SIGCMD_RELOAD;

				/* bad command name given */
				if (cmd == 0)
					help(argv[0]);
				break;
			case 'u':
				run_as_user = optarg;
				break;
			default:
				help(argv[0]);
				break;
		}
	}

	if (cmd) {
		sendsignal("upsmon", cmd);
		exit(0);
	}

	argc -= optind;
	argv += optind;

	openlog("upsmon", LOG_PID, LOG_FACILITY);

	initnotify();
	loadconfig();

	if (shutdowncmd == NULL)
		printf("Warning: no shutdown command defined!\n");

	/* we may need to get rid of a flag from a previous shutdown */
	if (powerdownflag != NULL)
		clearpdf();

	if (totalpv < minsupplies) {
		printf("\nFatal error: insufficient power configured!\n\n");

		printf("Sum of power values........: %d\n", totalpv);
		printf("Minimum value (MINSUPPLIES): %d\n", minsupplies);

		printf("\nEdit your upsmon.conf and change the values.\n");
		exit(1);
	}

	if (debuglevel < 1)
		background();

	/* === root parent and unprivileged child split here === */

	/* only do the pipe stuff if the user hasn't disabled it */
	if (use_pipe)
		start_pipe(run_as_user);
	else {
		upslog(LOG_INFO, "Not using split process model.");
		writepid("upsmon");
	}

	/* ignore sigpipes before the connection opens */
	setupsignals();

	/* reopen the log for the child process */
	closelog();
	openlog("upsmon", LOG_PID, LOG_FACILITY);

	for (;;) {
		utype	*ups;

		/* check flags from signal handlers */
		if (userfsd)
			forceshutdown();

		if (reload)
			reload_conf();

		sleepval = pollfreq;
		for (ups = firstups; ups != NULL; ups = ups->next)
			pollups(ups);

		recalc();

		/* make sure the parent hasn't died */
		if (use_pipe)
			check_parent();

		/* reap children that have exited */
		waitpid(-1, NULL, WNOHANG);

		sleep(sleepval);
	}

	/* NOTREACHED */
}
