/*vim ts=4*/

/*
 * NUT Oneac EG and ON model specific drivers for UPS units using
 * the Oneac Advanced Interface.  If your UPS is equipped with the
 * Oneac Basic Interface, use the genericups driver
*/

/*
   Copyright (C) 2002  by Eric Lawson <elawson@inficad.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
 * 13 Sept 2002.  Eric Lawson
 *	added BTEST0, BTEST1, RESETVOLT and SIMPWF instant commands.
 *		RESETVOLT resets the max and min measured voltage that
 *		has been measured by the UPS.
 *
 * 11 Sept 2002.  Eric Lawson
 *	Cleaned up code to take full advantage of upssend().
 *
 *	Added support for gathering ON specific values.
 *	The driver reports nominal utility voltage, nominal operating
 *	frequency, per cent of full load, current utility voltage,
 *	minimum and maximum utility voltage since the UPS last shut
 *	down, output UPS voltage, and if the UPS is boosting low utility
 *	voltage.
 *
 * 08 Sept 2002.
 *	 Initial release.
*/

#include "main.h"
#include "oneac.h"

void do_battery_test(void)
{
	char buffer[256];

	if (getval("testtime") == NULL)
		strlcpy(buffer, DEFAULT_BAT_TEST_TIME, 3);
	else {
		strlcpy(buffer, getval("testtime"),3);

	/*the UPS wants this value to always be two characters long*/
	/*so put a zero in front of the string, if needed....      */
		if (strlen(buffer) < 2) {
			buffer[2] = '\0';
			buffer[1] = buffer[0];
			buffer[0] = '0';
		}
	}
	upssend("%s%s%s",BAT_TEST_PREFIX,buffer,COMMAND_END);
}


/****************************************************************
 *below are the commands that are called by main (part of the   *
 *Above, are functions used only in this oneac driver           *
 ***************************************************************/

void instcmd (int auxcmd, int dlen, char *data)
{
	switch (auxcmd) {
		case CMD_SIMPWF :
			upssend("%s%s",SIM_PWR_FAIL,COMMAND_END);
			break;
		case CMD_BTEST1 :
			do_battery_test();
			break;
		case CMD_BTEST0 :
			upssend("%s00%s",BAT_TEST_PREFIX,COMMAND_END);
			break;
		case CMD_RSTMINMAX :
			upssend("%c%s",RESET_MIN_MAX, COMMAND_END);
			break;
		default:
			upslogx(LOG_INFO,"Oneac instcmd: unknown type 0x%04x", auxcmd);
	}
}


void upsdrv_initinfo(void)
{
	char buffer[256];

	addinfo(INFO_MFR, "ONEAC", 0, 0);
	addinfo(INFO_MODEL, "", 0, 0);
	addinfo(INFO_FIRMREV, "", 0, 0);
	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_XFERWHY, "", 0,0);
/*	addinfo(INFO_PDNGRACE, "", 0, 0); */
	addinfo(INFO_ACFREQ, "", 0, 0);
	addinfo(INFO_ALRM_GENERAL, "", 0, 0);

	addinfo(INFO_INSTCMD, "",0, CMD_BTEST1); /*start battery test*/
	addinfo(INFO_INSTCMD, "",0, CMD_BTEST0); /*stop battery test*/
	addinfo(INFO_INSTCMD, "",0, CMD_SIMPWF); /*15 second battery test*/
	addinfo(INFO_INSTCMD, "",0, CMD_RSTMINMAX);	/*reset min & max voltage */

	/*set some stuff that shouldn't change after initialization*/
	/*this stuff is common to both the EG and ON family of UPS */

	/*firmware revision*/
	upssend ("%c%s",GET_VERSION, COMMAND_END);
	upsrecv (buffer, sizeof(buffer),ENDCHAR,IGNCHARS);
	setinfo (INFO_FIRMREV, "%.3s",buffer);

	/*nominal AC frequency setting --either 50 or 60*/
	upssend ("%c%s",GET_NOM_FREQ,COMMAND_END);
	upsrecv (buffer, sizeof(buffer), ENDCHAR,IGNCHARS);
	setinfo (INFO_ACFREQ,"%.2s", buffer);


	/*UPS Family */

	upssend ("%c%s",GET_FAMILY,COMMAND_END);
	upsrecv (buffer, sizeof(buffer), ENDCHAR, IGNCHARS);
	setinfo (INFO_MODEL, "%.2s",buffer);
	printf("Found %.2s family of Oneac UPS\n", buffer);

	if (strncmp(buffer,FAMILY_ON,2) || strncmp(buffer,FAMILY_EG,2) == 0)
		printf("Unknown family of UPS. Assuming EG capabilities.\n");

	upsh.instcmd = instcmd;

	writeinfo();

	/*The ON series of UPS supports more stuff than does the EG.
	 *Take care of the ON only stuff here
	*/
	if(strncmp (getdata(INFO_MODEL), FAMILY_ON, 2) == 0) {
		addinfo(INFO_OUTVLTSEL,"", 0, 0);	/* selected output voltage*/
		addinfo(INFO_LOADPCT,"", 0, 0);		/* % of full power output*/
		addinfo(INFO_RUNTIME,"",0,0);		/* remaining run time*/
		addinfo(INFO_UTILITY,"",0,0);		/* current utility voltage*/
		addinfo(INFO_MINUTIL,"",0,0);		/*min utility voltage*/
		addinfo(INFO_MAXUTIL,"",0,0);		/*max utility voltage*/
		addinfo(INFO_OUTVOLT,"",0,0);		/*server outlet voltage*/

		/*now set the ON specific "static" parameters*/

			/*nominal input voltage*/

		upssend("%c%s", GET_NOM_VOLTAGE, COMMAND_END);
		upsrecv(buffer,sizeof(buffer),ENDCHAR,IGNCHARS);
		if(buffer[0] == '1') setinfo(INFO_OUTVLTSEL,"120");
		else if (buffer[0] == '2') setinfo(INFO_OUTVLTSEL, "240");
		else upslogx(LOG_INFO,"Oneac: Invalid voltage parameter from UPS");

		writeinfo();
	}
}

void upsdrv_updateinfo(void)
{
	char buffer[256];

	upssend ("%c%s",GET_ALL,COMMAND_END);
	upsrecv (buffer, sizeof(buffer), ENDCHAR, IGNCHARS);

	upsdebugx (2,"upsrecv_updateinfo: upsrecv returned: %s\n",buffer);

	status_init();
	/*take care of the UPS status information*/
	switch (buffer[12]) {
		case NORMAL :
			status_set("OL");
			break;
		case ON_BAT_LOW_LINE :
		case ON_BAT_HI_LINE  :
			status_set("OB");
			break;
		case LO_BAT_LOW_LINE :
		case LO_BAT_HI_LINE  :
			status_set("OB LB");
			break;
		case TOO_HOT :
			status_set("OVER OB LB");
			break;
		case FIX_ME :
			setinfo (INFO_ALRM_GENERAL, "ALARM ");
			break;
		case BAD_BAT :
			status_set("RB");
			break;
		default :				/*cry for attention, fake a status*/
								/*Would another status be better?*/
			upslogx (LOG_ERR, "Oneac: Unknown UPS status");
			status_set("OL");
	}

	/*take care of the reason why the UPS last transfered to battery*/
	switch (buffer[13]) {
		case XFER_BLACKOUT :
			setinfo (INFO_XFERWHY, "Blackout");
			break;
		case XFER_LOW_VOLT :
			setinfo (INFO_XFERWHY, "Low Input Voltage");
			break;
		case XFER_HI_VOLT :
			setinfo (INFO_XFERWHY, "High Input Voltage");
			break;
		case NO_VALUE_YET :
			setinfo (INFO_XFERWHY, "No transfer yet.");
			break;
		default :
			upslogx(LOG_INFO,"Oneac: Unknown reason for UPS battery transfer");
	}
	/* now update info for only the ON family of UPS*/

	if (strncmp(getdata(INFO_MODEL), FAMILY_ON, 2) == 0) {
		setinfo(INFO_LOADPCT,"0%.2s",buffer+31);

		/*run time left. */

/* I've commented this out until I either figure out how to convert
 * to an actual time, or I decide I can't do that and remove this code

		if(buffer[10] == YES)
			setinfo(INFO_RUNTIME,"0%.2s",buffer+33);
		else setinfo(INFO_RUNTIME,"100");
*/

		setinfo(INFO_UTILITY,"%.3s",buffer+35);
		setinfo(INFO_MINUTIL,"%.3s",buffer+38);
		setinfo(INFO_MAXUTIL,"%.3s",buffer+41);
		setinfo(INFO_OUTVOLT,"%.3s",buffer+44);
		if (buffer[47] == YES) status_set("BOOST");
	}

	status_commit();
	writeinfo();

}

void upsdrv_shutdown(void)
{
	upssend (SHUTDOWN);
}


void upsdrv_help(void)
{
	printf("\n---------\nNOTE:\n");
	printf("You must set the UPS interface card DIP switch to 9600BPS\n");
}

void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE,"testtime",
		"Change battery test time from 2 minute default.");
}

void upsdrv_banner(void)
{
printf("Network UPS Tools - Oneac EG/ON UPS driver 0.3 (%s)\n\n", UPS_VERSION);
experimental_driver = 1;		/*causes a warning to be printed*/
}

void upsdrv_initups(void)
{
	char buffer[256];
	
	open_serial(device_path, B9600);

	/*Is there anybody out there?*/

	upssend (COMMAND_END);
	sleep(1);
	upssend (COMMAND_END);
	sleep(1);
	upssend("%c%s",GET_MFR, COMMAND_END);
	upsrecv (buffer, sizeof(buffer), ENDCHAR, IGNCHARS);
	if (strncmp(buffer, MFGR, sizeof(MFGR)))
		fatalx("Unable to contact ONEAC UPS on %s\n",device_path);
}

/* tell main how many items you need */
int upsdrv_infomax(void)
{
	return 64;
}
