/* user.c - user handling functions for upsd

   Copyright (C) 2001  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "upsd.h"
#include "user.h"

#include "parseconf.h"

	extern	instcmds_t instcmds[];
	extern	acltype	*firstacl;
	ulist_t	*users = NULL;

	static	int	oldfilewarn = 0;
	static	ulist_t	*curr_user;

/* create a new user entry */
static void user_add(const char *un)
{
	ulist_t	*tmp, *last;

	if (!un)
		return;
	
	tmp = last = users;

	while (tmp) {
		last = tmp;

		if (!strcmp(tmp->username, un)) {
			fprintf(stderr, "Ignoring duplicate user %s\n", un);
			return;
		}

		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(ulist_t));
	tmp->username = xstrdup(un);
	tmp->firstacl = NULL;
	tmp->password = NULL;
	tmp->firstcmd = NULL;
	tmp->firstaction = NULL;
	tmp->next = NULL;

	if (last)
		last->next = tmp;
	else
		users = tmp;	

	/* remember who we're working on */
	curr_user = tmp;
}

static acllist *addallow(acllist *base, const char *acl)
{
	acllist	*tmp, *last;

	if (!acl)
		return base;

	tmp = last = base;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(acltype));
	tmp->aclname = xstrdup(acl);
	tmp->next = NULL;

	if (last) {
		last->next = tmp;
		return base;
	}

	return tmp;
}

/* attach allowed hosts to user */
static void user_add_allow(const char *host)
{
	if (!curr_user) {
		upslogx(LOG_WARNING, "Ignoring allowfrom definition outside "
			"user section");
		return;
	}

	curr_user->firstacl = addallow(curr_user->firstacl, host);
}

/* set password */
static void user_password(const char *pw)
{
	if (!curr_user) {
		upslogx(LOG_WARNING, "Ignoring password definition outside "
			"user section");
		return;
	}

	if (!pw)
		return;

	if (curr_user->password) {
		fprintf(stderr, "Ignoring duplicate password for %s\n", 
			curr_user->username);
		return;
	}

	curr_user->password = xstrdup(pw);
}

static instcmdlist *addinstcmd(instcmdlist *base, const char *cmd)
{
	instcmdlist	*tmp, *last;
	int	cmdnum = -1, i;

	if (!cmd)
		return base;

	tmp = last = base;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	/* 'all' is a special case */
	if (!strcasecmp(cmd, "all"))
		cmdnum = CMD_ALL;
	else {
		for (i = 0; instcmds[i].name != NULL; i++)
			if (!strcasecmp(instcmds[i].name, cmd)) {
				cmdnum = instcmds[i].cmd;
				break;
			}		
	}

	if (cmdnum == -1) {
		fprintf(stderr, "Ignoring unknown command name: %s\n", cmd);
		return base;
	}

	tmp = xmalloc(sizeof(instcmdlist));
	tmp->cmd = cmdnum;
	tmp->next = NULL;

	if (last) {
		last->next = tmp;
		return base;
	}

	return tmp;
}

/* attach allowed instcmds to user */
static void user_add_instcmd(const char *cmd)
{
	if (!curr_user) {
		upslogx(LOG_WARNING, "Ignoring instcmd definition outside "
			"user section");
		return;
	}

	if (!cmd)
		return;

	curr_user->firstcmd = addinstcmd(curr_user->firstcmd, cmd);
}

static actionlist *addaction(actionlist *base, const char *action)
{
	actionlist	*tmp, *last;

	if (!action)
		return base;

	tmp = last = base;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(actionlist));
	tmp->action = xstrdup(action);
	tmp->next = NULL;

	if (last) {
		last->next = tmp;
		return base;
	}

	return tmp;
}

/* attach allowed actions to user */
static void user_add_action(const char *act)
{
	if (!curr_user) {
		upslogx(LOG_WARNING, "Ignoring action definition outside "
			"user section");
		return;
	}

	curr_user->firstaction = addaction(curr_user->firstaction, act);
}

static void flushacl(acllist *first)
{
	acllist	*ptr, *next;

	ptr = first;

	while (ptr) {
		next = ptr->next;

		if (ptr->aclname)
			free(ptr->aclname);

		free(ptr);
		ptr = next;
	}
}

static void flushcmd(instcmdlist *first)
{
	instcmdlist	*ptr, *next;

	ptr = first;

	while (ptr) {
		next = ptr->next;

		free(ptr);

		ptr = next;
	}
}

static void flushaction(actionlist *first)
{
	actionlist	*ptr, *next;

	ptr = first;

	while (ptr) {
		next = ptr->next;

		if (ptr->action)
			free(ptr->action);
		free(ptr);

		ptr = next;
	}
}

/* flush all user attributes - used during reload */
void user_flush(void)
{
	ulist_t	*ptr, *next;

	ptr = users;

	while (ptr) {
		next = ptr->next;

		if (ptr->username)
			free(ptr->username);

		if (ptr->firstacl)
			flushacl(ptr->firstacl);

		if (ptr->password)
			free(ptr->password);

		if (ptr->firstcmd);
			flushcmd(ptr->firstcmd);

		if (ptr->firstaction);
			flushaction(ptr->firstaction);

		free(ptr);

		ptr = next;
	}

	users = NULL;
}	

int user_matchacl(ulist_t *user, struct sockaddr_in *addr)
{
	acllist	*tmp = user->firstacl;

	/* no acls means no access (fail-safe) */
	if (!tmp)
		return 0;		/* good */

	while (tmp) {
		if (checkacl(tmp->aclname, addr) == 1)
			return 1;	/* good */
	
		tmp = tmp->next;
	}

	return 0;	/* fail */
}

int user_matchinstcmd(ulist_t *user, int cmd)
{
	instcmdlist	*tmp = user->firstcmd;

	/* no commands means no access (fail-safe) */
	if (!tmp)
		return 0;	/* fail */

	while (tmp) {
		if ((tmp->cmd == cmd) || (tmp->cmd == CMD_ALL))
			return 1;	/* good */
		tmp = tmp->next;
	}

	return 0;	/* fail */
}

int user_checkinstcmd(struct sockaddr_in *addr, char *username, char *password,
               int cmd)
{
	ulist_t	*tmp = users;

	while (tmp) {

		/* let's be paranoid before we call strcmp */

		if ((!tmp->username) || (!tmp->password) ||
		   (!username) || (!password)) {
			tmp = tmp->next;
			continue;
		}

		if (!strcmp(tmp->username, username)) {
			if (!strcmp(tmp->password, password)) {
				if (!user_matchacl(tmp, addr))
					return 0;		/* fail */

				if (!user_matchinstcmd(tmp, cmd))
					return 0;		/* fail */

				/* passed all checks */
				return 1;	/* good */
			}

			/* password mismatch */
			return 0;	/* fail */
		}

		tmp = tmp->next;
	}		

	/* username not found */
	return 0;	/* fail */
}

int user_matchaction(ulist_t *user, char *action)
{
	actionlist	*tmp = user->firstaction;

	/* no actions means no access (fail-safe) */
	if (!tmp)
		return 0;	/* fail */

	while (tmp) {
		if (!strcasecmp(tmp->action, action))
			return 1;	/* good */
		tmp = tmp->next;
	}

	return 0;	/* fail */
}

int user_checkaction(struct sockaddr_in *addr, char *username, char *password,
                     char *action)
{
	ulist_t	*tmp = users;

	while (tmp) {

		/* let's be paranoid before we call strcmp */

		if ((!tmp->username) || (!tmp->password) ||
		   (!username) || (!password)) {
			tmp = tmp->next;
			continue;
		}

		if (!strcmp(tmp->username, username)) {
			if (!strcmp(tmp->password, password)) {

				if (!user_matchacl(tmp, addr)) {
					upsdebugx(2, "user_matchacl: failed");
					return 0;		/* fail */
				}

				if (!user_matchaction(tmp, action)) {
					upsdebugx(2, "user_matchaction: failed");
					return 0;		/* fail */
				}

				/* passed all checks */
				return 1;	/* good */
			}

			/* password mismatch */
			upsdebugx(2, "user_checkaction: password mismatch");
			return 0;	/* fail */
		}

		tmp = tmp->next;
	}		

	/* username not found */
	return 0;	/* fail */
}

/* handle "upsmon master" and "upsmon slave" for nicer configurations */
static void set_upsmon_type(char *type)
{
	/* master: login, master, fsd */
	if (!strcasecmp(type, "master")) {
		user_add_action("login");
		user_add_action("master");
		user_add_action("fsd");
		return;
	}

	/* slave: just login */
	if (!strcasecmp(type, "slave")) {
		user_add_action("login");
		return;
	}

	upslogx(LOG_WARNING, "Unknown upsmon type %s", type);
}

/* actually do something with the variable + value pairs */
static void parse_var(char *var, char *val)
{
	if (!strcasecmp(var, "password")) {
		user_password(val);
		return;
	}

	if (!strcasecmp(var, "instcmds")) {
		user_add_instcmd(val);
		return;
	}

	if (!strcasecmp(var, "actions")) {
		user_add_action(val);
		return;
	}

	if (!strcasecmp(var, "allowfrom")) {
		user_add_allow(val);
		return;
	}

	/* someone did 'upsmon = type' - allow it anyway */
	if (!strcasecmp(var, "upsmon")) {
		set_upsmon_type(val);
		return;
	}

	upslogx(LOG_NOTICE, "Unrecognized user setting %s", var);
}

/* parse first var+val pair, then flip through remaining vals */
static void parse_rest(char *var, char *fval, char **arg, int next, int left)
{
	int	i;

	/* no globals supported yet, so there's no sense in continuing */
	if (!curr_user)
		return;

	parse_var(var, fval);

	if (left == 0)
		return;

	for (i = 0; i < left; i++)
		parse_var(var, arg[next + i]);
}

void user_parseconf(int numargs, char **arg)
{
	char	*ep;

	if ((numargs == 0) || (!arg))
		return;

	/* TODO: remove this after awhile */

	/* don't attempt to parse any old files */
	if (oldfilewarn == 1)
		return;

	/* detect old file format */
	if ((!strcasecmp(arg[0], "user")) && (oldfilewarn == 0)) {
		upslogx(LOG_ERR, "Unable to parse old style upsd.users\n");
		oldfilewarn = 1;
		return;
	}

	/* handle 'foo=bar' (compressed form) */

	ep = strchr(arg[0], '=');
	if (ep) {
		*ep = '\0';

		/* parse first var/val, plus subsequent values (if any) */

		/*      0       1       2  ... */
		/* foo=bar <rest1> <rest2> ... */

		parse_rest(arg[0], ep+1, arg, 1, numargs - 1);
		return;
	}

	/* look for section headers - [username] */
	if ((arg[0][0] == '[') && (arg[0][strlen(arg[0])-1] == ']')) {
                
		arg[0][strlen(arg[0])-1] = '\0';
		user_add(&arg[0][1]);

                return;
        }

	if (numargs < 2)
		return;

	if (!strcasecmp(arg[0], "upsmon"))
		set_upsmon_type(arg[1]);

	/* everything after here needs arg[1] and arg[2] */
	if (numargs < 3)
		return;

	/* handle 'foo = bar' (split form) */
	if (!strcmp(arg[1], "=")) {

		/*   0 1   2      3       4  ... */
		/* foo = bar <rest1> <rest2> ... */

		/* parse first var/val, plus subsequent values (if any) */
		
		parse_rest(arg[0], arg[2], arg, 3, numargs - 3);
		return;
	}

	/* ... unhandled ... */
}

void user_parseerror(int linenum, char *errtext)
{
	if (linenum != 0)
		upslogx(LOG_ERR, "%s/upsd.users line %d: %s",
			CONFPATH, linenum, errtext);
	else
		upslogx(LOG_ERR, "%s/upsd.users: %s",
			CONFPATH, errtext);
}

void user_load(void)
{
	char	ufn[SMALLBUF];

	curr_user = NULL;

	snprintf(ufn, sizeof(ufn), "%s/upsd.users", CONFPATH);

	oldfilewarn = 0;
	parseconf(ufn, user_parseconf, user_parseerror);
}
