#include <stdlib.h>
#include <string.h>
#if defined (WIN32)
# include <windows.h>
#endif

#include <stdio.h>
#include <sql.h>
#include <sqlext.h>

#define IN_HELPER_C
#include "helper.h"

#define TABLE_ROWS 50


/************************************************************************/
/*                                                                      */
/*  do_create_table                                                     */
/*  ===============                                                     */
/*                                                                      */
/*  Delete the named table and create a new one.                        */
/*                                                                      */
/************************************************************************/
SQLRETURN do_create_table(
    SQLHDBC *hdbc,
    char *table_name,
    unsigned long flags,
    SQLINTEGER length,
    char *type,
    char *params)
{
    SQLHSTMT    hstmt;                          /* statement handle */
    SQLRETURN   ret;                            /* function status return */
    char        qbuf[1024];                     /* query buffer */
    
    printf("---------- do_create_table ----------\n");
    /*
     *  Allocate a statement and delete existing table.
     */
    if (SQLAllocHandle(SQL_HANDLE_STMT, hdbc, &hstmt) != SQL_SUCCESS)
    {
        do_a_error(SQL_HANDLE_DBC, hdbc, "SQLAllocHandle");
        return SQL_ERROR;
    }
    sprintf(qbuf, "drop table %s", table_name);
    printf("\"%s\"\n", qbuf);
    if (!SQL_SUCCEEDED(SQLExecDirect(hstmt, qbuf, SQL_NTS)))
    {
        do_a_error(SQL_HANDLE_STMT, hstmt, "SQLExecDirect");
    }
    if (flags & CREATE_TABLE_BIGCOL)
    {
        if (strstr(params, "length"))
        {
            sprintf(qbuf, "create table %s "
                    "(a INTEGER , b %s(%ld))",
                    table_name, type, length);
        }
        else
        {
            sprintf(qbuf, "create table %s "
                    "(a INTEGER , b %s)",
                    table_name, type);
        }
    }
    else
    {
        sprintf(qbuf, "create table %s "
                "(a INTEGER PRIMARY KEY, b CHARACTER VARYING(20))",
                table_name);
    }
    printf("\"%s\"\n", qbuf);
    ret = SQLExecDirect(hstmt, qbuf, SQL_NTS);
    if (!SQL_SUCCEEDED(ret))
        do_a_error(SQL_HANDLE_STMT, hstmt, "SQLExecDirect");
    
    ret = SQLFreeStmt(hstmt, SQL_DROP);
    if (!SQL_SUCCEEDED(ret))
    {
        do_a_error(SQL_HANDLE_STMT, hstmt, "SQLFreeStmt");
    }
    
    return ret;
}


/************************************************************************/
/*                                                                      */
/*  do_type_info                                                        */
/*                                                                      */
/************************************************************************/
SQLRETURN do_type_info(
    SQLHDBC *hdbc,
    typeinfo_t *typeinfo)
{
    struct types_s
    {
        SQLSMALLINT       type;
        char              *name;
    };
    struct types_s        types[] =
    {
        {SQL_ALL_TYPES,"ALL"},          
        {SQL_CHAR,"CHAR"},
        {SQL_NUMERIC,"NUMERIC"},
        {SQL_DECIMAL,"DECIMAL"},
        {SQL_INTEGER,"INTEGER"},
        {SQL_SMALLINT,"SMALLINT"},
        {SQL_FLOAT,"FLOAT"},
        {SQL_REAL,"REAL"},
        {SQL_DOUBLE,"DOUBLE"},
        {SQL_DATETIME,"DATETIME"},
        {SQL_VARCHAR,"VARCHAR"},
        {SQL_TIME,"TIME"},
        {SQL_TIMESTAMP,"TIMESTAMP"},
        {SQL_LONGVARCHAR,"LONGVARCHAR"},
        {SQL_BINARY,"BINARY"},
        {SQL_VARBINARY,"VARBINARY"},
        {SQL_LONGVARBINARY,"LONGVARBINARY"},
        {SQL_BIGINT,"BIGINT"},
        {SQL_TINYINT,"TINYINT"},
        {SQL_BIT,"BIT"},
        {0,NULL}
    };
    SQLRETURN           ret;                    /* function status return */
    unsigned int        t_index;                /* type index */
    SQLHSTMT            hstmt;                  /* statement handle */
    unsigned int        varchar_found=0;        /* found SQL_VARCHAR */
    unsigned int        integer_found=0;        /* found SQL_INTEGER */
    SQLSMALLINT         columns;                /* no. of resultant columns */
    
    printf("---------- do_type_info ----------\n");
    /*
     *  Allocate a statement and delete existing table.
     */
    if (SQLAllocHandle(SQL_HANDLE_STMT, hdbc, &hstmt) != SQL_SUCCESS)
    {
        do_a_error(SQL_HANDLE_DBC, hdbc, "SQLAllocHandle");
        return SQL_ERROR;
    }
    printf("\tCollecting type information with SQLGetTypeInfo\n");
    t_index = 0;
    while(types[t_index].name)
    {
        char            local_name[50];
        char            type_name[50];        
        SQLSMALLINT     sql_data_type;
        SQLINTEGER      column_size;
        char            create_params[256];
        
        if (debug) printf("%s:\n", types[t_index].name);
        ret = SQLGetTypeInfo(hstmt, types[t_index].type);
        if (!SQL_SUCCEEDED(ret))
        {
            do_a_error(SQL_HANDLE_STMT, hstmt, "SQLGetTypeInfo");
            ret = SQLCloseCursor(hstmt);
            if (!SQL_SUCCEEDED(ret))
                do_a_error(SQL_HANDLE_STMT, hstmt, "SQLCloseCursor");
            t_index++;
            continue;
        }
        if (!SQL_SUCCEEDED(SQLNumResultCols(hstmt, &columns)))
        {
            do_a_error(SQL_HANDLE_STMT, hstmt, "SQLNumResultCols");
            return SQL_ERROR;
        }
        if ((columns < 19) && (debug))
        {
            fprintf(stderr,
                    "** Can't find right number of columns in the result **\n");
            fprintf(stderr, "** Found %d columns\n", columns);
        }
        while(SQL_SUCCEEDED(ret = SQLFetch(hstmt)))
        {
            SQLINTEGER          ind;
            
            local_name[0] = '\0';
            if (!SQL_SUCCEEDED(ret = SQLGetData(hstmt, 1, SQL_C_CHAR,
                                                type_name, sizeof(type_name),
                                                &ind)))
            
            {
                do_a_error(SQL_HANDLE_STMT, hstmt, "SQLGetData");
            }
            if (!SQL_SUCCEEDED(ret = SQLGetData(hstmt, 2, SQL_C_SHORT,
                                                &sql_data_type,
                                                sizeof(sql_data_type),
                                                &ind)))
            {
                do_a_error(SQL_HANDLE_STMT, hstmt, "SQLGetData");
            }
            if (!SQL_SUCCEEDED(ret = SQLGetData(hstmt, 3, SQL_C_LONG,
                                                &column_size,
                                                sizeof(column_size),
                                                &ind)))
            {
                do_a_error(SQL_HANDLE_STMT, hstmt, "SQLGetData");
            }
            /*
             *  We are specifically interested in colun size as we can use
             *  this to choose a big data type for blob testing. However,
             *  driver can legitimately return SQL_NULL_DATA (which we will
             *  class as 0 length) and SQL_NO_TOTAL which sort of implies
             *  very big (no limit).
             */
            if (ind == SQL_NO_TOTAL) column_size = SQL_NO_TOTAL;
            if (ind == SQL_NULL_DATA) column_size = 0;
            create_params[0] = '\0';
            if (!SQL_SUCCEEDED(ret = SQLGetData(hstmt, 6, SQL_C_CHAR,
                                                create_params,
                                                sizeof(create_params),
                                                &ind)))
            {
                do_a_error(SQL_HANDLE_STMT, hstmt, "SQLGetData");
            }
            if (!SQL_SUCCEEDED(ret = SQLGetData(hstmt, 13, SQL_C_CHAR,
                                                local_name, sizeof(local_name),
                                                &ind)))
            {
                do_a_error(SQL_HANDLE_STMT, hstmt, "SQLGetData");
            }
            if (debug)
            {
                if (column_size == SQL_NO_TOTAL)
                {
                    printf("%20s %20s %d oo (%16s)\n",
                           type_name, local_name, sql_data_type, create_params);
                }
                else
                {
                    printf("%20s %20s %d %ld (%16s)\n",
                           type_name, local_name, sql_data_type, column_size,
                           create_params);
                }
            }
            if (types[t_index].type == SQL_VARCHAR) varchar_found = 1;
            if (types[t_index].type == SQL_VARCHAR) integer_found = 1;
            if (!strcmp(types[t_index].name, "ALL"))
            {
                if (typeinfo)
                {
                    strcpy(typeinfo->local_name, local_name);
                    strcpy(typeinfo->type_name, type_name);
                    strcpy(typeinfo->create_params, create_params);
                    typeinfo->sql_data_type = sql_data_type;
                    typeinfo->column_size = column_size;
                    typeinfo++;
                }
            }
        }
        if (ret != SQL_NO_DATA)
        {
            do_a_error(SQL_HANDLE_STMT, hstmt, "SQLFetch");
        }
        ret = SQLCloseCursor(hstmt);
        if (!SQL_SUCCEEDED(ret))
            do_a_error(SQL_HANDLE_STMT, hstmt, "SQLCloseCursor");
        t_index++;
    }
    if (debug) printf("\n");

    ret = SQLFreeStmt(hstmt, SQL_DROP);
    if (!SQL_SUCCEEDED(ret))
    {
        do_a_error(SQL_HANDLE_STMT, hstmt, "SQLFreeStmt");
    }
    if (!varchar_found || !integer_found)
    {
        fprintf(stderr,
                "** Can't seem to locate the INTEGER and VARCHAR types **\n");
        return SQL_ERROR;
    }
    return ret;
    
}


/************************************************************************/
/*                                                                      */
/*  do_a_error                                                          */
/*                                                                      */
/************************************************************************/
SQLRETURN do_a_error(
    SQLSMALLINT type,
    SQLHANDLE handle,
    char *fn)
{
    SQLRETURN           ret;
    char                state[7];
    SQLCHAR             text[1024];
    SQLSMALLINT         len;
    int                 i=0;
    SQLINTEGER          native;
    char                cbuf[2048];
    SQLINTEGER          ndiags;
    
    printf("** Error from %s **\n", fn);
    ret = SQLGetDiagField(type, handle, 0, SQL_DIAG_NUMBER, &ndiags, 0, NULL);
    printf("%ld diagnostics found\n", ndiags);
    do
    {
        ret = SQLGetDiagRec(type, handle, ++i, state,
                            &native, text, sizeof(text), &len);
        if (SQL_SUCCEEDED(ret))
        {
            sprintf(cbuf, "** error: %s:%d:%ld:%s **\n",
                    state, i, native, text);
            fprintf(stderr, cbuf);
        }
    }
    while (SQL_SUCCEEDED(ret));
    return ret;
}
    

/************************************************************************/
/*                                                                      */
/*  do_get_info                                                         */
/*  ===========                                                         */
/*                                                                      */
/************************************************************************/
SQLRETURN do_get_info(
    SQLHDBC *hdbc,
    SQLUINTEGER *array_row_counts,
    SQLUINTEGER *array_selects,
    SQLUINTEGER *static_ca1,
    SQLUINTEGER *static_ca2)
{
    SQLRETURN           ret;

    printf("---------- do_get_info ----------\n");
    ret = SQLGetInfo(hdbc, SQL_PARAM_ARRAY_ROW_COUNTS, array_row_counts, 0, NULL);
    if (!SQL_SUCCEEDED(ret))
        do_a_error(SQL_HANDLE_DBC, hdbc, "SQLGetInfo");
    if (*array_row_counts == SQL_PARC_BATCH)
    {
        printf("Driver: SQL_PARAM_ARRAY_ROW_COUNTS = SQL_PARC_BATCH\n");
    }
    else if (*array_row_counts == SQL_PARC_NO_BATCH)
    {
        printf("Driver: SQL_PARAM_ARRAY_ROW_COUNTS = SQL_PARC_BATCH\n");
    }
    else
    {
        printf("Driver: SQL_PARAM_ARRAY_ROW_COUNTS = unknown, %lu\n",
               *array_row_counts);
    }

    ret = SQLGetInfo(hdbc, SQL_PARAM_ARRAY_SELECTS, array_selects, 0, NULL);
    if (!SQL_SUCCEEDED(ret))
        do_a_error(SQL_HANDLE_DBC, hdbc, "SQLGetInfo");
    if (*array_selects == SQL_PAS_BATCH)
    {
        printf("Driver: SQL_PARAM_ARRAY_SELECTS = SQL_PAS_BATCH\n");
    }
    else if (*array_selects == SQL_PAS_NO_BATCH)
    {
        printf("Driver: SQL_PARAM_ARRAY_SELECTS = SQL_PAS_BATCH\n");
    }
    else if (*array_selects == SQL_PAS_NO_SELECT)
    {
        printf("Driver: SQL_PARAM_ARRAY_SELECTS = SQL_PAS_NO_SELECT\n");
    }
    else
    {
        printf("Driver: SQL_PARAM_ARRAY_SELECTS = unknown, %lu\n",
               *array_selects);
    }

    if (static_ca1)
    {
        ret = SQLGetInfo(hdbc, SQL_STATIC_CURSOR_ATTRIBUTES1, static_ca1,
                         0, NULL);
        if (!SQL_SUCCEEDED(ret))
            do_a_error(SQL_HANDLE_DBC, hdbc, "SQLGetInfo");
        if (*static_ca1 & SQL_CA1_POS_POSITION)
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says SQL_CA1_POS_POSITION"
                   "\nand so a static cursor can be positioned with SQLSetPos\n");
        }
        else
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says !SQL_CA1_POS_POSITION"
                   "\nand so a static cursor can NOT be positioned with SQLSetPos\n");
        }
        if (*static_ca1 & SQL_CA1_POS_UPDATE)
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says SQL_CA1_POS_UPDATE"
                   "\nand so a static cursor can be used to SQL_UPDATE with SQLSetPos\n");
        }
        else
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says !SQL_CA1_POS_UPDATE"
                   "\nand so a static cursor can NOT be used to SQL_UPDATE with SQLSetPos\n");
        }
        if (*static_ca1 & SQL_CA1_POS_DELETE)
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says SQL_CA1_POS_DELETE"
                   "\nand so a static cursor can be used to SQL_DELETE with SQLSetPos\n");
        }
        else
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says !SQL_CA1_POS_DELETE"
                   "\nand so a static cursor can NOT be used to SQL_DELETE with SQLSetPos\n");
        }
        if (*static_ca1 & SQL_CA1_POS_REFRESH)
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says SQL_CA1_POS_REFRESH"
                   "\nand so a static cursor can be used to SQL_REFRESH with SQLSetPos\n");
        }
        else
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES1 says !SQL_CA1_POS_REFRESH"
                   "\nand so a static cursor can NOT be used to SQL_REFRESH with SQLSetPos\n");
        }
        
    }
    if (static_ca2)
    {
        ret = SQLGetInfo(hdbc, SQL_STATIC_CURSOR_ATTRIBUTES2, static_ca2,
                         0, NULL);
        if (!SQL_SUCCEEDED(ret))
            do_a_error(SQL_HANDLE_DBC, hdbc, "SQLGetInfo");
        if (*static_ca2 & SQL_CA2_SENSITIVITY_ADDITIONS)
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES2 says SQL_CA2_SENSITIVITY_ADDITIONS"
                   "\nand so added rows are visible with a static cursor\n");
        }
        else
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES2 says !SQL_CA2_SENSITIVITY_ADDITIONS"
                   "\nand so added rows NOT are visible with a static cursor\n");
        }
        if (*static_ca2 & SQL_CA2_SENSITIVITY_DELETIONS)
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES2 says SQL_CA2_SENSITIVITY_DELETIONS"
                   "\nand so deleted rows are visible with a static cursor\n");
        }
        else
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES2 says !SQL_CA2_SENSITIVITY_DELETIONS"
                   "\nand so deleted rows NOT are visible with a static cursor\n");
        }
        if (*static_ca2 & SQL_CA2_SENSITIVITY_UPDATES)
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES2 says SQL_CA2_SENSITIVITY_UPDATES"
                   "\nand so updated rows are visible with a static cursor\n");
        }
        else
        {
            printf("Driver: SQL_STATIC_CURSOR_ATTRIBUTES2 says !SQL_CA2_SENSITIVITY_UPDATES"
                   "\nand so updated rows NOT are visible with a static cursor\n");
        }
    }
    
    return ret;
    
}    
