/*
#       lidsadm.c --- The Linux Intrusion Detection System Configuration Tool 
#       (C) Huagang Xie 1999-2001 All rights reserved.
#       EMail complaints to xie@lids.org 
#
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 2 of the License, or
#       (at your option) any later version.
#
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
*/
/* $Id: lidsconf.c,v 1.3 2004/02/21 22:41:22 xie Exp $ */
/* ------------------------------------------------------------------------- */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <../config.h>
#endif

/*#define _GNU_SOURCE*/
#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <getopt.h>
/* in order to remode the warning message */
#undef __attribute_used__
#undef __attribute_pure__
#include <linux/capability.h>
#include <linux/kdev_t.h>

#include "lidstools.h"
#include "lids_capflag.h"

#include "lidsif.h"

#define LIDS_STR2(x) #x
#define LIDS_STR(X) LIDS_STR2(X)

#ifdef DEBUG
#define LIDS_DBG(msg...)  printf("LIDS." __FUNCTION__ ".l" LIDS_STR(__LINE__) ": " ##msg)
#else
#define LIDS_DBG(msg...)
#endif

//#define PRINT_ACL(acl)        printf("inode= %d, dev= %d, file= %s, type= %d, inherit= %d, o_inode= %ld, o_dev= %d, o_file= %s, from:%ld, to_%ld, state: %d\n", acl.s_ino, acl.s_dev, acl.s_file, acl.type, acl.inherit, acl.o_ino, acl.o_dev, acl.o_file, acl.time[0][0], acl.time[1][0],acl.state)

//#define PRINT_ACL(acl)        printf("subject(%d,%d,%s), type= %d, object(%ld,%d,%s), state=%d\n", acl.s_ino, acl.s_dev, acl.s_file, acl.type, acl.inherit, acl.o_ino, acl.o_dev, acl.o_file, acl.time[0][0], acl.time[1][0],acl.state)
#define PRINT_ACL(acl)	\
	printf("\n%23s  %8s%s:%3d  %s  %20s %s%d\n\n",	\
	acl.s_file,						\
	lids_type_desc[get_type_str(abs(acl.type))],	\
	"",							\
	lids[i].inherit,					\
	time2str(lids[i].time),						\
	acl.o_file,										\
	strncmp("CAP_NET_BIND_SERVICE",acl.o_file,20) ?""  : disp_multi_data(lids[i].port,1) ,	\
	strncmp("LIDS_SOCKET_NF_MARK",acl.o_file,19)?0:lids[i].mark )

#define LIDS_WARNING(string) \
{	\
	if(verbose) {\
	fprintf(stderr,"-----------------------------------------------------------------------------\n"); \
	fprintf(stderr, string); \
	PRINT_ACL(lids[i]); \
	fprintf(stderr,"-----------------------------------------------------------------------------\n");\
	}\
}

/* ------------------------------------------------------------------------- */
char lids_cap_file[4][30] =
    { LIDS_GLOBAL_CAP_FILE, LIDS_BOOT_CAP_FILE, LIDS_POSTBOOT_CAP_FILE,
LIDS_SHUTDOWN_CAP_FILE };
char lids_acl_file[4][29] =
    { LIDS_GLOBAL_CONF_FILE, LIDS_BOOT_CONF_FILE, LIDS_POSTBOOT_CONF_FILE,
LIDS_SHUTDOWN_CONF_FILE };

/*  globals */
#define LIDS_TARGET_NUM 	8
#define LIDS_TYPE_NUM		7

char lids_type_desc[LIDS_TARGET_NUM][9] =
    { "DENY", "READONLY", "APPEND", "WRITE", "IGNORE", "DISABLE", "GRANT",
"ENABLE" };

int lids_type_val[LIDS_TYPE_NUM] = { LIDS_DENY,
	LIDS_DENY | LIDS_READONLY,
	LIDS_DENY | LIDS_READONLY | LIDS_APPEND,
	LIDS_DENY | LIDS_READONLY | LIDS_APPEND | LIDS_WRITE,
	LIDS_DENY | LIDS_READONLY | LIDS_APPEND | LIDS_WRITE,
	LIDS_SOCKET,
	LIDS_CAP
};

char lids_special_target_val[2] = { 0, 1 };

lids_t eft_lids[1024];
lids_t lids[1024];
static int last, eft_last;
int bindport[LIDS_PORT_ITEM][2];
int mark;
char acl_conf_file[256];
int state;
int lids_caps;
int verbose = 0;

/* ------------------------------------------------------------------------- */

void
exit_error(int status, char *msg)
{
	fprintf(stderr, "lidsconf: %s\n", msg);
	if (status == 3)
		perror("reason:");
	printf("\n");
	exit(status);
}

void
exit_version()
{
	printf("lidsconf version " VERSION " for the LIDS project\n");
	exit(1);
}

void
exit_normal()
{
	printf("lidsconf version " VERSION " for the LIDS project\n"
	       "Use 'lidsconf -h' for help\n");
	exit(1);
}

void
exit_help(int more)
{
	entry_t *entry;

	printf("lidsconf version " VERSION " for the LIDS project\n"
	       "       Huagang Xie <xie@lids.org>\n"
	       "       Philippe Biondi <philippe.biondi@webmotion.net>\n\n"
	       "Usage: lidsconf -A [acl_type] [-s subject] -o object [-d] [-t from-to] [-i level] -j ACTION\n"
	       "       lidsconf -D [acl_type] [-s file] [-o file] \n"
	       "       lidsconf -Z [acl_type]\n"
	       "       lidsconf -U\n"
	       "       lidsconf -L [acl_type] [-e]\n"
	       "       lidsconf -P\n"
	       "       lidsconf -v\n"
	       "       lidsconf -[h|H]\n\n"
	       "Commands:\n"
	       "       -A,--add	To add an entry\n"
	       "       -D,--delete	To delete an entry\n"
	       "       -Z,--zero	To delete all entries \n"
	       "       -U,--update	To update dev/inode numbers\n"
	       "       -L,--list	To list all entries \n"
	       "       -P,--passwd	To encrypt a password with RipeMD-160\n"
	       "       -v,--version	To show the version\n"
	       "       -h,--help	To list this help \n"
	       "       -H,--morehelp	To list this help with CAP/SOCKET name\n\n"
	       "subject: -s,--subject subj\n"
	       "       can be any program, must be a file\n"
	       "object: -o,--object [obj]\n"
	       "       can be a file, directory or Capability, Socket Name\n"
	       ""
	       "ACTION: -j,--jump\n"
	       "       DENY     deny access\n"
	       "       READONLY read only\n"
	       "       APPEND   append only\n"
	       "       WRITE    writable\n"
	       "       GRANT    grant capability to subject\n"
	       "       IGNORE   ignore any permissions set on this object\n"
	       "       DISABLE  disable some extersion feature\n"
	       "OPTION:\n"
	       "      -i,--inheritance Inheritance level\n"
	       "      -t,--time	Time dependency\n"
	       "      -e,--extended	Extended list\n");
	if (more) {
		printf("\nAvailable capabilities:\n");
		for_each_entry(cap_list, entry)
		    printf("%20s %s\n", entry->name, entry->desc);

		printf("\nAvailable Socket:\n");
		for_each_entry(socket_list, entry)
		    printf("%20s %s\n", entry->name, entry->desc);
	}

	exit(1);
}

void
lids_del_all()
{
	FILE *fd;

	if ((fd = fopen(acl_conf_file, "w")) == NULL) {
		exit_error(3, "cannot open acl file");
	}
	fwrite(LIDS_CONF_COMMENT, 1, strlen(LIDS_CONF_COMMENT), fd);
	fclose(fd);
}

/*
 *	lids_get_str get the ino,dev,filename from the buff.
 *	the format of buffer is "inode:dev:filename"
 */
int
lids_get_str(char *buffer, unsigned long *ino, int *dev, char *filename)
{
	char *p, *q;
	char ino_str[1024];
	char ino_dev[1024];

	memset(ino_str, '\0', 1024);
	memset(ino_dev, '\0', 1024);

	p = q = NULL;
	p = strchr(buffer, ':');
	if (p == NULL)
		return -1;
	*p = '\0';
	strncpy(ino_str, buffer, p - buffer);
	p++;
	q = strchr(p + 1, ':');
	if (q == NULL)
		return -1;
	strncpy(ino_dev, p, q - p);

	strcpy(filename, q + 1);
	*ino = atol(ino_str);
	*dev = atoi(ino_dev);
	return 0;
}

/*
 * check_format check if the given buf is the valid string for current version
 */
int
check_format(char *buf)
{
	int i, j;
	char *p = NULL;

	i = 0;
	p = strchr(buf, ':');
	while (p != NULL) {
		p = strchr(p + 1, ':');
		i++;
	}
	if (i != 8)
		return -1;
	return 0;
}

/*
 *	input: lids[1024] , include global acl and one state acl.
 *
 *  	output:	lids_effective[1024], include the real effective rules.
 *		
 *	side effect, warning, return the wrong rules 	
 */
void
compute_effective_rules(void)
{
	int curr_state = LIDS_STATE_GLOBAL;
	int i, default_rule, err;

	eft_last = 0;

	/* should check for all the ino = 0 rule first */
	for (i = 0; i < last; i++) {
		if (lids[i].s_ino == 0) {
			err = check_same_acl(lids[i]);
			if (err < 0) {
				err = -err - 1;
				LIDS_WARNING
				    ("Same rule found in same acl, previous one will be overwrited")
				    memcpy(&eft_lids[err], &lids[i],
					   sizeof (lids_t));
			}
			/* if difference rules found, overwrite it */
			if (err > 0) {
				err = err - 1;
				LIDS_WARNING
				    ("found same rules in Global acl, overwrite it\n");
				memcpy(&eft_lids[err], &lids[i],
				       sizeof (lids_t));
			} else {
				memcpy(&eft_lids[eft_last], &lids[i],
				       sizeof (lids_t));
				eft_last++;
			}
		}
	}
	while (1) {
		/* step 1.2 read the src inode != NULL's ACL */
		for (i = 0; i < last; i++) {
			if (lids[i].state == curr_state && lids[i].s_ino != 0) {
				if ((default_rule =
				     get_object_type(lids[i].s_file)) == -1)
					exit_error(2,
						   "the subject file is not protected");
				if (default_rule > LIDS_READONLY)
					exit_error(2,
						   "the subject files must be protected by READONLY or DENY");
				/* check the object */
				if (lids[i].type == LIDS_CAP) {
					if (cap_raised
					    (lids_caps, lids[i].o_dev)) {
						LIDS_WARNING
						    ("WARNING: The capability grant is already enable, this acl will not be effective. \n\tDisable the capablity in the cap file first.\n");
					}
				}
				if (lids[i].type != LIDS_CAP
				    && lids[i].type != LIDS_SOCKET) {
					if ((default_rule =
					     get_object_type(lids[i].o_file)) ==
					    -1) {
						exit_error(2,
							   "A default rules for object files is needed");
					}
					if (lids[i].type < default_rule) {
						LIDS_WARNING
						    ("WARNING: Default Permission on object already allow this rules, ignore\n");
					}

					err = check_same_acl(lids[i]);
					if (err != 0) {
						err = err - 1;
						err > 0 ? err : -err;
						LIDS_WARNING
						    ("WARNING: found same rules in acl, overwrite it\n");
						memcpy(&eft_lids[err], &lids[i],
						       sizeof (lids_t));
					} else {
						memcpy(&eft_lids[eft_last],
						       &lids[i],
						       sizeof (lids_t));
						eft_last++;
					}
				} else {
					/* CAP, SOCKET */
					memcpy(&eft_lids[eft_last], &lids[i],
					       sizeof (lids_t));
					eft_last++;
				}
			}
		}
		if (curr_state == state)
			break;
		else
			curr_state = state;
	}
}

/*
 *  lids read conf read /etc/lids.conf into a data structure.
 */
void
lids_do_read_conf(int curr_state)
{
	FILE *fd;
	int i = last;
	int type;
	char buffer[1024];
	char *p, *q;
	char *filename;

	filename = lids_acl_file[curr_state];

	if ((fd = fopen(filename, "r")) == NULL) {
		exit_error(3, "cannot open acl file");
	}

	while (fgets(buffer, 1024, fd) && i < 1024) {
		if (buffer[0] == '#')
			continue;
		if (buffer[strlen(buffer) - 1] == '\n')
			buffer[strlen(buffer) - 1] = '\0';
		if (check_format(buffer) < 0) {
			printf("syntax error in %s\n", filename);
			exit(-1);
		}
		p = strchr(buffer, ':');
		if (p == NULL)
			continue;
		q = strchr(p + 1, ':');
		if (q == NULL)
			continue;
		p = strchr(q + 1, ':');
		if (p == NULL || *(p + 1) == ':')
			continue;
		*p = '\0';
		/* get the permission type */
		q = strchr(p + 1, ':');
		if (q == NULL)
			continue;

		*q = '\0';
		lids[i].type = atoi(p + 1);

		/* get the inherit level */
		p = strchr(q + 1, ':');
		if (p == NULL || *(p + 1) == ':')
			continue;
		*p = '\0';
		lids[i].inherit = atoi(q + 1);

		/* get the time like :19283-89283 */
		q = rindex(p + 1, ':');
		*q = '\0';

		if (lids_get_time(q + 1, lids[i].time) < 0)
			continue;

		if (lids_get_str
		    (buffer, &lids[i].s_ino, &lids[i].s_dev,
		     lids[i].s_file) < 0)
			continue;

		if (lids_get_str
		    (p + 1, &lids[i].o_ino, &lids[i].o_dev, lids[i].o_file) < 0)
			continue;
		if ((lids[i].type == LIDS_CAP)
		    && (lids[i].o_dev ==
			getentrybyname(cap_list, "CAP_NET_BIND_SERVICE")->val))
			str2data(p + 1, lids[i].port, 1, LIDS_PORT_ITEM);
		if ((lids[i].type == LIDS_SOCKET)
		    && (lids[i].o_dev == LIDS_SOCKET_NF_MARK)) {

			lids[i].mark = atoi(p + 1);
		}
		lids[i].state = curr_state;
		i++;

	}
	last = i;

	fclose(fd);
}

/*
 *	read the capablity file from global cap and state cap
 *	merge them into a 32bits 
 *
 */
void
lids_do_read_cap()
{
	int curr_state = LIDS_STATE_GLOBAL;
	int i, cap;
	char *p;
	FILE *fd;
	char buffer[1024];

	lids_caps = 0;

	while (1) {
		/* read the global caps */
		if ((fd = fopen(lids_cap_file[curr_state], "r")) == NULL) {
			exit_error(3, "cannot open cap file");
		}
//              printf("ok...file=%s\n",lids_cap_file[curr_state]);
		while (fgets(buffer, 1024, fd)) {
			if (buffer[0] == '#')
				continue;
			if (buffer[0] != '+' && buffer[0] != '-') {
				continue;
			}
			if (buffer[strlen(buffer) - 1] == '\n')
				buffer[strlen(buffer) - 1] = '\0';

			p = strchr(buffer, ':');
			if (p == NULL)
				continue;
			*p = '\0';
			cap = atoi(buffer + 1);
			if (cap < 0 || cap > 32) {
				fprintf(stderr,
					"WARNING: invalid cap number in cap file \n");
				continue;
			}
			if (buffer[0] == '-') {
				cap_lower(lids_caps, cap);
			} else {
				cap_raise(lids_caps, cap);
			}
		}
		if (curr_state == state)
			break;

		if (curr_state == LIDS_STATE_GLOBAL)
			curr_state = state;
	}

	printf("\t effective capability = 0x%.8x\n\n", lids_caps);

}

/*	read the cap and acl file
 *	compute the effective rules and capability
 *	check each rules for intergrity.
 */
void
lids_read_conf()
{
	eft_last = last = 0;

	lids_do_read_cap();
	lids_do_read_conf(LIDS_STATE_GLOBAL);
	/* if it is not global, we will need to read it */
	if (state != LIDS_STATE_GLOBAL) {
		lids_do_read_conf(state);
	}
	compute_effective_rules();
}

/*
 *	performace the ACL rules interigitry checking
 *
 */
void
lids_check()
{

	for (state = LIDS_STATE_GLOBAL; state <= LIDS_STATE_SHUTDOWN; state++) {
		printf("checking  %s\n", lids_acl_file[state]);
		lids_read_conf();
	}
}

void
lids_write_file()
{
	int i = 0;
	FILE *fd;

	if ((fd = fopen(acl_conf_file, "w")) == NULL) {
		exit_error(3, "cannot open acl file");
	}
	fwrite(LIDS_CONF_COMMENT, 1, strlen(LIDS_CONF_COMMENT), fd);

	for (i = 0; i < last; i++) {

		if (lids[i].state == state && lids[i].type != LIDS_DELETE_FLAG) {
			if (lids[i].type == LIDS_CAP &&
			    lids[i].o_dev == getentrybyname(cap_list,
							    "CAP_NET_BIND_SERVICE")->
			    val)

				fprintf(fd, "%d:%d:%s:%d:%d:%s:%d:%s:%s\n",
					lids[i].s_ino, lids[i].s_dev,
					lids[i].s_file, lids[i].type,
					lids[i].inherit,
					disp_multi_data(lids[i].port, 1),
					lids[i].o_dev,
					lids[i].o_file,
					disp_multi_data(lids[i].time, 0), 0);
			else
				fprintf(fd, "%d:%d:%s:%d:%d:%d:%d:%s:%s\n",
					lids[i].s_ino, lids[i].s_dev,
					lids[i].s_file, lids[i].type,
					lids[i].inherit,
					lids[i].o_ino, lids[i].o_dev,
					lids[i].o_file,
					disp_multi_data(lids[i].time, 0), 0);
		}
	}

	fclose(fd);

}

/*
 *	get the str desc number from type number
 */
int
get_type_str(int type)
{
	int k, j;

	k = -1;
	for (j = 0; j < LIDS_TYPE_NUM; j++) {
		if (type == lids_type_val[j]) {
			k = j;
			break;
		}
	}
	if (k == -1)
		exit_error(2, "type mismatch in acl file");
	return k;
}

void
lids_list_file(int type)
{
	int i, j, k;
	static char anyfile[] = "Any file";
	char *src, *obj;

	lids_read_conf();

	if (type == LIDS_EXTEND)
		printf("Subj ino,dev   Obj ino,dev  type  ");
	printf
	    ("                Subject   ACCESS  inherit time        Object\n");

	printf
	    ("----------------------------------------------------------------------------\n");
	for (i = 0; i < last; i++) {
		if (lids[i].state == state) {
			src = anyfile;
			obj = anyfile;

			if (*lids[i].s_file)
				src = lids[i].s_file;

			if (*lids[i].o_file)
				obj = lids[i].o_file;

			if (type == LIDS_EXTEND)
				printf("%6d,%6d %6d,%6d %6d",
				       lids[i].s_ino, lids[i].s_dev,
				       lids[i].o_ino, lids[i].o_dev,
				       lids[i].type);

			printf("%23s  %8s%s:%3d  %s  %20s %s%d\n",
			       src,
			       lids_type_desc[get_type_str(abs(lids[i].type))],
			       "",
			       lids[i].inherit,
			       time2str(lids[i].time),
			       obj,
			       strncmp("CAP_NET_BIND_SERVICE", obj,
				       20) ? "" : disp_multi_data(lids[i].port,
								  1),
			       strncmp("LIDS_SOCKET_NF_MARK", obj,
				       19) ? 0 : lids[i].mark);
		}
	}
	printf("\n\n");
}

/*
 *	lids_search_file()
 *
 *	lids search the file by given file and type.
 *
 */
int
lids_search_file(char *s_file, char *o_file)
{
	int i, flag, number = 0;

	for (i = 0; i < last; i++) {
		flag = 0;
		if (*s_file != '\0') {
			flag = strcmp(lids[i].s_file, s_file);
		}
		if (*o_file != '\0')
			flag |= strcmp(lids[i].o_file, o_file);
		if (!flag) {
			number++;
			lids[i].type = LIDS_DELETE_FLAG;
		}
	}
	return number;
}

/*
 *	lids_search_inode()
 *
 *	lids search the file by given inode , dev  and type.
 *
 */
int
check_same_acl(lids_t acl)
{
	int i;
	int err = 0;

	for (i = 0; i < eft_last; i++) {
		if (eft_lids[i].s_ino == acl.s_ino &&
		    eft_lids[i].s_dev == acl.s_dev &&
		    eft_lids[i].o_ino == acl.o_ino &&
		    eft_lids[i].o_dev == acl.o_dev) {
			if (eft_lids[i].state == acl.state) {
				err = -(i + 1);
				break;
			} else {
				err = (i + 1);
			}
		}
	}
	return err;
}

void
lids_del_file(char *subject_file, char *object_file)
{
	int i;
	char canonical_s_path[PATH_MAX];
	char canonical_o_path[PATH_MAX];

	lids_read_conf();

	i = lids_search_file(subject_file, object_file);

	if (!i) {
		realpath(subject_file, canonical_s_path);
		realpath(object_file, canonical_o_path);
		i = lids_search_file(canonical_s_path, canonical_o_path);
		if (!i)
			exit_error(2, "the file does not exit in acl file");
	}
	printf("delete %d items\n", i);

	lids_write_file();
}

/*
 *	get special type from typestr, when lidsadm -A -s ..-o .. -t -j ..
 */

int
get_object_type(char *filename)
{
	int i;
	char o_file[1024];
	char *p = NULL;
	char root[2];
	char *fileptr;

	strcpy(o_file, filename);
	strcat(o_file, "/");
	root[0] = '/';
	root[1] = '\0';

	while (p != o_file) {
		p = rindex(o_file, '/');
		if (!p)
			break;
		*p = '\0';
		if (*o_file != '\0') {
			fileptr = o_file;
		} else {
			fileptr = root;
		}
		for (i = 0; i < eft_last; i++) {
			if (eft_lids[i].s_ino == 0
			    && strcmp(fileptr, eft_lids[i].o_file) == 0) {
				return eft_lids[i].type;
			}
		}
	}
	return -1;

}

/*
 *	lids_add_file 
 */
void
lids_add_file(char *subject_file, char *object_file, int type, int inherit,
	      time_t timescale[][2])
{
	struct stat s_st, o_st, z_st;
	char canonical_s_path[PATH_MAX];
	char canonical_o_path[PATH_MAX];
	int default_rule = -1;
	int sys_cap = 0;	/* 1 for system cap, 0 for individual cap */
	entry_t *entry;
	lids_t acl;
	int err;

	lids_read_conf();

	if (last >= 1024) {
		exit_error(2, "cannot exceed 1024 entries.");
	}

	LIDS_DBG("sub=[%s],type=%d\n", subject_file, type);

	if (*subject_file == '\0') {
		sys_cap = 1;
	}

	if (!sys_cap) {
		realpath(subject_file, canonical_s_path);
		if (stat(canonical_s_path, &s_st)) {
			exit_error(2, " cannot find the subject file");
		}
		if (!S_ISREG(s_st.st_mode)) {
			exit_error(2,
				   " subject file need to be a regular file\n");
		}

		/* check if the subject file is executable or not */
	} else {
		s_st.st_ino = s_st.st_dev = 0;
	}
	if (type == LIDS_CAP) {
		o_st.st_ino = -1;	/* special type's ino */
		if (entry = getentrybyname(cap_list, object_file)) {
			o_st.st_dev = entry->val;
			if (cap_raised(lids_caps, o_st.st_dev)) {
				exit_error(2,
					   "The capability grant is already enable, this acl will not be effective. Disable the capablity in the cap file first.");
			}
		} else {
			exit_error(2,
				   "special type must be one of CAP_XXX_XXX. use lidsadm -h for details.");
		}
		strcpy(canonical_o_path, object_file);
	} else if (type == LIDS_SOCKET) {
		o_st.st_ino = -1;	/* special type's ino */
		if (entry = getentrybyname(socket_list, object_file)) {
			o_st.st_dev = entry->val;
		} else {
			exit_error(2,
				   "special type must be one of LIDS_SOCKET_XXX_XXX. use lidsconf -h for details.");
		}
		if (o_st.st_dev == LIDS_SOCKET_NF_MARK) {
			o_st.st_ino = mark;
		}
		strcpy(canonical_o_path, object_file);
	} else {
		realpath(object_file, canonical_o_path);
		if (stat(canonical_o_path, &o_st)) {
			exit_error(2, " cannot find the object file");
		}
	}
	/* fill up the ACL */
	acl.s_ino = s_st.st_ino;
	acl.s_dev = s_st.st_dev;
	acl.o_ino = o_st.st_ino;
	acl.o_dev = o_st.st_dev;

	acl.type = type;
	acl.inherit = inherit;
	acl.state = state;

	strcpy(acl.s_file, canonical_s_path);
	strcpy(acl.o_file, canonical_o_path);

	memcpy(acl.time, timescale, LIDS_TIME_ITEM * 2 * sizeof (int *));
	memcpy(acl.port, bindport, LIDS_PORT_ITEM * 2 * sizeof (int *));
	/* check the subject */
	if (!sys_cap) {
		if ((default_rule = get_object_type(acl.s_file)) == -1)
			exit_error(2, "the subject file is not protected");
		if (default_rule > LIDS_READONLY)
			exit_error(2,
				   "the subject files must be protected by READONLY or DENY");
	}
	/* check the object */
	if (type != LIDS_CAP && type != LIDS_SOCKET) {
		if ((default_rule = get_object_type(acl.o_file)) == -1) {
			if (!sys_cap)
				exit_error(2,
					   "you must define the default rules for object files");
		}
		/* if current type have less persmission of defualt _rule */
		if (type < default_rule && !sys_cap) {
			exit_error(2,
				   "the type is less than default permssion, this rule is useless");
		}
	}

	LIDS_DBG("canonical_o_file=%s,o_ino =%d\n", canonical_o_path,
		 o_st.st_ino);

	err = check_same_acl(acl);

	if (err < 0)
		exit_error(2,
			   "a file with the same inode and dev is already referenced.");
	if (err > 0)
		printf("found same acl in the global file.\n");

	memcpy(&lids[last], &acl, sizeof (lids_t));

	last++;
	lids_write_file();
}

void
lids_update()
{
	struct stat s_st, o_st;
	int i;
	lids_read_conf();

	for (i = 0; i < last; i++) {
		/*
		 *      check subject files
		 */
		if (*lids[i].s_file != '\0') {
			if (stat(lids[i].s_file, &s_st)) {
				printf
				    ("%s doesn't exist anymore. not removed.\n",
				     lids[i].s_file);
			} else {
				if ((s_st.st_ino != lids[i].s_ino)
				    || (s_st.st_dev != lids[i].s_dev)) {
					printf
					    ("subject file %s was (%u:%u inode %lu) instead of (%u:%u %lu). corrected.\n",
					     lids[i].s_file,
					     MAJOR(lids[i].s_dev),
					     MINOR(lids[i].s_dev),
					     lids[i].s_ino,
					     MAJOR((unsigned long) s_st.st_dev),
					     MINOR((unsigned long) s_st.st_dev),
					     s_st.st_ino);
					lids[i].s_ino = s_st.st_ino;
					lids[i].s_dev = s_st.st_dev;
				}
			}
		}
		/*
		 *      check object files
		 */
		if (*lids[i].o_file != '\0' && stat(lids[i].o_file, &o_st)) {
			if ((getentrybyname(cap_list, lids[i].o_file) < 0)
			    && lids[i].type != LIDS_CAP
			    && lids[i].type != LIDS_SOCKET) {
				printf
				    ("%s doesn't exist anymore. not removed.\n",
				     lids[i].o_file);
			}
		} else {
			if (o_st.st_ino != lids[i].o_ino
			    || o_st.st_dev != lids[i].o_dev) {
				printf
				    ("object file %s was (%u:%u inode %lu) instead of (%u:%u %lu). corrected.\n",
				     lids[i].o_file, MAJOR(lids[i].o_dev),
				     MINOR(lids[i].o_dev), lids[i].o_ino,
				     MAJOR((unsigned long) o_st.st_dev),
				     MINOR((unsigned long) o_st.st_dev),
				     o_st.st_ino);

				lids[i].o_ino = o_st.st_ino;
				lids[i].o_dev = o_st.st_dev;
			}
		}
	}
	lids_write_file();
}

void
lids_make_rmd160_passwd()
{
	char passwd[64 * 2];
	int times = 0;
	int ok;
	FILE *fd;

	memset(passwd, '\0', 64 * 2);
	while (times < 3) {
		times++;
		if ((ok = read_rmd160_passwd(passwd, 1, 2)) == 0)
			break;
	}

	if (ok == 0) {
		fd = fopen(LIDS_PW_FILE, "w");
		if (fd == NULL)
			return;
		fwrite(passwd, 1, strlen(passwd), fd);
		fclose(fd);
		printf("wrote password to %s\n", LIDS_PW_FILE);
	}

}

void
get_acl_type(int argc, char **argv)
{
	char acl_type_str[16];

	memset(acl_type_str, 0, 16);
	if (optind < argc) {
		strncpy(acl_type_str, argv[optind], 8);
	}
	if (strlen(acl_type_str) == 0 || acl_type_str[0] == '-') {
		strcpy(acl_conf_file, LIDS_GLOBAL_CONF_FILE);
		state = LIDS_STATE_GLOBAL;
	} else if (strncmp(acl_type_str, "POSTBOOT", 8) == 0) {
		state = LIDS_STATE_POSTBOOT;
		strcpy(acl_conf_file, LIDS_POSTBOOT_CONF_FILE);
		optind++;
	} else if (strncmp(acl_type_str, "BOOT", 4) == 0) {
		state = LIDS_STATE_BOOT;
		strcpy(acl_conf_file, LIDS_BOOT_CONF_FILE);
		optind++;
	} else if (strncmp(acl_type_str, "SHUTDOWN", 4) == 0) {
		state = LIDS_STATE_SHUTDOWN;
		strcpy(acl_conf_file, LIDS_SHUTDOWN_CONF_FILE);
		optind++;
	} else {
		printf("acl_type_str [%s] len = %d\n", acl_type_str,
		       strlen(acl_type_str));
		exit_error(2, "Must be BOOT, POSTBOOT , SHUTDOWN or NULL");
	}
	printf("Using ACL FILE: %s\n", acl_conf_file);
}

static char shortopts[] = "U::A::D::Z::L::CPs:o::j:i:dehHt:vV";
static struct option longopts[] = {
	{"update", 2, 0, 'U'},
	{"add", 2, 0, 'A'},
	{"delete", 2, 0, 'D'},
	{"zero", 2, 0, 'Z'},
	{"list", 2, 0, 'L'},
	{"passwd", 0, 0, 'P'},
	{"subject", 1, 0, 's'},
	{"object", 2, 0, 'o'},
	{"jump", 1, 0, 'j'},
	{"inheritance", 1, 0, 'i'},
	{"extended", 0, 0, 'e'},
	{"time", 1, 0, 't'},
	{"help", 0, 0, 'h'},
	{"morehelp", 0, 0, 'H'},
	{"version", 0, 0, 'v'},
	{"verbose", 0, 0, 'V'},
	{"check", 0, 0, 'C'},
	{0, 0, 0, 0}
};

main(int argc, char **argv)
{
	int command = LIDS_NONE;
	int type = LIDS_NONE;
	char subject_file[1024], object_file[1024];
	char del_file[1024];
	char sw_flag[16];
	char lids_time[176];
	char acl_type_str[16];
	int c, i;
	int index = 0;
	int cap_type = 0;	/* program capability type */
	int inherit = 0;	/* if the acl is inheritable */
	time_t timescale[LIDS_TIME_ITEM][2];

	setentry(cap_list);
	setentry(socket_list);
	memset(subject_file, '\0', 1024);
	memset(object_file, '\0', 1024);
	memset(timescale, 0, sizeof (timescale));
	if (getuid() != 0) {
		exit_error(2, "you must be root to run this program");
	}
	while ((c = getopt_long(argc, argv, shortopts, longopts, &index)) != -1) {
		switch (c) {
		case 'U':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_UPDATE;
			get_acl_type(argc, argv);
			break;
		case 'A':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_ADD;
			get_acl_type(argc, argv);
			break;
		case 'D':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_DELETE;
			get_acl_type(argc, argv);
			break;
		case 'L':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_LIST;
			get_acl_type(argc, argv);
			break;
		case 'e':
			if (command != LIDS_LIST)
				exit_error(2, "error commands specified");
			type = LIDS_EXTEND;
			break;
		case 'Z':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_DELETE_ALL;
			get_acl_type(argc, argv);
			break;
		case 'o':	/* for object */
			if (command == LIDS_NONE)
				exit_error(2, "no comands specified");

			if (optind < argc) {
				strcpy(object_file, argv[optind++]);
			}
			if (!strcmp(object_file, "CAP_NET_BIND_SERVICE")) {
				if (optind != argc) {

					if (str2data
					    (argv[optind], bindport, 1,
					     LIDS_PORT_ITEM) < 0)
						exit_error(2,
							   "The binding port is invalid");
					optind++;
				} else
					bindport[0][0] = bindport[0][1] = 0;
			}
			if (!strcmp(object_file, "LIDS_SOCKET_NF_MARK")) {
				mark = 0;
				if (optind != argc) {
					mark = atoi(argv[optind]);
					if (mark < 1)
						exit_error(2,
							   "the mark value must > 0");
					optind++;
					printf("set NF MARK as %d\n", mark);
				}
			}
			if (strchr(object_file, ':'))
				exit_error(2, "filename can not contain :");
			break;
		case 's':	/* program capability subject */
			if (command == LIDS_NONE)
				exit_error(2, "no comands specified");
			strcpy(subject_file, optarg);
			if (strchr(object_file, ':'))
				exit_error(2, "filename must not contain :");
			break;
		case 't':	/* time scale restrition */
			if (command == LIDS_NONE)
				exit_error(2, "no comands specified");
			if (strlen(optarg) > 176)
				exit_error(2, "invalid time syntax");
			memset(lids_time, '\0', 176);
			strncpy(lids_time, optarg, 176);
			/* use '-1" to indicate the default value */
			memset(timescale, '\0',
			       LIDS_TIME_ITEM * 2 * sizeof (time_t *));
			/* FIXME */

			if (str2data(lids_time, timescale, 0, LIDS_TIME_ITEM) <
			    0)
				exit_error(2, "invalid time format");
			break;
		case 'i':
			if (command == LIDS_NONE)
				exit_error(2, "no comands specified");
			inherit = atoi(optarg);
			break;
		case 'd':
			exit_error(2, "domain not supported now");
			break;
		case 'j':
			if (command == LIDS_NONE)
				exit_error(2, "no comands specified");

			if (type != LIDS_NONE)
				exit_error(2, "multiple types specified");
			type = -1;

			for (i = 0; i < LIDS_TARGET_NUM; i++) {
				if (!strcmp(optarg, lids_type_desc[i])) {
					type = lids_type_val[i];
				}
			}
			if (type == -1)
				exit_error(2,
					   "cap type must be READONLY, WRITE, APPEND, DENY , ENABLE, DISABLE or GRANT");
			break;
		case 'P':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_MK_PASSWD;
			break;
		case 'C':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_CHECK;
			break;
		case 'V':
			verbose = 1;
			break;
		case 'v':
			exit_version();
			break;
		case 'h':
			exit_help(0);
		case 'H':
			exit_help(1);
		default:
			break;
		}
	}
	if (optind < argc)
		exit_error(2, "unknown arguments found on commandline");
	if ((command == LIDS_NONE) || (argc < 2))
		exit_normal();
	if ((command == LIDS_ADD) && (type == LIDS_NONE))
		exit_normal();
	if ((subject_file[0] != 0) && (object_file[0] == 0)
	    && (command == LIDS_ADD))
		exit_error(2, "defined subject must be related to an object");

	switch (command) {
	case LIDS_ADD:
		printf("ADD\n");
		lids_add_file(subject_file, object_file, type, inherit,
			      timescale);
		break;
	case LIDS_DELETE:
		printf("DELETE\n");
		lids_del_file(subject_file, object_file);
		break;
	case LIDS_LIST:
		printf("LIST\n");
		lids_list_file(type);
		break;
	case LIDS_DELETE_ALL:
		printf("DELETE_ALL\n");
		lids_del_all();
		break;
	case LIDS_MK_PASSWD:
		printf("MAKE PASSWD\n");
		lids_make_rmd160_passwd();
		break;
	case LIDS_UPDATE:
		printf("UPDATE\n");
		lids_update();
		break;
	case LIDS_CHECK:
		printf("INTERIGITY CHECK\n");
		verbose = 1;
		lids_check();
		break;
	}
	exit(0);
}
