# Module: moduleMultipleLogins
# Author: Craig H. Rowland <crowland@psionic.com>
# Created: 11-09-98
#
# Send all changes/modifications/bugfixes to the above address.  
# 
# This software is Copyright(c) 1997-98 Craig H. Rowland
# 
# Disclaimer:
# 
# All software distributed by Craig H. Rowland ("the author") and
# Psionic Systems is distributed AS IS and carries NO WARRANTY or
# GUARANTEE OF ANY KIND. End users of the software acknowledge that         
# they will not hold the author, Psionic Systems, and any employer of    
# the author liable for failure or non-function of a software
# product. YOU ARE USING THIS PRODUCT AT YOUR OWN RISK
# 
# Licensing restrictions apply. See the license that came with this
# file for more information or visit http://www.psionic.com for more
# information.
# 
# This software is NOT GPL NOR PUBLIC DOMAIN so please read the license   
# before modifying or distributing. Contact the above address if you have
# any questions.
# 
# Description:
# 
# This module will issue an alert if a user is already logged in and they
# are coming from a different system than what they were logged into. This
# can indicate a compromised account and hackers sharing account information
# logging in from different sites.
# 
# $Id: moduleMultipleLogins.py,v 1.3 1999/03/22 04:57:27 crowland Exp crowland $


from hostSentryCore import *
import hostSentryConfig
import hostSentryUser
import hostSentryLog
import hostSentryTTYDB
import hostSentryTTY
import hostSentryDB

import sys
import string
import re
import array

ALLOW_FILE = '/moduleMultipleLogins.allow'
MAX_ALLOW = 99999

class moduleMultipleLogins(hostSentryCore):

	def __init__(self):
		self.setLogLevel()
		self.__result = None

	def login(self, userObj, loginStamp):
		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleMultipleLogins: login: processing user: ' + userObj.getUsername())		

		loginIP, loginHostname, loginTTY, loginTime, logoutTime =  string.split(loginStamp, '@')


		try:
			config = hostSentryConfig.hostSentryConfig()
			config.configInit()
			ttydbFile = config.parseToken('DB_TTY_FILE')
			if ttydbFile == None:
				hostSentryLog.log('adminalert: moduleMultipleLogins: login: DB_TTY_FILE token not found in config. Aborting')
				raise hostSentryError('adminalert: moduleMultipleLogins: login: DB_TTY_FILE token not found in config. Aborting')
		except:
				config.close()
				hostSentryLog.log('adminalert: moduleMultpleLogins: login: Error parsing DB_TTY_FILE from config: %s' % sys.exc_value[0])
				raise hostSentryError('adminalert: moduleMultpleLogins: login: Error parsing DB_TTY_FILE from config: %s' % sys.exc_value[0])

		try:
			allowPath = config.parseToken('MODULE_PATH') 
                        if allowPath == None:
				hostSentryLog.log('adminalert: moduleMultipleLogins: MODULE_PATH token not found in config.')
				raise hostSentryError('adminalert: moduleMultipleLogins: MODULE_PATH token not found in config.')
			else:
				allowFilename = allowPath + ALLOW_FILE

			if self.getLogLevel() > 0:
				hostSentryLog.log('debug: moduleMultipleLogins: login: looking for allow file: %s' % allowFilename)

			allow = open(allowFilename)
			config.close()
		except:
			config.close()
			hostSentryLog.log('adminalert: moduleMultipleLogins: login: ERROR: Allow file open failed: %s' % sys.exc_value[0])
			raise hostSentryError('adminalert: moduleMultipleLogins: login: ERROR: Allow file open failed: %s' % sys.exc_value[0])


		# Read in all allowed hosts in one shot.
		try:
			allowData = []
			while 1:
				line = allow.readline()[:-1]

				if len(line) < 1:
					break
				elif line[0][0] == '#':
					pass
				else:
					allowData.append(line)
			allow.close()
		except:
			hostSentryLog.log('adminalert: moduleMultipleLogins: login: ERROR: Error reading in allow file into array: %s' % sys.exc_value[0])
			raise hostSentryError('adminalert: moduleMultipleLogins: login: ERROR: Error reading in allow file into array: %s' % sys.exc_value[0])

		# Crawl down the array looking for a match to ignore.
		try:			
			for x in range(len(allowData)):
				if re.search(allowData[x], loginHostname) != None:
					return
		except:
			hostSentryLog.log('adminalert: moduleMultipleLogins: login: ERROR: Error during regex compare (first ignore match): %s' % sys.exc_value[0])
			raise hostSentryError('adminalert: moduleMultipleLogins: login: ERROR: Error during regex compare (first ignore match): %s' % sys.exc_value[0])
	
		try:
			ttyDBObj = hostSentryTTYDB.hostSentryTTYDB(ttydbFile)
			ttyDBObj.setLogLevel(self.getLogLevel())
			ttyDBObj.open()

			ttys = ttyDBObj.dump()
			gotOneData = None

			# Check the ttyDB for matching logged in users
			# and parse them against the ignore host file.
			for x in range(len(ttys)):
				ttyRetObj = ttyDBObj.get(ttys[x])
				if ttyRetObj.getUsername() == userObj.getUsername():
					try:			
						loginData = string.split(ttyRetObj.getLoginStamp(), '@')
						for x in range(len(allowData)):
							if re.search(allowData[x], loginData[1]) != None:
								ttyDBObj.close()
								return
							# Concurrent logins from same host
							elif loginData[1] == loginHostname:
								pass
							else:
								gotOneData = loginData[1]
					except:
						hostSentryLog.log('adminalert: moduleMultipleLogins: login: ERROR: Error during regex compare.: %s' % sys.exc_value[0])
						raise hostSentryError('adminalert: moduleMultipleLogins: login: ERROR: Error during regex compare.: %s' % sys.exc_value[0])
			if gotOneData:
				hostSentryLog.log('securityalert: moduleMultipleLogins: login: user %s is logged in from multiple hosts: %s AND %s' % (userObj.getUsername(), loginHostname, gotOneData))
				self.setResult('securityalert: moduleMultipleLogins: login: user %s is logged in from multiple hosts: %s AND %s' % (userObj.getUsername(), loginHostname, gotOneData))

			ttyDBObj.close()
		except:
			hostSentryLog.log('adminalert: moduleMultipleLogins: login: ERROR: Error during processing of TTYDB: %s' % sys.exc_value[0])
			raise hostSentryError('adminalert: moduleMultipleLogins: login: ERROR: Error during processing of TTYDB: %s' % sys.exc_value[0])


	def logout(self, userObj, logoutStamp):
		if self.getLogLevel() > 0:
			hostSentryLog.log('debug: moduleMultipleLogins: logout: processing user: ' + userObj.getUsername())		

	def setResult(self, result):
		self.__result = result

	def getResult(self):
		return self.__result


if __name__ == '__main__':

	test = moduleMultipleLogins()
#	test.setLogLevel(99)
	user = hostSentryUser.hostSentryUser()
	user.setUsername('hostSentrytest')
	user.setTotalLogins(1)

	tty = hostSentryTTY.hostSentryTTY()
	tty.setUsername('hostSentrytest')
	# Fake an active login on bogus terminal
	tty.setTty('tty0007')
	tty.setLoginStamp('192.168.2.1@somewherebad.com@tty0007@999999999@')


	config = hostSentryConfig.hostSentryConfig()
	config.configInit()
	dbFile = config.parseToken('DB_FILE')
	ttydbFile = config.parseToken('DB_TTY_FILE')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.store(user)
	db.close()

	ttydb = hostSentryTTYDB.hostSentryTTYDB(ttydbFile)
	ttydb.store(tty)
	ttydb.close()


	test.login(user, '192.168.2.1@somewhere.com@tty0001@999999999@')
	test.logout(user, '192.168.2.1@somewhere.com@tty0001@999999999@98989898')

	db = hostSentryDB.hostSentryDB(dbFile)
	db.delete(user.getUsername())
	db.close()

	ttydb = hostSentryTTYDB.hostSentryTTYDB(ttydbFile)
	ttydb.delete('tty0007')
	ttydb.close()

