/*

  authc-securid.c

  Author:   Graeme Ahokas <gahokas@ssh.com>


  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SecurID authentication, client side.

*/
#include "ssh2includes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "readpass.h"
#include "sshclient.h"
#include "sshconfig.h"
#include "sshfsm.h"
#include "sshmsgs.h"





#ifdef SSH_CLIENT_WITH_SECURID

#define SSH_DEBUG_MODULE "Ssh2AuthSecurIDClient"

typedef struct SshClientSecurIDPinInfoRec
{

  unsigned int min_pin_len;
  unsigned int max_pin_len;
  unsigned int user_selectable;
  Boolean alphanumeric;
  char *system_pin;
  int system_pin_len;

  char *final_pin;

} *SshClientSecurIDPinInfo;

 
typedef struct SshClientSecurIDAuthRec
{
  SshFSM fsm;
  SshFSMThread main_thread;
  SshClient client;

  SshBuffer server_packet;
  unsigned int server_packet_type;

  SshClientSecurIDPinInfo pin_info;

  SshBuffer response_packet;

  char *user;

  void **state_placeholder;
  SshAuthClientCompletionProc completion_proc;
  void *completion_context;
} *SshClientSecurIDAuth;


SSH_FSM_STEP(ssh_securid_suspend)
{
  return SSH_FSM_SUSPENDED;
}


/* processes a server packet. */
SSH_FSM_STEP(ssh_securid_process_packet)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);
  SSH_PRECOND(gdata);

  switch (gdata->server_packet_type)
  {
    case SSH_MSG_USERAUTH_SECURID_CHALLENGE:
      /* user must enter passphrase after token change */
      ssh_buffer_free(gdata->server_packet);
      SSH_FSM_SET_NEXT("ssh_securid_challenge");
      return SSH_FSM_CONTINUE;

    case SSH_MSG_USERAUTH_SECURID_NEW_PIN_REQD:

      /* user chooses or accepts or must choose new pin */
      SSH_FSM_SET_NEXT("ssh_securid_new_pin");
      return SSH_FSM_CONTINUE;

    default:
      ssh_buffer_free( gdata->server_packet );
      /* unknown message, abort authentication */
      SSH_FSM_SET_NEXT("ssh_securid_finish");
      return SSH_FSM_CONTINUE; 

  }
  /* never reached */
}

/* finish by shutting down the fsm */
SSH_FSM_STEP(ssh_securid_finish)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);
  SSH_PRECOND(gdata);

  *gdata->state_placeholder = NULL;

  ssh_fsm_destroy(gdata->fsm);
  gdata->fsm = NULL;

  return SSH_FSM_FINISH;
}


/* sends the newly selected pin to the server. */
SSH_FSM_STEP(ssh_securid_send_pin)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);

  gdata->response_packet = ssh_buffer_allocate();

  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_BOOLEAN,
                    TRUE,
                    SSH_FORMAT_UINT32_STR,
                    gdata->pin_info->final_pin,
                    strlen(gdata->pin_info->final_pin),
                    SSH_FORMAT_END);

  (*gdata->completion_proc)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE,
                            gdata->user, gdata->response_packet,
                            gdata->completion_context);

  ssh_buffer_free(gdata->response_packet);

  SSH_FSM_SET_NEXT("ssh_securid_finish");
  return SSH_FSM_CONTINUE;
}


/* asks the user if they want to accept the system assigned pin.
   sends empty string as new pin if the user wants to leave token
   in new pin mode */
SSH_FSM_STEP(ssh_securid_accept_system_pin)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);
  char buf[100];
  Boolean decision;



  snprintf(buf, sizeof(buf), "Would you like to accept the "\
           "server assigned PIN (yes/no)? ");

  /* XXX make this async */
  decision = ssh_read_confirmation(buf, FALSE);




  if (decision == FALSE)
      gdata->pin_info->final_pin = ssh_xstrdup("");
  else
    {
      /* now tell the user this new pin*/

      ssh_informational("Your new PIN is %s .\r\n",gdata->pin_info->system_pin);



      gdata->pin_info->final_pin = ssh_xstrdup(gdata->pin_info->system_pin);
    }

  SSH_FSM_SET_NEXT("ssh_securid_send_pin");
  return SSH_FSM_CONTINUE;
}


/* The user has decided / must enter a new pin. informs the user of
   the restrictions on the pin and gets the new pin*/
SSH_FSM_STEP(ssh_securid_select_new_pin)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);

  char buf[100];
  char buf2[100];
  char *pin1 = NULL;
  char *pin2 = NULL;


  snprintf(buf, sizeof(buf), "New PIN: ");
  snprintf(buf2, sizeof(buf2), "Confirm new PIN: ");

  do
    {

      if (gdata->pin_info->alphanumeric)
        ssh_informational("Enter your new PIN containing %d to %d " \
                          "digits or characters\r\nor press <return> to leave "\
                          "your token in NEW PIN mode.\r\n",
                          gdata->pin_info->min_pin_len,
                          gdata->pin_info->max_pin_len);

      else
        ssh_informational("Enter your new PIN containing %d to %d " \
                          "digits\r\nor press <return> to leave " \
                          "your token in NEW PIN mode.\r\n",
                          gdata->pin_info->min_pin_len,
                          gdata->pin_info->max_pin_len);

      pin1 = ssh_read_passphrase(buf, FALSE);

      if (strlen(pin1) == 0)
        break;

      /* get confirmation */
      pin2 = ssh_read_passphrase(buf2, FALSE);

      if (!strcmp(pin1, pin2))
        break;

      /* pins did not match so ask again */
      ssh_informational("The PIN codes entered do not match. Try again.\r\n");
    } while (1);













  gdata->pin_info->final_pin = ssh_xstrdup(pin1);

  ssh_xfree(pin1);

  if (pin2)
    ssh_xfree(pin2);

  SSH_FSM_SET_NEXT("ssh_securid_send_pin");
  return SSH_FSM_CONTINUE;
}


/* The user's token is in new pin mode. according to the restrictions
   on the token, we will go to various states to get the user's selection */
SSH_FSM_STEP(ssh_securid_new_pin)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);
  Boolean decision;
  char buf[200];

  /* decode the packet to get the min, max, alpha allowed
     information from the server */

  if (ssh_decode_buffer(gdata->server_packet,
                        SSH_FORMAT_UINT32, &gdata->pin_info->min_pin_len,
                        SSH_FORMAT_UINT32, &gdata->pin_info->max_pin_len,
                        SSH_FORMAT_CHAR, &gdata->pin_info->user_selectable,
                        SSH_FORMAT_BOOLEAN, &gdata->pin_info->alphanumeric,
                        SSH_FORMAT_UINT32_STR, &gdata->pin_info->system_pin,
                        &gdata->pin_info->system_pin_len,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_free( gdata->server_packet );
      SSH_FSM_SET_NEXT("ssh_securid_error");
      return SSH_FSM_CONTINUE;
    }

  ssh_buffer_free( gdata->server_packet );


  ssh_informational("Your token is in new PIN mode.\r\n");




  if (gdata->pin_info->user_selectable == SSH_SECURID_CANNOT_CHOOSE_PIN)
    {
      SSH_FSM_SET_NEXT("ssh_securid_accept_system_pin");
      return SSH_FSM_CONTINUE;
    }
  else if (gdata->pin_info->user_selectable == SSH_SECURID_MUST_CHOOSE_PIN)
    {
      SSH_FSM_SET_NEXT("ssh_securid_select_new_pin");
      return SSH_FSM_CONTINUE;
    }
  else
    {

      ssh_informational("You may create your own PIN or accept a server "\
                         "assigned PIN.\r\n");
      snprintf(buf, sizeof(buf), "Would you like to create your " \
               "own new PIN (yes/no)? ");
      decision = ssh_read_confirmation(buf, FALSE);



      if (decision == FALSE)
        {
          /* go to accept system pin state*/
          SSH_FSM_SET_NEXT("ssh_securid_accept_system_pin");
          return SSH_FSM_CONTINUE;
        }

      SSH_FSM_SET_NEXT("ssh_securid_select_new_pin");
      return SSH_FSM_CONTINUE;
    }

  /* not reached */
}


SSH_FSM_STEP(ssh_securid_challenge)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);
  char *passphrase;
  char buf[100];


  snprintf(buf, sizeof(buf), "Wait for token to change and enter PASSCODE: ");

  /* XXX make this async at some point*/
  SSH_TRACE(2, ("Starting SecurID challenge query..."));          
  passphrase = ssh_read_passphrase(buf, FALSE);









  /* have passphrase, send to server */

  gdata->response_packet = ssh_buffer_allocate();

  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_BOOLEAN,
                    TRUE, 
                    SSH_FORMAT_UINT32_STR, passphrase, strlen(passphrase),
                    SSH_FORMAT_END);

  *gdata->state_placeholder = NULL;
  (*gdata->completion_proc)(SSH_AUTH_CLIENT_SEND,
                            gdata->user, gdata->response_packet,
                            gdata->completion_context);

  ssh_buffer_free(gdata->response_packet);
  ssh_xfree(passphrase);

  /* now terminate fsm */
  SSH_FSM_SET_NEXT("ssh_securid_finish");
  return SSH_FSM_CONTINUE;
}


/* abort authentication and stop fsm */
SSH_FSM_STEP(ssh_securid_error)
{
  SSH_FSM_GDATA(SshClientSecurIDAuth);

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                            gdata->user, NULL, gdata->completion_context);

  SSH_FSM_SET_NEXT("ssh_securid_finish");

  return SSH_FSM_CONTINUE;
}


SshFSMStateMapItemStruct securid_states[] =
{
  { "ssh_securid_new_pin", "User must choose or accept new pin",
     ssh_securid_new_pin },
  { "ssh_securid_send_pin", "Sends new pin to server",
     ssh_securid_send_pin },
  { "ssh_securid_select_new_pin", "Asks the user for a new pin",
     ssh_securid_select_new_pin },
  { "ssh_securid_accept_system_pin", "Asks the user to accept server pin",
     ssh_securid_accept_system_pin },
  { "ssh_securid_challenge", "Get user passphrase" \
     " after token change",  ssh_securid_challenge },
  { "ssh_securid_suspend", "Client waiting for server packet",
     ssh_securid_suspend },
  { "ssh_securid_process_packet", "Processing server packet",
     ssh_securid_process_packet },
  { "ssh_securid_error", "Error occured, abort ",
     ssh_securid_error },
  { "ssh_securid_finish", "Finished authentication",
     ssh_securid_finish },
};


void securid_destructor( void *gdata_ctx )
{
  SshClientSecurIDAuth gdata = (SshClientSecurIDAuth) gdata_ctx;
  SSH_PRECOND(gdata);

  if (gdata->user)
    ssh_xfree(gdata->user);

  if (gdata->pin_info->system_pin)
    ssh_xfree(gdata->pin_info->system_pin);

  if (gdata->pin_info->final_pin)
    ssh_xfree(gdata->pin_info->final_pin);

  if (gdata->pin_info)
  ssh_xfree(gdata->pin_info);
}


/* securid authentication, client-side. */

void ssh_client_auth_securid(SshAuthClientOperation op,
                            const char *user,
                            unsigned int packet_type,
                            SshBuffer packet_in,
                            const unsigned char *session_id,
                            size_t session_id_len,
                            void **state_placeholder,
                            SshAuthClientCompletionProc completion,
                            void *completion_context,
                            void *method_context)
{
  SshBuffer b;
/*  SshConfig clientconf = ((SshClient)method_context)->config; */
  SshClientSecurIDAuth state = *state_placeholder;
  SshFSM fsm;
  char buf[100]; 
  char *passcode;

  SSH_DEBUG(6, ("auth_securid op = %d  user = %s", op, user));
  switch (op)
    {

    case SSH_AUTH_CLIENT_OP_START:


      snprintf(buf, sizeof(buf), "Enter PASSCODE: ");

      /* XXX make this async at some point*/
      SSH_TRACE(2, ("Starting SecurID passphrase query..."));          
      passcode = ssh_read_passphrase(buf, FALSE);









      /* have passphrase, send to server */

      b = ssh_buffer_allocate();

      ssh_encode_buffer(b,
                        SSH_FORMAT_BOOLEAN,
                        FALSE,
                        SSH_FORMAT_UINT32_STR, passcode, strlen(passcode),
                        SSH_FORMAT_END);

      (*completion)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE,
                    user, b, completion_context);

      ssh_buffer_free(b);
      ssh_xfree(passcode);

      return;

    case SSH_AUTH_CLIENT_OP_CONTINUE:

      SSH_ASSERT(*state_placeholder == NULL);

      fsm = ssh_fsm_allocate(sizeof(*state),
                             securid_states,
                             SSH_FSM_NUM_STATES(securid_states),
                             securid_destructor );

      state = ssh_fsm_get_gdata_fsm(fsm);

      memset(state, 0, sizeof(*state));

      state->fsm = fsm;

      state->client = (SshClient)method_context;
      state->completion_proc = completion;
      state->completion_context = completion_context;
      state->state_placeholder = state_placeholder;
      state->server_packet = ssh_buffer_allocate();

      /* Copy packet_in (it will be freed after this function
         returns). */
      ssh_buffer_append(state->server_packet, ssh_buffer_ptr(packet_in),
                        ssh_buffer_len(packet_in));

      state->server_packet_type = packet_type;
      state->response_packet = NULL;
      state->user = ssh_xstrdup(user);
      *state_placeholder = state;
      state->pin_info = ssh_xcalloc(1, sizeof(*state->pin_info));

      state->main_thread = ssh_fsm_spawn(state->fsm,
                                         0, "ssh_securid_suspend",
                                         NULL, NULL);

      /* wake up thread to process packet. */
      ssh_fsm_set_next(state->main_thread, "ssh_securid_process_packet");
      ssh_fsm_continue(state->main_thread);

      return;

    case SSH_AUTH_CLIENT_OP_START_NONINTERACTIVE:
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;

    case SSH_AUTH_CLIENT_OP_ABORT:
      /* Clean up state. */
      if (state)
        if (state->main_thread)
          {
            ssh_fsm_set_next(state->main_thread, "ssh_securid_finish");
            ssh_fsm_continue(state->main_thread);
          }
      *state_placeholder = NULL;

      return;

     default:
      SSH_TRACE(0, ("unknown op %d", (int)op));
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
    }
}
#endif /* SSH_CLIENT_WITH_SECURID */
