/*

sshauthmethods.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SSH2 authentication methods for the server.

*/

#include "ssh2includes.h"
#include "sshauth.h"
#include "auths-passwd.h"
#include "auths-pubkey.h"
#include "auths-hostbased.h"
#ifdef DAEMON_WITH_PAM
#include "auths-pam.h"
#endif /* DAEMON_WITH_PAM */
#include "auths-kerberos.h"
#include "auths-kerberos-tgt.h"

#ifdef SSH_SERVER_WITH_SECURID
#include "auths-securid.h"
#endif /* SSH_SERVER_WITH_SECURID */

#include "sshdllist.h"

#define SSH_DEBUG_MODULE "SshAuthMethodServer"

static SshAuthServerMethodStruct server_methods[] =
{

  { SSH_AUTH_HOSTBASED, ssh_server_auth_hostbased },
#ifdef DAEMON_WITH_PAM
  { SSH_AUTH_PAM, ssh_server_auth_pam },
#endif /* DAMON_WITH_PAM */
#ifdef KERBEROS
  { SSH_AUTH_KERBEROS_TGT, ssh_server_auth_kerberos_tgt },
  { SSH_AUTH_KERBEROS, ssh_server_auth_kerberos },
#endif /* KERBEROS */

  { SSH_AUTH_PUBKEY, ssh_server_auth_pubkey },
  { SSH_AUTH_PASSWD, ssh_server_auth_passwd },
#ifdef SSH_SERVER_WITH_SECURID
  { SSH_AUTH_SECURID, ssh_server_auth_securid },
#endif /* SSH_SERVER_WITH_SECURID */
  { NULL, NULL }
};

/* Initializes the authentication methods array for the server. */
/* XXX This is basically the same as in sshauthmethodc.c . For
   maintainability and code clarity, these should be combined
   somehow. */
SshAuthServerMethod
ssh_server_authentication_initialize(SshDlList conf_methods)
{
  SshAuthServerMethod methods;
  SshDlList new_list;
  int i = 0, j = 0;
  char *cur_method = NULL, *existing = NULL;
  
  SSH_PRECOND(conf_methods);
  
  methods = ssh_xcalloc(sizeof(server_methods)/sizeof(*server_methods),
                        sizeof(*server_methods));

  /* Delete duplicate methods. */
  new_list = ssh_dllist_allocate();
  ssh_dllist_rewind(conf_methods);
  cur_method = ssh_dllist_current(conf_methods);

  for (;cur_method; ssh_dllist_fw(conf_methods, 1))
    {
      for (ssh_dllist_rewind(new_list);
           (existing = ssh_dllist_current(new_list)) != NULL;
           ssh_dllist_fw(new_list, 1))
        {
          SSH_ASSERT(existing);
          if (strcmp(existing, cur_method) == 0)
            break;
        }

      if (!existing)
        {
          ssh_dllist_add_item(new_list, cur_method, SSH_DLLIST_END);
          SSH_DEBUG(4, ("Added method \"%s\" to candidates.", cur_method));
        }
      
      cur_method = ssh_dllist_current(conf_methods);
    }

  /* Add the methods to the array in the order they are presented in
     the config. */
  ssh_dllist_rewind(new_list);
  cur_method = ssh_dllist_current(new_list);

  for (;cur_method; ssh_dllist_fw(new_list, 1),
         cur_method = ssh_dllist_current(new_list))
    {
      for (i = 0; server_methods[i].name != NULL; i++)
        {
          if (strcmp(server_methods[i].name, cur_method) == 0)
            {
              methods[j++] = server_methods[i];
              SSH_DEBUG(2, ("Added \"%s\" to usable methods.",
                            server_methods[i].name));
            }
        }
    }

  ssh_dllist_free(new_list);
  return methods;
}

/* Frees the returned authentication method array. */

void ssh_server_authentication_uninitialize(SshAuthServerMethod methods)
{
  memset(methods, 'F', sizeof(server_methods));
  ssh_xfree(methods);
}
