/*
  sshfc_overwrite.c

  Author: Tuomas Pelkonen <tpelkone@ssh.com>

  Copyright (C) 1999-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions for handling file overwrite
 */

#define SSH_DEBUG_MODULE "SshFileCopyOverwrite"

#include "sshincludes.h"
#include "sshfc_overwrite.h"

/* allocate a SshFileCopyOverwriteDir structure */
SshFileCopyOverwriteDir ssh_fc_overwrite_dir_allocate()
{
  SshFileCopyOverwriteDir dir;

  SSH_DEBUG(6, ("Allocating SshFileCopyOverwriteDir structure..."));
  dir = ssh_xcalloc(1, sizeof(*dir));

  dir->file_list = ssh_dllist_allocate(); 
  dir->dir_name = NULL;
  
  return dir;
}

/* destroy a SshFileCopyOverwriteDir structure */
void ssh_fc_overwrite_dir_destroy(SshFileCopyOverwriteDir dir)
{  
  ssh_dllist_rewind(dir->file_list);
  while(ssh_dllist_is_current_valid(dir->file_list))
    {
      ssh_xfree(ssh_dllist_current(dir->file_list));
      ssh_dllist_fw(dir->file_list, 1);
    }
  ssh_dllist_free(dir->file_list);
  if (dir->dir_name != NULL)
    ssh_xfree(dir->dir_name);
  ssh_xfree(dir);
}

/* allocate a SshFileCopyOverwriteFileList structure */
SshFileCopyOverwriteFileList ssh_fc_overwrite_file_list_allocate()
{
  SshFileCopyOverwriteFileList list;

  SSH_DEBUG(6, ("Allocating SshFileCopyOverwriteFileList structure..."));
  list = ssh_xcalloc(1, sizeof(*list));
  
  list->file_list = ssh_dllist_allocate();  
  list->current_path = ssh_xmalloc(255); 
  list->current_path[0] = '\0';
  list->current_dir = ssh_fc_overwrite_dir_allocate();
  
  return list;
}

/* destroy a SshFileCopyOverwriteFileList structure */
void ssh_fc_overwrite_file_list_destroy(SshFileCopyOverwriteFileList list)
{
  ssh_dllist_rewind(list->file_list);
  while(ssh_dllist_is_current_valid(list->file_list))
    {
      ssh_fc_overwrite_file_destroy((SshFileCopyOverwriteFile)
                                    ssh_dllist_current(list->file_list));
      ssh_dllist_fw(list->file_list, 1);
    }
  ssh_dllist_free(list->file_list);
  ssh_xfree(list->current_path);
  ssh_xfree(list->destination);
  ssh_xfree(list->slash);
  ssh_fc_overwrite_dir_destroy(list->current_dir);
  ssh_xfree(list);
}

/* allocate a SshFileCopyOverwriteFile structure */
SshFileCopyOverwriteFile ssh_fc_overwrite_file_allocate()
{
  SshFileCopyOverwriteFile file;

  SSH_DEBUG(6, ("Allocating SshFileCopyOverwriteFile structure..."));
  file = ssh_xcalloc(1, sizeof(*file));

  file->file_name = ssh_xmalloc(255);
  
  return file;
}

/* destroy a SshFileCopyOverwriteFile structure */
void ssh_fc_overwrite_file_destroy(SshFileCopyOverwriteFile file)
{
  ssh_xfree(file->file_name);
  ssh_xfree(file);
}

void *ssh_fc_overwrite_list_mapcar_subdir(void *item, void *ctx)
{
  SshFileCopyFile file = (SshFileCopyFile) item;
  SshFileCopyOverwriteFileList list = (SshFileCopyOverwriteFileList)ctx;
  SshFileCopyOverwriteFile overwrite_file;
  SshDlList current_list = list->current_list;
  
  char *temp;
  int len = strlen(list->current_path);

  temp = ssh_xmalloc(len + 1);
  strncpy(temp, list->current_path, len);
  temp[len] = '\0';
     
  list->current_path = ssh_xrealloc(list->current_path,
                                    len + 
                                    strlen(ssh_file_copy_file_get_name(file))
                                    + 2);
  strcat(list->current_path, ssh_file_copy_file_get_name(file));  

  if (file->dir_entries)
    {      
      /* 'file' is actually directory */
      strcat(list->current_path, list->slash);
      list->current_list = file->dir_entries->file_list;
      ssh_dllist_mapcar(file->dir_entries->file_list,
                        ssh_fc_overwrite_list_mapcar_subdir, ctx);    
      list->current_list = current_list;
    }
  else
    {
      /* add 'file' to file_list */
      overwrite_file = ssh_fc_overwrite_file_allocate();
      overwrite_file->file_name = ssh_xrealloc(overwrite_file->file_name,
                                               strlen(list->current_path) + 1);
      strcpy(overwrite_file->file_name, list->current_path);      
      overwrite_file->file = file;
      overwrite_file->parent_list = current_list;
      ssh_dllist_add_item(list->file_list,
                          (void *)overwrite_file, 
                          SSH_DLLIST_END);
    }
   
  strncpy(list->current_path, temp, len);
  list->current_path[len] = '\0';
  ssh_xfree(temp);

  return item;
}

void *ssh_fc_overwrite_list_mapcar(void *item, void *ctx)
{
  SshFileCopyFileListItem list_item = (SshFileCopyFileListItem) item;
  SshFileCopyOverwriteFileList list = (SshFileCopyOverwriteFileList)ctx;

  list->current_list = list_item->files->file_list;
  ssh_dllist_mapcar(list_item->files->file_list,
                    ssh_fc_overwrite_list_mapcar_subdir, ctx);
  
  return item;
}

/* the function that start the checking of file that already exists in the
   destination end. 'client' is the SshFileClient of the destination.
   'file_list' is the same file list that ssh_file_copy_recurse_dirs returns
   in the callback. 'file_list' contains all the file that will be transfered.
   Type of the items in the 'file_list' is SshFileCopyFileListItem.
   'destination' is the name of the destination directory that files will be
   transfered to. It must included the whole path from the root down. 'slash'
   is use to separate windows and unix. It can be "\\" or "/". 'overwrite_cb'
   is called when file already exists. This callback MUST call
   ssh_file_copy_overwrite_file with the same 'file' and 'list' in order to
   continue checking the files. 'done_cb' is called when all file are checked.
   'context' is the context which user can defined and the same context is
   passed in the callbacks. */   
void ssh_file_copy_overwrite(SshFileClient client,
                             SshDlList file_list,
                             const char *destination,
                             const char *slash,
                             SshFileCopyOverwriteCallback overwrite_cb,
                             SshFileCopyOverwriteDoneCallback done_cb,
                             void *context)
{
  SshFileCopyOverwriteFileList list = ssh_fc_overwrite_file_list_allocate();
  
  list->current_list = file_list;
  list->slash = ssh_xstrdup(slash);
  ssh_dllist_rewind(file_list);  

  /* transform 'file_list' which has SshFileCopyFileListItem items into
     a single list that has SshFileCopyOverwriteFile items */
  ssh_dllist_mapcar(file_list, ssh_fc_overwrite_list_mapcar, (void*)list);

  list->destination = ssh_xstrdup(destination);
  list->context = context;
  list->overwrite_cb = overwrite_cb;
  list->done_cb = done_cb;
  list->client = client;
  list->no_to_all = FALSE;
  
  /* start checking the files in the next(first) directory) */
  ssh_fc_overwrite_read_next_dir((void*)list); 
}

void ssh_fc_overwrite_read_next_dir(void *context)
{ 
  SshFileCopyOverwriteFileList list = (SshFileCopyOverwriteFileList)context;
  SshFileCopyOverwriteFile overwrite_file;
  char *temp;

  if (ssh_dllist_is_empty(list->file_list))
    {
      /* there are no more files */
      (list->done_cb)(list->context);
      ssh_fc_overwrite_file_list_destroy(list);
      return;
    }

  ssh_dllist_rewind(list->file_list);  
  overwrite_file = (SshFileCopyOverwriteFile)ssh_dllist_current(list->file_list);  
  
  temp = ssh_xmalloc(strlen(list->destination) +
                     strlen(overwrite_file->file_name) + 2);  
  strcpy(temp, list->destination);
  strcat(temp, list->slash);
  strcat(temp, overwrite_file->file_name);    

  ssh_fc_overwrite_get_path(temp);

  list->current_dir->dir_name = ssh_xrealloc(list->current_dir->dir_name,
                                             strlen(temp) + 1);
  strcpy(list->current_dir->dir_name, temp);

  ssh_dllist_clear(list->current_dir->file_list);

  /* open the next directory */
  ssh_file_client_opendir(list->client,
                          temp, 
                          ssh_fc_overwrite_opendir_cb, 
                          context);

  ssh_xfree(temp);
}

void ssh_fc_overwrite_get_path(char *file)
{
  char *temp;
  int pos;

  temp = strrchr(file, '/');
  if (temp != NULL)
    {
      pos = temp - file;
      if (pos >= 0 && pos < strlen(file))
        file[pos] = '\0';
      return;
    }

  temp = strrchr(file, '\\');
  if (temp != NULL)
    {
      pos = temp - file;
      if (pos >= 0 && pos < strlen(file))
        file[pos] = '\0';
    }
}

void ssh_fc_overwrite_get_filename(char *file)
{
  char *temp;

  temp = strrchr(file, '/');
  if (temp != NULL)
    {
      strcpy(file, temp + 1); 
      return;
    }

  temp = strrchr(file, '\\');
  if (temp != NULL)
    {
      strcpy(file, temp + 1);
    }
}

void ssh_fc_overwrite_opendir_cb(SshFileClientError error,
                              SshFileHandle handle,
                              void *context)
{
  SshFileCopyOverwriteFileList list = (SshFileCopyOverwriteFileList)context;
  SshFileCopyOverwriteFile overwrite_file;
  char *temp;
  if (error == SSH_FX_OK)
    {
      /* opening the directory was succesful. now it can be read */
      list->current_dir->handle = handle;
      ssh_file_client_readdir(handle, ssh_fc_overwrite_readdir_cb, context);
    }
  else
    {
      /* Directory didn't exists. Delete all the files that have this directory
         their path */
      while (ssh_dllist_is_current_valid(list->file_list))
        {
          overwrite_file = (SshFileCopyOverwriteFile)
                            ssh_dllist_current(list->file_list);
          temp = ssh_xmalloc(strlen(list->destination) +
                             strlen(overwrite_file->file_name) + 2);  
          strcpy(temp, list->destination);
          strcat(temp, list->slash);
          strcat(temp, overwrite_file->file_name);        
          ssh_fc_overwrite_get_path(temp);

          ssh_dllist_fw(list->file_list, 1);

          if (strcmp(temp, list->current_dir->dir_name) == 0)
            {
              ssh_dllist_delete_item(list->file_list, (void*)overwrite_file);
              ssh_fc_overwrite_file_destroy(overwrite_file);
            }
          ssh_xfree(temp);
        }
      /* move to the next directory */
      ssh_fc_overwrite_read_next_dir(context);
    }
}

void ssh_fc_overwrite_readdir_cb(SshFileClientError error,
                              const char *name,
                              const char *long_name,
                              SshFileAttributes attrs,
                              void *context)
{
  SshFileCopyOverwriteFileList list = (SshFileCopyOverwriteFileList)context;
  char *file;
  
  if (error != SSH_FX_EOF && name != NULL)
    {
      if ((attrs->permissions & SSH_UNIX_S_IFMT) != SSH_UNIX_S_IFDIR &&
          (attrs->permissions & SSH_UNIX_S_IFMT) != SSH_UNIX_S_IFLNK)
        {
          /* It is a file so add it to the file list */
          file = ssh_xstrdup(name);
          ssh_dllist_add_item(list->current_dir->file_list,
                              (void *)file, 
                              SSH_DLLIST_END);  
        }

      /* Read the next file in this directory */
      ssh_file_client_readdir(list->current_dir->handle,
                              ssh_fc_overwrite_readdir_cb, 
                              context);
    }
  else
    {
      /* No more files existed so close the directory */
      ssh_file_client_close(list->current_dir->handle,
                            ssh_fc_overwrite_file_status_cb, 
                            context);
    }
}

void ssh_fc_overwrite_file_status_cb(SshFileClientError error,
                                     void *context)
{
  SshFileCopyOverwriteFileList list = (SshFileCopyOverwriteFileList)context;
  ssh_dllist_rewind(list->file_list); 

  /* All files have been read from current directory so start checking which
     files that will be transfered to this directory already exist */
  ssh_fc_overwrite_next_file(list);  
}

void ssh_fc_overwrite_next_file(SshFileCopyOverwriteFileList list)
{
  SshFileCopyOverwriteFile overwrite_file;
  char *temp;
  char *file;
  int i;
  
  while (ssh_dllist_is_current_valid(list->file_list))
    {
      overwrite_file = (SshFileCopyOverwriteFile)
                        ssh_dllist_current(list->file_list);
    
      temp = ssh_xmalloc(strlen(list->destination) +
                         strlen(overwrite_file->file_name) + 2);  
      strcpy(temp, list->destination);
      strcat(temp, list->slash);
      strcat(temp, overwrite_file->file_name);        
      ssh_fc_overwrite_get_path(temp);

      ssh_dllist_fw(list->file_list, 1);

      if (strcmp(temp, list->current_dir->dir_name) == 0)
        {
          /* file will be transfered to this directory */
          ssh_dllist_delete_item(list->file_list, (void*)overwrite_file);
          ssh_dllist_rewind(list->current_dir->file_list);      
          while (ssh_dllist_is_current_valid(list->current_dir->file_list))
            {
              strcpy(temp, overwrite_file->file_name);
              ssh_fc_overwrite_get_filename(temp);
              file = (char*)ssh_dllist_current(list->current_dir->file_list);   
              
              for (i=0;i<strlen(file);i++)
                file[i] = tolower(file[i]);
              for (i=0;i<strlen(temp);i++)
                temp[i] = tolower(temp[i]);

              if (strcmp(file, temp) == 0)
                {         
                  /* file already exists in this directory */
                  ssh_xfree(temp);
                  if (list->no_to_all)
                    ssh_file_copy_overwrite_file(list,
                                                 overwrite_file, 
                                                 SSH_FC_OVERWRITE_NO);
                  else 
                    (list->overwrite_cb)(list, overwrite_file, list->context);
                  return;
                }
              ssh_dllist_fw(list->current_dir->file_list, 1);   
            }    
          ssh_fc_overwrite_file_destroy(overwrite_file);
        }
      ssh_xfree(temp);
    }
  
  /* start reading the next destination directory */
  ssh_fc_overwrite_read_next_dir((void*)list);
}

/* This function determines what to do with the file that already exists.
   'list' must be the same list that is passed in the callback. 'file' must
   be the same file that is passed in the callback. 'overwrite' determines
   what to do with file. Possible options are overwrite it, do not overwrite
   it, overwrite everything or cancel the whole file transfer */
void ssh_file_copy_overwrite_file(SshFileCopyOverwriteFileList list, 
                                  SshFileCopyOverwriteFile file, 
                                  SshFileCopyOverwrite overwrite)
{
  if (overwrite == SSH_FC_OVERWRITE_YES_TO_ALL)
    {
      (list->done_cb)(list->context);
      ssh_fc_overwrite_file_list_destroy(list);
      return;
    }
  else if (overwrite == SSH_FC_OVERWRITE_NO)
    {    
      ssh_dllist_delete_item(file->parent_list, (void*)file->file);
      ssh_file_copy_file_destroy((void*)file->file);
      ssh_fc_overwrite_file_destroy(file);
    }
  else if (overwrite == SSH_FC_OVERWRITE_CANCEL)
    {
      ssh_fc_overwrite_file_list_destroy(list);
      return;
    }
  else if (overwrite == SSH_FC_OVERWRITE_NO_TO_ALL)
    {
      list->no_to_all = TRUE;
    }
  
  ssh_fc_overwrite_next_file(list);
}

/* Returns the name of the 'file' */
const char *ssh_file_copy_overwrite_file_name(SshFileCopyOverwriteFile file)
{
  return file->file_name;
}

