/*

  sshrand.h

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (c) 2000 SSH Communications Security Corp
  All rights reserved.

  Created Wed Oct 18 22:51:22 2000.

  */

/* A simple (deterministic) pseudo-random number generator.

   Purpose of this file is to introduce a simple and standard mechanism
   for SSH software to generate random numbers in test programs in a
   deterministic way.

   For the convenience of test programs the caller can use the
   state passing approach, or the global state.

*/

#ifndef SSH_RAND_H
#define SSH_RAND_H

/***** Private structures. */

/* This section contains definitions of the internal representation
   of the pseudo-random number generator. This may change and applications
   should not rely on details of this section. */

/* The random number generation is based on following approach:

   Lagged Fibonacci generator (LFG) is used for generating individual
   pseudo-random numbers. The benefit is high-speed, and LFG's have
   good statistical justification. (At the moment the LFG uses a
   simple extension, which may cause unstable behaviour. More study on
   this will be conducted.)

   Inversive Congruental generator (ICG) is used to seed the
   LGF. ICG's are statistically very stable generator, however, in
   practice very slow.  */
  
/* The Lagged Fibonacci generator vector size. This was selected
   to achieve a period close to 2^128. */
#define SSH_RAND_LGF_VSIZE 98
struct SshLFGStateRec
{
  /* The lagged Fibonacci vector and the current position. */
  SshUInt32 v[SSH_RAND_LGF_VSIZE], pos;
};

typedef struct SshLFGStateRec    SshLFGStateStruct;

/* The main random state structure. */
struct SshRandStructRec
{
  SshLFGStateStruct lfg;
};

/***** Public interface. */

/* This section contains the public interface of the pseudo-random
   number generator. */

/* The random state context. */
typedef struct SshRandStructRec *SshRand;
typedef struct SshRandStructRec  SshRandStruct;

/* Seed the global generator state. Any value from [0, 2^32) is
   allowed for the seed, including the value zero. The caller may
   trust that no pair of seeds will produce pseudo-random number
   sequences that are highly correlated.

   Note. There is no such proof that correlation would not be
   possible, however, it is unlikely.

   Seeding is not a fast operation, it is approximately 1000 times
   slower than obtaining a new pseudo-random number from the
   generator.  */
void ssh_rand_seed(SshUInt32 seed);

/* Obtain a pseudo-random number uniformly selected from [0,2^32). It
   is guaranteed that the sequence of numbers produced has a high
   period, and passes most empirical statistical randomness tests.

   This operation updates the global state irreversibly.
*/
SshUInt32 ssh_rand(void);

/* The local state interface. */

/* As a general rule a variable `state' may be NULL, and it will be
   interpreted as the global state. */

/* Re-seed a state. This operation destroys the previous state
   irreversibly. The `state' may be NULL in which case the global state
   is reseeded.

   See also the comments above for `ssh_rand_seed'. */
void ssh_rand_state_seed(SshRand state, SshUInt32 seed);

/* Obtain a pseudo-random number uniformly selected from [0,2^32). The
   `state' may be NULL in which case the global state is used to obtain
   the number.

   This operation updates the state irreversibly.
*/
SshUInt32 ssh_rand_state(SshRand state);

/* Make a copy of the current state. This can be used to backtrack in
   a program that uses randomness to direct its state transition.
   The states `dst' and `src' may either or both be NULL, in which case
   NULL is replaced by the global state.

   This operation is efficient and does not require reseeding.
*/
void ssh_rand_state_copy(SshRand dst, SshRand src);

#endif /* SSH_RAND_H */

