/*
  sshthread.h

  Author: Jukka Aittokallio <jai@ssh.com>

  Copyright (C) 2000 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Generic interface to threads. Some systems may offer some
  synchronous function interfaces (example of this is the
  smart card API in Microsoft Windows). Because all the time
  consuming functions within SSH libraries should be asynchronous,
  the only way is to "threadify" those synchronous calls. Be aware that
  most of the modules in SSH library are not thread safe, so
  threads are to be used very carefully. Generally, use threads
  only to make external synchronous APIs asynchronous.
*/
#ifndef SSH_THREAD_H
#define SSH_THREAD_H

#ifdef __cplusplus
extern "C" {
#endif

/* Handle to executing thread. This handle is valid till
   SshThreadDoneCB is called. You never have to free the
   context because it is done automatically by the thread library.
   This handle is only used with ssh_thread_wait function. */
typedef struct SshThreadRec *SshThread;

/* When you call ssh_thread_fork, it creates a new system thread and
   calls this type of function from it. The value you return from
   this callback function is given as an argument to SshThreadDoneCB
   callback which is called when thread execution has been finished. */
typedef void * (*SshThreadFunc)(void *context);

/* This callback is called when the execution of a thread has been
   finished. 'thread_context' is the context you gave as an
   argument to ssh_thread_fork function. 'return_value' is a
   value returned by the SshThreadFunc. */
typedef void (*SshThreadDoneCB)(void *thread_context,
                                void *return_value);

/* Forks a new thread. 'func' is executed in a new thread. 'thread_done'
   is called after thread has finished execution. It is called from
   the main event loop thread. 'thread_done' can be also NULL in case
   of which the only way to know when thred is finished is to call
   the ssh_thread_wait function. 'context' is given as an argument to
   SshThreadFunc function and also to SshThreadDoneCB callback. */
SshThread
ssh_thread_fork(SshThreadFunc func,
                SshThreadDoneCB thread_done,
                void *context);

/* Waits a thread to finish its execution. This function can be called
   from any thread that knows the valid thread handle. This function
   stops the execution of the caller until the thread has finished its 
   execution. SshThreadDoneCB is called before this function 
   returns to caller. */
void
ssh_thread_wait(SshThread thread);

/* This function MUST NOT BE called from the main event loop thread.
   This function stops the execution of the thread for a specific
   amount of time. */
void
ssh_thread_sleep(SshUInt32 milliseconds);

#ifdef __cplusplus
}
#endif

#endif /* SSH_THREAD_H */
