/*

sshasyncfilterstream.h

Authors: Tatu Ylonen <ylo@ssh.fi>
         Tomi Salo <ttsalo@ssh.com>

Copyright (c) 1997,2000 SSH Communications Security, Finland
                        All rights reserved

An object for filtering a data stream bidirectionally.  The object
allows callbacks to modify and filter data bidirectionally, and allow
them to disconnect the connection.

*/

#ifndef SSHASYNCFILTERSTREAM_H
#define SSHASYNCFILTERSTREAM_H

#include "sshstream.h"
#include "sshbuffer.h"

typedef struct SshStreamAsyncFilterRec *SshStreamAsyncFilter;

/* Operation codes returned by filter functions.  */

/* Indicates that the given number of bytes should be passed through.
   The number should not be greater than the number of bytes in the
   buffer.  The value zero indicates to keep all bytes in the buffer,
   and call the filter function again when more data has been
   received.  If ``nbytes'' is less than the number of bytes in
   buffer, the remaining bytes are kept in the buffer. */
#define SSH_ASYNC_FILTER_ACCEPT(nbytes)  (nbytes)

/* Instructs to keep the data in buffer, and call the filter again when
   more data has been received.  Returning this has the same effect as
   accepting zero bytes. */
#define SSH_ASYNC_FILTER_HOLD            0

/* Indicates that the stream should be immediately disconnected.  All bytes
   in the buffer are thrown away, and EOF will be returned by the stream
   in both directions.  No more data will be accepted.  The filter functions
   will not be called again (but ``destroy'' will be called when the
   application closes the stream). */
#define SSH_ASYNC_FILTER_DISCONNECT      -1

/* Indicates that the stream should be shortcircuited in both directions.
   Data still in buffers is flushed in both directions, and from then on,
   any data will be directly transmitted through.  The filter functions
   will not be called again (but ``destroy'' will be called when the
   application closes the stream). */
#define SSH_ASYNC_FILTER_SHORTCIRCUIT    -2

/* Data direction definitions.
   SSH_ASYNC_FILTER_INBOUND refers to the data going into the stream,
   SSH_ASYNC_FILTER_OUTBOUND to the data coming from the stream. */
typedef enum {
  SSH_ASYNC_FILTER_INBOUND,
  SSH_ASYNC_FILTER_OUTBOUND
} SshStreamAsyncFilterDataDir;

/* Filter function. The filter stream will call this function for
   every piece of data passing to/from the stream (two different
   function pointers must be given when creating the filter, one will
   be called for outbound data and one for inbound). ``data'' contains
   data received from the stream so far.  Data up to ``offset'' has
   been already accepted, and should not be touched.  This function is
   allowed to modify data after offset, and even add/remove data
   there.  When accepting bytes, data before ``offset'' should not be
   counted.  It has already been accepted, and is only kept in the
   buffer until it can be written out. The filter function can modify
   the unaccepted data and when ready to accept (some of) the data, it
   should call the function ``ssh_stream_async_filter_accept''.
     `data'         data received so far; contents may be modified
     `offset'       offset where unaccepted data starts
     `eof_received' TRUE if EOF has already been received in this direction
     'filter_context' context for this filter
     `context'      context argument from creating the filter. */
typedef void (*SshAsyncFilterProc)(SshBuffer data, size_t offset,
                                   Boolean eof_received,
                                   SshStreamAsyncFilter filter_context,
                                   void *context);

/* The filter function (or some other part of the program) must call
   this function to accept the data in the buffer given to it by the
   filter stream. The operation codes are explained in the beginning
   of this file. Direction should be set according to which buffer
   (inbound or outbound) is in question. This function can be called
   straight from the filter function or later from elsewhere. */

void ssh_stream_async_filter_accept(SshStreamAsyncFilter filter_context,
                                    SshStreamAsyncFilterDataDir direction,
                                    int operation);

/* Returns the current offset to the start of the unaccepted data in the buffer. */

size_t
ssh_stream_async_filter_get_offset(SshStreamAsyncFilter sf,
                                   SshStreamAsyncFilterDataDir direction);

/* Creates a stream that can be used to filter data to/from another
   stream.  ``stream'' is an already existing stream whose data is to
   be filter.  It is wrapped into the filter stream, and will be
   closed automatically when the filter stream is closed.
   ``to_stream_filter'', if non-NULL, is a filter to call whenever
   data is written to the returned stream (and is on its way to
   ``stream'').  ``from_stream_filter'' (if non-NULL) is called
   whenever data is received from ``stream''.  ``destroy'' (if
   non-NULL) is called when the returned stream is closed; it can be
   used to free ``context''.  The filter functions must ensure that the
   buffer does not grow unboundedly.
     `stream'             stream whose data is to be filtered
     `max_buffer_size'    maximum number of bytes to buffer
     `to_stream_filter'   filter for data going to ``stream'', or NULL
     `from_stream_filter' filter for data coming from ``stream'', or NULL
     `destroy'            called when the returned stream is closed, or NULL
     `context'            context argument to pass to the functions.
   The filter functions are not allowed to directly destroy the stream. */
SshStream ssh_stream_async_filter_create(SshStream stream,
                                         size_t max_buffer_size,
                                         SshAsyncFilterProc to_stream_filter,
                                         SshAsyncFilterProc from_stream_filter,
                                         void (*destroy)(void *context),
                                         void *context);

#endif /* SSHASYNCFILTERSTREAM_H */
