/*

  Author: Tim Rochford <rochford@ssh.fi>

  Copyright (C) 2000 SSH Communications Security Oy, Helsinki, Finland
  All rights reserved.


        L2TP test functions.

  */

#define SSH_DEBUG_MODULE "SshL2TP"

#include "sshincludes.h"
#include "sshbuffer.h"
#include "ssheloop.h"
#include "sshtimeouts.h"
#include "sshfdstream.h"
#include "sshpacketstream.h"

#include "l2tp.h" /* XXX ../sshutil/tests/ mv to includes */
#include "ssh_ppp_framing.h"

#include <netinet/in.h> 

/* #define BASIC_TEST_INITIATE_TUNNEL_ONLY */
/* #define BASIC_TEST_MANY_INCOMING_CALLS  */

#define MAX_MESG_SIZE 1024

void ssh_l2tp_can_send_cb(void *context);

void ssh_l2tp_eof_cb(void *context);

void ssh_l2tp_packet_recv(SshPacketType type,const unsigned char *data,
                          size_t len, void *context); 

unsigned char *ssh_l2tp_get_error_mesg(l2tp_tunnel *tun);

/* Use this only */
#define SSH_L2TP_SPD_FILE_PATH "/home/tim/dev/tmp/src/lib/sshutil/tests/sshl2tp.spd" 


#define SSH_L2TP_SPDFILE "sshl2tp.spd"

/* 16 + ZLB */
#define NUM_L2TP_MESGS 17

/* From book */
#define SEQ_NUM_LT(a, b) ((a) != (b) && ((((a) +1) - (b)) &0x8000))

SshUInt16 SSH_AVP_TYPE_MESSAGETYPE = 0x0000;
SshUInt16 SSH_AVP_TYPE_RANDOMVECTOR = 0x0000;    /* XXX  avp value */
SshUInt16 SSH_AVP_TYPE_RESULTCODE = 0x0001;
SshUInt16 SSH_AVP_TYPE_PROTOCOLVERSION = 0x0002;
SshUInt16 SSH_AVP_TYPE_FRAMINGCAPABILITIES = 0x0003;
SshUInt16 SSH_AVP_TYPE_BEARERCAPABILITIES = 0x0004;
SshUInt16 SSH_AVP_TYPE_TIEBREAKER = 0x0005;
SshUInt16 SSH_AVP_TYPE_FIRMWAREREVISION = 0x0006;
SshUInt16 SSH_AVP_TYPE_HOSTNAME = 0x0007;
SshUInt16 SSH_AVP_TYPE_VENDORNAME = 0x0008;
SshUInt16 SSH_AVP_TYPE_ASSIGNEDTUNNELID = 0x0009;
SshUInt16 SSH_AVP_TYPE_RECEIVEWINDOWSIZE = 0x000A;
SshUInt16 SSH_AVP_TYPE_CHALLENGERESPONSE = 0x000B;
SshUInt16 SSH_AVP_TYPE_CAUSECODE = 0x000C;
SshUInt16 SSH_AVP_TYPE_ASSIGNEDSESSIONID = 0x000E;
SshUInt16 SSH_AVP_TYPE_CALLSERIALNUMBER = 0x000F;
SshUInt16 SSH_AVP_TYPE_MINIMUMBPS = 0x0010;
SshUInt16 SSH_AVP_TYPE_MAXIMUMBPS = 0x0011;
SshUInt16 SSH_AVP_TYPE_BEARERTYPE = 0x0012;
SshUInt16 SSH_AVP_TYPE_FRAMINGTYPE = 0x0013;
SshUInt16 SSH_AVP_TYPE_CALLEDNUMBER = 0x0014;
SshUInt16 SSH_AVP_TYPE_CALLINGNUMBER = 0x0015;
SshUInt16 SSH_AVP_TYPE_SUBADDRESS = 0x0016;
SshUInt16 SSH_AVP_TYPE_TXCONNECTSPEED = 0x0017;
SshUInt16 SSH_AVP_TYPE_RXCONNECTSPEED  = 0x0018;
SshUInt16 SSH_AVP_TYPE_PHYSICALCHANNELID  = 0x0019;
SshUInt16 SSH_AVP_TYPE_PRIVATEGROUPID  = 0x0020;
SshUInt16 SSH_AVP_TYPE_SEQUENCINGREQUIRED = 0x0021;

SshFSMStateMapItemStruct l2tp_states[] =
{
  /*  Connection establishment */
  { "L2TP idle", 
    /* "Waiting for a local open a recv a good sccrq " */ NULL , 
    l2tp_idle},
  { "L2TP wait_ctl_reply", 
    /* "can now timeout, gd sccrp, loose tie or stop ccn" */ NULL , 
    l2tp_wait_ctl_reply},
  { "L2TP established", 
    /* "need hello, recv a strt cont con msg" */ NULL , 
    l2tp_established},
  { "L2TP cleanup", 
    /* "now recvd ZLB for StopCCN, free(), resets" */ NULL , 
    l2tp_cleanup} ,
  { "L2TP cleanupwait", 
    /* "sent StopCCN, but wait for ack," */ NULL ,  
    l2tp_cleanupwait},
  { "L2TP wait_ctl_conn", 
    /* "sent sccrp, wait goodscccn" */ NULL  , 
    l2tp_wait_ctl_conn},

  /* incoming calls, restricted to tun->setLNS == FALSE */
  { "L2TP IN idle", 
    /* "Waiting for a local open a recv a good Iccrq "*/ NULL , 
    l2tp_incoming_idle},
  { "L2TP IN wait_ctl_reply", 
    /* "can now timeout, gd Icrp, loose tie or CDN" */ NULL , 
    l2tp_incoming_wait_ctl_reply},
  { "L2TP IN established", 
    /* "recv a strt cont con msg" */ NULL , 
    l2tp_incoming_established},
  { "L2TP IN cleanup", 
    /* "now recvd ZLB for CDN, free(), resets" */ NULL , 
    l2tp_incoming_cleanup} ,
  { "L2TP IN cleanupwait", 
    /* "sent StopCCN, but wait for ack," */ NULL  , 
    l2tp_incoming_cleanupwait},
  { "L2TP IN wait_ctl_conn", 
    /* "sent sccrp, wait good Iccn" */ NULL , 
    l2tp_incoming_wait_ctl_conn},

  /* outgoing calls, restricted to tun->setLNS == TRUE */
  { "L2TP OUT idle", 
    /* "Waiting for a local open a recv a good ocrq " */ NULL , 
    l2tp_outgoing_idle},
  { "L2TP OUT wait_ctl_reply", 
    /* "can now timeout, gd ocrp, loose tie or CDN" */ NULL  , 
    l2tp_outgoing_wait_ctl_reply},
  { "L2TP OUT established", 
    /* "recv a strt cont con msg" */ NULL  , 
    l2tp_outgoing_established},
  { "L2TP OUT cleanup", 
    /* "now recvd ZLB for CDN, free(), resets" */ NULL, 
    l2tp_outgoing_cleanup} ,
  { "L2TP OUT cleanupwait", 
    /* "sent StopCCN, but wait for ack." */ NULL , 
    l2tp_outgoing_cleanupwait},
  { "L2TP OUT wait_ctl_conn", 
    /* "sent sccrp, wait good occcn" */ NULL , 
    l2tp_outgoing_wait_ctl_conn}
};

typedef struct t_tas_context
{
  SshStream pstream;
  SshBuffer pbuffer;
}* t_tas_context;

static void t_tas_stream_callback(SshStreamNotification notification, 
                                  void* pdata);

int main(int argc, char *argv[])
{
  l2tp_tunnel *tun;
  SshFSM fsm;
  SshFSMThread thread;

  /* TAS context */
  t_tas_context pcontext = 0;

  /* TAS allocate and initialize context */
  pcontext = ssh_xmalloc(sizeof (*pcontext));
  memset(pcontext, 0, sizeof (*pcontext));
  
  /* TAS allocate buffer */
  pcontext->pbuffer = ssh_buffer_allocate();
 
  ssh_debug_set_global_level(0); 
  

  /* if(argc != 1)
    {      
      ssh_debug("t-l2tp <<<<stick your pppd data here>>>>");
      }*/
        
  ssh_event_loop_initialize();

  /* tun = ssh_l2tp_control_channel_init(); */ 
  tun = ssh_l2tp_control_channel_init((unsigned char *)argv[1], 
                                      (unsigned char *)argv[2], 
                                      (unsigned char *)argv[3], 
                                      (unsigned char *)argv[4], 
                                      (unsigned char *)argv[5]); 

  if(tun == NULL)
    {
      ssh_debug("Tunnel creation failed");
      return -1;
    }

  fsm = ssh_fsm_allocate(sizeof(tun), l2tp_states, 
                         SSH_FSM_NUM_STATES(l2tp_states), NULL);
  thread = (void *)ssh_fsm_spawn(fsm, 0, "L2TP idle", NULL, NULL);      
        
  tun->state = "L2TP idle";
  ssh_fsm_push_tdata(thread, tun);
  tun = ssh_fsm_get_tdata(thread);

  /* TAS open stdin/stdout stream */
  /* pcontext->pstream = ssh_stream_fd_stdio(); */

  /* TAS set stream callback */
  /* ssh_stream_set_callback(pcontext->pstream,
                          t_tas_stream_callback, 
                          pcontext); 
  */ 

  ssh_event_loop_run();

  ssh_event_loop_uninitialize();  
  
  /* free buffer */
  ssh_buffer_free(pcontext->pbuffer);

  /* free context */
  ssh_xfree(pcontext);

  return 0;
}

/* The data could be extrated from a SshBuffer before it gets to this function.  
 */ 
size_t ssh_l2tp_send_data(l2tp_tunnel *tun)
{
  static SshUInt16 l2tpData[MAX_MESG_SIZE]; 
  size_t message_length;
  int i;
  SshUInt16 *buf;
  static unsigned char *data;


  data = ssh_buffer_ptr(tun->pbuffer);
  /* XXX offset and other hdr fields (optional) */
  /* T bit(0), L bit(1), Seq(0), P (0), Ver 0x2   */
  message_length = 8; 
  /* default */
  l2tpData[0] = htons(0x4002); 

  /* Add seq num to data channel */
  if(tun->seq_data)
    message_length =+ 4;

  /* Do we need to pad the data ? */
  for(i = strlen((const char *) data); i > 0; i=-2)
    {
      if(i == 1)
        {
          /* Set offset bit */
          l2tpData[0] = htons(0x4402);
          message_length =+ 3;
        }
    }
  /* Allocate the gen purpose buffer for strogin data */
  tun->pbuffer = ssh_buffer_allocate();
  
  l2tpData[2] = htons(tun->tunnel_id); 
  l2tpData[3] = htons(tun->call_id);
  /* If Add seq num to data channel */
  if(tun->seq_data)
    {
      l2tpData[4] = htons(tun->dataSs); 
      l2tpData[5] = htons(tun->dataSr);
    } 
  /* if loop counter still 1, offset padding present, the l2tp data will begin 
     this many bytes beyond the l2tp header. */
  if(i == 1)
    {
      l2tpData[6] = htons(i);
      l2tpData[7] = 0x0000; /* XXX really only an octet */
      /* ssh_debug("we pad"); */
    }
  l2tpData[1] = htons(message_length + strlen((const char *)data)  ); 

  ssh_buffer_append(tun->pbuffer, (unsigned char *)l2tpData, message_length); 
  ssh_buffer_append(tun->pbuffer, data, strlen((const char *)data));
  
  /* ssh_ppp_framing_encode(tun->pbuffer); */
      
  buf = (SshUInt16 *) ssh_buffer_ptr(tun->pbuffer);

  ssh_udp_send((void *)tun->myListener, tun->remote_address, tun->remote_port, 
               (const unsigned char *)&buf, ssh_buffer_len(tun->pbuffer));

  /* XXX how reliably do we send the payload data ... */
  ssh_debug("SENT DATA (%d)", message_length + strlen((const char *)data)); 
  return message_length + strlen((const char *)data);
}

size_t ssh_l2tp_send_control(size_t message_type, 
                             l2tp_tunnel *tun)
{
  size_t message_length; 
  SshUInt16 l2tpData[MAX_MESG_SIZE]; 
  SshUInt16 *key_holder;   
  SshUInt32 bc_data;

  switch(message_type)
    {
      case SSH_SCCRQ:
        message_length = 52 + strlen(tun->myHostName); 
        if(tun->receive_window_size_avp)
          message_length += 8;
        if(tun->bearer_capabilities_avp)
          message_length += 10;
        if(tun->vendor_name_avp)
          message_length += 6 + strlen(tun->myVendorName);
        if(tun->firmware_revision_avp)
          message_length += 8;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_SCCRQ);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_PROTOCOLVERSION, 0x0100);
        ssh_add_avp_vl(tun, l2tpData, tun->running_length_count, 
                       SSH_AVP_TYPE_HOSTNAME, tun->myHostName, 
                       strlen(tun->myHostName));
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_FRAMINGCAPABILITIES, 0x00000001);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDTUNNELID, tun->tunnel_id);
        if(tun->receive_window_size_avp)
          ssh_add_avp(tun, l2tpData,tun->running_length_count, 
                      SSH_AVP_TYPE_RECEIVEWINDOWSIZE, tun->rws);
        if(tun->bearer_capabilities_avp)
          {
            bc_data = tun->support_asynchronous & tun->support_synchronous;   
            ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                        SSH_AVP_TYPE_BEARERCAPABILITIES, bc_data);
          }
        if(tun->vendor_name_avp)          
          ssh_add_avp_vl(tun, l2tpData, tun->running_length_count, 
                       SSH_AVP_TYPE_VENDORNAME, tun->myVendorName, 
                       strlen(tun->myVendorName));
        if(tun->firmware_revision_avp)
           ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_FIRMWAREREVISION, tun->firmware_rev);
        break;

      case SSH_SCCRP:
        /* Mandatory  */
        message_length = 52 + strlen(tun->myHostName); 
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_SCCRP);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_PROTOCOLVERSION, 0x0100);
        ssh_add_avp_vl(tun, l2tpData, tun->running_length_count, 
                       SSH_AVP_TYPE_HOSTNAME, tun->myHostName, 
                       strlen(tun->myHostName));
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_FRAMINGCAPABILITIES, 0x00000001);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDTUNNELID, tun->tunnel_id);
        break;
      case SSH_SCCCN:
        if(strcmp(tun->state, "L2TP wait_ctl_reply") != 0 )
          {
            ssh_debug("[%s] not in correct state to send mesg(%d)", tun->state, 
                      message_type );
            exit(1);
          }
        message_length = 20;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_SCCCN);
        break;
      case SSH_STOPCCN:
        message_length = 38;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_STOPCCN);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDTUNNELID, tun->tunnel_id);
        /* XXX result code, error code. If a error message is included then 
           the  ssh_add_avp_vl() function should be used instead. */
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_RESULTCODE, 0x00010001);

        break;

      case SSH_HELLO:
        message_length = 20;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_HELLO);
        break;

      case SSH_OCRQ:
        ssh_debug("SSH_OCRQ");
        message_length = 88;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_OCRQ);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDSESSIONID, tun->call_id);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_CALLSERIALNUMBER, tun->call_serial_number);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_MINIMUMBPS, tun->minbps);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_MAXIMUMBPS, tun->maxbps);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_BEARERTYPE, tun->bearer_type);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_FRAMINGTYPE, tun->framing_type);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_CALLEDNUMBER, tun->callednumber);
        break;
      case SSH_OCRP:
        ssh_debug("SSH_OCRP");
        message_length = 28;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_OCRQ);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDSESSIONID, tun->call_id);
        break;
      case SSH_OCCN:
        message_length = 40;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_OCCN);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_TXCONNECTSPEED, 0x12345678);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_FRAMINGTYPE, tun->framing_type);
        break;
      case SSH_ICRQ:
        message_length = 38;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_ICRQ);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDSESSIONID, tun->call_id);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_CALLSERIALNUMBER, tun->call_serial_number);
        break;

      case SSH_ICRP:
        message_length = 28;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_ICRP);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDSESSIONID, tun->call_id);
        break;
      case SSH_ICCN:
        message_length = 40;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_ICCN);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_TXCONNECTSPEED, 0x12345678);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_FRAMINGTYPE, tun->framing_type);
        break;

      case SSH_CDN:
        message_length = 38;
        ssh_add_control_header(tun, l2tpData, message_length);
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_MESSAGETYPE, SSH_CDN);
        ssh_add_avp32(tun, l2tpData, tun->running_length_count, 
                      SSH_AVP_TYPE_RESULTCODE, 0x00010001); 
        ssh_add_avp(tun, l2tpData, tun->running_length_count, 
                    SSH_AVP_TYPE_ASSIGNEDSESSIONID, tun->call_id);
        break;

      case SSH_WEN:
        /* only if lac to lns */
        if(tun->setLNS)
          {
            ssh_debug("ERROR - LNS not allowed to send this message type");
            exit(1);
          }
        else
            ssh_debug("SSH_WEN");
        break;
      case SSH_SLI:
        ssh_debug("SSH_SLI");
        break;
      case ZLB:
        message_length = 12;
        ssh_add_control_header(tun, l2tpData, message_length);
    }
  ssh_udp_send((void *)tun->myListener, tun->remote_address, tun->remote_port, 
               (const unsigned char *)&l2tpData, message_length); 
  /* Were going to treat keepalive messages differently */
  if(message_type != ZLB)
    {
      tun->mesg = (void *)&l2tpData;
      tun->mesg_len = (void *)message_length;
      /* XXX  Test out adding a vlaue to the hash table */
      key_holder = (SshUInt16 *)tun->Ss;
      ssh_mapping_put_vl(tun->unackedTable, (void *) &key_holder, 
                         sizeof(key_holder), (void *)&tun->mesg, message_length);
      /* ssh_debug("[%s] Waiting for Ack for Mesg(%x)", tun->state, key_holder ); 
        
      ssh_register_timeout(tun->timeoutValue[tun->reTx++],0, sender_timeout, 
                           tun); 
      */ 
      tun->mesg = (void *)message_type;
      if(message_type == SSH_HELLO )
        {
          ssh_cancel_timeouts(keep_alive_timeout, SSH_ALL_CONTEXTS);
          ssh_register_timeout(tun->keepAliveTimerValue,0, 
                               keep_alive_timeout, tun);
          
          ssh_register_timeout(tun->timeoutValue[tun->reTx],0, sender_timeout, 
                               tun); 
        }
      else
        {
          ssh_register_timeout(tun->timeoutValue[tun->reTx],0, sender_timeout, 
                               tun); 
        }
      tun->Ss++;
    }
  return message_length;
}

/* This procedure will be called when a tunnel needs a hello message */
void keep_alive_timeout(void *context)
{
  l2tp_tunnel *tun;

  /* Can only send hellos in an established state */
  if(strcmp(tun->state,"L2TP established") ==  0
     || strcmp(tun->state,"L2TP IN established") ==  0
     || strcmp(tun->state,"L2TP OUT established") ==  0)
    {
      tun = (l2tp_tunnel *)context;
      ssh_debug("[%s] Sending hello mesg", tun->state);
      ssh_l2tp_send_control(SSH_HELLO, tun);
    }
  else
    {
      if(strcmp(tun->state,"L2TP established") ==  0)
        ssh_l2tp_send_control(SSH_STOPCCN, tun);
      else
        ssh_l2tp_send_control(SSH_CDN, tun);
    }
}

/* We have timed out, meaning that a message has been sent, but the alloted 
   time has passed. Check if the sequence number of the timed out message 
   has been received, ie, in the hash table. */
void sender_timeout(void *context)
{
  l2tp_tunnel *tun;
  SshUInt16 *rvalue; 
  size_t rlen;  
  SshUInt16 *key_holder;

  tun = (l2tp_tunnel *)context;
  if(tun->reTx < tun->maxReTx)
    {
      tun->reTx++;
      /* XXX The problem with incrementing the Seq Num Ss in the send control 
         function is the assumption that a UDP message will get through 
         (common case), but here we want to retransmit the previous mesg, so 
         tun->Ss--  */
      key_holder = (void *)((SshUInt16 *)tun->Ss - (SshUInt16 *)1);
      ssh_debug("[%s] Searching for key, %x", tun->state, key_holder );
      if(ssh_mapping_copy_value_vl(tun->unackedTable, (void **)&rvalue, 
                                   rlen))
        ssh_debug("XXX hello worldly ");
      /*   check that the sequence number I timed out on is in the hash table */
      if(ssh_mapping_get_vl(tun->unackedTable, (void *) &key_holder, 
                            sizeof((int *)key_holder),(void **)&rvalue, &rlen))
        {

          ssh_debug("[%s] Retransmitting %x (%d retx)", tun->state, key_holder, 
                    tun->reTx);
          /* if(!ssh_mapping_copy_value_vl(tun->unackedTable,(void *)&rvalue, 
             &rlen)) */
          /* Should the same mechanism be used to resend or just a udp_send() 
             ??? Check can use rlen in next function */
          ssh_udp_send((void *)tun->myListener, tun->remote_address, 
                       tun->remote_port, (const unsigned char *)tun->mesg, 
                       rlen);
          ssh_register_timeout(tun->timeoutValue[tun->reTx],0, 
                               sender_timeout, tun);
          
        }
      /* Else the message sent does not have a key in the table, ie it has 
         been removed becuase it has been acknowledged, do nothing */
    }
  else 
    {
      /* Close the whole show down */
      ssh_debug("[%s] ERROR - Timed out ... closing tunnel", tun->state);
      /* XXX not sure about what to send here */
      if(strcmp(tun->state,"L2TP established") ==  0
         || strcmp(tun->state,"L2TP IN established") ==  0
         || strcmp(tun->state,"L2TP OUT established") ==  0)
        ssh_l2tp_send_control(SSH_CDN, tun);
      else
        ssh_l2tp_send_control(SSH_STOPCCN, tun);
      tun->Ss++;
      /* ssh_debug("[%s] Closing tunnel", tun->state); */
      /* XXX Uninit() */
      /* call not estlablished in time - cdn err msg */
      tun->error_code = 10;

      exit(1); 
    }
}

/* This function is called upon receiving a valid L2TP control message. 
   The Nr value of the received message needs to be checked against 
   the unacknowledged messages in the hash table. */
size_t ssh_ack(l2tp_tunnel *tun, size_t mesg)
{
  SshUInt16 *key_holder;
  SshUInt16 *rvalue; 
  size_t rlen;  

  key_holder = (void *)((SshUInt16 *)mesg - (SshUInt16 *)1);
  /* ssh_debug("[%s] Received ACK for(%x)", tun->state, key_holder); */
  /* check that the sequence number received is in the hash table */
  if(ssh_mapping_get_vl(tun->unackedTable, (void *) &key_holder, 
                        sizeof(key_holder), (void *)&rvalue, &rlen))
    {
      ssh_debug("[%s] recvd ack (%x)", tun->state, key_holder); 
      ssh_cancel_timeouts(sender_timeout, tun);
    }
  return 0;
}


/* XXX under construction ... */
size_t process_avp(l2tp_tunnel *tun, SshUInt16 *buffer, size_t received_len)
{
  unsigned char *tmp = 0;
  SshBuffer *inbound_msg_ptr;

  inbound_msg_ptr = (void *)ssh_buffer_allocate();
  ssh_buffer_init((void *)inbound_msg_ptr);
  ssh_buffer_clear((void *)inbound_msg_ptr);

  ssh_buffer_append((void *)inbound_msg_ptr, (unsigned char *)buffer, 
                    received_len);
  tmp = ssh_buffer_ptr((void *)inbound_msg_ptr);


  /* WILL BE MOVED TO END OF THE FUNCTION WHEN COMPLETED */
  /* So far has been good, about to return, Sr++, set reTx = 0 */
  ssh_ack(tun, ntohs(buffer[5]));
  tun->Sr++;
  tun->reTx = 0;
  return 0;

  if(ntohs(buffer[0]) != 0xC802) { ssh_debug("bad control"); return 1; }
  if(ntohs(buffer[1]) != received_len) 
    { 
      ssh_debug("bad length field"); 
      return 1;       
    }
  /* check if is a SCCRQ, if is then get tunnel id */
  if(ntohs(buffer[8]) == SSH_AVP_TYPE_MESSAGETYPE 
     && ntohs(buffer[9]) ==  0x0001)
    {
      ssh_debug("sccrq tunid = %x", ntohs(buffer[35]));
      tun->tunnel_id = ntohs(buffer[35]);
    }
  /* check that tunnel id is correct for this tunnel, exception is SCCRQ */
  if(tun->tunnel_id != ntohs(buffer[2]))
    {
      ssh_debug("[%s] bad tunnel id, tun->tunnel_id==%x, ntohs(buffer[2])==%x.", 
                (char *)tun->state, tun->tunnel_id, ntohs(buffer[2])  );
      return 1;
    }
  if(tun->call_id != ntohs(buffer[3]) && (char *)tun->state != "ICRQ") 
    {  
      ssh_debug("bad call id"); 
      return 1;         
    }
  /* call retransmit function if the --Nr does not equal the Ns */
  ssh_debug("[%s] all okie dokie", tun->state);
  /* Increment the Nr value as everything was cool   */
}

Boolean establishedState(l2tp_tunnel *tun)
{
  if(strcmp(tun->state, "L2TP established ") == 0 
     || strcmp(tun->state, "L2TP IN established ") == 0
     || strcmp(tun->state, "L2TP OUT established ") == 0)
    return TRUE;
  else
    return FALSE;
}


/* l2tp_tunnel *ssh_l2tp_control_channel_init(void) */ 
l2tp_tunnel *ssh_l2tp_control_channel_init(unsigned char *type,
                                           unsigned char *srcaddr,
                                           unsigned char *srcport,
                                           unsigned char *dstaddr,
                                           unsigned char *dstport)
{
  l2tp_tunnel *tun;
  l2tp_tunnel myT;
  SshUdpListener myHost;

  tun = &myT;
  tun = (void *)(struct l2tp_tunnel *)ssh_xmalloc(sizeof(myT));
  if(!tun)
    {
      ssh_debug("tunnel creation failed...exiting");
      ssh_event_loop_uninitialize();
      exit(1);
    }
  /* XXX temp */
  tun->setLNS = FALSE;
  if(strcmp(type , "lns") == 0)
     tun->setLNS = TRUE;
  tun->local_address = srcaddr;
  tun->local_port = srcport;
  tun->remote_address = dstaddr;
  tun->remote_port = dstport;
  tun->myHostName = "chef.hel.fi.ssh.com ";

  /* 
     ssh_l2tp_parse_config(tun);
     ssh_debug("parsed config file");
  */
  /* XXX set up a hash table which will use the Ss value sent as the key, 
     it will point to a pointer holding the message which was sent. This 
     will then be resent upon a sender_timeout event, or the message will 
     be deleted from the hash table upon receipt of a acknowledgement ZLB
     from the other peer */
  /* key lengt & value length ignored with VL pointers */
  tun->unackedTable = ssh_mapping_allocate(SSH_MAPPING_TYPE_POINTER_VL, 1, 1);
  if(tun->unackedTable == NULL)
    {
      ssh_debug("Failure to create hash table ");
      exit(1);
    }
  /* This is probably not needed XXX - check */
  ssh_mapping_clear(tun->unackedTable);  
  myHost = ssh_udp_make_listener(tun->local_address, tun->local_port, NULL, 
     NULL, NULL, NULL); 
  if (myHost == NULL)
    {
      ssh_debug("Listener creation failed.%s %s", tun->local_address, 
                tun->local_port );
      ssh_event_loop_uninitialize();
      exit(1);
      return NULL;
    }

  /* taken from file  */   
  tun->rws = 2;
  tun->maxReTx = 4;
  tun->vendor_id = 0x0000; 
  /* Will send peer a Hello after xyz secs */
  tun->keepAliveTimerValue = 40;
 
  tun->state = "L2TP idle"; 
  tun->reTx = 0;
  tun->myListener = (void *)myHost;
  tun->tunnel_id = 2;  
  tun->call_id = 1;
  tun->Ss = 0;
  tun->Sr = 0;
  tun->timeoutValue[0] = 1; 
  tun->timeoutValue[1] = 2;
  tun->timeoutValue[2] = 4;
  tun->timeoutValue[3] = 8;
  /* Default to reserved  */
  tun->error_code = 0; 
  /* no seq num on l2tp data packets */
  tun->seq_data = FALSE;
  tun->running_length_count = 0;
  tun->support_asynchronous = 0x00000000;
  tun->support_synchronous = 0x00000001; 

  /* Non mandatory AVPs */
  tun->bearer_capabilities_avp = FALSE;
  tun->receive_window_size_avp = TRUE;
  tun->tie_breaker_avp = FALSE;
  tun->firmware_revision_avp = FALSE;
  tun->vendor_name_avp = FALSE;
  tun->bearer_type_avp = FALSE;
  tun->physical_channel_id_avp = FALSE;
  tun->calling_number_avp = FALSE;
  tun->called_number_avp = FALSE;
  tun->sub_address_avp = FALSE;

  /* XXX changed type */
  tun->call_serial_number = 0x12345678;
  tun->connect_speed = 0x12345678;
  tun->framing_type = 0x12121212;  
  tun->minbps = 0x1234ABCD;
  tun->maxbps = 0x345678AB;
  tun->bearer_type = 0xABCDE012;
  tun->callednumber = 0x04082717; 
  tun->firmware_rev = 0x0001;
  return tun;
}

size_t ssh_add_control_header(l2tp_tunnel *tun, 
                              SshUInt16 *l2tpData, 
                              size_t mesg_len)
{
  /* XXX if message_length mod SshUInt16 != 0 add in offset flag and 
     padding */
  
  tun->running_length_count = 0;
  
  l2tpData[0] = htons(0xC802);
  l2tpData[1] = htons(mesg_len);

  if((strcmp(tun->state, "L2TP idle") == 0) && !tun->setLNS)
    {
      l2tpData[2] = htons(0x0000);
      l2tpData[3] = htons(0x0000);
    } 
  else
  if((strcmp(tun->state, "L2TP IN idle") == 0) && !tun->setLNS)
      {
        l2tpData[2] = htons(tun->tunnel_id);
        l2tpData[3] = htons(0x0000);
      }
    else
      {
        l2tpData[2] = htons(tun->tunnel_id);
        l2tpData[3] = htons(tun->call_id);
      }
  l2tpData[4] = htons(tun->Ss);
  l2tpData[5] = htons(tun->Sr);
  tun->running_length_count += 6;
  return 0;
}

  /* Generic avp ???. Takes in the tunnel, the data buffer, the start position 
     of avp (offset), message_type and the data. Note it only works with AVPs 
     which have data of 16 bits in length  */
size_t ssh_add_avp(l2tp_tunnel *tun, 
                   SshUInt16 *l2tpData, 
                   size_t offset, 
                   size_t message_type, 
                   SshUInt16 data)
{
  l2tpData[offset] = htons(0x8008);
  l2tpData[offset+1] = htons(tun->vendor_id);
  l2tpData[offset+2] = htons(message_type);
  l2tpData[offset+3] = htons(data);
  tun->running_length_count += 4;
  return 4;
}

size_t ssh_add_avp32(l2tp_tunnel *tun, 
                     SshUInt16 *l2tpData, 
                     size_t offset, 
                     size_t message_type, 
                     SshUInt32 data)
{
  SshUInt16 part_uno, part_duo;

  part_uno = data >> 16;
  part_duo = data >> 31; /* XXX check if this should 32, compiler warning  */
  l2tpData[offset] = htons(0x800A);
  l2tpData[offset+1] = htons(tun->vendor_id);
  l2tpData[offset+2] = htons(message_type);
  l2tpData[offset+3] = htons(part_uno);
  l2tpData[offset+4] = htons(part_duo);
  tun->running_length_count += 5;
  return 5;
}

/* Note that data is a byte long. Need to convert this to 16 bits when doing 
   running_length_count assignment */
size_t ssh_add_avp_vl(l2tp_tunnel *tun,
                SshUInt16 *l2tpData,
                size_t offset,
                size_t message_type,
                unsigned char *data,
                size_t data_len)
{
  size_t i,j ;

  l2tpData[offset] = htons(0x6 + data_len);
  l2tpData[offset+1] = htons(tun->vendor_id);
  l2tpData[offset+2] = htons(message_type);
  j = 0;
  for(i = 0; i < data_len; i++)
    {
      l2tpData[i + offset + 3] = htons(data[j+ 1]) + ( htons(data[j]) >> 8 );
      j+=2;
    }
  tun->running_length_count += 3 + (data_len / 2);
  return (data_len / 2) + 3;
}

void Process_avp(l2tp_tunnel *tun, SshUInt16 *buffer, size_t received_len)
{
  if(process_avp(tun, buffer, received_len) != 0)
    {
      ssh_debug("processed avp was bad");
      /* Cancel all timeouts and send StopCCN - note should recv a ZLB 
         for stopCCN */
      ssh_l2tp_send_control(SSH_STOPCCN, tun);
    }
}

/* Increments the identity be one */
size_t getID(l2tp_tunnel *tun, unsigned char type)
{

  /* We need to initiate a new Tunnel in the following state else initiate a new 
     call. Assuming that a call or tunnel can only be initiated when an existing 
     call or tunnel is already estabished or in an idle state. 
  */
  if(strcmp(tun->state, "L2TP idle") == 0 
     || strcmp(tun->state, "L2TP established") == 0)
    return tun->tunnel_id++;
  if(strcmp(tun->state, "L2TP IN idle") == 0
     || strcmp(tun->state, "L2TP OUT idle") == 0
     || strcmp(tun->state, "L2TP IN established") == 0
     || strcmp(tun->state, "L2TP OUT established") == 0) 
    return tun->call_id++;
  else
    {
      ssh_debug("getID - improper argument, must be a 'c' or a 't'");
      return -1; /* XXX or just uninit() and die */
    }
}

SSH_FSM_STEP(l2tp_idle)
{
  l2tp_tunnel *tun;
  size_t rlen;
  unsigned char *rvalue;
  

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  if(!tun->setLNS)
    {
      ssh_l2tp_send_control(SSH_SCCRQ, tun);
      SSH_FSM_SET_NEXT("L2TP wait_ctl_reply");
      return SSH_FSM_CONTINUE;
    }
  else
    {     
      /* recv a valid SCCRQ */
      tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen);
      if(tun->mesg_recvd == -1)
        {
          SSH_FSM_SET_NEXT((char *)tun->state); 
          return SSH_FSM_CONTINUE;
        }
      /* Check was a good SCCRQ */
      if(tun->mesg_recvd == SSH_SCCRQ )
        {       
          tun->Sr++;
          SSH_FSM_SET_NEXT("L2TP wait_ctl_conn");
          ssh_l2tp_send_control(SSH_SCCRP, tun); 
          return SSH_FSM_CONTINUE;            
        }         
      if(tun->mesg_recvd == SSH_STOPCCN )
        {
          ssh_l2tp_send_control(SSH_STOPCCN, tun);
          SSH_FSM_SET_NEXT("L2TP cleanupwait");
          return SSH_FSM_CONTINUE;
        }
      SSH_FSM_SET_NEXT("L2TP idle");
    }
  SSH_FSM_SET_NEXT((char *)tun->state);
  return SSH_FSM_CONTINUE;
}

/* Sent a gd SCCRQ, need to recv a gd SCCRP. */
SSH_FSM_STEP(l2tp_wait_ctl_reply)
{  
  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue;
  
  SshFSM tmpfsm;
  SshFSMCondition got_mesg;

  tun = ssh_fsm_get_tdata(thread);
  /* Get the underlying FSM for a thread. */
  tmpfsm = ssh_fsm_get_fsm(thread);

  /* Set up a conditional variable */
  got_mesg = ssh_fsm_condition_create(tmpfsm);

  tun->state = ssh_fsm_get_thread_current_state(thread); 
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen);  
  /* XXX fix the following line so that waits for mesg_recvd > 0 */
  if(tun->mesg_recvd == -1)
    {
      SSH_FSM_SET_NEXT((char *)tun->state); 
      return SSH_FSM_CONTINUE;
    }
  if(tun->mesg_recvd == SSH_SCCRP )
    {
      SSH_FSM_SET_NEXT("L2TP established");
      /* ssh_ack(tun, 1); */ 
      tun->Sr++;
      ssh_l2tp_send_control(SSH_SCCCN, tun);
    }
  if(tun->mesg_recvd == SSH_STOPCCN )
    {
      SSH_FSM_SET_NEXT("L2TP cleanupwait"); 
      ssh_l2tp_send_control(SSH_STOPCCN, tun);
    }
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(l2tp_established)
{
  l2tp_tunnel *tun;

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);

  /* XXX what do I do about keepalives ? */ 
  ssh_cancel_timeouts(keep_alive_timeout, SSH_ALL_CONTEXTS); 
  ssh_register_timeout(tun->keepAliveTimerValue,0, keep_alive_timeout, tun); 

#ifdef BASIC_TEST_INITIATE_TUNNEL_ONLY
  SSH_FSM_SET_NEXT(ssh_fsm_get_thread_current_state(thread));    
  return SSH_FSM_CONTINUE;
#endif

/* Fork here and one thread will handle setting up the Incoming call 
   connection and the other will handle the l2tp_established state's 
   duties such as HELL0 and checking that mesgs received are correct 
   according to the l2tp state machine diagram. This thread will be 
   suspended until events:
   i)   Keepalive timeout.
   ii)  Recv a bad mesgs eg a SCCRP etc. 
   iii) Close down tunnel request SSH_STOPCCN.
*/

  if(tun->setLNS)
    {
      /* wait to receive an incoming call or make a outgoing call */
      if(1) /* XXX */ 
        {
          tun->state = "L2TP IN idle"; 
          /* SSH_FSM_FORK(0, "L2TP IN idle", NULL, NULL); */
          SSH_FSM_SET_NEXT("L2TP IN idle"); 
          return SSH_FSM_CONTINUE;
        }
      else      
        {
          tun->state = "L2TP OUT idle"; 
          /* SSH_FSM_FORK(0, "L2TP OUT idle", NULL, NULL); */
          SSH_FSM_SET_NEXT("L2TP OUT idle"); 
          return SSH_FSM_CONTINUE;
        }
    }
  else
    {      
      SSH_FSM_SET_NEXT("L2TP IN idle"); 
      /* SSH_FSM_FORK(0, "L2TP IN idle", NULL, NULL); */ 
    }
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(l2tp_cleanup)
{
  l2tp_tunnel *tun;

  tun = ssh_fsm_get_tdata(thread);
  tun->state =  ssh_fsm_get_thread_current_state(thread);
  ssh_debug("ready to end");
  /* Clear all timeouts and uninit() */
  ssh_cancel_timeouts(SSH_ALL_CALLBACKS, SSH_ALL_CONTEXTS);

  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(l2tp_cleanupwait)
{
  size_t rlen = 0;
  unsigned char *rvalue; 
  l2tp_tunnel *tun;
  

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);

  /* make sure that the Stopccn was recvd, so wait for ZLB here */
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen); 
  if(tun->mesg_recvd == -1)
    {
      SSH_FSM_SET_NEXT((char *)tun->state); 
      return SSH_FSM_CONTINUE;
    }
  if(tun->mesg_recvd == ZLB)
    {
      ssh_debug("received a ZLB");
      SSH_FSM_SET_NEXT("L2TP cleanup");
      return SSH_FSM_CONTINUE;
    }
  SSH_FSM_SET_NEXT((char *)ssh_fsm_get_thread_current_state(thread));
  return SSH_FSM_CONTINUE;
}

/* Have now sent the SCCRP, need to recv a ZLB or a gd SCCN */
SSH_FSM_STEP(l2tp_wait_ctl_conn)
{
  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue;
  

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  /* make sure that the Stopccn was recvd, so wait for ZLB here */
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen);
  if(tun->mesg_recvd == SSH_SCCCN)
    {
      tun->Sr++;
      SSH_FSM_SET_NEXT("L2TP established");
      return SSH_FSM_CONTINUE;
    }
  if(tun->mesg_recvd == -1)
    {
      SSH_FSM_SET_NEXT((char *)tun->state); 
      return SSH_FSM_CONTINUE;
    }
  if(tun->mesg_recvd != ZLB)
    {
      SSH_FSM_SET_NEXT("L2TP cleanupwait");   
      ssh_debug("[%s] Receieved a wrong mesg %d.", tun->state, tun->mesg_recvd);
    }
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(l2tp_incoming_idle)
{
  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue; 
  
  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  /* For now LAC initiate a Inc. Call */
  if(!tun->setLNS) 
    {
      ssh_l2tp_send_control(SSH_ICRQ, tun);
      tun->state = "L2TP IN wait_ctl_reply";
    }
  else
    {      
      /* recv a valid ICCRQ  */
      tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen); 
      if(tun->mesg_recvd == -1)
        {
          SSH_FSM_SET_NEXT((char *)tun->state); 
          return SSH_FSM_CONTINUE;
        }
      if(tun->mesg_recvd == SSH_ICRQ)
        {
          tun->Sr++;
          tun->state = "L2TP IN wait_ctl_conn";
          ssh_l2tp_send_control(SSH_ICRP, tun);
          SSH_FSM_SET_NEXT((char *)tun->state);
          return SSH_FSM_CONTINUE;
        }
      else
        {
          ssh_l2tp_send_control(SSH_STOPCCN, tun);
          tun->state = "L2TP IN cleanupwait"; 
          SSH_FSM_SET_NEXT((char *)tun->state);
          return SSH_FSM_CONTINUE;
        }
    }
  SSH_FSM_SET_NEXT((char *)tun->state);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(l2tp_incoming_wait_ctl_reply)
{
  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue;

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen);
  if(tun->mesg_recvd == -1)
    {
      SSH_FSM_SET_NEXT((char *)tun->state); 
      return SSH_FSM_CONTINUE;
    }
  if(tun->mesg_recvd == SSH_ICRP )
    {
      tun->Sr++; 
      tun->state = "L2TP IN established";
      ssh_debug("[%s] Receieved a ICRP (%d)", tun->state, 
                strlen( (void *)&rlen));
      ssh_l2tp_send_control(SSH_ICCN, tun);
    }
  else
    {
      tun->state = "L2TP IN cleanupwait";
      ssh_l2tp_send_control(SSH_STOPCCN, tun);
      
    }   
  SSH_FSM_SET_NEXT((char *)tun->state);   
  return SSH_FSM_CONTINUE;
}

void ssh_l2tp_can_send_cb(void *context)
{
  ssh_debug("called when more packets can be sent, rws etc");
}

void ssh_l2tp_eof_cb(void *context)
{
  ssh_debug("eof");
}


void ssh_l2tp_packet_recv(SshPacketType type,const unsigned char *data,
                             size_t len, void *context)
{
  ssh_debug("recv packet"); 
}  
  
 
SSH_FSM_STEP(l2tp_incoming_established)
{
  l2tp_tunnel *tun;
  static unsigned char *buf;
  size_t rlen = 0;
  unsigned char *rvalue;

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  if(tun->setLNS)
    {
      /* ssh_debug("[%s] waiting to recv data on call number %d", tun->state, 
         tun->call_id); */ 
      
      /* Another call request, a CDN for an existing call  a bad control packet, 
         a ZLB or a Hello messgae */
      tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen); 
      if(tun->mesg_recvd == -1)
        {
          SSH_FSM_SET_NEXT((char *)tun->state); 
          return SSH_FSM_CONTINUE;
        }
      if(tun->mesg_recvd == SSH_ICRQ) 
        {
          ssh_debug("[%s] Receieved a ICRQ (%d)", tun->state, 
                    strlen( (void *)&rlen));
          tun->Sr++;
          SSH_FSM_SET_NEXT("L2TP IN wait_ctl_conn");
          return SSH_FSM_CONTINUE;        
        }
      if(tun->mesg_recvd == SSH_CDN) 
        {
          tun->Sr++;
          ssh_l2tp_send_control(ZLB, tun);
          SSH_FSM_SET_NEXT("L2TP IN cleanup"); 
          return SSH_FSM_CONTINUE; 
        }  
      /* L2TP Data Packet */
      ssh_debug("[%s] Received %d bytes of data on call %d", 
                    tun->state, rlen, tun->call_id);      
      /*  stay here */
      SSH_FSM_SET_NEXT((char *)ssh_fsm_get_thread_current_state(thread)); 
      return SSH_FSM_CONTINUE;
    }
  else
    {
      buf = "~}#@!}!}!} }4}\"}&} } } } }%}&x!b}2}'}\"}(}\"c~";       
  
      tun->pbuffer = ssh_buffer_allocate();
      ssh_buffer_append(tun->pbuffer, buf, strlen((const char *)buf));
      buf = "";
      buf = ssh_buffer_ptr(tun->pbuffer); 
      ssh_l2tp_send_data(tun);

      /*  check that sent atleast the data we have given this func */
      /* if(ssh_l2tp_send_data(tun, buf) < ssh_buffer_len(tun->pbuffer))
        {
          ssh_debug("send error"); 
          exit(1); 
          } */ 
      ssh_buffer_free(tun->pbuffer);
      /* XXX okay now sent data mv from here to cleanup and */
      ssh_l2tp_send_control(SSH_CDN, tun);
      SSH_FSM_SET_NEXT("L2TP IN cleanupwait");   
      return SSH_FSM_CONTINUE;  
    }

#ifdef BASIC_TEST_MANY_INCOMING_CALLS
  /* TIM This code attempts initiating a new call (incoming) while there is 
     already an incoming call established. This code can be removed. Use the fsm 
     to do this: 
     XXX thread = (void *)ssh_fsm_spawn(fsm, 0, "L2TP IN idle", NULL, NULL); */ 
  if(!tun->setLNS) 
    {
      tun->call_id = getID(tun, 'c'); 
      ssh_l2tp_send_control(SSH_ICRQ, tun);
      SSH_FSM_SET_NEXT("L2TP IN wait_ctl_reply");
      return SSH_FSM_CONTINUE; 
    }
#else
  SSH_FSM_SET_NEXT((char *)tun->state);
#endif
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(l2tp_incoming_cleanup)
{
  l2tp_tunnel *tun;

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);

  /* XXX, not need ssh_cancel_timeouts(sender_timeout, tun); */ 
  ssh_cancel_timeouts(SSH_ALL_CALLBACKS, SSH_ALL_CONTEXTS);
  /* Move to the control channel est state 
  SSH_FSM_SET_NEXT("L2TP established");
  return SSH_FSM_CONTINUE; */

  return SSH_FSM_FINISH;
}

/* Make sure that the Stopccn was recvd, so wait for ZLB here in this state */
SSH_FSM_STEP(l2tp_incoming_cleanupwait)
{
  size_t rlen = 0;
  unsigned char *rvalue;
  l2tp_tunnel *tun;
  
  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen); 
  if(tun->mesg_recvd == -1)
    {      
      SSH_FSM_SET_NEXT((char *)tun->state); 
      return SSH_FSM_CONTINUE;
    }
  if(tun->mesg_recvd == ZLB)
    {
      SSH_FSM_SET_NEXT("L2TP IN cleanup");
      return SSH_FSM_CONTINUE;
    }
  else
    {
      ssh_debug("[%s] recvd a wrong control mesg, closing anyway");
      SSH_FSM_SET_NEXT("L2TP IN cleanup");
      return SSH_FSM_CONTINUE;
    }
  
}

SSH_FSM_STEP(l2tp_incoming_wait_ctl_conn)
{
  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue;

  SSH_FSM_SET_NEXT("L2TP IN established");
  tun = ssh_fsm_get_tdata(thread);
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen);
  if(tun->mesg_recvd == -1)
    {
      SSH_FSM_SET_NEXT((char *)tun->state); 
      return SSH_FSM_CONTINUE;
    }
  if(tun->mesg_recvd == SSH_ICCN)
    {
      tun->Sr++;
      SSH_FSM_SET_NEXT("L2TP IN established");
      ssh_l2tp_send_control(ZLB, tun);
    }
  else
    {
      SSH_FSM_SET_NEXT("L2TP IN cleanupwait");   
      ssh_debug("[%s] Receieved a wrong message", tun->state);
    }
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(l2tp_outgoing_idle)
{
  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue; 
  

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  if(!tun->setLNS)
    {
      ssh_l2tp_send_control(SSH_OCRQ, tun);
      SSH_FSM_SET_NEXT("L2TP OUT wait_ctl_reply");
    }
  else
    {
      /* recv a valid OCRQ  */
    tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen); 
      if(tun->mesg_recvd == SSH_OCRQ) 
        {
          tun->Sr++;
          SSH_FSM_SET_NEXT("L2TP OUT wait_ctl_conn");
          ssh_l2tp_send_control(SSH_OCRP, tun);
          return SSH_FSM_CONTINUE;
        }
      else
        {
          ssh_l2tp_send_control(SSH_STOPCCN, tun);
          SSH_FSM_SET_NEXT("L2TP OUT cleanupwait"); 
          return SSH_FSM_CONTINUE;
        }
    }  
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(l2tp_outgoing_wait_ctl_reply)
{

  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue; 
  

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen);
  if(rlen > 0)
    {
      /* ZLB support, RWS Check */
      if(tun->mesg_recvd == SSH_OCRP) 
        {
          tun->Sr++;
          SSH_FSM_SET_NEXT("L2TP OUT established");
          ssh_debug("[%s] Receieved a OCRP", tun->state);
          ssh_l2tp_send_control(SSH_OCCN, tun);
        }
      else
        {
          ssh_l2tp_send_control(SSH_CDN, tun);
          SSH_FSM_SET_NEXT("L2TP OUT cleanupwait"); 
        }
    }
  return SSH_FSM_CONTINUE;
}

/* XXX */
SSH_FSM_STEP(l2tp_outgoing_established)
{
  l2tp_tunnel *tun;
  SshFSMCondition got_mesg;
  SshFSM tmpfsm;

  /* Get the underlying FSM for a thread. */
  tmpfsm = ssh_fsm_get_fsm(thread);

  /* Set up a conditional variable */
  got_mesg = ssh_fsm_condition_create(tmpfsm);

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  /* XXX  Now ready to txmit data */

  SSH_FSM_CONDITION_BROADCAST(got_mesg);
  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(l2tp_outgoing_cleanup)
{
  l2tp_tunnel *tun;

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  return SSH_FSM_FINISH;
}


SSH_FSM_STEP(l2tp_outgoing_cleanupwait)
{
  size_t rlen = 0;
  unsigned char *rvalue; 
  l2tp_tunnel *tun;
  

  tun = ssh_fsm_get_tdata(thread);
  tun->state = ssh_fsm_get_thread_current_state(thread);
  /* make sure that the Stopccn was recvd, so wait for ZLB here */
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen); 
  if(tun->mesg_recvd == ZLB)
    {
      ssh_debug("recvd a ZLB");
    }
  SSH_FSM_SET_NEXT((char *)ssh_fsm_get_thread_current_state(thread));
  return SSH_FSM_CONTINUE;
}


SSH_FSM_STEP(l2tp_outgoing_wait_ctl_conn)
{
  l2tp_tunnel *tun;
  size_t rlen = 0;
  unsigned char *rvalue;
  

  ssh_debug("I am l2tp_outgoing_wait_ctl_conn");
  tun = ssh_fsm_get_tdata(thread);
  /* make sure that the Stopccn was recvd, so wait for ZLB here */
  tun->mesg_recvd = ssh_l2tp_recv_mesg(tun, &rvalue, &rlen); 
  if(tun->mesg_recvd == SSH_OCCN)
    {
      tun->Sr++;
      SSH_FSM_SET_NEXT("L2TP OUT established");
      ssh_debug("[%s] Receieved a good OCCN", tun->state);
    }
  else
    {
      SSH_FSM_SET_NEXT("L2TP OUT cleanupwait");   
      ssh_debug("[%s] Receieved a wrong message", tun->state);
    }
  return SSH_FSM_CONTINUE;
}

int ssh_l2tp_recv_mesg(l2tp_tunnel *tun, unsigned char **rvalue, size_t *rlen)
{
  static unsigned char received_data[256];  
  size_t received_len;       
  char remote_address[256]; 
  char remote_port[16];
  SshUdpError error;
  SshUInt16 dat[MAX_MESG_SIZE];
  int i, j;
  struct message *msg_checker;
  struct avp *searcher;
  /* unsigned short lower_v;
     unsigned short upper_v; */ 
  size_t offset;
  int msg_type = -1 ;
  unsigned short tmpmsg[50]; /* XXX aahh ! */

  /* This structure records the avp_number, whether it is optional or manadatory 
   its length (a length of zero indicates that the avp is of a variable length, 
   such as the hostname, vendorname or possibly the result code avp). It also 
   includes a handler which might be used at some point in the future for more 
   specific checks. 
  */
 struct avp 
 {
   SshUInt16 avp_number;
   Boolean mandatory;
   int avp_length;
   unsigned char *avp_name;
   int (*handler)(l2tp_tunnel *tun, char *data, size_t data_len );
 };

  /* This structure is the link between the AVPs and the messages received and 
     sent. Each avp is defined in 'struct avp' but this structure captures the
     link between messages and the AVPs. 
  */
  struct message
  {
    int mesg_num;
    unsigned char *mesg_name;
    int valid_avps[10]; 
  };

  struct message messages[] = 
  {
    { SSH_SCCRQ,   "SCCRQ",   { SSH_AVP_TYPE_MESSAGETYPE,  
                                SSH_AVP_TYPE_PROTOCOLVERSION,
                                SSH_AVP_TYPE_FRAMINGCAPABILITIES,
                                SSH_AVP_TYPE_BEARERCAPABILITIES,
                                SSH_AVP_TYPE_HOSTNAME, 
                                SSH_AVP_TYPE_VENDORNAME, 
                                SSH_AVP_TYPE_ASSIGNEDTUNNELID,
                                SSH_AVP_TYPE_RECEIVEWINDOWSIZE}},

    { SSH_SCCRP,   "SCCRP",   { SSH_AVP_TYPE_MESSAGETYPE, 
                                SSH_AVP_TYPE_PROTOCOLVERSION,
                                SSH_AVP_TYPE_FRAMINGCAPABILITIES,
                                SSH_AVP_TYPE_BEARERCAPABILITIES,
                                SSH_AVP_TYPE_HOSTNAME, 
                                SSH_AVP_TYPE_VENDORNAME, 
                                SSH_AVP_TYPE_ASSIGNEDTUNNELID,
                                SSH_AVP_TYPE_RECEIVEWINDOWSIZE} },

    { SSH_SCCCN,   "SCCCN",   { SSH_AVP_TYPE_MESSAGETYPE} },    

    { SSH_STOPCCN, "StopCCN", { SSH_AVP_TYPE_MESSAGETYPE,
                                SSH_AVP_TYPE_ASSIGNEDTUNNELID,
                                SSH_AVP_TYPE_RESULTCODE} },

    { SSH_HELLO,   "HELLO",   { SSH_AVP_TYPE_MESSAGETYPE} },

    { SSH_ICRQ,    "ICRQ",    { SSH_AVP_TYPE_MESSAGETYPE,
                                SSH_AVP_TYPE_ASSIGNEDSESSIONID,
                                SSH_AVP_TYPE_CALLSERIALNUMBER } },

    { SSH_ICRP,    "ICRP",    { SSH_AVP_TYPE_MESSAGETYPE,
                                SSH_AVP_TYPE_ASSIGNEDSESSIONID } },

    { SSH_ICCN,    "ICCN",    { SSH_AVP_TYPE_MESSAGETYPE,
                                SSH_AVP_TYPE_TXCONNECTSPEED,
                                SSH_AVP_TYPE_FRAMINGTYPE } },

    { SSH_CDN,     "CDN",     { SSH_AVP_TYPE_MESSAGETYPE,
                                SSH_AVP_TYPE_RESULTCODE,
                                SSH_AVP_TYPE_ASSIGNEDTUNNELID} }
  };
 
 /* zero length means variable */
 struct avp avps[] = 
 {
   { SSH_AVP_TYPE_MESSAGETYPE,         TRUE,  8,"Message Type", NULL },
   { SSH_AVP_TYPE_RESULTCODE,          FALSE, 0, "Result Code", NULL },  
   { SSH_AVP_TYPE_PROTOCOLVERSION,     TRUE,  8, "Protocol Version", NULL },
   { SSH_AVP_TYPE_FRAMINGCAPABILITIES, TRUE,  10, "Framing Capabilities", NULL },
   { SSH_AVP_TYPE_BEARERCAPABILITIES,  FALSE, 10, "Bearer Capabilities", NULL },
   { SSH_AVP_TYPE_FIRMWAREREVISION,    FALSE, 8, "Firmware Revision", NULL },
   { SSH_AVP_TYPE_HOSTNAME,            TRUE,  0, "Host Name", NULL },
   { SSH_AVP_TYPE_VENDORNAME,          FALSE, 0, "Vendor Name", NULL }, 
   { SSH_AVP_TYPE_ASSIGNEDTUNNELID,    TRUE,  8, "Assigned Tunnel ID", NULL },
   { SSH_AVP_TYPE_RECEIVEWINDOWSIZE,   FALSE, 8, "Receive Window Size", NULL },
   { SSH_AVP_TYPE_CHALLENGERESPONSE,   FALSE, 0, "Challenge Response", NULL },
   { SSH_AVP_TYPE_ASSIGNEDSESSIONID,   TRUE,  8,  "Assigned Session ID", NULL },
   { SSH_AVP_TYPE_MINIMUMBPS,          FALSE, 10, "Minimum Bps", NULL }
 };

 
  error = ssh_udp_read((void *)tun->myListener, remote_address, 256,
                       remote_port, 16,
                       received_data, 256, &received_len);
  /* As Using Thread library this function will return many SSH_UDP_NO_DATA */
  if(error == SSH_UDP_HOST_UNREACHABLE 
     || error == SSH_UDP_PORT_UNREACHABLE )
    {
      ssh_debug("error: Host/port unreachable");
      tun->error_code = 6;
      exit(1);
    }
  if(error == SSH_UDP_NO_DATA)
    return -1;
    
  /* load data into a easier format */
  j = 0;
  for(i = 0; i < received_len / 2; i++)
    {
      dat[i] = ntohs(received_data[j+ 1]) + (ntohs(received_data[j]) >> 8 ); 
      j+=2;
    }

  /* Check if it was data. If ntohs(dat[0]) is less than 0x8000 then could be 
     data, as can't be a control mesg. Then check that tunnel and call identity 
     are correct. 
   */
  if(ntohs(dat[0]) < 0x8000  )
    {
      ssh_debug("XXX (%d),dat1(%d)", received_len, ntohs(dat[1]));
      if(ntohs(dat[1]) != received_len)
        {
          ssh_debug("error, header length not match recvd data %x", 
                    ntohs(dat[1]));
          tun->error_code = 2;
          return -1;
        }
      /* Check Identity.*/
      if(ntohs(dat[2]) != tun->tunnel_id ||  ntohs(dat[3]) != tun->call_id)
        {
          ssh_debug("error, incorrect identity");
          tun->error_code = 2; /* XXX check */
          return -1;
        }
      if(tun->seq_data)
        {
          if(ntohs(dat[4]) != tun->dataSs || ntohs(dat[5]) != tun->dataSr)
            {
              ssh_debug("error, incorrect identity");
              tun->error_code = 2; /* XXX check */
              return -1;
            }
        }
      else
        /* If the offset bit not set then we have data @ next dat[x] */
        {
          ssh_debug("XXX @ data.");
        }
    }

  /* Check for a ZLB here */
  if(received_len == 12)
    {
      /* Is the call idenity zero ie no call being / been established */
      if(ntohs(dat[3]) == 0)
        {
          if(ntohs(dat[0]) == 0xC802  
             && ntohs(dat[1]) == 0xc
             && ntohs(dat[2]) == tun->tunnel_id
             && ntohs(dat[4]) == tun->Sr
             && ntohs(dat[5]) == tun->Ss
             && (strncmp(tun->state, "L2TP I", 6) == 0 
                 || strncmp(tun->state, "L2TP O", 6) == 0 ))
            {
              /* Remove Seq Numbers from hash table and return */
              /* ssh_debug("[%s] ZLB Ss=%d Sr=%d", tun->state, tun->Ss, tun->Sr);
               */ 
              return ZLB;
            }
        }
      else 
        {
          if(ntohs(dat[3]) == tun->call_id)
            {         
              if(ntohs(dat[0]) == 0xC802  
                 && ntohs(dat[1]) == 0xc
                 && ntohs(dat[2]) == tun->tunnel_id
                 && ntohs(dat[4]) == tun->Sr
                 && ntohs(dat[5]) == tun->Ss
                 && (strncmp(tun->state, "L2TP I", 6) == 0 
                     || strncmp(tun->state, "L2TP O", 6) == 0 ))
                {
                  /* Remove Seq Numbers from hash table and return */
                  ssh_debug("[%s] ZLB Ss=%d Sr=%d", tun->state, tun->Ss, tun->Sr);
                  return ZLB;
                }
            }
          else
            {
              /* error dat[3] not zero or call id stop */
              ssh_debug("Call id error in ZLB");
              *rlen = -1;
              memset(*rvalue, 0, received_len);
              return -1;
            }
        }
    }
  
  
  if(dat[0] == htons(0xC802)
     || dat[1] == htons(received_len))
    {
      for(i = 6; i < received_len; i++) 
        {
          /* Test that the L2TP bits are correct, and that the next byte is the 
             vendor identity. Note for all the AVPs defined in RFC 2661, the
             vendor id must be zero. */
          if(dat[i] > htons(0x8) && dat[i + 1] == 0) 
            {
              
              for(searcher = avps; searcher->avp_number < 13; searcher++ )
                {
                  /* get the AVP type and check that it is of the correct length 
                     and that all avps for this message have been received. */
                  if(dat[i+2] == htons(searcher->avp_number))
                    {                 
                      if(searcher->avp_length == 8)
                        {
                          /* ssh_debug("avp16 is %s(avp num %d), value == %x", 
                                    searcher->avp_name, searcher->avp_number, 
                                    htons(dat[i+3])); */ 
                          if(searcher->avp_number == SSH_AVP_TYPE_MESSAGETYPE)
                            {
                              for(msg_checker = messages; 
                                  msg_checker->mesg_num < NUM_L2TP_MESGS; 
                                  msg_checker++)
                                {
                                  if(msg_checker->mesg_num == ntohs(dat[i+3]) )
                                    {
                                      /* Store the mesg_num in msg_type, if any 
                                         of the other avp attributes are 
                                         incorrect then set this value from the 
                                         good value to the -1. */
                                      /* ssh_debug("[%s] got mesg %s", 
                                         tun->state, 
                                         msg_checker->mesg_name); */ 
                                      msg_type = msg_checker->mesg_num;
                                    }
                                }
                            }
                          /* Check that the assisgned [call|tunnel] id is good 
                             for this tunnel (exception SCCRQ, ICCRQ id == 0)  
                          */
                          if(searcher->avp_number == 
                             SSH_AVP_TYPE_ASSIGNEDTUNNELID)
                            {
                              if(tun->tunnel_id != ntohs(dat[i+3]) )
                                {
                                  ssh_debug("tunnel id error");
                                  return -1;
                                }
                            }
                          if(searcher->avp_number == 
                             SSH_AVP_TYPE_ASSIGNEDSESSIONID)
                            {
                              if(tun->call_id != ntohs(dat[i+3]) )
                                {
                                  ssh_debug("call id error");
                                  return -1;
                                }
                            }
                        }
                      if(searcher->avp_length == 10)
                        {
                          /* ssh_debug("avp32 is %s, value == %x%x", 
                                    searcher->avp_name, 
                                    ntohs(dat[i+3]), ntohs(dat[i+4])); */
                        }
                      /* Variable length avps are the exception, if the 
                         avp_length is zero then this means that the avp can 
                         be a variable length. The "Result Code" avp can be 
                         10 in length of variable. Offset is the distance in 
                         octets from the current location to the start of 
                         the message. It is always 3 unless the avp is a 
                         "result code avp". 
                      */
                      offset = 3;
                      if(searcher->avp_number == SSH_AVP_TYPE_RESULTCODE)
                        {              
                          offset = 5;
                          tun->error_code = ntohs(dat[i+2]);
                          ssh_debug("[%s] error %s", tun->state,
                                    ssh_l2tp_get_error_mesg(tun));
                          if(ntohs(dat[i] - htons(0x8006)) == 4 
                             || ntohs(dat[i] - htons(0x8006))  == 2)
                            {
                              continue;
                            }
                          else
                            {
                              j = 0;                      
                              for(j = 0; 
                                  j< (ntohs(dat[i]) - 6)/ 2; 
                                  j++)
                                {                       
                                  /* lower_v = htons(dat[j+offset+i]) >> 0;
                                     upper_v = htons(dat[j+offset+i]) >> 8; */ 
                                  tmpmsg[j] = dat[j+5+i];
                                  ssh_debug("%c", dat[j+5+i] );
                                }       
                              ssh_debug("[%s] additional err msg", tun->state, 
                                        searcher->avp_name, tmpmsg);
                              i = i + j;
                            }
                        }
                      /* All other variable legnth avps here */
                      if(searcher->avp_length == 0 
                         && searcher->avp_number != SSH_AVP_TYPE_RESULTCODE)
                        {
                          j = 0;                      
                          for(j = 0; 
                              j< (ntohs(dat[i]) - 6)/ 2; 
                              j++)
                            {                       
                              /* lower_v = htons(dat[j+offset+i]) >> 0;
                                 upper_v = htons(dat[j+offset+i]) >> 8; */  
                              tmpmsg[j] = dat[j+offset+i]; 
                            }       
                          ssh_debug("[%s] avp %s (%s)", tun->state, 
                                    searcher->avp_name, tmpmsg);
                          i = i + j;
                        }
                      
                    }   
                }                 
            }                                
        }
    }
  *rlen = received_len;
  *rvalue = received_data;
  /* Ack the mesg, if is good, if not a zlb*/
  if(msg_type != -1 )
    ssh_ack(tun, tun->Sr);
  return msg_type;
}



unsigned char *ssh_l2tp_get_error_mesg(l2tp_tunnel *tun)
{    
  /* Errors for CDN  */
  /* If the 6th character of the L2TP state name is either a 'I'|'O' then it means
     that the control channel has already been established and there fore the 
     error message must be for a call. */
  if(strncmp(tun->state, "L2TP I" ,6) == 0 
     ||strncmp(tun->state, "L2TP O" ,6) == 0 )
    {
      switch(tun->error_code)
        {
        case 0:            
          return "Reserved"; 
        case 1:            
          return "Call disconnected due to loss of carrier";        
        case 2 :           
          return "Call disconnected for the reason indicated in error code"; 
        case 3:            
          return "Call disconnected for administrative reasons";        
        case 4:            
          return "Call failed due to lack of appropriate facilities - temp";    
        case 5:            
          return "Call failed - lack of appropriate facilities being - perm";
        case 6:            
          return "Invalid destination"; 
        case 7:            
          return "Call failed due to no carrier detected";      
        case 8:            
          return "Call failed due to detection of a busy signal";       
        case 9:            
          return "Call failed due to lack of a dial tone";      
        case 10:           
          return "Call was not established within time allotted by LAC";        
        case 11:           
          return "Call was connected but no appropriate framing was detected"; 
        }
    }
  else
    {      
      switch(tun->error_code)
        {
          /* Errors for StopCCN  */
        case 0:   
          return "Reserved";
        case 1:    
          return "General request to clear control connection"; 
        case 2:    
          return "General error--Error Code indicates the problem"; 
        case 3:    
          return "Control channel already exists"; 
        case 4:    
          return "Requester is not authorized to establish a control channel"; 
        case 5:    
          return "L2TP version of the peer is not supported see Error Code"; 
        case 6:    
          return "Requester is being shut down"; 
        case 7:    
          return "Finite State Machine error"; 
        }
    }     
  /* XXX SSH_NOT_REACHED ??? */
  return NULL;
}

/* XXX */
Boolean ssh_l2tp_check_received_data(l2tp_tunnel *tun, 
                                     unsigned char **data, 
                                     size_t *data_len)
{

  return TRUE;
}

void ssh_l2tp_set_type(l2tp_tunnel *tun, char *value)
{
  tun->setLNS = FALSE;
  if(strcmp(value, "lns") == 0)
    tun->setLNS = TRUE;
}

void ssh_l2tp_set_localaddr(l2tp_tunnel *tun, char *value)
{
  tun->local_address = (unsigned char *)value;
  ssh_debug("localaddr:%s", tun->local_address);
}

void ssh_l2tp_set_localport(l2tp_tunnel *tun, char *value)
{
  tun->local_port = (unsigned char *)value;
}

void ssh_l2tp_set_remoteaddr(l2tp_tunnel *tun, char *value)
{
  tun->remote_address = (unsigned char *)value;
}

void ssh_l2tp_set_remoteport(l2tp_tunnel *tun, char *value)
{
  tun->remote_port = (unsigned char *)value;
}

void ssh_l2tp_set_hostname(l2tp_tunnel *tun, char *value)
{
  tun->myHostName = value;
}

void ssh_l2tp_set_keepalivevalue(l2tp_tunnel *tun, char *value)
{
  tun->keepAliveTimerValue = atoi(value);
}

void ssh_l2tp_set_rws(l2tp_tunnel *tun, char *value)
{
  tun->rws = atoi(value);
}
void ssh_l2tp_set_maxretx(l2tp_tunnel *tun, char *value)
{
  tun->maxReTx = atoi(value);
}
void ssh_l2tp_set_vendorid(l2tp_tunnel *tun, char *value)
{
  tun->vendor_id = atoi(value);
}

int ssh_l2tp_parse_config(l2tp_tunnel *tun)
{
/* Definition of a keyword */ 
struct keyword 
{
  char *keyword;
  int (*handler)(l2tp_tunnel *tun, char *value);
};

/* XXX Better type casting ??? */
struct keyword words[] = {
  { "type",  (void *) &ssh_l2tp_set_type },
  { "local ip", (void *) &ssh_l2tp_set_localaddr },
  { "local port", (void *) &ssh_l2tp_set_localport },
  { "remote ip", (void *) &ssh_l2tp_set_remoteaddr },
  { "remote port", (void *) &ssh_l2tp_set_remoteport },
  { "hostname", (void *) &ssh_l2tp_set_hostname },
  { "keepalive value", (void *) &ssh_l2tp_set_keepalivevalue },
  { "rws", (void *) &ssh_l2tp_set_rws },
  { "vendor id", (void *) &ssh_l2tp_set_vendorid },
  { "maxretx", (void *) &ssh_l2tp_set_maxretx },
  { NULL, NULL }
};

  FILE *f;
  char buf[128];
  char *s, *t;
  int linenum=0;
  struct keyword *kw;

  f = fopen(SSH_L2TP_SPD_FILE_PATH, "r");  
  if(f == NULL)
    {
      ssh_debug("error opening file");
      return 0;
    }
  while(!feof(f)) 
    {
      fgets(buf,sizeof(buf),f);
      if (feof(f)) 
        break;
      linenum++;
      s=buf;
      /* Strip comments */
      while (*s && *s != ';') 
        s++; 
      *s=0;
      s=buf;
      if (!strlen(buf)) 
        continue;
      /* Skip over beginning white space */
      while((*s < 33) && *s) 
        s++;    
      t=s+strlen(s);
      /* Ditch trailing white space  */
      while((t >= s) && (*t < 33)) 
        *(t--)=0;       
      if (!strlen(s)) 
        continue;
      for (kw=words;kw->keyword;kw++) 
        {
          if(strncmp(s, kw->keyword, strlen(kw->keyword)) == 0)
            {
              /* Now matched the keyword, remove everything upto the '=' and then 
                 should remove all white space after this as well to make it a 
                 little less strict in what it accepts. */
              while(*s && *s != '=') 
                s++;
              /* Skip equals && check not fallen off end */
              if(s == NULL)
                {
                  ssh_debug("config file error(line %d), no equals sign !", 
                            linenum);
                  return -1;
                }
              s++;
              /* Skip any white space */
              while((*s < 33) && *s) 
                s++;    
              /* ssh_debug("we have a match 2 %s", s); */
              /* ssh_debug("kw->%s, data=%s.", kw->keyword, s); */
              kw->handler(tun, s);
              ssh_debug("after handler:local address:%s.", tun->local_address);
              break;
            }
        }
    }
  fclose(f);
  ssh_debug("end parse:%s,%s.", tun->local_address);
  return 0;
}

static void t_tas_stream_callback(SshStreamNotification  notification,
                                  void*                  pdata)
{
  t_tas_context pcontext = pdata;
  int           i;
  char          abuf[4096];
  
  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      /* read from stream */
      while ((i = ssh_stream_read(pcontext->pstream,
                                  abuf,
                                  4096)) > 0)
        {
          /* append read data to buffer */
          ssh_buffer_append(pcontext->pbuffer,
                            abuf,
                            i);
          
          /* call stream callback directly */
          t_tas_stream_callback(SSH_STREAM_CAN_OUTPUT,
                                pcontext);
        }
      
      /* check for EOF */
      if (0 == i)
        {
          /* got EOF, abort */
          ssh_event_loop_abort();
        }
      break;
    case SSH_STREAM_CAN_OUTPUT:
      /* check for available data */
      i = ssh_buffer_len(pcontext->pbuffer);

      /* got data? */
      if (i > 0)
        {
          /* write data to stream */
          i = ssh_stream_write(pcontext->pstream,
                               ssh_buffer_ptr(pcontext->pbuffer),
                               i);

          /* how much was written */
          if (i > 0)
            {
              /* consume data from buffer */
              ssh_buffer_consume(pcontext->pbuffer,
                                 i);
            }
        }
      break;
    case SSH_STREAM_DISCONNECTED:
      /* disconnected */
      ssh_event_loop_abort();
      break;
    default:
      SSH_NOTREACHED;
      break;
    }
}




























