//////////////////////////////////////////////////////////////////////////
//
//  IGATOR Copyright (C) 1997-98 RIT Research Labs
//
//  This programs is free for commercial and non-commercial use as long as
//  the following conditions are aheared to.
//
//  Copyright remains RIT Research Labs, and as such any Copyright notices
//  in the code are not to be removed. If this package is used in a
//  product, RIT Research Labs should be given attribution as the RIT Research
//  Labs of the parts of the library used. This can be in the form of a textual
//  message at program startup or in documentation (online or textual)
//  provided with the package.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//
//  1. Redistributions of source code must retain the copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. All advertising materials mentioning features or use of this software
//     must display the following acknowledgement:
//     "Based on IGATOR by RIT Research Labs."
//
//  THIS SOFTWARE IS PROVIDED BY RIT RESEARCH LABS "AS IS" AND ANY EXPRESS
//  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//  DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
//  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
//  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
//  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
//  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
//  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//  The licence and distribution terms for any publically available
//  version or derivative of this code cannot be changed. i.e. this code
//  cannot simply be copied and put under another distribution licence
//  (including the GNU Public Licence).
//
//////////////////////////////////////////////////////////////////////////

{$V-}

unit Config;

interface
uses Windows, SysUtils, Classes, Utils, Logger;

type
     TXTable = Array [char] of Char;

     TXLT = class
       Name: String[50];
       Cset: String[50];
       T: TXTable;
       BT: TXTable;
       constructor Create(const AName, ACset, FName: String);
       function IsIt(const S: String; CS: Boolean): Boolean;
       function XString(const S: String): String;
       procedure XBuffer(var B; Len: Integer);
       procedure XlatFile(const FName: String; Back: Boolean);
       function BXString(const S: String): String;
       procedure BXBuffer(var B; Len: Integer);
     end;

     TUser = class
       Name: String;
       Addr: String;
       eMail: String;
       Home: String;
       Pop: String;
       PopPassword: String;
       PopLastread: LongInt;
       WriteInfo: Boolean;
       BytesIn: LongInt;
       BytesOut: LongInt;
       MsgIn: LongInt;
       MsgOut: LongInt;
       Encode: Integer;
       Decode: Integer;
       _Zone, _Net, _Node, _Point: Word;
       Modified: Boolean;
       XName: String;
       XTable: TXLT;
       procedure XInFile(const FName: String);
       procedure XOutFile(const FName: String);
       function XKludge(const S: String): String;
     end;

     TMsgHdr = packed record
       F_Name: Array[0..35] of Char;
       T_Name: Array[0..35] of Char;
       Subj: Array[0..71] of Char;
       DateTime: Array[0..19] of Char;
       TimesRead: SmallWord;
       DestNode: SmallWord;
       OrigNode: SmallWord;
       cost: SmallWord;
       origNet: SmallWord;
       DestNet: SmallWord;
       DestZone: SmallWord;
       OrigZone: SmallWord;
       DestPoint: SmallWord;
       OrigPoint: SmallWord;
       ReplyTo: SmallWord;
       Attr: SmallWord;
       NextReply: SmallWord;
     end;

     TXlatTable = Array [char] of packed record
                                C1: Char;
                                C2: Char;
                               end;

     TIdxRec = packed record
       Info: Array[0..15] of Byte;
       Name: Array[0..253] of Char;
       XX: Array[0..5] of Char;
     end;

const
     XTag = 'IGator (v1.4)';
     XLogo = 'IGator v1.4 (Win32) - The Internet <-> Fido Gateway';
     MainAddr: String = '';
     NetPath: String = '';
     SMTP: String = '';
     RRQFile: String = '';
     HelpFile: String = '';
     MaxMsg: LongInt = 0;
     LogFile: String = 'IGATOR.LOG';
     XLTs: TList = nil;
     ConvertAt: Boolean = False;

var
     Users: TList;
     M_Zone,
     M_Net,
     M_Node,
     M_Point: Word;

procedure ReadIni;
procedure UpdateIni;
function FindXlat(const S: String; CS: Boolean): TXLT;

implementation
uses IniFiles;

function FindXLat;
  var SS: String;
      I: Integer;
begin
  Result := nil;
  if XLTs = nil then Exit;
  SS := UpperCase(S);
  for I := 0 to XLTs.Count-1 do
    if TXLT(XLTs[I]).IsIt(SS, CS) then
      begin
        Result := XLTs[I];
        Exit;
      end;
end;

constructor TXLT.Create;
  var SS: TFileStream;
      C: Char;
      A: TXlatTable;
      W: Integer;
      B: Boolean;
begin
  inherited Create;
  SS := nil;
  try
    SS := TFileStream.Create(FName, fmOpenReadWrite);
    Name := AName; CSet := ACset;
    for C := #0 to #255 do
      begin
        T[C] := C;
        BT[C] := C;
      end;
    W := Min(SizeOf(A), SS.Size);
    B := W = SS.Read(A, W);
    if B then
      begin
        if XLTs = nil then XLTs := TList.Create;
        XLTs.Add(Self);
        for C := #0 to Char((W div 2)-1) do
          begin
            T[A[C].C1] := A[C].C2;
            BT[A[C].C2] := A[C].C1;
          end;
      end;
  finally
    SS.Free;
  end;
end;

function TXLT.IsIt;
begin
  if CS then Result := UpperCase(Cset) = S
        else Result := UpperCase(Name) = S;
end;

function TXLT.XString(const S: String): String;
begin
  Result := S;
  if Length(S) > 0 then XBuffer(Result[1], Length(S));
end;

procedure TXLT.XBuffer(var B; Len: Integer);
  var P: PChar;
      I: Integer;
begin
  P := @B;
  for I := 0 to Len-1 do P[I] := T[P[I]];
end;

function TXLT.BXString(const S: String): String;
begin
  Result := S;
  if Length(S) > 0 then BXBuffer(Result[1], Length(S));
end;

procedure TXLT.BXBuffer(var B; Len: Integer);
  var P: PChar;
      I: Integer;
begin
  P := @B;
  for I := 0 to Len-1 do P[I] := BT[P[I]];
end;

procedure TXLT.XlatFile;
  var S: TFileStream;
      B: PChar;
      I,L: LongInt;
begin
  S := nil; B := nil; L := 0;
  try
    S := TFileStream.Create(FName, fmOpenReadWrite);
    L := S.Size;
    GetMem(B, L);
    S.Read(B[0], L);
    if Back then BXBuffer(B[0], L) else XBuffer(B[0], L);
    S.Position := 0;
    S.Write(B[0], L);
  finally
    FreeMem(B, L);
    S.Free;
  end;
end;

procedure TUser.XInFile;
begin
  if XTable <> nil then XTable.XlatFile(FName, False);
end;

procedure TUser.XOutFile;
begin
  if XTable <> nil then XTable.XlatFile(FName, True);
end;

function TUser.XKludge;
  var I: Integer;
      A: String[255];
      CC: String[3];
      B: Boolean;
      C: Char;
begin
  Result := S; B := True;
  if XTable = nil then Exit;
  for I := 1 to Length(S) do
    if S[I] in [#0..#31, #128..#255] then
      begin B := False; Break; end;
  if B then Exit;
  A := Xtable.BXString(S);
  I := 1;
  while I <= Length(A) do
    begin
      C := A[I];
      if (C = '=') or (C = '?') then
        begin
          CC := Hex2(Byte(C));
          A[I] := '=';
          Insert(CC, A, I+1);
          Inc(I, 2);
        end;
      Inc(I);
    end;
  Result := Format('=?%s?Q?%s?=', [XTable.Cset, A]);
end;

procedure ReadIni;
  var S: TIniFile;
      I,J: Integer;
      U: TUser;

  procedure Get(var SS: String; const T: String);
  begin
    SS := S.ReadString('General', T, '');
    if SS = '' then
      begin
        MessageBeep(MB_ICONEXCLAMATION);
        Log('ERROR: `'+T+''' variable is not defined', True);
        Halt(2);
      end;
  end;

  procedure GetUInfo(N: Integer; var SS: String; const T: String; Fail: Boolean);
  begin
    Str(N, SS); SS := 'User '+SS;
    SS := S.ReadString(SS, T, '');
    if (SS = '') and Fail then
      begin
        MessageBeep(MB_ICONEXCLAMATION);
        Log('ERROR: `'+T+''' variable is not defined for User #'+Itos(N), True);
        Halt(3);
      end;
  end;

  function GetN(N: Integer; const Nm: String): LongInt;
  begin
    GetN := S.ReadInteger(Format('User %d', [N]), Nm, 0);
  end;

  procedure ReadXLTs;
    var I, C: Integer;
        XN: String;
        Cn, Nm, Fn: String;
        X: TXLT;
        U: TUser;
  begin
    Xn := 'XLT';
    C := S.ReadInteger(Xn, 'Count', 0);
    for I := 1 to C do
      begin
        Nm := S.ReadString(Xn, Format('Table %d', [I]), '');
        CN := S.ReadString(Xn, Format('Charset %d', [I]), '');
        Fn := S.ReadString(Xn, Format('XLat %d', [I]), '');
        if (Nm <> '') and (Fn <> '') and (CN <> '') then X := TXLT.Create(Nm, Cn, Fn);
      end;
    for I := 0 to Users.Count-1 do
      begin
        U := Users[I];
        U.XTable := nil;
        U.XTable := FindXlat(U.XName, False);
      end;
  end;

begin
  S := TIniFile.Create('.\IGATOR.INI');
  Get(MainAddr, 'Address');
  ParseAddress(MainAddr, M_Zone, M_Net, M_Node, M_Point);
  Get(NetPath, 'NetmailPath');
  Get(Smtp, 'SMTP');
  RRQFile := S.ReadString('General', 'RRQ', '');
  HelpFile := S.ReadString('General', 'Help', '');
  I := S.ReadInteger('General', 'Users', 0);
  if I = 0 then
    begin
      MessageBeep(MB_ICONEXCLAMATION);
      Log('ERROR: `Users'' variable is not defined', True);
      Halt(2);
    end;
  for J := 1 to I do
    begin
      U := TUser.Create;
      GetUInfo(J, U.Name, 'Name', True);
      GetUInfo(J, U.Addr, 'Address', True);
      with U do
        begin
          ParseAddress(MainAddr, _Zone, _Net, _Node, _Point);
          ParseAddress(Addr, _Zone, _Net, _Node, _Point);
        end;
      GetUInfo(J, U.Email, 'eMail', True);
      GetUInfo(J, U.Home, 'Home', False);
      GetUInfo(J, U.Pop, 'POP', False);
      GetUInfo(J, U.PopPassword, 'POP Password', False);
      U.PopLastread := GetN(J, 'POP Lastread');
      U.Decode := GetN(J, 'Decode');
      U.Encode := GetN(J, 'Encode');
      U.BytesIn := GetN(J, 'Bytes In');
      U.BytesOut := GetN(J, 'Bytes Out');
      U.MsgIn := GetN(J, 'Msgs In');
      U.MsgOut := GetN(J, 'Msgs Out');
      U.WriteInfo := Boolean(GetN(J, 'Full Info'));
      GetUInfo(J, U.XName, 'Xlat', False);
      Users.Add(U);
    end;
  ReadXlts;
  S.Free;
end;

procedure UpdateIni;
  var F: TIniFile;
      I: Integer;

  procedure DoPut(U: TUser);
    var S: String;
  begin
    if U.Modified then
      begin
        S := Format('User %d', [I+1]);
        F.WriteInteger(S, 'POP Lastread', U.PopLastread);
        F.WriteInteger(S, 'Full Info', Byte(U.WriteInfo));
        F.WriteInteger(S, 'Decode', U.Decode);
        F.WriteInteger(S, 'Encode', U.Encode);
        F.WriteInteger(S, 'Bytes In', U.BytesIn);
        F.WriteInteger(S, 'Bytes Out', U.BytesOut);
        F.WriteInteger(S, 'Msgs In', U.MsgIn);
        F.WriteInteger(S, 'Msgs Out', U.MsgOut);
      end;
  end;

begin
  F := TIniFile.Create('.\IGATOR.INI');
  for I := 0 to Users.Count-1 do DoPut(Users[I]);
  F.Free;
end;


end.
