//////////////////////////////////////////////////////////////////////////
//
//  IGATOR Copyright (C) 1997-98 RIT Research Labs
//
//  This programs is free for commercial and non-commercial use as long as
//  the following conditions are aheared to.
//
//  Copyright remains RIT Research Labs, and as such any Copyright notices
//  in the code are not to be removed. If this package is used in a
//  product, RIT Research Labs should be given attribution as the RIT Research
//  Labs of the parts of the library used. This can be in the form of a textual
//  message at program startup or in documentation (online or textual)
//  provided with the package.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are
//  met:
//
//  1. Redistributions of source code must retain the copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. All advertising materials mentioning features or use of this software
//     must display the following acknowledgement:
//     "Based on IGATOR by RIT Research Labs."
//
//  THIS SOFTWARE IS PROVIDED BY RIT RESEARCH LABS "AS IS" AND ANY EXPRESS
//  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//  DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
//  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
//  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
//  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
//  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
//  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//  The licence and distribution terms for any publically available
//  version or derivative of this code cannot be changed. i.e. this code
//  cannot simply be copied and put under another distribution licence
//  (including the GNU Public Licence).
//
//////////////////////////////////////////////////////////////////////////

unit WinSock;

interface

uses Windows;

type
  u_char = Char;
  u_short = Word;
  u_int = Integer;
  u_long = Longint;

{ The new type to be used in all
  instances which refer to sockets. }
  TSocket = u_int;

const
  FD_SETSIZE     =   64;

type
  PFDSet = ^TFDSet;
  TFDSet = packed record
    fd_count: u_int;
    fd_array: array[0..FD_SETSIZE-1] of TSocket;
  end;

  PTimeVal = ^TTimeVal;
  TTimeVal = packed record
    tv_sec: Longint;
    tv_usec: Longint;
  end;

const
  IOCPARM_MASK = $7f;
  IOC_VOID     = $20000000;
  IOC_OUT      = $40000000;
  IOC_IN       = $80000000;
  IOC_INOUT    = (IOC_IN or IOC_OUT);

  FIONREAD     = IOC_OUT or { get # bytes to read }
    ((Longint(SizeOf(Longint)) and IOCPARM_MASK) shl 16) or
    (Longint(Byte('f')) shl 8) or 127;
  FIONBIO      = IOC_IN or { set/clear non-blocking i/o }
    ((Longint(SizeOf(Longint)) and IOCPARM_MASK) shl 16) or
    (Longint(Byte('f')) shl 8) or 126;
  FIOASYNC     = IOC_IN or { set/clear async i/o }
    ((Longint(SizeOf(Longint)) and IOCPARM_MASK) shl 16) or
    (Longint(Byte('f')) shl 8) or 125;

type
  PHostEnt = ^THostEnt;
  THostEnt = packed record
    h_name: PChar;
    h_aliases: ^PChar;
    h_addrtype: Smallint;
    h_length: Smallint;
    h_addr_list: ^PChar;
  end;

  PNetEnt = ^TNetEnt;
  TNetEnt = packed record
    n_name: PChar;
    n_aliases: ^PChar;
    n_addrtype: Smallint;
    n_net: u_long;
  end;

  PServEnt = ^TServEnt;
  TServEnt = packed record
    s_name: PChar;
    s_aliases: ^PChar;
    s_port: Smallint;
    s_proto: PChar;
  end;

  PProtoEnt = ^TProtoEnt;
  TProtoEnt = packed record
    p_name: PChar;
    p_aliases: ^Pchar;
    p_proto: Smallint;
  end;

const

{ Protocols }

  IPPROTO_IP     =   0;
  IPPROTO_ICMP   =   1;
  IPPROTO_GGP    =   2;
  IPPROTO_TCP    =   6;
  IPPROTO_PUP    =   12;
  IPPROTO_UDP    =   17;
  IPPROTO_IDP    =   22;
  IPPROTO_ND     =   77;

  IPPROTO_RAW    =   255;
  IPPROTO_MAX    =   256;

{ Port/socket numbers: network standard functions}

  IPPORT_ECHO    =   7;
  IPPORT_DISCARD =   9;
  IPPORT_SYSTAT  =   11;
  IPPORT_DAYTIME =   13;
  IPPORT_NETSTAT =   15;
  IPPORT_FTP     =   21;
  IPPORT_TELNET  =   23;
  IPPORT_SMTP    =   25;
  IPPORT_TIMESERVER  =  37;
  IPPORT_NAMESERVER  =  42;
  IPPORT_WHOIS       =  43;
  IPPORT_MTP         =  57;

{ Port/socket numbers: host specific functions }

  IPPORT_TFTP        =  69;
  IPPORT_RJE         =  77;
  IPPORT_FINGER      =  79;
  IPPORT_TTYLINK     =  87;
  IPPORT_SUPDUP      =  95;

{ UNIX TCP sockets }

  IPPORT_EXECSERVER  =  512;
  IPPORT_LOGINSERVER =  513;
  IPPORT_CMDSERVER   =  514;
  IPPORT_EFSSERVER   =  520;

{ UNIX UDP sockets }

  IPPORT_BIFFUDP     =  512;
  IPPORT_WHOSERVER   =  513;
  IPPORT_ROUTESERVER =  520;

{ Ports < IPPORT_RESERVED are reserved for
  privileged processes (e.g. root). }

  IPPORT_RESERVED    =  1024;

{ Link numbers }

  IMPLINK_IP         =  155;
  IMPLINK_LOWEXPER   =  156;
  IMPLINK_HIGHEXPER  =  158;

type
  SunB = packed record
    s_b1, s_b2, s_b3, s_b4: u_char;
  end;

  SunW = packed record
    s_w1, s_w2: u_short;
  end;

  PInAddr = ^TInAddr;
  TInAddr = packed record
    case integer of
      0: (S_un_b: SunB);
      1: (S_un_w: SunW);
      2: (S_addr: u_long);
      3: (ClassA, ClassB, ClassC, ClassD: Byte);
  end;

  PSockAddrIn = ^TSockAddrIn;
  TSockAddrIn = packed record
    case Integer of
      0: (sin_family: u_short;
          sin_port: u_short;
          sin_addr: TInAddr;
          sin_zero: array[0..7] of Char);
      1: (sa_family: u_short;
          sa_data: array[0..13] of Char)
  end;

const
  INADDR_ANY       = $00000000;
  INADDR_LOOPBACK  = $7F000001;
  INADDR_BROADCAST = $FFFFFFFF;
  INADDR_NONE      = $FFFFFFFF;

const
  WSADESCRIPTION_LEN     =   256;
  WSASYS_STATUS_LEN      =   128;

type
  PWSAData = ^TWSAData;
  TWSAData = packed record
    wVersion: Word;
    wHighVersion: Word;
    szDescription: array[0..WSADESCRIPTION_LEN] of Char;
    szSystemStatus: array[0..WSASYS_STATUS_LEN] of Char;
    iMaxSockets: Word;
    iMaxUdpDg: Word;
    lpVendorInfo: PChar;
  end;

  PTransmitFileBuffers = ^TTransmitFileBuffers;
  TTransmitFileBuffers = packed record
      Head: Pointer;
      HeadLength: DWORD;
      Tail: Pointer;
      TailLength: DWORD;
  end;

const

{ Options for use with [gs]etsockopt at the IP level. }

  IP_OPTIONS          = 1;
  IP_MULTICAST_IF     = 2;           { set/get IP multicast interface   }
  IP_MULTICAST_TTL    = 3;           { set/get IP multicast timetolive  }
  IP_MULTICAST_LOOP   = 4;           { set/get IP multicast loopback    }
  IP_ADD_MEMBERSHIP   = 5;           { add  an IP group membership      }
  IP_DROP_MEMBERSHIP  = 6;           { drop an IP group membership      }

  IP_DEFAULT_MULTICAST_TTL   = 1;    { normally limit m'casts to 1 hop  }
  IP_DEFAULT_MULTICAST_LOOP  = 1;    { normally hear sends if a member  }
  IP_MAX_MEMBERSHIPS         = 20;   { per socket; must fit in one mbuf }

{ This is used instead of -1, since the
  TSocket type is unsigned.}

  INVALID_SOCKET                = TSocket(NOT(0));
  SOCKET_ERROR                  = -1;

{ Types }

  SOCK_STREAM     = 1;               { stream socket }
  SOCK_DGRAM      = 2;               { datagram socket }
  SOCK_RAW        = 3;               { raw-protocol interface }
  SOCK_RDM        = 4;               { reliably-delivered message }
  SOCK_SEQPACKET  = 5;               { sequenced packet stream }

{ Option flags per-socket. }

  SO_DEBUG        = $0001;          { turn on debugging info recording }
  SO_ACCEPTCONN   = $0002;          { socket has had listen() }
  SO_REUSEADDR    = $0004;          { allow local address reuse }
  SO_KEEPALIVE    = $0008;          { keep connections alive }
  SO_DONTROUTE    = $0010;          { just use interface addresses }
  SO_BROADCAST    = $0020;          { permit sending of broadcast msgs }
  SO_USELOOPBACK  = $0040;          { bypass hardware when possible }
  SO_LINGER       = $0080;          { linger on close if data present }
  SO_OOBINLINE    = $0100;          { leave received OOB data in line }

  SO_DONTLINGER  =   $ff7f;

{ Additional options. }

  SO_SNDBUF       = $1001;          { send buffer size }
  SO_RCVBUF       = $1002;          { receive buffer size }
  SO_SNDLOWAT     = $1003;          { send low-water mark }
  SO_RCVLOWAT     = $1004;          { receive low-water mark }
  SO_SNDTIMEO     = $1005;          { send timeout }
  SO_RCVTIMEO     = $1006;          { receive timeout }
  SO_ERROR        = $1007;          { get error status and clear }
  SO_TYPE         = $1008;          { get socket type }

{ Options for connect and disconnect data and options.  Used only by
  non-TCP/IP transports such as DECNet, OSI TP4, etc. }

  SO_CONNDATA     = $7000;
  SO_CONNOPT      = $7001;
  SO_DISCDATA     = $7002;
  SO_DISCOPT      = $7003;
  SO_CONNDATALEN  = $7004;
  SO_CONNOPTLEN   = $7005;
  SO_DISCDATALEN  = $7006;
  SO_DISCOPTLEN   = $7007;

{ Option for opening sockets for synchronous access. }

  SO_OPENTYPE     = $7008;

  SO_SYNCHRONOUS_ALERT    = $10;
  SO_SYNCHRONOUS_NONALERT = $20;

{ Other NT-specific options. }

  SO_MAXDG        = $7009;
  SO_MAXPATHDG    = $700A;

{ TCP options. }

  TCP_NODELAY     = $0001;
  TCP_BSDURGENT   = $7000;

{ Address families. }

  AF_UNSPEC       = 0;               { unspecified }
  AF_UNIX         = 1;               { local to host (pipes, portals) }
  AF_INET         = 2;               { internetwork: UDP, TCP, etc. }
  AF_IMPLINK      = 3;               { arpanet imp addresses }
  AF_PUP          = 4;               { pup protocols: e.g. BSP }
  AF_CHAOS        = 5;               { mit CHAOS protocols }
  AF_IPX          = 6;               { IPX and SPX }
  AF_NS           = 6;               { XEROX NS protocols }
  AF_ISO          = 7;               { ISO protocols }
  AF_OSI          = AF_ISO;          { OSI is ISO }
  AF_ECMA         = 8;               { european computer manufacturers }
  AF_DATAKIT      = 9;               { datakit protocols }
  AF_CCITT        = 10;              { CCITT protocols, X.25 etc }
  AF_SNA          = 11;              { IBM SNA }
  AF_DECnet       = 12;              { DECnet }
  AF_DLI          = 13;              { Direct data link interface }
  AF_LAT          = 14;              { LAT }
  AF_HYLINK       = 15;              { NSC Hyperchannel }
  AF_APPLETALK    = 16;              { AppleTalk }
  AF_NETBIOS      = 17;              { NetBios-style addresses }
  AF_VOICEVIEW    = 18;              { VoiceView }

  AF_MAX          = 19;

type
  { Structure used by kernel to store most addresses. }

  PSockAddr = ^TSockAddr;
  TSockAddr = TSockAddrIn;

  { Structure used by kernel to pass protocol information in raw sockets. }
  PSockProto = ^TSockProto;
  TSockProto = packed record
    sp_family: u_short;
    sp_protocol: u_short;
  end;

const
{ Protocol families, same as address families for now. }

  PF_UNSPEC       = AF_UNSPEC;
  PF_UNIX         = AF_UNIX;
  PF_INET         = AF_INET;
  PF_IMPLINK      = AF_IMPLINK;
  PF_PUP          = AF_PUP;
  PF_CHAOS        = AF_CHAOS;
  PF_NS           = AF_NS;
  PF_IPX          = AF_IPX;
  PF_ISO          = AF_ISO;
  PF_OSI          = AF_OSI;
  PF_ECMA         = AF_ECMA;
  PF_DATAKIT      = AF_DATAKIT;
  PF_CCITT        = AF_CCITT;
  PF_SNA          = AF_SNA;
  PF_DECnet       = AF_DECnet;
  PF_DLI          = AF_DLI;
  PF_LAT          = AF_LAT;
  PF_HYLINK       = AF_HYLINK;
  PF_APPLETALK    = AF_APPLETALK;
  PF_VOICEVIEW    = AF_VOICEVIEW;

  PF_MAX          = AF_MAX;

type
{ Structure used for manipulating linger option. }
  PLinger = ^TLinger;
  TLinger = packed record
    l_onoff: u_short;
    l_linger: u_short;
  end;

const
{ Level number for (get/set)sockopt() to apply to socket itself. }

  SOL_SOCKET      = $ffff;          {options for socket level }

{ Maximum queue length specifiable by listen. }

  SOMAXCONN       = 5;

  MSG_OOB         = $1;             {process out-of-band data }
  MSG_PEEK        = $2;             {peek at incoming message }
  MSG_DONTROUTE   = $4;             {send without using routing tables }

  MSG_MAXIOVLEN   = 16;

  MSG_PARTIAL     = $8000;          {partial send or recv for message xport }

{ Define constant based on rfc883, used by gethostbyxxxx() calls. }

  MAXGETHOSTSTRUCT        = 1024;

{ Define flags to be used with the WSAAsyncSelect() call. }

  FD_READ         = $01;
  FD_WRITE        = $02;
  FD_OOB          = $04;
  FD_ACCEPT       = $08;
  FD_CONNECT      = $10;
  FD_CLOSE        = $20;

{ All Windows Sockets error constants are biased by WSABASEERR from the "normal" }

  WSABASEERR              = 10000;

{ Windows Sockets definitions of regular Microsoft C error constants }

  WSAEINTR                = (WSABASEERR+4);
  WSAEBADF                = (WSABASEERR+9);
  WSAEACCES               = (WSABASEERR+13);
  WSAEFAULT               = (WSABASEERR+14);
  WSAEINVAL               = (WSABASEERR+22);
  WSAEMFILE               = (WSABASEERR+24);

{ Windows Sockets definitions of regular Berkeley error constants }

  WSAEWOULDBLOCK          = (WSABASEERR+35);
  WSAEINPROGRESS          = (WSABASEERR+36);
  WSAEALREADY             = (WSABASEERR+37);
  WSAENOTSOCK             = (WSABASEERR+38);
  WSAEDESTADDRREQ         = (WSABASEERR+39);
  WSAEMSGSIZE             = (WSABASEERR+40);
  WSAEPROTOTYPE           = (WSABASEERR+41);
  WSAENOPROTOOPT          = (WSABASEERR+42);
  WSAEPROTONOSUPPORT      = (WSABASEERR+43);
  WSAESOCKTNOSUPPORT      = (WSABASEERR+44);
  WSAEOPNOTSUPP           = (WSABASEERR+45);
  WSAEPFNOSUPPORT         = (WSABASEERR+46);
  WSAEAFNOSUPPORT         = (WSABASEERR+47);
  WSAEADDRINUSE           = (WSABASEERR+48);
  WSAEADDRNOTAVAIL        = (WSABASEERR+49);
  WSAENETDOWN             = (WSABASEERR+50);
  WSAENETUNREACH          = (WSABASEERR+51);
  WSAENETRESET            = (WSABASEERR+52);
  WSAECONNABORTED         = (WSABASEERR+53);
  WSAECONNRESET           = (WSABASEERR+54);
  WSAENOBUFS              = (WSABASEERR+55);
  WSAEISCONN              = (WSABASEERR+56);
  WSAENOTCONN             = (WSABASEERR+57);
  WSAESHUTDOWN            = (WSABASEERR+58);
  WSAETOOMANYREFS         = (WSABASEERR+59);
  WSAETIMEDOUT            = (WSABASEERR+60);
  WSAECONNREFUSED         = (WSABASEERR+61);
  WSAELOOP                = (WSABASEERR+62);
  WSAENAMETOOLONG         = (WSABASEERR+63);
  WSAEHOSTDOWN            = (WSABASEERR+64);
  WSAEHOSTUNREACH         = (WSABASEERR+65);
  WSAENOTEMPTY            = (WSABASEERR+66);
  WSAEPROCLIM             = (WSABASEERR+67);
  WSAEUSERS               = (WSABASEERR+68);
  WSAEDQUOT               = (WSABASEERR+69);
  WSAESTALE               = (WSABASEERR+70);
  WSAEREMOTE              = (WSABASEERR+71);

  WSAEDISCON              = (WSABASEERR+101);

{ Extended Windows Sockets error constant definitions }

  WSASYSNOTREADY          = (WSABASEERR+91);
  WSAVERNOTSUPPORTED      = (WSABASEERR+92);
  WSANOTINITIALISED       = (WSABASEERR+93);

{ Error return codes from gethostbyname() and gethostbyaddr()
  (when using the resolver). Note that these errors are
  retrieved via WSAGetLastError() and must therefore follow
  the rules for avoiding clashes with error numbers from
  specific implementations or language run-time systems.
  For this reason the codes are based at WSABASEERR+1001.
  Note also that [WSA]NO_ADDRESS is defined only for
  compatibility purposes. }

{ Authoritative Answer: Host not found }

  WSAHOST_NOT_FOUND       = (WSABASEERR+1001);

{ Non-Authoritative: Host not found, or SERVERFAIL }

  WSATRY_AGAIN            = (WSABASEERR+1002);

{ Non recoverable errors, FORMERR, REFUSED, NOTIMP }

  WSANO_RECOVERY          = (WSABASEERR+1003);

{ Valid name, no data record of requested type }

  WSANO_DATA              = (WSABASEERR+1004);

{ no address, look for MX record }

  WSANO_ADDRESS           = WSANO_DATA;


{ Socket function prototypes }

function recv(s: TSocket; var Buf; len, flags: Integer): Integer;
function WSAStartup(wVersionRequired: word; var WSData: TWSAData): Integer;
function socket(af, struct, protocol: Integer): TSocket;
function WSAGetLastError: Integer;
function closesocket(s: TSocket): Integer;
function htons(hostshort: u_short): u_short;
function bind(s: TSocket; var addr: TSockAddr; namelen: Integer): Integer;
function inet_addr(cp: PChar): u_long;
function gethostbyname(name: PChar): PHostEnt;
function connect(s: TSocket; var name: TSockAddr; namelen: Integer): Integer;
function send(s: TSocket; const Buf; len, flags: Integer): Integer;
function shutdown(s: TSocket; how: Integer): Integer;
function WSACleanup: Integer;


implementation

const
  WinSocketDLL =  'wsock32.dll';
  WinSocket2DLL = 'ws2_32.dll';

type
  T_recv            = function (s: TSocket; var Buf; len, flags: Integer): Integer; stdcall;
  T_WSAStartup      = function (wVersionRequired: word; var WSData: TWSAData): Integer; stdcall;
  T_socket          = function (af, struct, protocol: Integer): TSocket; stdcall;
  T_WSAGetLastError = function : Integer; stdcall;
  T_closesocket     = function (s: TSocket): Integer; stdcall;
  T_htons           = function (hostshort: u_short): u_short; stdcall;
  T_bind            = function (s: TSocket; var addr: TSockAddr; namelen: Integer): Integer; stdcall;
  T_inet_addr       = function (cp: PChar): u_long; stdcall;
  T_gethostbyname   = function (name: PChar): PHostEnt; stdcall;
  T_connect         = function (s: TSocket; var name: TSockAddr; namelen: Integer): Integer; stdcall;
  T_send            = function (s: TSocket; const Buf; len, flags: Integer): Integer; stdcall;
  T_shutdown        = function (s: TSocket; how: Integer): Integer; stdcall;
  T_WSACleanup      = function : Integer; stdcall;

var
  Frecv             : T_recv;
  FWSAStartup       : T_WSAStartup;
  Fsocket           : T_socket;
  FWSAGetLastError  : T_WSAGetLastError;
  Fclosesocket      : T_closesocket;
  Fhtons            : T_htons;
  Fbind             : T_bind;
  Finet_addr        : T_inet_addr;
  Fgethostbyname    : T_gethostbyname;
  Fconnect          : T_connect;
  Fsend             : T_send;
  Fshutdown         : T_shutdown;
  FWSACleanup       : T_WSACleanup;
  
  WinSockModule: Integer;


function GetWinSockFuncAddr(A: PChar): Pointer;
begin
  Result := nil;
  if WinSockModule = 0 then WinSockModule := LoadLibrary(WinSocket2DLL);
  if WinSockModule = 0 then WinSockModule := LoadLibrary(WinSocketDLL);
  if WinSockModule > 0 then Result := GetProcAddress(WinSockModule, A);
end;



function recv(s: TSocket; var Buf; len, flags: Integer): Integer;
begin
  Result := SOCKET_ERROR;
  if not Assigned(Frecv) then Frecv := GetWinSockFuncAddr('recv');
  if Assigned(Frecv) then Result := Frecv(s, Buf, len, flags);
end;

function WSAStartup(wVersionRequired: word; var WSData: TWSAData): Integer;
begin
  Result := WSASYSNOTREADY;
  if not Assigned(FWSAStartup) then FWSAStartup := GetWinSockFuncAddr('WSAStartup');
  if Assigned(FWSAStartup) then Result := FWSAStartup(wVersionRequired, WSData);
end;

function socket(af, struct, protocol: Integer): TSocket;
begin
  Result := INVALID_SOCKET;
  if not Assigned(Fsocket) then Fsocket := GetWinSockFuncAddr('socket');
  if Assigned(Fsocket) then Result := Fsocket(af, struct, protocol);
end;

function WSAGetLastError: Integer;
begin
  Result := WSASYSNOTREADY;
  if not Assigned(FWSAGetLastError) then FWSAGetLastError := GetWinSockFuncAddr('WSAGetLastError');
  if Assigned(FWSAGetLastError) then Result := FWSAGetLastError;
end;

function closesocket(s: TSocket): Integer;
begin
  Result := SOCKET_ERROR;
  if not Assigned(Fclosesocket) then Fclosesocket := GetWinSockFuncAddr('closesocket');
  if Assigned(Fclosesocket) then Result := Fclosesocket(s);
end;

function htons(hostshort: u_short): u_short;
begin
  Result := 0;
  if not Assigned(Fhtons) then Fhtons := GetWinSockFuncAddr('htons');
  if Assigned(Fhtons) then Result := Fhtons(hostshort);
end;

function bind(s: TSocket; var addr: TSockAddr; namelen: Integer): Integer;
begin
  Result := SOCKET_ERROR;
  if not Assigned(Fbind) then Fbind := GetWinSockFuncAddr('bind');
  if Assigned(Fbind) then Result := Fbind(s, addr, namelen);
end;

function inet_addr(cp: PChar): u_long;
begin
  Result := INADDR_NONE;
  if not Assigned(Finet_addr) then Finet_addr := GetWinSockFuncAddr('inet_addr');
  if Assigned(Finet_addr) then Result := Finet_addr(cp);
end;

function gethostbyname(name: PChar): PHostEnt;
begin
  Result := nil;
  if not Assigned(Fgethostbyname) then Fgethostbyname := GetWinSockFuncAddr('gethostbyname');
  if Assigned(Fgethostbyname) then Result := Fgethostbyname(name);
end;

function connect(s: TSocket; var name: TSockAddr; namelen: Integer): Integer;
begin
  Result := SOCKET_ERROR;
  if not Assigned(Fconnect) then Fconnect := GetWinSockFuncAddr('connect');
  if Assigned(Fconnect) then Result := Fconnect(s, name, namelen);
end;

function send(s: TSocket; const Buf; len, flags: Integer): Integer;
begin
  Result := SOCKET_ERROR;
  if not Assigned(Fsend) then Fsend := GetWinSockFuncAddr('send');
  if Assigned(Fsend) then Result := Fsend(s, buf, len, flags);
end;

function shutdown(s: TSocket; how: Integer): Integer;
begin
  Result := SOCKET_ERROR;
  if not Assigned(Fshutdown) then Fshutdown := GetWinSockFuncAddr('shutdown');
  if Assigned(Fshutdown) then Result := Fshutdown(s, how);
end;

function WSACleanup: Integer;
begin
  Result := WSANOTINITIALISED;
  if not Assigned(FWSACleanup) then FWSACleanup := GetWinSockFuncAddr('WSACleanup');
  if Assigned(FWSACleanup) then Result := FWSACleanup;
  if Result <> 0 then Exit;
  FreeLibrary(WinSockModule);
  Frecv             := nil;
  FWSAStartup       := nil;
  Fsocket           := nil;
  FWSAGetLastError  := nil;
  Fclosesocket      := nil;
  Fhtons            := nil;
  Fbind             := nil;
  Finet_addr        := nil;
  Fgethostbyname    := nil;
  Fconnect          := nil;
  Fsend             := nil;
  Fshutdown         := nil;
  FWSACleanup       := nil;
  WinSockModule     := 0;

end;



end.
